/** \file DS_Hashtable.h
    Hashtable data structure

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.
Please contact the author concerning commercial usage.
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_DS_HASHTABLE
#define H_DS_HASHTABLE

#include <DS_Vector.h>

#ifndef BASE_HASHTABLE
  #define BASE_HASHTABLE void
  #define HASH_EXTRA     void
#endif

/** A hashtable is a set of unique elements.
    Each element is identified by its hash value (see Containable)
    Note: In non-destructive hashtables, elements must not be deleted
          before they are removed from the table.
    All methods that accept "Data" ( except put() )  do not require
    the actual item (this would be rediculous) but a containable
    that has the same hash value as the required item.
*/
class Hashtable : public Container
{
public:
  /** Default Constructor: Construct an empty hash table. */
  Hashtable(const unsigned flags=0);

  /** Copy Constructor: Construct a copy of a hash table. */
  Hashtable(const Hashtable&);

  /** Construct a hashtable from any container. 
      Note that only the first of any duplicate entries (according to 
      Containable::equals()) will be added from the container. */
  Hashtable(const Container&);

  /** Construct a hashtable from an enumeration. */
  Hashtable(const Enumeration&);
  Hashtable(Enumeration& e, int temp=0);

  /** Contruct a hashtable from from a vector. */
  Hashtable(const Vector&);
  Hashtable(Vector& v);

  /** Destructor. */
  ~Hashtable();

  /** Assignment: Copy a hash table. */
  Hashtable& operator=(const Hashtable&);



  /** Load this hashtable with the elements from a vector. */
  virtual int load(const Vector& v);



// Implement pure and overridden functions of Container.  See <Container.h>

   /** Add a data item to this hashtable.

       When index == -1 (the default),
          if an item with the same key exists, it is replaced (and deleted if 
          this hashtable isDestuctive()) and zero is returned; 
          otherwise, (new key) the item is added to the hashtable and 1 is returned.

       When index >= 0 and index < size(), after the item at that index is removed (and
          deleted if this container isDestructive()), the item is added as above.  Note
          that this may cause another item to be removed from the table (if one with the
          same key exists).  Returns -1 if two items are removed, otherwise 0 (no change
          in table size).

       If index < -1 or index > size(), IndexOutOfBounds is thrown. */
   virtual int         put(const Data item, const int index=-1);

   /** Load this hashtable from an enumeration. */
           int         load(const Enumeration& e)    { return Hashtable::load((Enumeration&)e, 0); }

   /** Load this hashtable with a non-const enumeration and optionally discard source
       if temp is non-zero. Invaildates e if temp. */
   virtual int load(Enumeration&, int temp=0);


   /** Retrieve a data item from this hashtable.
       If an item is found in this hashtable with the same key
       (as determined by Containable::equals==), it is returned;
       otherwise, (not found) NULL is returned. */
   virtual Data        get(const Data item) const;

   /** Retrieve an item by index */
   virtual Data        get(const int index) const;

   /** Pop: Get and remove an item. */
   virtual Data        pop();

   /** Remove an item.  item must conform to Container::get() rules.
       If the data item exists in this hashtable, it is removed and
       non-zero is returned; otherwise, (not found) zero is returned. */
   virtual DS_BOOL     remove(const Data item);

   /** Remove an item by index. */
   virtual void        remove(const int index);

   /** Returns non-zero if the item is in this hashtable. */
   virtual DS_BOOL     contains(const Data item) const;

   /** Number of elements in this hashtable. */
   virtual int         size() const;

   /** Remove all elements from this hashtable. */
   virtual  void       clear();

   /** Return an enumeration of all elements in this hashtable.
       This is a copy (snapshot) of the elements, so this hashtable can 
       be modified while iterating through the enumeration. */
   virtual Enumeration elements() const;

protected:
  BASE_HASHTABLE *m_Table;
  HASH_EXTRA     *m_Extra;
};

#endif // H_DS_HASHTABLE
