/** \file UTIL_Filters.h
    Definition of iostream filters.
    All streams in this library are BINARY only.
    This is useful for creating generic filters for compression,
    encryption, DSP, etc.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.
Please contact the author concerning commercial usage.
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_UTIL_FILTERS
#define H_UTIL_FILTERS

#include <fstream.h>
#include <UTIL_GenStreamBuf.h>
#include <UTIL_Misc.h>

/** The base for all ostream filters.  Implements the Write
    abstract function to use the generic stream buffer object */
class IOSOutputFilter : public OutputFilter, public ostream
{
public:
  IOSOutputFilter(GenStreamBuf* sbuf);
  ~IOSOutputFilter();

          int Write(char* Buffer, int Len);
  virtual void update(int Bytes);
          int  getTotalBytes();
protected:
  int m_Bytes;
};

/** The base for all istream filters.  Implements the Read
    abstract function to use the generic stream buffer object */
class IOSInputFilter : public InputFilter, public istream
{
public:
  IOSInputFilter(GenStreamBuf* sbuf);
  ~IOSInputFilter();

  int Read(char* Buffer, int Len);
};


///////////////////////////////////////////////////////////
// Block filters
///////////////////////////////////////////////////////////

/** Provides a stream for reading a certain area in another stream.
    The block starting position is the stream current position
    Example:  Limit a stream input to the first 500 bytes (Virtual EOF) */
class BlockStream : public IOSInputFilter
{
public:
  BlockStream(istream& Input, int Size);
};

/** Provides a BlockStream specific to files.
    Adds seeking capabilities. */
class FileBlockStream : public IOSInputFilter
{
public:
  FileBlockStream(ifstream& Input, int Size);
};

///////////////////////////////////////////////////////////
// Memory Filters
///////////////////////////////////////////////////////////

/** An input stream that gets its data from memory.
    This is usually a source terminator of the filter chain. */
class MemoryInputStream : public IOSInputFilter
{
public:
  MemoryInputStream(char* Buffer, int Size);
};

/** An output stream that puts its data to memory.
    This is usually a sink terminator of the filter chain. */
class MemoryOutputStream : public IOSOutputFilter
{
public:
  MemoryOutputStream(char* Buffer, int Size);
};

///////////////////////////////////////////////////////////
// Compression filters
///////////////////////////////////////////////////////////
/** Compression filter.  Destructor must be executed to indicate
    no further input and end of compressed data. */
class DeflateStream : public IOSOutputFilter
{
public:
  DeflateStream(ostream& Output);
};

/** Decompression filter. */
class InflateStream : public IOSInputFilter
{
public:
  InflateStream(istream& Input);
};

///////////////////////////////////////////////////////////
// Encryption filters
///////////////////////////////////////////////////////////
/** Encrypts all data written to it. */
class EncryptionStream : public IOSOutputFilter
{
public:
  EncryptionStream(ostream& Output, const char *Password);
};

/** Decrypts all data read from it. */
class DecryptionStream : public IOSInputFilter
{
public:
  DecryptionStream(istream& Input, const char *Password);
};

#pragma comment(lib, "libUTIL.lib")

#endif // H_UTIL_FILTERS