//
//
//                                               
//     Ŀ                                          
//              Ŀ                        
//        Ŀ                      
//                                       
//    O O                               
//             O O            
//         O O
//                            C++ Music, Sound, and Effects Library
//                               Microsoft Windows 95/98/NT Version
//
//  Copyright (c) 1995-1999 by Dan Higdon, Tim Little, and Chuck Walbourn
//
//
//
// This file and all associated files are subject to the terms of the
// GNU Lesser General Public License version 2 as published by the
// Free Software Foundation (http://www.gnu.org).   They remain the
// property of the authors: Dan Higdon, Tim Little, and Chuck Walbourn.
// See LICENSE.TXT in the distribution for a copy of this license.
//
// THE AUTHORS MAKE NO WARRANTIES, EXPRESS OR IMPLIED, AS TO THE CORRECTNESS
// OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE IT.  THE AUTHORS
// PROVIDE THE CODE ON AN "AS-IS" BASIS AND EXPLICITLY DISCLAIMS ANY
// LIABILITY, INCLUDING CONSEQUENTIAL AND INCIDENTAL DAMAGES FOR ERRORS,
// OMISSIONS, AND OTHER PROBLEMS IN THE CODE.
//
//
//
//                        http://www.mythos-engine.org/
//
//
//
// Created by Dan Higdon & Chuck Walbourn
//
// bachdigi.hpp
//
// BachSample - Base sample container class
//
// BachStaticSample - Static sample container class
//
// BachSample3D - 3D static sample container class
//
// BachFileSample - Sample that plays from a file
//
// BachDigital - Digital sound source class
//
// BachDigital3D - Digital 3D sound source class
//
//

#ifndef __BACHDIGI_HPP
#define __BACHDIGI_HPP    1

#ifdef _MSC_VER
#pragma pack(push,1)
#endif

//
//
//                                Includes
//
//

#include <dsound.h>

// Charybdis headers
#include <debug.h>
#include <portable.h>
#include <ivory.hpp>
#include <xfile.hpp>

//Ŀ
// Bach Library Includes                                                    
//
#include "bachdefs.h"

//
//
//                                Classes
//
//

class BachDigital;
class BachDigital3D;

//
// Sample Classes 
//

//Ŀ
// BachSample - Base class for digital samples                              
//
class BachSample
{
protected:
    //Ŀ
    // Protected data members                                               
    //
    IDirectSound        *dsound;            // DSound instanace
    IDirectSoundBuffer  *sbuffer;           // DSound buffer

    //Ŀ
    // Internal routines                                                    
    //
    enum { db_table_size=128 };

    static short db_table[db_table_size];
    static short db_table_find(int i);

    virtual bach_err_codes createbuffer(WAVEFORMATEX *wfmt)=0;

public:
    //Ŀ
    // Public data members.                                                 
    //
    dword           flags;                  // Sample flags
    bach_digi_fmt   fmt;                    // Sample format
    ulong           rate;                   // Sample rate
    void            *app_data;              // Application data

    //Ŀ
    // Constructor/Destructor                                               
    //
    BachSample (BachDigital *digi);
    BachSample (BachSample const &b);

    virtual ~BachSample();

    //Ŀ
    // Operations                                                           
    //
    BachSample &operator = (BachSample const &that);

    // Free all associated data
    virtual void release ();

    // Play controls
    virtual bach_err_codes play (int loop = 0)=0;
    virtual bach_err_codes stop ()=0;
    virtual bach_err_codes pause()=0;
    virtual bach_err_codes resume()=0;
    virtual dword status() const=0;

    //Ŀ
    // Utility functions.                                                   
    //
    void set_app_data(void *p) { app_data = p; }

    void set_flags(dword f) { flags = f; }
};


//Ŀ
// BachStaticSample - Static sample class                                   
//
class BachStaticSample : public BachSample
{
protected:
    //Ŀ
    // Internal routines                                                    
    //
    virtual bach_err_codes createbuffer(WAVEFORMATEX *wfmt);
    bach_err_codes restore();

public:
    //Ŀ
    // Public data members.                                                 
    //
    ulong           length;                 // Sample length
    IvoryHandle     handle;                 // Sample data

    //Ŀ
    // Constructor/Destructor                                               
    //
    BachStaticSample (BachDigital *digi);
    virtual ~BachStaticSample();

    //Ŀ
    // Operations                                                           
    //

    // Base class interface
    virtual void release ();
    virtual bach_err_codes play (int loop = 0);
    virtual bach_err_codes stop ();
    virtual bach_err_codes pause();
    virtual bach_err_codes resume();
    virtual dword status() const;

    // Additional controls for static samples
    bach_err_codes set_volume (long vol);
    long get_volume () const;

    bach_err_codes set_pan (long heading);
    long get_pan () const;

    bach_err_codes set_playrate (ulong r);
    ulong get_playrate() const;

    bach_err_codes set_position (ulong p);
    ulong get_position() const;

    // Create sample from data buffer
    bach_err_codes create(void *p, ulong l,
                          bach_digi_fmt f=BACH_DIGI_MONO_8,
                          ulong r=BACH_DIGI_DRIVERRATE);

    //Ŀ
    // Utility functions.                                                   
    //
    ulong get_playtime () const;

    //Ŀ
    // I/O routines.                                                        
    //

    // Load sample from disk
    bach_err_codes load_raw(const char *fname,
                            bach_digi_fmt f=BACH_DIGI_MONO_8,
                            ulong r=BACH_SMP_DEFAULTRATE);
    bach_err_codes load_raw(XFile &xf,
                            bach_digi_fmt f=BACH_DIGI_MONO_8,
                            ulong r=BACH_SMP_DEFAULTRATE);

    bach_err_codes load_wav(const char *fname);
    bach_err_codes load_wav(XFParseIFF &iff);
};


//Ŀ
// BachSample3D - 3D static sample class                                    
//
class BachSample3D : public BachStaticSample
{
protected:
    //Ŀ
    // Internal routines                                                    
    //
    virtual bach_err_codes createbuffer(WAVEFORMATEX *wfmt);

public:
    //Ŀ
    // Public data members.                                                 
    //
    IDirectSound3DListener  *listener;       // DSound listener interface
    IDirectSound3DBuffer    *sbuffer3d;      // DSound buffer 3D interface

    //Ŀ
    // Constructor/Destructor                                               
    //
    BachSample3D (BachDigital3D *digi);
    BachSample3D (BachSample3D const &b);
    virtual ~BachSample3D();

    //Ŀ
    // Operations                                                           
    //
    BachSample3D &operator = (BachSample3D const &that);

    // Base class interface
    virtual void release ();

    // 3D controls
    bach_err_codes set(DS3DBUFFER &settings, int recalc=1);

    bach_err_codes set_mode(DWORD mode=DS3DMODE_NORMAL, int recalc=1);

    bach_err_codes set_cone(ulong ia, ulong oa, int recalc=1);
    bach_err_codes set_cone_orient(float i, float j, float k, int recalc=1);
    bach_err_codes set_cone_volume(long vol, int recalc=1);

    bach_err_codes set_min_distance(float dist, int recalc=1);
    bach_err_codes set_max_distance(float dist, int recalc=1);

    bach_err_codes set_position(float x, float y, float z, int recalc=1);
    bach_err_codes set_velocity(float i, float j, float k, int recalc=1);

    //Ŀ
    // Utility functions.                                                   
    //
    void get(DS3DBUFFER &settings) const;

    DWORD get_mode() const;

    void get_cone(ulong &ia, ulong &oa) const;
    void get_cone_orient(float &i, float &j, float &k) const;
    long get_cone_volume() const;

    float get_min_distance() const;
    float get_max_distance() const;

    void get_position(float &x, float &y, float &z) const;
    void get_velocity(float &i, float &j, float &k) const;
};


//Ŀ
// BachFileSample - File streamed sample class                              
//
class BachFileSample : BachSample
{
protected:
    //Ŀ
    // Protected data members                                               
    //
    IDirectSoundBuffer  *sbuffer_next;      // DSound buffer

    HANDLE              thread_handle;
    CRITICAL_SECTION    critical_section;

    enum
    {
        START   = 0,
        STOP    = 1,
        MAX
    };
    HANDLE              thread_event[MAX];

    ulong               buffer_size;

    XFile               *xfile;
    ulong               playrate;
    ulong               start;
    ulong               offset;
    ulong               length;

    XFile               *xfile_next;
    bach_digi_fmt       fmt_next;
    ulong               rate_next;
    ulong               playrate_next;
    ulong               start_next;
    ulong               length_next;

    //Ŀ
    // Internal routines                                                    
    //
    virtual bach_err_codes createbuffer(WAVEFORMATEX *wfmt);
    static DWORD WINAPI thread_proc (LPVOID);

    // We cannot allow copying because of the critical section
    BachFileSample (BachFileSample const &);
    BachFileSample &operator= (BachFileSample const &);

public:
    //Ŀ
    // Constructor/Destructor                                               
    //
    BachFileSample (class BachDigital *digi, ulong bsize = 16384);
    virtual ~BachFileSample ();

    //Ŀ
    // Operations                                                           
    //

    // Base class interface
    virtual void release();
    virtual bach_err_codes play (int loop = 0);
    virtual bach_err_codes stop ();
    virtual bach_err_codes pause();
    virtual bach_err_codes resume();
    virtual dword status() const;

    // Initializes class
    bach_err_codes init();

    // Prepares files for streaming
    bach_err_codes ready_raw (const char *fname,
                              bach_digi_fmt f=BACH_DIGI_MONO_8,
                              ulong r=BACH_SMP_DEFAULTRATE);
    bach_err_codes ready_raw (XFile *xf,
                              bach_digi_fmt f=BACH_DIGI_MONO_8,
                              ulong r=BACH_SMP_DEFAULTRATE, ulong l=0,
                              int owns=1);

    bach_err_codes ready_wav (const char *fname);
    bach_err_codes ready_wav (XFile *xf, XFParseIFF *iff, int owns=0);

    // Skip to next, if any
    bach_err_codes next();
};



//
// Digitial Class 
//

//Ŀ
// BachDigital - Digital output object.                                     
//
class BachDigital
{
public:
    //Ŀ
    // Public data members.                                                 
    //
    dword               flags;              // System flags
    IDirectSound        *dsound;            // DirectSound instance
    IDirectSoundBuffer  *primary;           // DirectSound playback buffer
    ulong               driverRate;         // Driver playback rate
    bach_digi_fmt       driverFmt;          // Driver playback format

    //Ŀ
    // Constructor/Destructor                                               
    //
    BachDigital(ulong drate=BACH_DIGI_DRIVERRATE,
                bach_digi_fmt fmt=BACH_DIGI_STEREO_16);
    virtual ~BachDigital();

    //Ŀ
    // Operations                                                           
    //
    virtual bach_err_codes init (HWND hWnd, DWORD coop_level = DSSCL_PRIORITY, int cont_play=1);
    virtual void close ();

    //Ŀ
    // Utility functions.                                                   
    //
    bach_err_codes get_caps(DSCAPS &caps) const;
};


//Ŀ
// BachDigital3D - Digital 3D output object.                                
//
class BachDigital3D : public BachDigital
{
public:
    //Ŀ
    // Public data members.                                                 
    //
    IDirectSound3DListener  *listener;      // DS listener interface

    //Ŀ
    // Constructor/Destructor                                               
    //
    BachDigital3D(ulong drate=BACH_DIGI_DRIVERRATE,
                  bach_digi_fmt fmt=BACH_DIGI_STEREO_16);
    virtual ~BachDigital3D();

    //Ŀ
    // Operations                                                           
    //
    virtual bach_err_codes init (HWND hWnd, DWORD coop_level = DSSCL_PRIORITY, int cont_play=1);
    virtual void close ();

    // 3D listener controls
    bach_err_codes set(DS3DLISTENER &settings, int recalc=1);
    bach_err_codes set_distance(float factor, int recalc=1);
    bach_err_codes set_doppler(float factor, int recalc=1);
    bach_err_codes set_rolloff(float factor, int recalc=1);

    bach_err_codes set_orient(float fi, float fj, float fk,
                                float ti, float tj, float tk, int recalc=1);
    bach_err_codes set_position(float x, float y, float z, int recalc=1);
    bach_err_codes set_velocity(float i, float j, float k, int recalc=1);

    bach_err_codes recalc();

    //Ŀ
    // Utility functions.                                                   
    //
    void get(DS3DLISTENER &settings) const;

    float get_distance() const;
    float get_doppler() const;
    float get_rolloff() const;

    void get_orient(float &fi, float &fj, float &fk, float &ti, float &tj, float &tk) const;
    void get_position(float &x, float &y, float &z) const;
    void get_velocity(float &i, float &j, float &k) const;
};

#ifdef _MSC_VER
#pragma pack(pop)
#endif

#endif

// End of header - bachdigi.hpp 
