/*
 * PANVIEW.C
 *
 * Displays a cylindrical panoramic image using the Panodome engine.
 *
 * Copyright (c) 1998-1999, ITB CompuPhase. You may use/modify/distribute this
 * file or portions of it. It is provided as an example for the use of the
 * Panodome toolkit. There are no waranties on the correct behavior of this
 * program.
 */
#include <windows.h>
#include <commdlg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "..\include\panodome.h"
#include "panview.h"

static char Caption[] = "Panodome demo";

extern HWND CreateLogo(HINSTANCE hInstance, HWND hwndParent);
static HINSTANCE hinstApp;

static char *pan_err_strings[] = {
  "*** no error ***",
  "incorrect source image size",
  "viewport is too big (should zoom)",
  "insufficient memory",
  "unsupported file format (not a 256 colour BMP file)",
  "invalid parameter",
  "invalid zoom factor",
  "invalid panorama angle",
  "invalid coordinates",
  "invalid index parameter"
};

static int ExtraX,ExtraY;

static void set_extension(char *filename,char *extension)
{
  char *ptr = strrchr(filename, '.');
  if (ptr!=NULL && strchr(ptr,'\\')==NULL && strlen(ptr)<=4)
    *ptr = '\0';        /* erase extension */
  strcat(filename, extension);
}

LRESULT CALLBACK _export MainProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
static PANINFO paninfo;
static HWND hwndCompass;
static int maxzoom;
  LPBITMAPINFO lpImage;
  PANATTRIB attribs;
  PAINTSTRUCT ps;
  RECT rect;
  OPENFILENAME ofn;
  char string[256];
  HWND hwndPan;
  HMENU hmenu;
  BOOL autoscale;
  int width, height;
  int need_width, need_height;
  LPSTR label;

  switch (msg) {
  case WM_COMMAND:
    switch (wParam) {
    case ID_CLOSE:
      pan_Delete(paninfo,TRUE);
      paninfo=NULL;
      InvalidateRect(hwnd,NULL,TRUE);
      break;
    case ID_OPEN:
      string[0]='\0';
      memset(&ofn,0,sizeof ofn);
      ofn.lStructSize=sizeof ofn;
      ofn.hwndOwner=hwnd;
      ofn.lpstrFilter="BMP files\0*.bmp\0";
      ofn.nFilterIndex=1;
      ofn.lpstrFile=string;
      ofn.nMaxFile=256;
      ofn.Flags=OFN_FILEMUSTEXIST;
      if (GetOpenFileName(&ofn)) {
        pan_Delete(paninfo,TRUE);
        paninfo=NULL;
        // open the image and load the attributes
        lpImage = pan_LoadDIB(string, NULL, NULL, NULL);
        set_extension(string, ".PAN");
        pan_AttribFile(string, &attribs);
        GetClientRect(hwnd, &rect);
        // see if the image must be scaled, see whether it is allowed to do so
        pan_MinImageSize(&need_width, &need_height, &rect, attribs.angle);
        width = lpImage->bmiHeader.biWidth;
        height = lpImage->bmiHeader.biHeight;
        if (height < need_height || width < need_width || (width % 4)!=0) {
          // must scale
          hmenu=GetMenu(hwnd);
          autoscale=(GetMenuState(hmenu,ID_AUTOSCALE,MF_BYCOMMAND) & MF_CHECKED) != 0;
          if (autoscale) {
            if (height < need_height) {
              height = need_height;
              width = (int)(((long)width * height) / lpImage->bmiHeader.biHeight);
            } /* if */
            if (width < need_width) {
              int orgwidth = width;
              width = need_width;
              height = (int)(((long)width * height) / orgwidth);
            } /* if */
            width = (width + 3) & ~3;   /* round to multiple of 4 pixels */
            lpImage = pan_ScaleDIB(lpImage, NULL, NULL, width, height, TRUE);
          } else {
            if (height < need_height || width < need_width)
              MessageBox(hwnd,"De viewport is too large for the image. "
                              "Either reduce the viewport or toggle the "
                              "\"Autoscale\" option on.", Caption, MB_OK);
            else
              MessageBox(hwnd,"De image width is not a multiple of 4 pixels. "
                              "Please toggle the \"Autoscale\" option on.",
                              Caption, MB_OK);
            break;
          } /* if */
        } /* if */

        paninfo=pan_Create(lpImage,NULL,&rect,&attribs);
        if (paninfo==NULL) {
          char str[60];
          wsprintf(str, "Error: %s.", (LPSTR)pan_err_strings[pan_Error(NULL)]);
          MessageBox(hwnd, str, Caption, MB_OK);
          pan_Delete(paninfo, TRUE);
          paninfo = NULL;
          break;
        } /* if */
        pan_SetValue(paninfo,PAN_VALUE_ORIENTATION,0);

        hwndPan = pan_CreateWindow(paninfo, WS_VISIBLE|WS_CHILD, hwnd);
        if (hwndPan == NULL) {
          MessageBox(hwnd, "Error creating navigation window", Caption, MB_OK);
          pan_Delete(paninfo, TRUE);
          paninfo = NULL;
          break;
        } /* if */
        SetFocus(hwndPan);
        if (IsWindow(hwndCompass)) {
          int orientation = pan_GetValue(paninfo,PAN_VALUE_ORIENTATION);
          PostMessage(hwndCompass,PAN_COMMAND,PC_ORIENTATION,orientation);
        } /* if */
        pan_SetValue(paninfo,PAN_VALUE_MAXZOOM,maxzoom);
      } /* if */
      break;
    case ID_EXIT:
      // ??? save .INI settings
      DestroyWindow(hwnd);
      break;
    case ID_COMPASS:
      hmenu=GetMenu(hwnd);
      if (IsWindow(hwndCompass)) {
        DestroyWindow(hwndCompass);
        hwndCompass=NULL;
        CheckMenuItem(hmenu,ID_COMPASS,MF_BYCOMMAND|MF_UNCHECKED);
      } else {
        GetWindowRect(hwnd,&rect);
        hwndCompass=pan_CreateCompass(rect.left+20,rect.top+100,hwnd,FALSE);
        CheckMenuItem(hmenu,ID_COMPASS,MF_BYCOMMAND|MF_CHECKED);
      } /* if */
      break;
    case ID_SCALE320:
      pan_Delete(paninfo,TRUE);
      hmenu=GetMenu(hwnd);
      CheckMenuItem(hmenu,ID_SCALE320,MF_BYCOMMAND|MF_CHECKED);
      CheckMenuItem(hmenu,ID_SCALE640,MF_BYCOMMAND|MF_UNCHECKED);
      SetWindowPos(hwnd,NULL,0,0,320+ExtraX,240+ExtraY,SWP_NOMOVE|SWP_NOZORDER);
      InvalidateRect(hwnd,NULL,TRUE);
      break;
    case ID_SCALE640:
      pan_Delete(paninfo,TRUE);
      hmenu=GetMenu(hwnd);
      CheckMenuItem(hmenu,ID_SCALE320,MF_BYCOMMAND|MF_UNCHECKED);
      CheckMenuItem(hmenu,ID_SCALE640,MF_BYCOMMAND|MF_CHECKED);
      SetWindowPos(hwnd,NULL,0,0,640+ExtraX,480+ExtraY,SWP_NOMOVE|SWP_NOZORDER);
      InvalidateRect(hwnd,NULL,TRUE);
      break;
    case ID_AUTOSCALE:
      hmenu=GetMenu(hwnd);
      autoscale=(GetMenuState(hmenu,ID_AUTOSCALE,MF_BYCOMMAND) & MF_CHECKED) != 0;
      CheckMenuItem(hmenu,ID_AUTOSCALE,
                    MF_BYCOMMAND | (autoscale ? MF_UNCHECKED : MF_CHECKED));
      break;
    case ID_MAXZOOM0:
      hmenu=GetMenu(hwnd);
      CheckMenuItem(hmenu,ID_MAXZOOM0,MF_BYCOMMAND|MF_CHECKED);
      CheckMenuItem(hmenu,ID_MAXZOOM5,MF_BYCOMMAND|MF_UNCHECKED);
      CheckMenuItem(hmenu,ID_MAXZOOM10,MF_BYCOMMAND|MF_UNCHECKED);
      maxzoom=0;
      if (paninfo!=NULL)
        pan_SetValue(paninfo,PAN_VALUE_MAXZOOM,maxzoom);
      break;
    case ID_MAXZOOM5:
      hmenu=GetMenu(hwnd);
      CheckMenuItem(hmenu,ID_MAXZOOM0,MF_BYCOMMAND|MF_UNCHECKED);
      CheckMenuItem(hmenu,ID_MAXZOOM5,MF_BYCOMMAND|MF_CHECKED);
      CheckMenuItem(hmenu,ID_MAXZOOM10,MF_BYCOMMAND|MF_UNCHECKED);
      maxzoom=5;
      if (paninfo!=NULL)
        pan_SetValue(paninfo,PAN_VALUE_MAXZOOM,maxzoom);
      break;
    case ID_MAXZOOM10:
      hmenu=GetMenu(hwnd);
      CheckMenuItem(hmenu,ID_MAXZOOM0,MF_BYCOMMAND|MF_UNCHECKED);
      CheckMenuItem(hmenu,ID_MAXZOOM5,MF_BYCOMMAND|MF_UNCHECKED);
      CheckMenuItem(hmenu,ID_MAXZOOM10,MF_BYCOMMAND|MF_CHECKED);
      maxzoom=10;
      if (paninfo!=NULL)
        pan_SetValue(paninfo,PAN_VALUE_MAXZOOM,maxzoom);
      break;
    case ID_ABOUT:
      CreateLogo(hinstApp, hwnd);
      break;
    } /* switch */
    break;

  case WM_CREATE:
    paninfo=NULL;
    GetWindowRect(hwnd,&rect);
    hwndCompass=pan_CreateCompass(rect.left+20,rect.top+100,hwnd,FALSE);
    /* set menu items */
    hmenu=GetMenu(hwnd);
    if (IsWindow(hwndCompass))
      CheckMenuItem(hmenu,ID_COMPASS,MF_BYCOMMAND|MF_CHECKED);
    GetClientRect(hwnd,&rect);
    if (rect.right<640)
      CheckMenuItem(hmenu,ID_SCALE320,MF_BYCOMMAND|MF_CHECKED);
    else
      CheckMenuItem(hmenu,ID_SCALE640,MF_BYCOMMAND|MF_CHECKED);
    CheckMenuItem(hmenu,ID_MAXZOOM0,MF_BYCOMMAND|MF_CHECKED);
    maxzoom=0;
    break;

  case WM_DESTROY:
    pan_Delete(paninfo,TRUE);
    PostQuitMessage(0);
    break;

  case WM_PAINT:
    BeginPaint(hwnd,&ps);
    GetClientRect(hwnd,&rect);
    /* panorama window will lie on top of this */
    MoveToEx(ps.hdc,0,0,NULL);
    LineTo(ps.hdc,rect.right,rect.bottom);
    MoveToEx(ps.hdc,rect.right,0,NULL);
    LineTo(ps.hdc,0,rect.bottom);
    SetBkMode(ps.hdc,TRANSPARENT);
    TextOut(ps.hdc,rect.right/2-105,25,"Select \"File/Open...\" on the menu",33);
    EndPaint(hwnd,&ps);
    break;

  case PAN_NOTIFY:
    switch (wParam) {
    case PN_SELHOTSPOT:
      label = pan_GetData(paninfo, PAN_DATA_HOTSPOTLBL, LOWORD(lParam));
      if (stricmp(label,"@close")==0) {
        PostMessage(hwnd, WM_COMMAND, ID_CLOSE, 0);
        MessageBox(hwnd, "Panorama closed by user", Caption, MB_OK);
      } else {
        MessageBox(hwnd, "Uncaught hotspot", Caption, MB_OK);
      } /* if */
      break;
    case PN_MOVE:
      if (IsWindow(hwndCompass)) {
        int orientation = pan_GetValue(paninfo,PAN_VALUE_ORIENTATION);
        PostMessage(hwndCompass,PAN_COMMAND,PC_ORIENTATION,orientation);
      } /* if */
      break;
    } /* switch */
    break;

  default:
    return DefWindowProc(hwnd,msg,wParam,lParam);
  } /* switch */
  return 0L;
}


#if defined(__BORLANDC__)
#  pragma argsused
#endif
int PASCAL WinMain(HINSTANCE hInstance,HINSTANCE hPrevInst,LPSTR lpCmdLine,int nCmdShow)
{
  HWND hwnd;
  MSG msg;
  RECT rect;

  hinstApp=hInstance;
  if (hPrevInst==0) {
    WNDCLASS WndClass;

    memset(&WndClass, 0, sizeof WndClass);
    WndClass.style         = 0L;
    WndClass.lpfnWndProc   = MainProc;
    WndClass.hInstance     = hInstance;
    WndClass.hIcon         = LoadIcon(hInstance,"appicon");
    WndClass.hCursor       = LoadCursor(NULL,IDC_ARROW);
    WndClass.lpszMenuName  = "PAN_MENU";
    WndClass.hbrBackground = GetStockObject(LTGRAY_BRUSH);
    WndClass.lpszClassName = "panorama:main";
    if (!RegisterClass(&WndClass)) {
      MessageBox(NULL, "Error: failed to create application window class",
                 Caption, MB_OK);
      return 0;
    } /* if */
  } /* if */

  SetRect(&rect,0,0,0,0);
  AdjustWindowRect(&rect,WS_POPUPWINDOW|WS_CAPTION,TRUE);
  ExtraX=rect.right-rect.left;
  ExtraY=rect.bottom-rect.top;
  hwnd = CreateWindow("panorama:main", Caption,
                      WS_POPUPWINDOW|WS_CAPTION|WS_VISIBLE,
                      20, 20, 320+ExtraX, 240+ExtraY,
                      0, 0, hInstance, NULL);
  if (hwnd==0) {
    MessageBox(NULL, "Error: failed to create application window", Caption, MB_OK);
    return 0;
  } /* if */

  while (GetMessage(&msg, NULL, NULL, NULL)) {
    TranslateMessage(&msg);         /* translate "virtual key" codes */
    DispatchMessage(&msg);          /* send message to right window */
  } /* while */

  return 0;
}

