{***************************************************************}
{ FIBPlus - component library for direct access to Firebird and }
{ Interbase databases                                           }
{                                                               }
{    FIBPlus is based in part on the product                    }
{    Free IB Components, written by Gregory H. Deatz for        }
{    Hoagland, Longo, Moran, Dunst & Doukas Company.            }
{    mailto:gdeatz@hlmdd.com                                    }
{                                                               }
{    Copyright (c) 1998-2001 Serge Buzadzhy                     }
{    Contact: buzz@devrace.com                                  }
{                                                               }
{ ------------------------------------------------------------- }
{    FIBPlus home page      : http://www.fibplus.net/           }
{    FIBPlus support e-mail : fibplus@devrace.com               }
{ ------------------------------------------------------------- }
{                                                               }
{  Please see the file License.txt for full license information }
{***************************************************************}


unit FIBMiscellaneous;

(*
 * Compiler defines
 *)
{$A+}                           (* Aligned records: On *)
{$B-}                           (* Short circuit boolean expressions: Off *)
{$G+}                           (* Imported data: On *)
{$H+}                           (* Huge Strings: On *)
{$J-}                           (* Modification of Typed Constants: Off *)
{$M+}                           (* Generate run-time type information: On *)
{$Q-}                           (* Overflow checks: Off *)
{$R-}                           (* Range checks: Off *)
{$T+}                           (* Typed address: On *)
{$U+}                           (* Pentim-safe FDIVs: On *)
{$X+}                           (* Extended syntax: On *)
{$Z1}                           (* Minimum Enumeration Size: 1 Byte *)

interface

{$I FIBPlus.inc}
uses
 {$IFDEF MSWINDOWS}
  Windows, SysUtils, Classes, ibase,IB_Intf, ib_externals,
  DB, fib, FIBDatabase, FIBQuery, StdFuncs,IB_ErrorCodes;
 {$ENDIF}
 {$IFDEF LINUX}
  Types,Libc, SysUtils, Classes, ibase,IB_Intf, IB_Externals,
  DB, fib, FIBDatabase, FIBQuery, StdFuncs,IB_ErrorCodes;
 {$ENDIF}

const
  DefaultBlobSegmentSize = 16 * 1024; // By default, let's try to read in 16k blocks.

type
  (* TFIBBlobStream *)
  TFIBBlobStream = class(TStream)
  private
    FDatabase:TFIBDatabase;
    FTransaction:TFIBTransaction;
    FBlobID: TISC_QUAD;
    FBlobMaxSegmentSize,           // Maximum segment size
    FBlobNumSegments,              // How many segments?
    FBlobSize: Long;               // Blob size
    FOldBlobSize: Long;
    FBlobType: Short;              // 0 = segmented, 1 = streamed.
    FBlobSubType: Long;  // ivan_ra
    FBuffer: PChar;
    FOldBuffer: PChar;
    FBlobInitialized: Boolean;     // Has the blob been "opened" yet?
    FHandle: TISC_BLOB_HANDLE;
    FMode: TBlobStreamMode;        // (bmRead, bmWrite, bmReadWrite);
    FModified: Boolean;            // When finalize is called, does it need to do anything?
    FPosition: Long;              // The current position in the stream.

    FBlobStreamList:TList;
    FIndexInList:integer;
    FFieldNo    :integer;
  protected
    procedure DoOnDatabaseFree(Sender:TObject);
    procedure CreateBlob;
    procedure EnsureBlobInitialized;
    procedure GetBlobInfo;
    function  GetDatabase: TFIBDatabase;
    function  GetDBHandle: PISC_DB_HANDLE;
    function  GetTransaction: TFIBTransaction;
    function  GetTRHandle: PISC_TR_HANDLE;
    procedure CheckHandles;    
    procedure OpenBlob;
    procedure SetBlobID(Value: TISC_QUAD);
    procedure SetDatabase(Value: TFIBDatabase);
    procedure SetMode(Value: TBlobStreamMode);
    procedure SetTransaction(Value: TFIBTransaction);
    function  GetAsString: string;


  public
    constructor CreateNew(aFieldNo:integer;aBlobStreamList:TList);
    constructor Create;
    destructor Destroy; override;
    function  Call(ErrCode: ISC_STATUS; RaiseError: Boolean): ISC_STATUS;
    procedure CheckReadable;
    procedure CheckWritable;
    procedure DoFinalize(ClearModified:Boolean);
    procedure Finalize;
    procedure CloseBlob;
    procedure Cancel; 
    procedure LoadFromFile(Filename: String);
    procedure LoadFromStream(Stream: TStream);
    function  Read(var Buffer; Count: Longint): Longint; override;
    procedure SaveToFile(Filename: String);
    procedure SaveToStream(Stream: TStream);
    function  Seek(Offset: Longint; Origin: Word): Longint; override;
    procedure SetSize(NewSize: Long); override;
    procedure Truncate;
    function  Write(const Buffer; Count: Longint): Longint; override;
    // properties
    property BlobInitialized:boolean read FBlobInitialized; 
    property Handle: TISC_BLOB_HANDLE read FHandle;
    property BlobID: TISC_QUAD read FBlobID write SetBlobID;
    property BlobMaxSegmentSize: Long read FBlobMaxSegmentSize;
    property BlobNumSegments: Long read FBlobNumSegments;
    property BlobSize: Long read FBlobSize;
    property BlobType: Short read FBlobType;
    property BlobSubType: Long read FBlobSubType write FBlobSubType;    // ivan_ra    
    property Database: TFIBDatabase read GetDatabase write SetDatabase;
    property DBHandle: PISC_DB_HANDLE read GetDBHandle;
    property Mode: TBlobStreamMode read FMode write SetMode;
    property Modified: Boolean read FModified;
    property Transaction: TFIBTransaction read GetTransaction write SetTransaction;
    property TRHandle: PISC_TR_HANDLE read GetTRHandle;
    property AsString:string read GetAsString;
    property FieldNo :integer read FFieldNo;
  end;

  procedure GetBlobInfo(hBlobHandle: PISC_BLOB_HANDLE;
    var NumSegments, MaxSegmentSize, TotalSize: Long; var BlobType: Short);
  procedure ReadBlob(hBlobHandle: PISC_BLOB_HANDLE; Buffer: PChar;
    BlobSize: Long);
  procedure WriteBlob(hBlobHandle: PISC_BLOB_HANDLE; Buffer: PChar;
    BlobSize: Long);

{$IFDEF MSWINDOWS}
type
  (* TFIBOutputDelimitedFile *)
  TFIBOutputDelimitedFile = class(TFIBBatchOutputStream)
  protected
    FHandle: THandle;
    FOutputTitles: Boolean;
    FColDelimiter,
    FRowDelimiter: String;
  public
    destructor Destroy; override;
    procedure ReadyStream; override;
    function WriteColumns: Boolean; override;
    property ColDelimiter: String read FColDelimiter write FColDelimiter;
    property OutputTitles: Boolean read FOutputTitles
                                   write FOutputTitles;
    property RowDelimiter: String read FRowDelimiter write FRowDelimiter;
  end;

  (* TFIBInputDelimitedFile *)
  TFIBInputDelimitedFile = class(TFIBBatchInputStream)
  protected
    FColDelimiter,
    FRowDelimiter: String;
    FEOF: Boolean;
    FFile: TFileStream;
    FLookAhead: Char;
    FReadBlanksAsNull: Boolean;
    FSkipTitles: Boolean;
  public
    destructor Destroy; override;
    function GetColumn(var Col: String): Integer;
    function ReadParameters: Boolean; override;
    procedure ReadyStream; override;
    property ColDelimiter: String read FColDelimiter write FColDelimiter;
    property ReadBlanksAsNull: Boolean read FReadBlanksAsNull
                                       write FReadBlanksAsNull;
    property RowDelimiter: String read FRowDelimiter write FRowDelimiter;
    property SkipTitles: Boolean read FSkipTitles write FSkipTitles;
  end;

  (* TFIBOutputRawFile *)
  TFIBOutputRawFile = class(TFIBBatchOutputStream)
  protected
    FHandle: THandle;
  public
    destructor Destroy; override;
    procedure  ReadyStream; override;
    function   WriteColumns: Boolean; override;
  end;

  (* TFIBInputRawFile *)
  TFIBInputRawFile = class(TFIBBatchInputStream)
  protected
    FHandle: THandle;
  public
    destructor Destroy; override;
    function  ReadParameters: Boolean; override;
    procedure ReadyStream; override;
  end;

{$ENDIF}

implementation

uses
  FIBDataSet,IBBlobFilter;

procedure GetBlobInfo(hBlobHandle: PISC_BLOB_HANDLE;
  var NumSegments, MaxSegmentSize, TotalSize: Long; var BlobType: Short);
var
  items: array[0..3] of Char;
  results: array[0..99] of Char;
  i, item_length: Integer;
  item: Integer;
begin
  items[0] := Char(isc_info_blob_num_segments);
  items[1] := Char(isc_info_blob_max_segment);
  items[2] := Char(isc_info_blob_total_length);
  items[3] := Char(isc_info_blob_type);

  if isc_blob_info(StatusVector, hBlobHandle, 4, @items[0], SizeOf(results),
                    @results[0]) > 0 then
    IBError(nil);

  i := 0;
  while (i < SizeOf(results)) and (results[i] <> Char(isc_info_end)) do
  begin
    item := Integer(results[i]); Inc(i);
    item_length := isc_vax_integer(@results[i], 2); Inc(i, 2);
    case item of
      isc_info_blob_num_segments:
        NumSegments := isc_vax_integer(@results[i], item_length);
      isc_info_blob_max_segment:
        MaxSegmentSize := isc_vax_integer(@results[i], item_length);
      isc_info_blob_total_length:
        TotalSize := isc_vax_integer(@results[i], item_length);
      isc_info_blob_type:
        BlobType := isc_vax_integer(@results[i], item_length);
    end;
    Inc(i, item_length);
  end;
end;

procedure ReadBlob(hBlobHandle: PISC_BLOB_HANDLE; Buffer: PChar;
  BlobSize: Long);
var
  CurPos: Long;
  BytesRead, SegLen: UShort;
  LocalBuffer: PChar;
begin
  CurPos := 0;
  LocalBuffer := Buffer;
  SegLen := UShort(DefaultBlobSegmentSize);
  while (CurPos < BlobSize) do
  begin
    if (CurPos + SegLen > BlobSize) then
      SegLen := BlobSize - CurPos;
    if not ((isc_get_segment(
               StatusVector, hBlobHandle, @BytesRead, SegLen,
               LocalBuffer) = 0) or
            (StatusVectorArray[1] = isc_segment)) then
      IBError(nil);
    Inc(LocalBuffer, BytesRead);
    Inc(CurPos, BytesRead);
    BytesRead := 0;
  end;
end;

procedure WriteBlob(hBlobHandle: PISC_BLOB_HANDLE; Buffer: PChar;
  BlobSize: Long);
var
  CurPos, SegLen: Long;
begin
  CurPos := 0;
  SegLen := DefaultBlobSegmentSize;
  while (CurPos < BlobSize) do
  begin
    if (CurPos + SegLen > BlobSize) then
      SegLen := BlobSize - CurPos;
    if isc_put_segment(StatusVector, hBlobHandle, SegLen,
         PChar(@Buffer[CurPos])) > 0 then
      IBError(nil);
    Inc(CurPos, SegLen);
  end;
end;

(* TFIBBlobStream *)
procedure TFIBBlobStream.DoOnDatabaseFree(Sender: TObject);
begin
  FDatabase   :=nil;
  FTransaction:=nil;
end;

constructor TFIBBlobStream.CreateNew(aFieldNo:integer;aBlobStreamList:TList);
begin
  inherited Create;
  FBuffer         := nil;
  FBlobSize       := 0;
  FOldBuffer      := nil;
  FOldBlobSize    := 0;
  FBlobInitialized:=false;
  FBlobStreamList :=aBlobStreamList;
  FFieldNo        :=aFieldNo;
  if Assigned(FBlobStreamList) then
   FIndexInList:=FBlobStreamList.Add(Self)
end;

constructor TFIBBlobStream.Create;
begin
 CreateNew(-1,nil)
end;

destructor TFIBBlobStream.Destroy;
begin
  CloseBlob;
  SetSize(0);
  ReallocMem(FOldBuffer, 0);
  FOldBuffer := nil;
  FOldBlobSize := 0;
  if Assigned(FBlobStreamList) then
  with FBlobStreamList do
  begin
    begin
     if FIndexInList<Count-1 then
     begin
      FBlobStreamList[FIndexInList]:=FBlobStreamList[Count-1];
      TFIBBlobStream(FBlobStreamList[FIndexInList]).FIndexInList:=FIndexInList;
     end;
     Delete(Count-1)
    end;
  end;  
  inherited Destroy;
end;


function TFIBBlobStream.Call(ErrCode: ISC_STATUS; RaiseError: Boolean): ISC_STATUS;
begin
  Result := 0;
  if Transaction <> nil then
    Result := Transaction.Call(ErrCode, RaiseError)
  else if RaiseError and (ErrCode > 0) then
    IBError(Self);
end;

procedure TFIBBlobStream.CheckReadable;
begin
  if FMode = bmWrite then FIBError(feBlobCannotBeRead, [nil]);
end;

procedure TFIBBlobStream.CheckWritable;
begin
  if FMode = bmRead then FIBError(feBlobCannotBeWritten, [nil]);
end;

procedure TFIBBlobStream.CloseBlob;
begin
  Finalize;
  if (FHandle <> nil) and
     (Call(isc_close_blob(StatusVector, @FHandle), False) > 0) then
    IBError(Self);
  FHandle:=nil;
  FBlobInitialized:=false;
end;

procedure TFIBBlobStream.CreateBlob;
begin
  CheckWritable;
  FBlobID.gds_quad_high := 0;
  FBlobID.gds_quad_low := 0;
  Truncate;
end;

procedure TFIBBlobStream.EnsureBlobInitialized;
begin
  if not FBlobInitialized then
  begin
    case FMode of
      bmWrite:
        CreateBlob;
      bmReadWrite:
      begin
        if (FBlobID.gds_quad_high = 0) and
           (FBlobID.gds_quad_low = 0) then
          CreateBlob
        else
          OpenBlob;
      end;
    else
        OpenBlob;
    end;
    FBlobInitialized := True;
  end;
end;


procedure TFIBBlobStream.DoFinalize(ClearModified:Boolean);
begin
  if (not FBlobInitialized) or (FMode = bmRead)  or (not FModified)  then Exit;
  CheckHandles;
  // We need to start writing to a blob, so first create one.
  Call(isc_create_blob2(StatusVector, DBHandle, TRHandle, @FHandle,
       @FBlobID, 0, nil), True);
  ReallocMem(FOldBuffer, FBlobSize);
  FOldBlobSize := FBlobSize;        
  Move(FBuffer[0], FOldBuffer[0], FBlobSize);
  IBFilterBuffer(Database,FBuffer, FBlobSize, FBlobSubType, true);
  FIBMiscellaneous.WriteBlob(@FHandle, FBuffer, FBlobSize);
  Call(isc_close_blob(StatusVector, @FHandle), True);
  if ClearModified then FModified := False;
end;

procedure TFIBBlobStream.Finalize;
begin
 DoFinalize(True)
end;

procedure TFIBBlobStream.Cancel;
begin
  if (not FBlobInitialized) or (FMode = bmRead) then  Exit;
  SetSize(FOldBlobSize);
  Move(FOldBuffer[0], FBuffer[0], FBlobSize);
  FModified := False;
end;

procedure TFIBBlobStream.GetBlobInfo;
var
  iBlobSize: Long;
begin
  FIBMiscellaneous.GetBlobInfo(@FHandle, FBlobNumSegments,
    FBlobMaxSegmentSize,    iBlobSize, FBlobType
  );
  SetSize(iBlobSize);
end;

function TFIBBlobStream.GetDatabase: TFIBDatabase;
begin
  Result := FDatabase;
end;

function TFIBBlobStream.GetDBHandle: PISC_DB_HANDLE;
begin
  if Assigned(FDatabase)  and Assigned(FDatabase.Handle) then
   Result := @FDatabase.Handle
  else
   Result :=nil;  
end;

function TFIBBlobStream.GetTransaction: TFIBTransaction;
begin
  Result := FTransaction;
end;

function TFIBBlobStream.GetTRHandle: PISC_TR_HANDLE;
begin
  if Assigned(FTransaction) and Assigned(FTransaction.Handle) then
   Result := @FTransaction.Handle
  else
   Result := nil
end;

procedure TFIBBlobStream.CheckHandles;
begin
 if (GetDBHandle=nil) then
   FIBError(feDatabaseNotAssigned, ['BlobStream'])
 else
 if (GetTRHandle=nil) then
   FIBError(feTransactionNotAssigned, ['BlobStream']);
end;

procedure TFIBBlobStream.LoadFromFile(Filename: String);
var
  Stream: TStream;
begin
  Stream := TFileStream.Create(FileName, fmOpenRead);
  try
    LoadFromStream(Stream);
  finally
    Stream.Free;
  end;
end;

procedure TFIBBlobStream.LoadFromStream(Stream: TStream);
begin
  CheckWritable;
  EnsureBlobInitialized;
  Stream.Position := 0;
  SetSize(Stream.Size);
  if FBlobSize <> 0 then Stream.ReadBuffer(FBuffer^, FBlobSize);
  FModified := True;
end;

procedure TFIBBlobStream.OpenBlob;
begin
  CheckReadable;
  CheckHandles;
  Call(isc_open_blob2(StatusVector, DBHandle, TRHandle, @FHandle,
                     @FBlobID, 0, nil), True);
  try
    GetBlobInfo;
    SetSize(FBlobSize);
    FIBMiscellaneous.ReadBlob(@FHandle, FBuffer, FBlobSize);
    IBFilterBuffer(Database,FBuffer, FBlobSize, FBlobSubType, false);  
    ReallocMem(FOldBuffer, FBlobSize);
    FOldBlobSize := FBlobSize;
    Move(FBuffer[0], FOldBuffer[0], FBlobSize); 
  except
    Call(isc_close_blob(StatusVector, @FHandle), False);
    raise;
  end;
  Call(isc_close_blob(StatusVector, @FHandle), True);
end;

function TFIBBlobStream.Read(var Buffer; Count: Longint): Longint;
begin
  CheckReadable;
  EnsureBlobInitialized;
  if (Count <= 0) then
  begin
    Result := 0;
    Exit;
  end;
  if (FPosition + Count > FBlobSize) then
    Result := FBlobSize - FPosition
  else
    Result := Count;
  Move(FBuffer[FPosition], Buffer, Result);
  Inc(FPosition, Result);
end;

procedure TFIBBlobStream.SaveToFile(Filename: String);
var
  Stream: TStream;
begin
  Stream := TFileStream.Create(FileName, fmCreate);
  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  end;
end;

procedure TFIBBlobStream.SaveToStream(Stream: TStream);
begin
  CheckReadable;
  EnsureBlobInitialized;
  if FBlobSize <> 0 then
  begin
    Seek(0, soFromBeginning);
    Stream.WriteBuffer(FBuffer^, FBlobSize);
  end;
end;

function TFIBBlobStream.Seek(Offset: Longint; Origin: Word): Longint;
begin
  EnsureBlobInitialized;
  case Origin of
    soFromBeginning     : FPosition := Offset;
    soFromCurrent	: Inc(FPosition, Offset);
    soFromEnd           : FPosition := FBlobSize + Offset;
  end;
  Result := FPosition;
end;

procedure TFIBBlobStream.SetBlobID(Value: TISC_QUAD);
begin
  System.Move(Value, FBlobID, SizeOf(TISC_QUAD));
  FBlobInitialized := False;
end;

procedure TFIBBlobStream.SetDatabase(Value: TFIBDatabase);
begin
  if Assigned(FDatabase) then
   FDatabase.RemoveEvent(DoOnDatabaseFree,detBeforeDestroy); 
  FDatabase := Value;
  if Assigned(FDatabase) then
   FDatabase.AddEvent(DoOnDatabaseFree,detBeforeDestroy);
  FBlobInitialized := False;
end;

procedure TFIBBlobStream.SetMode(Value: TBlobStreamMode);
begin
  FMode := Value;
  FBlobInitialized := False;
end;

procedure TFIBBlobStream.SetSize(NewSize: Long);
begin
  if (NewSize <> FBlobSize) then
  begin
    ReallocMem(FBuffer, NewSize);
    FBlobSize := NewSize;
    // Guarantee that FBuffer is nil, if size is 0.
    if NewSize = 0 then
      FBuffer := nil;
  end;
end;

procedure TFIBBlobStream.SetTransaction(Value: TFIBTransaction);
begin
  if Assigned(FTransaction) then
   FTransaction.RemoveEvent(DoOnDatabaseFree,tetBeforeDestroy);


  FBlobInitialized := False;

  if Assigned(FTransaction) then
   FTransaction.AddEvent(DoOnDatabaseFree,tetBeforeDestroy);
  FTransaction := Value;
end;

function  TFIBBlobStream.GetAsString: string;
var
  Len: Integer;
begin
  CheckReadable;
  EnsureBlobInitialized;
  if FBlobSize <> 0 then
    begin
      Seek(0, soFromBeginning);
      Len := Size;
      SetString(Result, nil, Len);
      ReadBuffer(Pointer(Result)^, Len);
    end
  else
    Result:='';
end;


procedure TFIBBlobStream.Truncate;
begin
  SetSize(0);
end;

function TFIBBlobStream.Write(const Buffer; Count: Longint): Longint;
begin
  CheckWritable;
  EnsureBlobInitialized;
  Result := Count;
  if Count <= 0 then  Exit;
  if (FPosition + Count > FBlobSize) then
    SetSize(FPosition + Count);
  Move(Buffer, FBuffer[FPosition], Count);
  Inc(FPosition, Count);
  FModified := True;
end;
{$IFDEF MSWINDOWS}

(*
 * TFIBOutputDelimitedFile
 *)

destructor TFIBOutputDelimitedFile.Destroy;
begin
  if FHandle <> 0 then
  begin
    FlushFileBuffers(FHandle);
    CloseHandle(FHandle);
  end;
  inherited Destroy;
end;

const NULL_TERMINATOR = #0;
      TAB  = #9;
      CR   = #13;
      LF   = #10;



procedure TFIBOutputDelimitedFile.ReadyStream;
var
  i: Integer;
  BytesWritten: DWORD;
  st: String;
begin
  if FColDelimiter = '' then
    FColDelimiter := TAB;
  if FRowDelimiter = '' then
    FRowDelimiter := CRLF;
  FHandle := CreateFile(PChar(Filename), GENERIC_WRITE, 0, nil, CREATE_ALWAYS,
                        FILE_ATTRIBUTE_NORMAL, 0);
  if FHandle = INVALID_HANDLE_VALUE then
    FHandle := 0;
  if FOutputTitles then
  begin
    for i := 0 to Columns.Count - 1 do
      if i = 0 then
        st := String(Columns[i].Data^.aliasname)
      else
        st := st + FColDelimiter + String(Columns[i].Data^.aliasname);
    st := st + FRowDelimiter;
    WriteFile(FHandle, st[1], Length(st), BytesWritten, nil);
  end;
end;

function TFIBOutputDelimitedFile.WriteColumns: Boolean;
var
  i: Integer;
  BytesWritten: DWORD;
  st: String;
begin
  Result := False;
  if FHandle <> 0 then
  begin
    st := '';
    for i := 0 to Columns.Count - 1 do
    begin
      if i > 0 then
        st := st + FColDelimiter;
      st := st + StripString(Columns[i].AsString,
                             FColDelimiter + FRowDelimiter);
    end;
    st := st + FRowDelimiter;
    WriteFile(FHandle, st[1], Length(st), BytesWritten, nil);
    if BytesWritten = DWORD(Length(st)) then
      Result := True;
  end
end;

(*
 * TFIBInputDelimitedFile
 *)
destructor TFIBInputDelimitedFile.Destroy;
begin
  FFile.Free;
  inherited Destroy;
end;

function TFIBInputDelimitedFile.GetColumn(var Col: String): Integer;
var
  c: Char;
  BytesRead: Integer;

  procedure ReadInput;
  begin
    if FLookAhead <> NULL_TERMINATOR then
    begin
      c := FLookAhead;
      BytesRead := 1;
      FLookAhead := NULL_TERMINATOR;
    end
    else
      BytesRead := FFile.Read(c, 1);
  end;

  procedure CheckCRLF(Delimiter: String);
  begin
    if (c = CR) and (Pos(LF, Delimiter) > 0) then
    begin
      BytesRead := FFile.Read(c, 1);
      if (BytesRead = 1) and (c <> #10) then
        FLookAhead := c
    end;
  end;

begin
  Col := '';
  Result := 0;
  ReadInput;
  while BytesRead <> 0 do
  begin
    if Pos(c, FColDelimiter) > 0 then
    begin
      CheckCRLF(FColDelimiter);
      Result := 1;
      break;
    end
    else
    if Pos(c, FRowDelimiter) > 0 then
    begin
      CheckCRLF(FRowDelimiter);
      Result := 2;
      break;
    end
    else
      Col := Col + c;
    ReadInput;
  end;
end;

function TFIBInputDelimitedFile.ReadParameters: Boolean;
var
  i, curcol: Integer;
  Col: String;
begin
  Result := False;
  if not FEOF then
  begin
    curcol := 0;
    repeat
      i := GetColumn(Col);
      if (i = 0) then FEOF := True;
      if (curcol < Params.Count) then
      begin
        try
          if (Col = '') and (ReadBlanksAsNull) then
            Params[curcol].IsNull := True
          else
            Params[curcol].AsString := Col;
          Inc(curcol);
        except
          on E: Exception do
          begin
            if not (FEOF and (curcol = Params.Count)) then
              raise;
          end;
        end;
      end;
    until (FEOF) or (i = 2);
    Result := ((FEOF) and (curcol = Params.Count)) or (not FEOF);
  end;
end;

procedure TFIBInputDelimitedFile.ReadyStream;
begin
  if FColDelimiter = '' then    FColDelimiter := TAB;
  if FRowDelimiter = '' then    FRowDelimiter := CRLF;
  FLookAhead := NULL_TERMINATOR;
  FEOF := False;
  if FFile <> nil then  FFile.Free;
  FFile := TFileStream.Create(FFilename, fmOpenRead or fmShareDenyWrite);
  if FSkipTitles then  ReadParameters;
end;

(* TFIBOutputRawFile *)
destructor TFIBOutputRawFile.Destroy;
begin
  if FHandle <> 0 then
  begin
    FlushFileBuffers(FHandle);
    CloseHandle(FHandle);
  end;
  inherited Destroy;
end;

const VerRowFile:string='FIB$BATCH_ROW1';

procedure TFIBOutputRawFile.ReadyStream;
var 
    BytesWritten:DWord;
begin
  FHandle := CreateFile(PChar(Filename), GENERIC_WRITE, 0, nil, CREATE_ALWAYS,
                        FILE_ATTRIBUTE_NORMAL, 0);
  if FHandle = INVALID_HANDLE_VALUE then
  begin
    FState  :=bsInError;
    FHandle := 0;
  end
  else
  begin
   WriteFile(FHandle,VerRowFile[1], Length(VerRowFile),
                BytesWritten, nil);
   FState  :=bsFileReady                
  end;
end;

function TFIBOutputRawFile.WriteColumns: Boolean;
var
  i: Integer;
  BytesWritten: DWord;
  b:boolean;
  Buffer :Pointer;
  bs : TMemoryStream;
  Bytes: integer;
begin
  Result := False;
  bs :=nil;
  if FHandle <> 0 then
  try
    FState :=bsInProcess;
    for i := 0 to Columns.Count - 1 do
    begin
      b:=Columns[i].IsNull;
      WriteFile(FHandle,b, SizeOf(Boolean), BytesWritten, nil);
      if BytesWritten<> SizeOf(Boolean) then Exit;
      if not b then
      with Columns[i].Data^ do
      begin
       Buffer :=sqldata;
       case sqltype and (not 1) of
        SQL_VARYING:Bytes:=sqllen+2;
        SQL_BLOB: begin
                   if bs=nil then bs := TMemoryStream.Create;
                   Columns[i].SaveToStream(bs);
                   Bytes := bs.Size;
                   WriteFile(FHandle, Bytes,
                         SizeOf(Integer) , BytesWritten, nil
                   );
                   Buffer :=bs.Memory;
                  end;
       else
        Bytes:=sqllen
       end;
       WriteFile(FHandle, Buffer ^,  Bytes , BytesWritten, nil);
       if BytesWritten <> DWORD(Bytes) then Exit;
      end;
    end;
    Result := True;
  finally
    bs.Free;
  end;
end;

(* TFIBInputRawFile *)
destructor TFIBInputRawFile.Destroy;
begin
  if FHandle <> 0 then
    CloseHandle(FHandle);
  inherited;
end;

function TFIBInputRawFile.ReadParameters: Boolean;
var
  i: Integer;
  BytesRead: DWord;
  b:boolean;
  Bytes:DWORD;
  Buffer :Pointer;
  bs : TMemoryStream;  
begin
  Result := False;
  bs :=nil;
  if FHandle <> 0 then
  try
    FState :=bsInProcess;
    for i := 0 to Params.Count - 1 do
    begin
      ReadFile(FHandle, b, SizeOf(Boolean), BytesRead, nil);
      if BytesRead<> SizeOf(Boolean) then Exit;
      Params[i].IsNull:=b;
      if not b then
      with Params[i].Data^ do
      begin
       Buffer :=sqldata;
       case sqltype and (not 1) of
        SQL_VARYING:Bytes:=sqllen+2;
        SQL_BLOB: begin
                   if bs=nil then bs := TMemoryStream.Create;
                    ReadFile(FHandle, Bytes,
                         SizeOf(Integer) , BytesRead, nil
                    );
                    Params[i].IsNull:=BytesRead=0;
                    if Params[i].IsNull then Continue;
                    bs.Size:=Bytes;
                    Buffer :=bs.Memory;
                  end;
       else
        Bytes:=sqllen;
       end;

       ReadFile(FHandle, Buffer ^, Bytes, BytesRead, nil);
       if BytesRead <> Bytes then    Exit;
       if (sqltype and (not 1))=SQL_BLOB  then
         Params[i].LoadFromStream(bs);
      end;
    end;
    Result := True;
  finally
   bs.Free
  end;
end;

procedure TFIBInputRawFile.ReadyStream;
var s:string;
    BytesRead:DWORD;
begin
  if FHandle <> 0 then
    CloseHandle(FHandle);
  FHandle := CreateFile(PChar(Filename), GENERIC_READ, 0, nil, OPEN_EXISTING,
                        FILE_FLAG_SEQUENTIAL_SCAN, 0);
  if FHandle = INVALID_HANDLE_VALUE then
  begin
   FHandle := 0 ;
   FState  :=bsInError;
  end
  else
  begin
   SetLength(s,Length(VerRowFile));
   ReadFile(FHandle, s[1], Length(VerRowFile), BytesRead, nil);
   if (BytesRead<> DWORD(Length(VerRowFile))) or (s<>VerRowFile) then
   begin
    CloseHandle(FHandle);
    FHandle := 0;
    FState  := bsInError;    
   end
   else
    FState  :=bsFileReady
  end;
end;
{$ENDIF}
end.
