{***************************************************************}
{ FIBPlus - component library for direct access to Firebird and }
{ Interbase databases                                           }
{                                                               }
{    FIBPlus is based in part on the product                    }
{    Free IB Components, written by Gregory H. Deatz for        }
{    Hoagland, Longo, Moran, Dunst & Doukas Company.            }
{    mailto:gdeatz@hlmdd.com                                    }
{                                                               }
{    Copyright (c) 1998-2001 Serge Buzadzhy                     }
{    Contact: buzz@devrace.com                                  }
{                                                               }
{ ------------------------------------------------------------- }
{    FIBPlus home page      : http://www.fibplus.net/           }
{    FIBPlus support e-mail : fibplus@devrace.com               }
{ ------------------------------------------------------------- }
{                                                               }
{  Please see the file License.txt for full license information }
{***************************************************************}


unit pFIBDataInfo;

interface

{$I FIBPlus.inc}
uses
  SysUtils,Classes,DB,pFIBDataSet,FIBDataSet,FIBDataBase,FIBQuery,
  pFIBQuery,pFIBDataBase,pFIBProps
 {$IFDEF MSWINDOWS},Dialogs; {$ENDIF}
 {$IFDEF LINUX}   ,QDialogs; {$ENDIF}

 type   TpFIBFieldInfo=class
        private
         FIsComputed  :boolean;
         FDefaultValue:string;
         FCanIncToWhereClause:boolean;
         FDomainName:string;
         FDefaultValueEmptyString: boolean;
// Info from FIB$FIELDS_INFO
         FWithAdditionalInfo:boolean;
         FDisplayLabel:string;
         FVisible     :boolean;
         FEditFormat  :string;
         FDisplayFormat:string;
         FIsTriggered :boolean;
         FOtherInfo   :TStrings;
         FDisplayWidth:integer;
        public
         constructor Create; 
         destructor  Destroy;override;
         procedure SaveToStream(Stream:TStream);
         procedure LoadFromStream(Stream:TStream;FromBegin:boolean);
         property IsComputed:boolean read FIsComputed ;
         property DefaultValue:string read FDefaultValue ;
         property DomainName:string read FDomainName;
         property CanIncToWhereClause:boolean read FCanIncToWhereClause;
// Info from FIB$FIELDS_INFO
         property WithAdditionalInfo:boolean read FWithAdditionalInfo;
         property DisplayLabel:string  read FDisplayLabel;
         property Visible     :boolean read FVisible   ;
         property EditFormat  :string  read FEditFormat ;
         property DisplayFormat:string read FDisplayFormat;
         property IsTriggered:boolean  read FIsTriggered;
         property OtherInfo:TStrings   read FOtherInfo;
         property DefaultValueEmptyString:boolean read FDefaultValueEmptyString;
         property DisplayWidth:integer read FDisplayWidth ;
        end;



        TpFIBTableInfo=class
        private
         FDBName:string;
         FTableName:string;
         FPrimaryKeyFields:string;
         FFieldList :TStringList;
         FFormatNumber :integer;
         FFIVersion    :integer;//FIB$FIELDS_INFO
         FInternalTransaction:TpFIBTransaction;
         procedure GetInfoFields(const TableName:string;aTransaction:TFIBTransaction);
         procedure ClearFieldList;
         procedure GetAdditionalInfo(FromQuery:TpFIBDataset;
          const FieldName:string;ToFieldInfo:TpFIBFieldInfo );

         procedure   FillInfo
          (aDataBase:TFIBDataBase;const ATableName:string);
         function    IsActualInfo(aDataBase:TFIBDataBase) :boolean;
         function    GetPrimaryKeyFields: string;
        public
         constructor Create;
         destructor  Destroy;override;
         procedure   SaveToStream(Stream:TStream);
         procedure   LoadFromStream(Stream:TStream;FromBegin:boolean);

         function    FieldInfo(const FieldName:string):TpFIBFieldInfo;
         property    TableName:string read FTableName ;
         property    PrimaryKeyFields:string read GetPrimaryKeyFields;
         property    FieldList :TStringList read FFieldList;
        end;



       TpFIBTableInfoCollect= class(TComponent)
       private
        FDBNames    :TStringList;
        FListTabInfo:TStringList;
        FInternalTransaction:TpFIBTransaction;
       protected
         procedure Notification(AComponent: TComponent; Operation: TOperation); override;
       public
        constructor Create(AOwner:TComponent); override;
        destructor  Destroy;override;

        procedure   SaveToStream(Stream:TStream);
        procedure   LoadFromStream(Stream:TStream);

        procedure   SaveToFile(const FileName:string);
        procedure   LoadFromFile(const FileName:string);

        procedure   ValidateSchema(aDataBase:TFIBDataBase;Proc:TpFIBAcceptCacheSchema);

        function    FindTableInfo(const ADBName,ATableName:string):TpFIBTableInfo;
        function    GetTableInfo(aDataBase:TFIBDataBase;const ATableName:string):TpFIBTableInfo;
        function    GetFieldInfo(aDataBase:TFIBDataBase;
                       const ATableName,AFieldName:string
                    ):TpFIBFieldInfo;

        procedure   Clear;
        procedure   ClearForDataBase(aDatabase:TFIBDataBase);
        procedure   ClearForTable(const TableName:string)   ;
       end;


       TpDataSetInfo= class
       private
         FDBName:string;
         FSelectSQL:TStrings;
         FInsertSQL:TStrings;
         FUpdateSQL:TStrings;
         FDeleteSQL:TStrings;
         FRefreshSQL:TStrings;
         FKeyField:string;
         FGeneratorName:string;
         FDescription  :string;
       public
        constructor Create(DataSet:TpFIBDataSet);
        destructor  Destroy;override;
       end;

       TpDataSetInfoCollect= class
       private
        FListDataSetInfo:TStringList;
       public
        constructor Create;//(AOwner:TComponent);// override;
        destructor  Destroy;override;
        function    FindDataSetInfo(DataSet:TpFIBDataSet):TpDataSetInfo;
        function    GetDataSetInfo(DataSet:TpFIBDataSet):TpDataSetInfo;
        function    LoadDataSetInfo(DataSet:TpFIBDataSet):boolean;
        procedure   ClearDSInfo(DataSet:TpFIBDataSet);
       end;

       TpStoredProcCollect= class
       private
        FStoredProcNames:TStringList;
        FSPParamTxt:TStringList;
        function  IndexOfSP(DB:TFIBDatabase;const SPName:string;
         ForceReQuery:boolean):integer;
        function  GetParamsText(DB:TFIBDatabase;const SPName:string):string;
       public
        constructor Create;
        destructor  Destroy;override;
        function    GetExecProcTxt(DB:TFIBDatabase;
         const SPName:string;   ForceReQuery:boolean
        ):string;
        procedure   ClearSPInfo(DB:TFIBDatabase);
       end;


var
     ListTableInfo  :TpFIBTableInfoCollect;
     ListDataSetInfo:TpDataSetInfoCollect;
     ListSPInfo     :TpStoredProcCollect;

// Manage Developer Info tables
function    ExistFRepositaryTable(DB:TFIBDatabase):boolean;
procedure   CreateFRepositaryTable(DB:TFIBDatabase);
procedure   CreateDRepositaryTable(DB:TFIBDatabase);
function    ExistDRepositaryTable(DB:TFIBDatabase):boolean;
function    ExistBooleanDomain(DB:TFIBDatabase):boolean;

function    SaveFIBDataSetInfo(DataSet:TpFibDataSet):boolean;

// Routine function

function  GetFieldInfos(Field:TField):TpFIBFieldInfo;
function  GetOtherFieldInfo(Field:TField;const InfoName:string):string;

procedure   Update1RepositaryTable(Tr:TFIBTransaction);

implementation

uses StrUtil,FIBConsts,pFIBCacheQueries;

var ListOfDataBases:TStringList;

const
  SDefer='@FIB_DEFERRED';

function ExistRepositaryTable(DB:TFIBDatabase;Kind:byte):boolean;
var aTransaction:TFibTransaction;
    qry:TFIBQuery;
    Index:integer;

function RepositaryIsRegistered:boolean;
begin
 Index:=ListOfDataBases.IndexOfObject(DB);
 if Index>-1 then
 begin
   Result:=Pos(IntToStr(Kind),ListOfDataBases[Index])>0;
 end
 else
   Result:=false;
end;

begin
 Result:=RepositaryIsRegistered;
 if Result or (Index>-1) then Exit;

 aTransaction:=TFibTransaction.Create(nil);
 aTransaction.DefaultDatabase:=DB;
 qry:=TFIBQuery.Create(nil);
 with qry do
 try
  ParamCheck:=true;
  Database:=DB;  Transaction:=aTransaction;
  SQL.Text:='select COUNT(RDB$RELATION_NAME) '+
                    'from RDB$RELATIONS where RDB$FLAGS = 1 '+
                    'and RDB$RELATION_NAME=?RT';
  aTransaction.StartTransaction;
  Params[0].asString:='FIB$FIELDS_INFO';
  ExecQuery;
  with ListOfDataBases do
   if Fields[0].asInteger>0 then
    Index:=AddObject('1',DB)
   else
    Index:=AddObject('0',DB);

  Close;
  Params[0].asString:='FIB$DATASETS_INFO';
  ExecQuery;
  if Fields[0].asInteger>0 then
   ListOfDataBases[Index]:=ListOfDataBases[Index]+'2';
  Result:=RepositaryIsRegistered;
 finally
  aTransaction.Free;
  Free;
 end;
end;

function ExistDRepositaryTable(DB:TFIBDatabase):boolean;
begin
 Result:=ExistRepositaryTable(DB,2)
end;

function ExistBooleanDomain(DB:TFIBDatabase):boolean;
var
    Transaction:TFibTransaction;
    qry:TFIBQuery;
begin

 qry:=TFIBQuery.Create(nil);
 Transaction:=TFibTransaction.Create(nil);
 try
  Transaction.DefaultDatabase:=DB;
  qry.Transaction:=Transaction;
  with qry,qry.SQL do
  begin
   ParamCheck:=false;
   Database:=DB;
   ParamCheck:=false;
   Transaction.StartTransaction;
   Text:=
       'Select Count(*) FROM RDB$FIELDS FLD ' +
       'WHERE FLD.RDB$FIELD_NAME = ''FIB$BOOLEAN''' ;
   ExecQuery;
   Result:=qry.Fields[0].asInteger<>0;
  end
 finally
  Transaction.Free;
  qry.Free;
 end;
end;

function ExistFRepositaryTable(DB:TFIBDatabase):boolean;
begin
 Result:=ExistRepositaryTable(DB,1)
end;

procedure   Update1RepositaryTable(Tr:TFIBTransaction);
var qry:TFIBQuery;
begin
 qry:=TFIBQuery.Create(nil);
 with qry,qry.SQL do
 try
  Database:=Tr.DefaultDatabase;  Transaction:=Tr;
  ParamCheck:=false;
  if not Tr.InTransaction then Tr.StartTransaction;
  try
   Text:='CREATE GENERATOR FIB$FIELD_INFO_VERSION';
   ExecQuery;
  except
  end;
  try
   Text:='ALTER TABLE FIB$FIELDS_INFO ADD DISPLAY_WIDTH INTEGER DEFAULT 0';
   ExecQuery;
  except
  end;
  try
   Text:='ALTER TABLE FIB$FIELDS_INFO ADD FIB$VERSION INTEGER';
   ExecQuery;
  except
  end;

  try
   Text:=
   'CREATE TRIGGER FIB$FIELDS_INFO_BI FOR FIB$FIELDS_INFO '+
   'ACTIVE BEFORE INSERT POSITION 0 as '+#13#10+
   'begin '+#13#10+
     'new.fib$version=gen_id(fib$field_info_version,1);'+#13#10+
   'end';
   ExecQuery;
  except
  end;
  try
   Text:=
   'CREATE TRIGGER FIB$FIELDS_INFO_BU FOR FIB$FIELDS_INFO '+
   'ACTIVE BEFORE UPDATE POSITION 0 as '+#13#10+
   'begin '+#13#10+
     'new.fib$version=gen_id(fib$field_info_version,1);'+#13#10+
   'end';
   ExecQuery;
  except
  end;
 finally
  Tr.CommitRetaining;
  Free
 end;
end;

procedure   DoCreateRepositaryTable(DB:TFIBDatabase;Kind:byte);
var Index:integer;
    Transaction:TFibTransaction;
    qry:TFIBQuery;
begin
 qry:=TFIBQuery.Create(nil);
 Transaction:=TFibTransaction.Create(nil);
 try
  Transaction.DefaultDatabase:=DB;
  qry.Database:=DB;  qry.Transaction:=Transaction;
  qry.ParamCheck:=false;
  Transaction.StartTransaction;
  with qry,qry.SQL do
  case Kind of
  1: begin

      if not ExistBooleanDomain(DB) then
      begin
       Text:=
       'CREATE DOMAIN FIB$BOOLEAN AS SMALLINT DEFAULT 1 NOT NULL CHECK (VALUE IN (0,1))';
       ExecQuery;
      end;
      Text:=
       'CREATE TABLE FIB$FIELDS_INFO (TABLE_NAME VARCHAR(31) NOT NULL, '+
       'FIELD_NAME VARCHAR(31) NOT NULL, '+
       'DISPLAY_LABEL VARCHAR(25),'+
       'VISIBLE FIB$BOOLEAN DEFAULT 1 NOT NULL,'+
       'DISPLAY_FORMAT VARCHAR(15),'+
       'EDIT_FORMAT VARCHAR(15),'+
       'TRIGGERED FIB$BOOLEAN DEFAULT 0 NOT NULL,'+
       'CONSTRAINT PK_FIB$FIELDS_INFO PRIMARY KEY (TABLE_NAME, FIELD_NAME))';
      ExecQuery;
      Text:='GRANT SELECT ON TABLE FIB$FIELDS_INFO TO PUBLIC';
      ExecQuery;
      Update1RepositaryTable(Transaction);      
     end;
  2: begin
      Text:=
       'CREATE TABLE FIB$DATASETS_INFO (DS_ID INTEGER NOT NULL, '+
       'DESCRIPTION VARCHAR(40),'+
       'SELECT_SQL BLOB sub_type 1 segment size 80,'+
       'UPDATE_SQL BLOB sub_type 1 segment size 80,'+
       'INSERT_SQL BLOB sub_type 1 segment size 80,'+
       'DELETE_SQL BLOB sub_type 1 segment size 80,'+
       'REFRESH_SQL BLOB sub_type 1 segment size 80,'+
       'NAME_GENERATOR VARCHAR(31), '+
       'KEY_FIELD VARCHAR(31),'+
       'CONSTRAINT PK_FIB$DATASETS_INFO PRIMARY KEY (DS_ID))';
      ExecQuery;
      Text:=
       'GRANT SELECT ON TABLE FIB$DATASETS_INFO TO PUBLIC';
      ExecQuery;
     end;
  end; //case
  Transaction.Commit;
 finally
  Transaction.Free;
  qry.Free;
 end;
 with ListOfDataBases do
 begin
  Index:=IndexOfObject(DB);
  if Index=-1 then
   AddObject(IntToStr(Kind),DB)
  else
   ListOfDataBases[Index]:=ListOfDataBases[Index]+IntToStr(Kind);
 end;
end;

procedure   CreateDRepositaryTable(DB:TFIBDatabase);
begin
 DoCreateRepositaryTable(DB,2)
end;

procedure   CreateFRepositaryTable(DB:TFIBDatabase);
begin
 DoCreateRepositaryTable(DB,1)
end;


function   ExchangeDataSetInfo(DataSet:TpFibDataSet;DS_Info:TpDataSetInfo):boolean;
var
    vTransaction:TFibTransaction;
    DI:TFIBDataset;
    vDescription:string;
begin
 Result:=true;
 with DataSet do
 begin
   if DataSet_ID= 0 then
    raise Exception.Create(Name + SCompEditDataSet_ID);
   if DataBase=nil then  raise Exception.Create(SDataBaseNotAssigned);
   if not ExistDRepositaryTable(DataSet.Database) then
     raise Exception.Create(SCompEditDataSetInfoNotExists);

   DI:=TFIBDataset.Create(nil);
   vTransaction:=TFibTransaction.Create(nil);
   try
    vTransaction.DefaultDatabase:=DataSet.DataBase;
    DI.Database:=DataSet.DataBase;  DI.Transaction:=vTransaction;
    vTransaction.StartTransaction;
    DI.SelectSQL.Text:=
     'SELECT * FROM FIB$DATASETS_INFO WHERE DS_ID='+IntToStr(DataSet_ID);

    if DS_Info<>nil then
    begin
     DI.Open;
     if DI.RecordCount=0 then Exit;
     with DS_Info,DI do
     begin
      FSelectSQL.Text :=FieldByName('SELECT_SQL').asString;
      FUpdateSQL.Text :=FieldByName('UPDATE_SQL').asString;
      FInsertSQL.Text :=FieldByName('INSERT_SQL').asString;
      FDeleteSQL.Text :=FieldByName('DELETE_SQL').asString;
      FRefreshSQL.Text:=FieldByName('REFRESH_SQL').asString;
      FKeyField       :=FieldByName('KEY_FIELD').asString;
      FGeneratorName  :=FieldByName('NAME_GENERATOR').asString;
      FDescription    :=FieldByName('DESCRIPTION')   .asString;
     end;
    end
    else begin
     DI.InsertSQL.Text:=
      'INSERT INTO FIB$DATASETS_INFO (DS_ID) VALUES('+IntToStr(DataSet_ID)+')';
     DI.UpdateSQL.Text:=
      'UPDATE FIB$DATASETS_INFO SET SELECT_SQL=?SELECT_SQL,'+
     'DESCRIPTION=?DESCRIPTION,'+
     'UPDATE_SQL=?UPDATE_SQL,'+
     'INSERT_SQL=?INSERT_SQL,'+
     'DELETE_SQL=?DELETE_SQL,'+
     'REFRESH_SQL=?REFRESH_SQL,'+
     'NAME_GENERATOR=?NAME_GENERATOR,'+
     'KEY_FIELD=?KEY_FIELD '+
     'WHERE DS_ID='+IntToStr(DataSet_ID)
     ;
     DI.Open;

      if DI.RecordCount=0  then
      begin
       DI.QInsert.ExecQuery;
       DI.Close;DI.Open;
       if DI.RecordCount=0  then raise Exception.Create(SCompEditUnableInsertInfoRecord);
      end;
      vDescription:=DI.FieldByName('DESCRIPTION')   .asString;
      Result:=false;
      if not InputQuery(SCompEditSaveDataSetProperty, SCompEditDataSetDesc, vDescription
      ) then Exit;
      DI.Edit;
      DI.FieldByName('SELECT_SQL') .asString :=SelectSQL.Text;
      DI.FieldByName('INSERT_SQL') .asString :=InsertSQL.Text;
      DI.FieldByName('DELETE_SQL') .asString :=DeleteSQL.Text;
      DI.FieldByName('UPDATE_SQL') .asString :=UpdateSQL.Text;
      DI.FieldByName('REFRESH_SQL').asString :=RefreshSQL.Text;
      DI.FieldByName('KEY_FIELD')  .asString :=AutoUpdateOptions.KeyFields;
      DI.FieldByName('NAME_GENERATOR').asString  :=AutoUpdateOptions.GeneratorName;
      DI.FieldByName('DESCRIPTION')   .asString  :=vDescription;
      DI.Post;
    end;
    vTransaction.Commit;
    Result:=true;
   finally
    vTransaction.Free;
    DI.Free;
   end;
 end;
end;

function   SaveFIBDataSetInfo(DataSet:TpFibDataSet):boolean;
begin
 Result:=ExchangeDataSetInfo(DataSet,nil);
end;

function DBPrimaryKeyFields(const TableName:string;
  aTransaction:TFIBTransaction
 ):string;
var q:TpFIBQuery;
const
  SGetPrimary=   'select i.rdb$field_name '+
   'from    rdb$relation_constraints r, rdb$index_segments i '+
   'where   r.rdb$relation_name=:TN and '+
   'r.rdb$constraint_type=''PRIMARY KEY'' and '+
   'r.rdb$index_name=i.rdb$index_name ' +
   'order by i.rdb$field_position';

begin
  if (aTransaction=nil) or (aTransaction.DefaultDatabase=nil) then
  begin
   Result:=SDefer;
   Exit;
  end;
  q:=GetQueryForUse(aTransaction,SGetPrimary);
  Result:='';
  with q do  try
   ParamByName('TN').AsString:=TableName;
   if not Transaction.InTransaction then Transaction.StartTransaction;
   ExecQuery;
   Result:=Trim(Fields[0].asString); Next;
   while not eof do
   begin
    Result:=Result+';'+Trim(Fields[0].asString);
    Next
   end;
  finally
    FreeQueryForUse(q);
  end
 end;


constructor TpFIBFieldInfo.Create;
begin
  inherited Create;
  FWithAdditionalInfo:=false;
  FIsTriggered       :=false;
  FOtherInfo         :=TStringList.Create;
  FDefaultValueEmptyString := false;
  FDisplayWidth :=0;
end;

destructor  TpFIBFieldInfo.Destroy;
begin
 FOtherInfo.Free;
 inherited Destroy;
end;

procedure TpFIBFieldInfo.SaveToStream(Stream:TStream);
var L:integer;
    st:string;
begin
  with Stream do
  begin
   WriteBuffer(FIsComputed,SizeOf(boolean));
   WriteBuffer(FCanIncToWhereClause,SizeOf(boolean));
   WriteBuffer(FDefaultValueEmptyString,SizeOf(boolean));
   WriteBuffer(FWithAdditionalInfo,SizeOf(boolean));
   WriteBuffer(FVisible,SizeOf(boolean));
   WriteBuffer(FIsTriggered,SizeOf(boolean));
   WriteBuffer(FDisplayWidth,SizeOf(integer));

   L:=Length(FDefaultValue);
   WriteBuffer(L,SizeOf(integer));
   WriteBuffer(Pointer(FDefaultValue)^,L);


   L:=Length(FDomainName);
   WriteBuffer(L,SizeOf(integer));
   WriteBuffer(Pointer(FDomainName)^,L);

   L:=Length(FDisplayLabel);
   WriteBuffer(L,SizeOf(integer));
   WriteBuffer(Pointer(FDisplayLabel)^,L);

   L:=Length(FEditFormat);
   WriteBuffer(L,SizeOf(integer));
   WriteBuffer(Pointer(FEditFormat)^,L);

   L:=Length(FDisplayFormat);
   WriteBuffer(L,SizeOf(integer));
   WriteBuffer(Pointer(FDisplayFormat)^,L);
   st:=FOtherInfo.Text;
   L:=Length(st);
   WriteBuffer(L,SizeOf(integer));   
   if L>0 then
    WriteBuffer(Pointer(st)^,L);
  end;
end;

procedure TpFIBFieldInfo.LoadFromStream(Stream:TStream;FromBegin:boolean);
var L:integer;
   st:string ;
procedure RaizeErrStream;
begin
 raise Exception.Create(SCompEditFieldInfoLoadError);
end;

begin
  with Stream do
  begin
   if FromBegin then Seek(0,soFromBeginning);
   ReadBuffer(FIsComputed,SizeOf(boolean));
   ReadBuffer(FCanIncToWhereClause,SizeOf(boolean));
   ReadBuffer(FDefaultValueEmptyString,SizeOf(boolean));
   ReadBuffer(FWithAdditionalInfo,SizeOf(boolean));
   ReadBuffer(FVisible,SizeOf(boolean));
   ReadBuffer(FIsTriggered,SizeOf(boolean));
   ReadBuffer(FDisplayWidth,SizeOf(integer));

   ReadBuffer(L,SizeOf(Integer));
   SetLength(FDefaultValue,L);
   ReadBuffer(Pointer(FDefaultValue)^,L);

   ReadBuffer(L,SizeOf(Integer));
   SetLength(FDomainName,L);
   ReadBuffer(Pointer(FDomainName)^,L);

   ReadBuffer(L,SizeOf(Integer));
   SetLength(FDisplayLabel,L);
   ReadBuffer(Pointer(FDisplayLabel)^,L);

   ReadBuffer(L,SizeOf(Integer));
   SetLength(FEditFormat,L);
   ReadBuffer(Pointer(FEditFormat)^,L);

   ReadBuffer(L,SizeOf(Integer));
   SetLength(FDisplayFormat,L);
   ReadBuffer(Pointer(FDisplayFormat)^,L);

   ReadBuffer(L,SizeOf(Integer));
   if L>0 then
   begin
    SetLength(st,L);
    ReadBuffer(Pointer(st)^,L);
    FOtherInfo.Text:=st;
   end;
  end
end;

// TpFIBTableInfo
constructor TpFIBTableInfo.Create;
begin
 inherited Create;
 FFieldList:=TStringList.Create;
 FInternalTransaction:=TpFIBTransaction.Create(nil);

end;

procedure   TpFIBTableInfo.FillInfo
                 (aDataBase:TFIBDataBase;const ATableName:string);

begin
 inherited Create;
 FTableName:=ATableName;
 FDBName   :=aDataBase.DBName;
 FInternalTransaction.DefaultDatabase:=aDatabase;
 FPrimaryKeyFields:=SDefer;
{ FPrimaryKeyFields:=  DBPrimaryKeyFields(FTableName,
                       aDatabase,FInternalTransaction
                      );}
 GetInfoFields(FTableName,FInternalTransaction);
 if FInternalTransaction.InTransaction then
  FInternalTransaction.Commit;
end;



destructor TpFIBTableInfo.Destroy;//override;
begin
 ClearFieldList;
 FFieldList.Free;
 if FInternalTransaction.InTransaction then
  FInternalTransaction.Commit;
 FInternalTransaction.Free; 
 inherited Destroy;
end;

procedure   TpFIBTableInfo.SaveToStream(Stream:TStream);
var i,L:integer;
    fn:string;
begin
  with Stream do
  begin
   L:=Length(FDBName);
   WriteBuffer(L,SizeOf(Integer));
   WriteBuffer(Pointer(FDBName)^,L);
   L:=Length(FTableName);
   WriteBuffer(L,SizeOf(Integer));
   WriteBuffer(Pointer(FTableName)^,L);

   WriteBuffer(FFormatNumber,SizeOf(Integer)); //    
   WriteBuffer(FFIVersion   ,SizeOf(Integer)); //   FIB$FIELD_INFO

   L:=Length(FPrimaryKeyFields);
   WriteBuffer(L,SizeOf(Integer));
   WriteBuffer(Pointer(FPrimaryKeyFields)^,L);
   L:=FFieldList.Count;
   WriteBuffer(L,SizeOf(Integer));
   for i := 0  to Pred(FFieldList.Count) do
   begin
    fn:=FFieldList[i];
    L:=Length(fn);
    WriteBuffer(L,SizeOf(Integer));
    WriteBuffer(Pointer(fn)^,L);
    TpFIBFieldInfo(FFieldList.Objects[i]).SaveToStream(Stream)
   end;
  end;
end;

procedure   TpFIBTableInfo.LoadFromStream(Stream:TStream;FromBegin:boolean);
var i,fc,L:integer;
    fn:string;
    fi:TpFIBFieldInfo;
procedure RaizeErrStream;
begin
 raise Exception.Create(SCompEditFieldInfoLoadError);
end;

begin
  with Stream do
  begin
   if FromBegin then Seek(0,soFromBeginning);

   ReadBuffer(L,SizeOf(Integer));
   SetLength(FDBName,L);
   ReadBuffer(Pointer(FDBName)^,L);

   ReadBuffer(L,SizeOf(Integer));
   SetLength(FTableName,L);
   ReadBuffer(Pointer(FTableName)^,L);

    ReadBuffer(FFormatNumber,SizeOf(Integer));
    ReadBuffer(FFIVersion   ,SizeOf(Integer)); //   FIB$FIELD_INFO

   ReadBuffer(L,SizeOf(Integer));
   SetLength(FPrimaryKeyFields,L);
   ReadBuffer(FPrimaryKeyFields[1],L);
   ReadBuffer(fc,SizeOf(Integer));

   ClearFieldList;
   for i:=0 to Pred(fc) do
   begin
    ReadBuffer(L,SizeOf(Integer));
    SetLength(fn,L);
    ReadBuffer(fn[1],L);
    fi:=TpFIBFieldInfo.Create;
    fi.LoadFromStream(Stream,false);
    FFieldList.AddObject(fn,fi);
   end;
  end;
end;

function TpFIBTableInfo.GetPrimaryKeyFields: string;
begin
  if FPrimaryKeyFields=SDefer then
   FPrimaryKeyFields:=DBPrimaryKeyFields(FTableName,FInternalTransaction);
  Result:=FPrimaryKeyFields
end;


function TpFIBTableInfo.FieldInfo(const FieldName:string):TpFIBFieldInfo;
var Index:integer;
begin
 Result:=nil;
 Index:=FFieldList.IndexOf(Trim(ReplaceCIStr(FieldName,'"','')));
 if Index>-1 then Result:=TpFIBFieldInfo(FFieldList.Objects[Index]);
end;

procedure TpFIBTableInfo.ClearFieldList;
var i:integer;
begin
 for i:=Pred(FFieldList.Count) downto 0 do
  FFieldList.Objects[i].Free;
 FFieldList.Clear;
end;

procedure TpFIBTableInfo.GetAdditionalInfo(FromQuery:TpFIBDataset;
          const FieldName:string;ToFieldInfo:TpFIBFieldInfo);
var i:integer;
begin
 try
  with FromQuery,ToFieldInfo do       //
    if (FN('FIELD_NAME').asString=FieldName) or
     Locate('FIELD_NAME',TrimRight(FieldName),[]) then
    begin
     FWithAdditionalInfo:=true;
     for i:=0 to Pred(FieldCount) do
     with Fields[i] do
     begin
      if StringInArray(FieldName,['TABLE_NAME','FIELD_NAME','FIB$VERSION']) then
       Continue
      else
      case FieldName[1] of
       'D':
         case Length(FieldName) of
          13:if EquelNames(false,FieldName,'DISPLAY_LABEL')then
              FDisplayLabel:=asString
             else
             if EquelNames(false,FieldName,'DISPLAY_WIDTH') then
              FDisplayWidth:=asInteger
             else
              FOtherInfo.Values[FieldName]:=asString;
          14:if EquelNames(false,FieldName,'DISPLAY_FORMAT') then
              FDisplayFormat:=asString
             else
              FOtherInfo.Values[FieldName]:=asString;
         else
           FOtherInfo.Values[FieldName]:=asString;
         end;
      'E':if EquelNames(false,FieldName,'EDIT_FORMAT') then
           FEditFormat  :=asString
          else
           FOtherInfo.Values[FieldName]:=asString;
      'V':if EquelNames(false,FieldName,'VISIBLE') then
           FVisible     :=asInteger=1
          else
           FOtherInfo.Values[FieldName]:=asString;
      'T':if EquelNames(false,FieldName,'TRIGGERED') then
           FIsTriggered :=asInteger=1
          else
           FOtherInfo.Values[FieldName]:=asString;
      else
       FOtherInfo.Values[FieldName]:=asString;
      end
     end;  
    end
  except
  end
end;

const FormatNumberSQL=
  'select a1.RDB$RELATION_ID VER  from RDB$RELATIONS a1 '+
  'where a1.RDB$SYSTEM_FLAG = 0  and a1.rdb$relation_name=?TN '+
  'and not  a1.rdb$view_blr is null '+
  'union '+
  'select RDB$FORMAT from RDB$RELATIONS R '+
  'where R.RDB$SYSTEM_FLAG = 0    and R.rdb$relation_name=?TN '+
  'and  R.rdb$view_blr is null';

FormatNumbersSQL=
  'select a1.rdb$relation_name,a1.RDB$RELATION_ID VER  from RDB$RELATIONS a1 '+
  'where a1.RDB$SYSTEM_FLAG = 0  '+
  'and not  a1.rdb$view_blr is null '+
  'union '+
  'select R.rdb$relation_name,RDB$FORMAT from RDB$RELATIONS R '+
  'where R.RDB$SYSTEM_FLAG = 0    '+
  'and  R.rdb$view_blr is null   order by 1';

FI_VersionSQL='Select Max(fib$version) From FIB$FIELDS_INFO Where table_name=?TN';
FI_VersionsSQL=
 'Select table_name,Max(fib$version) From FIB$FIELDS_INFO group by table_name order by 1';

function  TpFIBTableInfo.IsActualInfo(aDataBase:TFIBDataBase) :boolean;
var q:TpFIBQuery;
begin
    if aDataBase.DBName<>FDBName then
    begin
     Result:=False; Exit; //    .    
    end
    else
     Result:=True; 
    with FInternalTransaction do
     if DefaultDatabase<>aDataBase  then
     begin
      if InTransaction then  Commit;
      DefaultDatabase:=aDataBase;
     end;


    if ExistFRepositaryTable(aDataBase) then
    begin
      q:=GetQueryForUse(FInternalTransaction,  FI_VersionSQL );
      with q do  try
       Options:=[qoStartTransaction];
       Params[0].asString:=TableName;
       try
        ExecQuery;
       except
        Result:=True;
        Update1RepositaryTable(FInternalTransaction);
        if FInternalTransaction.InTransaction then
         FInternalTransaction.Commit;
        Exit;
       end;
       Result:=FFIVersion=q.Fields[0].asInteger;
      finally
        Close;
        FreeQueryForUse(q);
      end;
    end;

    if  Result and not (EquelNames(false,FTableName,'ALIAS')) then
    begin
      q:=GetQueryForUse(FInternalTransaction,  FormatNumberSQL  );
      with q do  try
       Options:=[qoStartTransaction];
       Params[0].asString:=TableName;
       ExecQuery;
       Result:=FFormatNumber=q.Fields[0].asInteger;
      finally
        Close;
        FreeQueryForUse(q);
      end;
    end;
    if FInternalTransaction.InTransaction then
     FInternalTransaction.Commit
end;

procedure TpFIBTableInfo.GetInfoFields(const TableName:string;
  aTransaction:TFIBTransaction
);
var q:TpFIBQuery;
    fi:TpFIBFieldInfo;
    ExistAdInfo:boolean;
    d:TpFIBDataSet;
begin
  d:=nil;
  ExistAdInfo:=ExistFRepositaryTable(aTransaction.DefaultDatabase);
  if ExistAdInfo then
  try
     q:=GetQueryForUse(aTransaction,  FI_VersionSQL );
     with q do
     try
      Options:=[qoStartTransaction];
      Params[0].asString:=TableName;
      try
       q.ExecQuery;
       FFIVersion:=q.Fields[0].asInteger;
      except
       Update1RepositaryTable(aTransaction);
       FFIVersion:=0;
      end;
     finally
       Close;
       FreeQueryForUse(q);
     end;


    d:=TpFIBDataSet.Create(nil);
    d.DataBase:=aTransaction.DefaultDatabase;
    d.Transaction:=aTransaction;
    d.SelectSQL.Text:='Select * from FIB$FIELDS_INFO where TABLE_NAME='''+
                    TableName+''' ORDER BY FIELD_NAME';
    d.PrepareOptions:=[];
    d.Open;
  except
   d.Free;
   d:=nil;
   ExistAdInfo:=false;
  end;
  if EquelNames(false,TableName,'ALIAS') then
  begin
   if not ExistAdInfo then Exit;
   with d do try
    while not eof do
    begin
     fi:=TpFIBFieldInfo.Create;
     GetAdditionalInfo(d,Trim(Fields[1].asString),fi);
     FFieldList.AddObject(Trim(Fields[1].asString),fi);
     Next
    end;
    finally
     Free;
    end
  end
  else begin
    q:=GetQueryForUse(aTransaction,  FormatNumberSQL  );
    with q do  try
     Options:=[qoStartTransaction];
     Params[0].asString:=TableName;
     q.ExecQuery;
     FFormatNumber:=q.Fields[0].asInteger;
    finally
      Close;
      FreeQueryForUse(q);
    end;


    q:=GetQueryForUse(aTransaction,
         'Select R.RDB$FIELD_NAME,R.RDB$FIELD_SOURCE,F.RDB$COMPUTED_BLR, '+
     'R.RDB$DEFAULT_SOURCE DS,F.RDB$DEFAULT_SOURCE DS1, '+
     'F.RDB$FIELD_TYPE, F.RDB$DIMENSIONS '+
     'from  RDB$RELATION_FIELDS R '+
     'JOIN RDB$FIELDS F ON (R.RDB$FIELD_SOURCE = F.RDB$FIELD_NAME) '+
     'where  R.RDB$RELATION_NAME=:TN '+
     'order by R.RDB$FIELD_POSITION'
    );
    with q do  try
     Options:=[qoStartTransaction,qoTrimCharFields];
     Params[0].asString:=TableName;
     ExecQuery;
     if q.eof then
     begin
      Close;
      FreeQueryForUse(q);
      q:=GetQueryForUse(aTransaction,
       'Select RDB$PARAMETER_NAME,0,0,0,0, '+
       'RDB$PARAMETER_TYPE,0 '+
       'from RDB$PROCEDURE_PARAMETERS '+
       'WHERE RDB$PROCEDURE_NAME=:PN '+
       'AND RDB$PARAMETER_TYPE=1 '
      );
      Params[0].asString:=TableName;
      ExecQuery;
     end;
     ClearFieldList;
     while not eof do
     begin
      fi:=TpFIBFieldInfo.Create;
      with fi do
      begin
       FDomainName:=Fields[1].asString;
       FIsComputed:=not Fields[2].IsNull;
       FDefaultValue:=Trim(BlobAsString('DS'));
       if FDefaultValue='' then
        FDefaultValue:=Trim(BlobAsString('DS1'));
       if FDefaultValue<>'' then
       begin
        FDefaultValue:=Trim(Copy(FDefaultValue,8,MaxInt)); //Cut "DEFAULT"
        if FDefaultValue[1]in['''','"'] then
         FDefaultValue:=Copy(FDefaultValue,2,Length(FDefaultValue)-2);
         if FDefaultValue='' then
            FDefaultValueEmptyString := true;
         //Cut Leading Quote
       end;
       FCanIncToWhereClause:=
        (Fields[5].asInteger<>261)and(Fields[5].asInteger<>9)
        and(Fields[6].asInteger=0);
       if ExistAdInfo then
        GetAdditionalInfo(d,Trim(Fields[0].asString),fi);
      end;
      FFieldList.AddObject(Trim(Fields[0].asString),fi);
      Next
     end;
    finally
     if ExistAdInfo then d.Free;
     FreeQueryForUse(q)
    end
  end;
end;

//TpFIBTableInfoCollect
constructor TpFIBTableInfoCollect.Create(AOwner:TComponent);
begin
 inherited Create(AOwner);
 FListTabInfo:=TStringList.Create;
 FListTabInfo.Sorted:=true;
 FListTabInfo.Duplicates:=dupAccept;
 FDBNames    :=TStringList.Create;
 FDBNames    .Sorted:=true;
 FDBNames    .Duplicates:=dupAccept;
 FInternalTransaction:=TpFIBTransaction.Create(nil);
end;

destructor  TpFIBTableInfoCollect.Destroy;//override;
begin
 Clear;
 FListTabInfo.Free;
 FDBNames    .Free;
 FInternalTransaction.Free;
 inherited Destroy;
end;

procedure TpFIBTableInfoCollect.Notification(AComponent: TComponent; Operation: TOperation);
var Index:integer;
begin
 if Operation=opRemove then
  if AComponent is TFIBDataBase then
  begin
    ClearForDataBase(TFIBDataBase(AComponent));
    Index:=ListOfDataBases.IndexOfObject(AComponent);
    if Index<>-1 then ListOfDataBases.Delete(Index)
  end;
 inherited Notification(AComponent,Operation);
end;

const
 Signature='FIB$MCF';

procedure   TpFIBTableInfoCollect.SaveToStream(Stream:TStream);
var i,tc,L:integer;
    Version:integer;
    tn:string;
begin
  with Stream do
  begin
   Seek(0,soFromBeginning);
   L:=Length(Signature);
   tn:=Signature;
   WriteBuffer(Pointer(tn)^,L);
   Version:=3;
   WriteBuffer(Version,SizeOf(Integer));
   tc:=FListTabInfo.Count;
   WriteBuffer(tc,SizeOf(Integer));
   for i :=0  to Pred(tc) do
   begin
    tn:=FListTabInfo[i];
    L:=Length(tn);
    WriteBuffer(L,SizeOf(Integer));
    WriteBuffer(Pointer(tn)^,L);
    TpFIBTableInfo(FListTabInfo.Objects[i]).SaveToStream(Stream);
   end;
  end;
end;

procedure   TpFIBTableInfoCollect.LoadFromStream(Stream:TStream);
var i,tc,L:integer;
    Version:integer;
    tn:string;
    ti:TpFIBTableInfo;
begin
  Clear;
  with Stream do
  begin
   Seek(0,soFromBeginning);
   L:=Length(Signature);
   SetLength(tn,L);
   ReadBuffer(Pointer(tn)^,L); // Signature
   if tn<>Signature then Exit;
   ReadBuffer(Version,SizeOf(Integer));
   if Version<3 then  Exit;  // old style cache
   ReadBuffer(tc,SizeOf(Integer));
   for i:=0 to Pred(tc) do
   begin
    ReadBuffer(L,SizeOf(Integer));
    SetLength(tn,L);
    ReadBuffer(Pointer(tn)^,L); // TableName
    ti:=TpFIBTableInfo.Create;
    ti.LoadFromStream(Stream,false);
    FListTabInfo.AddObject(tn,ti)
   end;
  end;
end;

procedure   TpFIBTableInfoCollect.SaveToFile(const FileName:string);
var
  Stream: TFileStream;
begin
 try
  Stream := TFileStream.Create(FileName, fmCreate);
  try
   SaveToStream(Stream);
  finally
   Stream.Free;
  end;
 except
 end; 
end;

procedure   TpFIBTableInfoCollect.LoadFromFile(const FileName:string);
var
  Stream: TFileStream;
begin
  if not FileExists(FileName) then Exit;
  Stream := TFileStream.Create(FileName, fmOpenRead);
  try
   LoadFromStream(Stream);
  finally
   Stream.Free;
  end;
end;

{$WARNINGS OFF}
procedure   TpFIBTableInfoCollect.ValidateSchema(aDataBase:TFIBDataBase;Proc:TpFIBAcceptCacheSchema);
const MaxCountSeparate =10;
var i:integer;
    DelCache,rt:boolean;
    FiDs,TfDs: TpFIBDataSet;
//  FiDs -     FIB$FIELDS_INFO
//  TfDs -  o  
begin
 with FListTabInfo do
 begin
   if (Count<=MaxCountSeparate) or   Assigned(Proc) then
    rt:=False
   else
   begin
     TfDs:=TpFIBDataSet.Create(nil);
     with TfDs do
     begin
      Database:=aDataBase;
      Options :=[poStartTransaction,poTrimCharFields];
      Transaction:= FInternalTransaction;
      FInternalTransaction.DefaultDatabase := aDataBase;
      SelectSQL.Add(FormatNumbersSQL);
      PrepareOptions:=[];
      Open;
     end;
     rt:=
      TfDs.ExtLocate('RDB$RELATION_NAME','FIB$FIELDS_INFO',[eloInSortedDS]) ;
    if rt then
    begin
     FiDs:=TpFIBDataSet.Create(nil);
     with FiDs do
     begin
      Database:=aDataBase;
      Options :=[poStartTransaction,poTrimCharFields];
      Transaction:= FInternalTransaction;
      SelectSQL.Add(FI_VersionsSQL);
      PrepareOptions:=[];
      Open;
     end;
    end;
   end;
   try
     for i:=Count-1 downto 0 do
     begin
      if Assigned(Proc) then
      begin
       DelCache:=False;
       Proc(FListTabInfo[i],DelCache);
       DelCache:= not DelCache
      end
      else
      if Count>MaxCountSeparate then
      begin
        DelCache:=(not EquelNames(false,FListTabInfo[i],'ALIAS')) and  not (
         TfDs.ExtLocate('RDB$RELATION_NAME',FListTabInfo[i],[eloInSortedDS]) and
         (TfDs.Fields[1].asInteger=TpFIBTableInfo(Objects[i]).FFormatNumber));
        if rt and not DelCache  then
         DelCache:= not (
          FiDs.ExtLocate('TABLE_NAME',FListTabInfo[i],[eloInSortedDS]) and
          (FiDs.Fields[1].asInteger=TpFIBTableInfo(Objects[i]).FFIVersion));
      end
      else
       DelCache:=not TpFIBTableInfo(Objects[i]).IsActualInfo(aDataBase);
      if DelCache then
      begin
       Objects[i].Free;
       Delete(i)
      end;
     end;
   finally
    if FInternalTransaction.InTransaction then
     FInternalTransaction.Commit;
    TfDs.Free;
    if rt then  FiDs.Free;
   end;
 end;
end;

{$WARNINGS ON}

function    TpFIBTableInfoCollect.FindTableInfo
 (const ADBName,ATableName:string):TpFIBTableInfo;
var Index: Integer;
    i:integer;
begin
 Result:=nil;
 with FDBNames do
 if not Find(aDBName, Index) then
 begin
  Index:=Add(aDBName);
 end;
 with FListTabInfo do
  if Find(ATableName, Index) then
   if aDBName=TpFIBTableInfo(Objects[Index]).FDBName then
    Result:=TpFIBTableInfo(Objects[Index])
   else // other Database
    for i:=Index+1 to Pred(Count) do
     if (ATableName=Strings[i]) and
        (aDBName=TpFIBTableInfo(Objects[i]).FDBName)
     then
     begin
      Result:=TpFIBTableInfo(Objects[i]);
      Break;
     end
     else
     if (ATableName<>Strings[i]) then Exit;
end;

function    TpFIBTableInfoCollect.GetTableInfo(aDataBase:TFIBDataBase;
 const ATableName:string
):TpFIBTableInfo;
var TmpTableName: string;
begin
 TmpTableName := Trim(ReplaceCIStr(ATableName,'"',''));
 Result:=FindTableInfo(aDataBase.DBName,TmpTableName);
 if Result=nil then
 begin
  Result:=TpFIBTableInfo.Create;
  Result.FillInfo(aDataBase,TmpTableName);
  FListTabInfo.AddObject(TmpTableName,Result);
//  aDatabase.FreeNotification(Self)
 end;
end;

function  TpFIBTableInfoCollect.GetFieldInfo(aDataBase:TFIBDataBase;
                       const ATableName,AFieldName:string
                    ):TpFIBFieldInfo;
var ti:TpFIBTableInfo;
begin
  Result:=nil;
  ti:=GetTableInfo(aDataBase,ATableName);
  if ti=nil then Exit;
  Result:=ti.FieldInfo(AFieldName);
end;


procedure   TpFIBTableInfoCollect.Clear;
var i:integer;
begin
 for i:=0 to Pred(FListTabInfo.Count) do
  FListTabInfo.Objects[i].Free;
 FListTabInfo.Clear;
end;

procedure   TpFIBTableInfoCollect.ClearForDataBase(aDatabase:TFIBDataBase);
var i:integer;
begin
 if aDatabase=nil then Exit;
 with FListTabInfo do
 for i:=Pred(Count) downto 0 do
  if (TpFIBTableInfo(Objects[i]).FDBName=aDatabase.DBName)  then
  begin
   Objects[i].Free;
   Delete(i)
  end;
end;

procedure   TpFIBTableInfoCollect.ClearForTable(const TableName:string);
var Index: Integer;
begin
 with FListTabInfo do
  if Find(TableName, Index) then
  begin
   while (Index<Count) and
    (TpFIBTableInfo(Objects[Index]).FTableName=TableName)
   do
   begin
    Objects[Index].Free;    Delete(Index)
   end;
  end;
end;

// DataSets info

constructor TpDataSetInfo.
 Create(DataSet:TpFIBDataSet);
begin
 if (DataSet=nil) or (DataSet.DataBase=nil)
 or (DataSet.DataBase.DBName='')
 then
  Raise Exception.Create(SCantGetInfo+IntToStr(DataSet.DataSet_ID)+#13#10+
   SDataBaseNotAssigned
  );
 inherited Create;
 FDBName:=DataSet.DataBase.DBName;
 FSelectSQL:=TStringList.Create;
 FInsertSQL:=TStringList.Create;
 FUpdateSQL:=TStringList.Create;
 FDeleteSQL:=TStringList.Create;
 FRefreshSQL:=TStringList.Create;
 FKeyField:='';
 FGeneratorName:='';
end;

destructor  TpDataSetInfo.Destroy;
begin
 FSelectSQL.Free;
 FInsertSQL.Free;
 FUpdateSQL.Free;
 FDeleteSQL.Free;
 FRefreshSQL.Free;
 inherited Destroy;
end;

constructor TpDataSetInfoCollect.Create;//(AOwner:TComponent);
begin
 inherited Create;
 FListDataSetInfo:=TStringList.Create;
 FListDataSetInfo.Sorted:=true;
 FListDataSetInfo.Duplicates:=dupAccept
end;

destructor  TpDataSetInfoCollect.Destroy;
var i:integer;
begin
 with FListDataSetInfo do
 begin
  for i:=0 to Pred(Count) do Objects[i].Free;
  Free
 end;
 inherited;
end;


function    TpDataSetInfoCollect.FindDataSetInfo(DataSet:TpFIBDataSet):TpDataSetInfo;
var ID_Str:string;
    i,Index:integer;
begin
  Result:=nil;
  if (DataSet=nil) or (DataSet.DataSet_ID=0) then Exit;
  ID_Str:=IntToStr(DataSet.DataSet_ID);
 Result:=nil;
 with FListDataSetInfo do
  if Find(ID_Str, Index) then
   if DataSet.Database.DBName=TpDataSetInfo(Objects[Index]).FDBName then
    Result:=TpDataSetInfo(Objects[Index])
   else
    for i:=Index+1 to Pred(Count) do
     if (ID_Str=Strings[i]) and
        (DataSet.Database.DBName=TpDataSetInfo(Objects[i]).FDBName)
     then
     begin
      Result:=TpDataSetInfo(Objects[i]);
      Break;
     end
     else
     if (ID_Str<>Strings[i]) then Exit;
end;

function TpDataSetInfoCollect.GetDataSetInfo(DataSet:TpFIBDataSet):TpDataSetInfo;
begin
 Result:=nil;
 if (DataSet=nil) or (DataSet.DataBase=nil) or (DataSet.DataSet_ID=0)
 then Exit;
 Result:=FindDataSetInfo(DataSet);
 if Result=nil then
 begin
  Result:=TpDataSetInfo.Create(DataSet);
  ExchangeDataSetInfo(DataSet,Result);
  FListDataSetInfo.AddObject(IntToStr(DataSet.DataSet_ID),Result);
 end;
end;

function TpDataSetInfoCollect.LoadDataSetInfo(DataSet:TpFIBDataSet):boolean;
var   DSI:TpDataSetInfo;
begin
  Result:=false;
  DSI:=GetDataSetInfo(DataSet);
  if DSI=nil then Exit;
  with DataSet do
  begin
   if (DSI.FSelectSQL.Count>0) and (not SelectSQL.Equals(DSI.FSelectSQL)) then
   begin
    if not Conditions.Applied or (Conditions.PrimarySQL<>DSI.FSelectSQL.Text) then
     SelectSQL.Assign(DSI.FSelectSQL);
   end;
   if (DSI.FInsertSQL.Count>0) and (not InsertSQL.Equals(DSI.FInsertSQL)) then
    InsertSQL.Assign(DSI.FInsertSQL);
   if (DSI.FUpdateSQL.Count>0) and (not UpdateSQL.Equals(DSI.FUpdateSQL)) then
    UpdateSQL.Assign(DSI.FUpdateSQL);
   if (DSI.FDeleteSQL.Count>0) and (not DeleteSQL.Equals(DSI.FDeleteSQL)) then
    DeleteSQL.Assign(DSI.FDeleteSQL);
   if (DSI.FRefreshSQL.Count>0)and(not RefreshSQL.Equals(DSI.FRefreshSQL))then
    RefreshSQL.Assign(DSI.FRefreshSQL);
   with AutoUpdateOptions do
   begin
    SelectGenID:=(DSI.FKeyField<>'') and (DSI.FGeneratorName<>'');
    KeyFields :=DSI.FKeyField;
    GeneratorName:=DSI.FGeneratorName;
   end;
   Description :=DSI.FDescription
  end;
end;

procedure TpDataSetInfoCollect.ClearDSInfo(DataSet:TpFIBDataSet);
var ID_Str:string;
    Index:integer;
begin
 if (DataSet=nil) or (DataSet.DataSet_ID=0) then Exit;
 ID_Str:=IntToStr(DataSet.DataSet_ID);
 with FListDataSetInfo do
  if Find(ID_Str, Index) then
   if DataSet.Database.DBName=TpDataSetInfo(Objects[Index]).FDBName then
   begin
        Objects[Index].Free;
        Delete(Index);
   end;
end;
//

constructor TpStoredProcCollect.Create;
begin
 inherited Create;
 FStoredProcNames:=TStringList.Create;
 FStoredProcNames.Sorted:=true;
 FStoredProcNames.Duplicates:=dupIgnore;
 FSPParamTxt    :=TStringList.Create;
end;

destructor  TpStoredProcCollect.Destroy;//override;
begin
 FStoredProcNames.Free;
 FSPParamTxt  .Free;
 inherited Destroy;
end;

function  TpStoredProcCollect.GetParamsText(DB:TFIBDatabase;
           const SPName:string
          ):string;
var
  Qry: TpFIBQuery;
  Trans: TpFIBTransaction;
  lSQLDA: TFIBXSQLDA;
begin
    begin
      Result := '';
      Qry := TpFIBQuery.Create(nil);
      Trans := TpFIBTransaction.Create(nil);
      with Qry,Trans do
      try
        Database := DB;
        DefaultDatabase := Database;
        Transaction := Trans;
        SQL.Text :=
         'SELECT RDB$PARAMETER_NAME, RDB$PARAMETER_TYPE FROM '+
         'RDB$PROCEDURE_PARAMETERS WHERE RDB$PROCEDURE_NAME = ' +
         '''' + FormatIdentifierValue(Database.SQLDialect, SPName) + '''' +
         'ORDER BY RDB$PARAMETER_NUMBER';
        StartTransaction;
        try
          ExecQuery;
          lSQLDA := Current;
          while not Qry.Eof do
          begin
            if (lSQLDA.ByName['RDB$PARAMETER_TYPE'].AsInteger = 0) then
            begin
              if (Result <> '') then
                Result := Result + ', ';
              Result := Result + '?' + FormatIdentifier(Database.SQLDialect,
Trim(lSQLDA.ByName['RDB$PARAMETER_NAME'].AsString));
            end;
            lSQLDA := Next;
          end;
          Close;
        finally
          Commit;
        end;
      finally
        Qry.Free;
        Trans.Free;
      end;
    end;
    if Result<>'' then Result:='('+Result+')'
end;

function TpStoredProcCollect.IndexOfSP(DB:TFIBDatabase;const SPName:string;
 ForceReQuery:boolean
):integer;
var RegSPName:string;
begin
 RegSPName:=DB.DBName+'||'+SPName;
 with FStoredProcNames do
 begin
   if ForceReQuery or not Find(RegSPName,Result) then
   begin
    Result:=Add(RegSPName);
    FSPParamTxt.Insert(Result,GetParamsText(DB,SPName))
   end;
 end;
end;

function StripQuote(const Value: string): string;
begin
  if (Value <> EmptyStr) and  (Value[1] in ['"', '''']) then
   Result := Copy(Value, 2, length(Value) - 2)
  else
   Result := Value;
end;


function TpStoredProcCollect.GetExecProcTxt(DB:TFIBDatabase;
         const SPName:string;   ForceReQuery:boolean
        ):string;

begin
  Result:='EXECUTE PROCEDURE ' + SPName+' '+
   FSPParamTxt[IndexOfSP(DB,StripQuote(SPName),ForceReQuery)]
end;


procedure   TpStoredProcCollect.ClearSPInfo(DB:TFIBDatabase);
var i,c:integer;
begin
   c:=Pred(FStoredProcNames.Count);
   for i:=c downto 0 do
    if Pos(DB.DBName,FStoredProcNames[i])=1 then
    begin
      FStoredProcNames.Delete(i);
      FSPParamTxt.Delete(i);
    end;
end;

// Routine function

function GetFieldInfos(Field:TField):TpFIBFieldInfo;
var d:TpFIBDataSet;
    RelTable,RelField:string;
begin
 Result:=nil;
 if not (Field.DataSet is TpFIBDataSet) then Exit;
 d:=TpFIBDataSet(Field.DataSet);
 with d do
 begin
  RelTable:=GetRelationTableName(Field);
  RelField:=GetRelationFieldName(Field);
  Result:=
   ListTableInfo.GetFieldInfo(DataBase,
       RelTable,RelField
   );
 end;
 if Result<>nil then Exit;
 
 RelTable:='ALIAS';
 RelField:=Field.FieldName;
 with d do
 Result:=
   ListTableInfo.GetFieldInfo(DataBase,
       RelTable,RelField
   );
end;

function  GetOtherFieldInfo(Field:TField;const InfoName:string):string;
var vFI:TpFIBFieldInfo;
begin
 Result:='';
 vFI   :=GetFieldInfos(Field);
 if vFI=nil then Exit;
 Result:=vFI.FOtherInfo.Values[InfoName]
end;


initialization
 ListTableInfo:=TpFIBTableInfoCollect.Create(nil);
 ListDataSetInfo:=TpDataSetInfoCollect.Create;
 ListOfDataBases:=TStringList.Create;
 ListSPInfo     :=TpStoredProcCollect.Create;
finalization
 ListOfDataBases.Free;
 ListDataSetInfo.Free;
 ListSPInfo.Free ;
 ListTableInfo.Free;
end.
