VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
END
Attribute VB_Name = "Variables"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = True
Attribute VB_PredeclaredId = False
Attribute VB_Exposed = True
Attribute VB_Ext_KEY = "SavedWithClassBuilder" ,"Yes"
Attribute VB_Ext_KEY = "Collection" ,"I500Variable"
Attribute VB_Ext_KEY = "Member0" ,"I500Variable"
Attribute VB_Ext_KEY = "Top_Level" ,"Yes"
'**
' A collection for the storage and retrieval of Variable Objects
'
' @describe
'   This class is designed to simplify the storage and  retrieval of Variable
'   objects.  Variable values may be stored and retrieved with the Value
'   properties or the backward compatible SetValue/GetValue methods.
'
'   References to the Variable objects may be retrieved through the GetVariable
'   method.<br><br>
'
'   Variable tracing is also available at this level.
'
' @author       Andrew Friedl
'
' @copyright    Copyright 1997.04.10, BlackBox Software & Consulting
' @revised      2000.03.15, Implemented new event model.
'*
Option Explicit

'local variable to hold collection
Private mCol As Collection

'**
' Collection based retrieval
'*
Public Property Get Item(vntIndexKey As Variant) As Variable
Attribute Item.VB_UserMemId = 0
  Set Item = mCol(vntIndexKey)
End Property

'**
' Collection based object count
'*
Public Property Get Count() As Long
    'used when retrieving the number of elements in the
    'collection. Syntax: Debug.Print x.Count
    Count = mCol.Count
End Property

'**
' Collection based deletion
'*
Public Sub Remove(vntIndexKey As Variant)
    'used when removing an element from the collection
    'vntIndexKey contains either the Index or Key, which is why
    'it is declared as a Variant
    'Syntax: x.Remove(xyz)
    mCol.Remove vntIndexKey
End Sub

'**
' Collection based enumeration
'*
Public Property Get NewEnum() As IUnknown
Attribute NewEnum.VB_UserMemId = -4
Attribute NewEnum.VB_MemberFlags = "40"
    'this property allows you to enumerate
    'this collection with the For...Each syntax
    Set NewEnum = mCol.[_NewEnum]
End Property

'**
' Class Constructor
'*
Private Sub Class_Initialize()
    'creates the collection when this class is created
    Set mCol = New Collection
End Sub

'**
' Class Destructor
'*
Private Sub Class_Terminate()
    'destroys collection when this class is terminated
    Set mCol = Nothing
End Sub

'**
' Locates a named Variable object.
'
' @describe
'   This function seaches for and retreves a reference to a named Variable
'   contained within the collection.
'
' @param    Name    The name of the variable to be located.
' @param    Value   The reference that is to be set if the variable is found.
'
' @returns  True if the Variable is located, False otherwise.
'*
Public Function GetVariable(Name As String, Value As Variable) As Boolean
    On Error GoTo NotFound
    Set Value = mCol.Item(Name)
    GetVariable = (Err.Number = 0)
    Exit Function
NotFound:
    GetVariable = False
End Function

'**
' Sets or creates a named variable within the collection.
'
' @describe
'   This function sets or creates a named value within the collection.
'   If a value with the Name specified already exists, then it an attempt
'   is made to update it to the new value.  If no value exists then a new
'   Variable is created with the value specified.
'
' @param    Name        The name of the variable to be created or set.
' @param    Value       The value to be stored or updated.
' @param    ReadOnly    An optional parameter indicating the value os to be
'   readonly.  This parameter is only meaningful when a variable is first
'   created.
'
' @returns  True if the Variable is created or updated, False otherwise.
'*
Public Function SetValue(Name As String, Value As Variant, Optional ReadOnly As Variant) As Boolean
    Dim Val As Variable
    On Error GoTo NotFound
    Set Val = mCol.Item(Name)
    Val.SetValue Value
    SetValue = True
    Exit Function
NotFound:
    On Error Resume Next
    Set Val = New Variable
    Val.Name = Name
    Val.SetValue Value
    Val.ReadOnly = OptParmBool(False, ReadOnly)
    mCol.Add Val, Name
    SetValue = Err.Number = 0
End Function

'**
' Retrieves a named value from the collection.
'
' @describe
'   This function retrieves the value of a named variable and stores that
'   value in the Value parameter, provided the variable exists.
'
' @param    Name    The name of the variable whose value is to be retrieved.
' @param    Value   The variant where the value is to be stored.
'
' @returns  True if the Variable exists, False otherwise.
'*
Public Function GetValue(Name As String, Value As Variant) As Boolean
    Dim Val As Variable
    On Error GoTo NotFound
    Set Val = mCol.Item(Name)
    Val.GetValue Value
    GetValue = True
    Exit Function
NotFound:
    ' otherwise, variale returns it's own value
    GetValue = False
End Function

'**
' Retrieves a named value from the collection.
'
' @describe
'   This function retrieves the value of a named variable, assuming the
'   variable exists.
'
' @param    Name    The name of the variable whose value is to be retrieved.
' @returns  The value of the variable.
'
' @throws   Exception thrown if the named value does not exists.
'*
Public Property Get Value(Name As String) As Variant
    Dim Val As Variant
    If GetValue(Name, Val) Then
        If IsObject(Val) Then
            Set Value = Val
        Else
            Value = Val
        End If
    Else
        Err.Raise vbObjectError + 1, "VarStack", "unknown value '" & Name & "'"
    End If
End Property

'**
' Stores a named value (non object) to the collection.
'
' @describe
'   This function stores a new non-object value to the collection or updates
'   an existing variable with a given name.  If the variable exists, an attempt
'   is made to update it, otherwise it is created.
'
' @param    Name    The name of the variable whose value is to be retrieved.
' @param    Value   The value of the variable to be updated or stored.
' @throws   Exception thrown if the value does not exist or is ReadOnly
'*
Public Property Let Value(Name As String, Value As Variant)
    Call SetValue(Name, Value)
End Property

'**
' Stores a named value (object) to the collection.
'
' @describe
'   This function stores an object value to the collection or updates
'   an existing variable.  If the variable exists, an attempt is made
'   to update it, otherwise it is created.
'
' @param    Name    The name of the whose value is to be retrieved.
' @param    Value   The value of the variable to be updated or stored.
' @throws   Exception thrown if the value does not exist or is ReadOnly
'*
Public Property Set Value(Name As String, Value As Variant)
    Call SetValue(Name, Value)
End Property

'**
' Removes all variables form the collection.
'
' @describe
'   This subroutine causes all variables within the collection
'   to be deleted.  All variable traces in effect will receive
'   event notifications prior to deletion.
'*
Public Sub Clear()
    Call EndAllTraces
    Set mCol = Nothing
    Set mCol = New Collection
End Sub

'**
' Returns the collection storage for the variable set.
' @returns A Collection of Variable objects.
'*
Public Property Get Storage() As Collection
    Set Storage = mCol
End Property

'**
' Adds tracing to a named variable.
'
' @describe
'   This function creates a trace upon a variable contained within
'   the collection, provided the variable exists.
'
' @param    Name    The name of the variable to be traced.
' @returns  A VarTrace object capable of generating events.
'*
Public Function SetTrace(Name As String) As VarTrace
    Dim Var As Variable
    ' locate the container value object
    If GetVariable(Name, Var) Then
        Set SetTrace = Var.AddTrace()
    Else
        Set SetTrace = Nothing
    End If
End Function

'**
' Terminates all tracing within the collection.
'
' @describe
'   This subroutine removes all tracing from the variables
'   contained within the collection.
'*
Private Sub EndAllTraces()
    Dim Var As Variable
    For Each Var In mCol
        ' inform traces that the variable is dying
        Call Var.UnsetValue
        'unlock any remaining traces in existence
        Call Var.EndAllTraces
    Next
End Sub
