__perl__

#
# webcalng - a web based calendar program.
# Copyright 2003 - webcalng software solutions
#

chdir   '__install__';
use lib '__install__';
$::db_dir = "__db_dir__";

# Test for the existance and compilation of perl modules.  Print error
# and exit if anything is not working right already.
BEGIN {
	eval q{
		use Time::Local;
		use webcalng_subs;
		use webcalng_io;
		use HTML::Template;
		use Net::SMTP;
	};
	if ($@) {
		die "Error: webcalng_remind.pl could not load a necessary Perl module: $@\n";
	}
}

# Load standard Perl modules that we need.
use strict;
use Time::Local;
use HTML::Template;
use Net::SMTP;

# Load webcalng custom Perl modules.
use webcalng_subs;
use webcalng_io;

# Global variables.
my ($time,$date,@calendars,$startmark,$endmark,$itemsref,$exceptions_ref,$loop,$loopstart,$loopend);
my ($messagecounter);
$::calendar        = "";
%::webcalng_conf   = ();
%::preferences     = ();
%::language        = ();
$::language_filter = \&webcalng_subs::language;
$loop              = 0;
$messagecounter    = 1;

# Check command line.
$loop = 1 if (($ARGV[0]) && ($ARGV[0] eq "-loop"));

# If we are in loop mode we will continually look for reminders and send them every 
# 5 minutes.  Loop mode is really meant to be used when this is running as a service
# under windows, but there is no reason you could not do this in unix as well, instead
# of running from cron.
while (1) {
	# Set the start time for this loop.
	$loopstart = time;

	# Open the log file and redirect standard file handles to it.
	open FILE,   ">>$::db_dir/webcalng_remind.log" or die "Error opening log file: $!\n";
	open STDOUT, ">>$::db_dir/webcalng_remind.log" or log_and_die("Error redirecting STDOUT: $!");
	open STDERR, ">>$::db_dir/webcalng_remind.log" or log_and_die("Error redirecting STDERR: $!");

	# Make sure output goes immediately to the log file.
	select STDERR; $|++;
	select STDOUT; $|++;
	select FILE;   $|++;

	# Set date and time for this loop.
	$time = time;
	$date = localtime;
	log_message("Looking for reminders at $date");
	
	# Read the configuration file.
	webcalng_subs::read_webcalng_conf();

	# Connect to the database if need be.
	if ($::webcalng_conf{'DB'}) {
		require webcalng_rdbms;
		webcalng_rdbms::dbh_setup();
	}

	# Loop through the list of calendars, calling a subroutine that looks for
	# reminders to be sent on the list of items in each calendar.
	@calendars = webcalng_io::get_calendar_list();
	for $::calendar (@calendars) {
	
		# First, setup calendar specific information based on preferences.
		webcalng_subs::read_preferences();	
		next if (! $::preferences{'send_email'}{'value'});
		webcalng_subs::setup_language();
	
		# Look for any items that have an immediate notification set on them.
		$startmark = 0;
		$endmark   = 9999999999;
		($itemsref,undef) = webcalng_io::get_items($startmark,$endmark,undef,undef,'1',$::calendar);
		find_reminders($itemsref,'0');
	
		# We only need to look at items from yesterday through the next 4 weeks for reminders.
		$startmark = $time - 86400;
		$endmark   = $time + 2419200 + 86400;
		($itemsref,$exceptions_ref) = webcalng_io::get_items($startmark,$endmark,undef,undef,'0',$::calendar);
		find_reminders($itemsref,$exceptions_ref);
	}

	# Now send all messages right away from the message queue.
	send_messagequeue();
	
	# Disconnect from database if not using flat files.
	$::dbh->disconnect() if ($::webcalng_conf{'DB'});

	# If we are not in loop mode, break out of the loop.  Otherwise, sleep for an amount of 
	# time such that the loop runs in 5 minute intervals.
	if ($loop) {
		my ($runtime,$sleeptime);
		$loopend   = $time;
		$runtime   = $loopend - $loopstart;
		$sleeptime = 300 - $runtime;
		$sleeptime = 1 if ($sleeptime <= 0);
		log_message("Sleeping for $sleeptime seconds");
		sleep $sleeptime;
	} else {
		last;
	}

	# Close log file.
	close FILE;
}

exit 0;

#
# Loop through the items for a calendar.  If we find a reminder to be
# sent, call a subroutine to send it out.
#
sub find_reminders {
	my ($itemsref,$exceptions_ref) = (@_);
	my ($calendar,$itemid,$exceptionid,$day,$month,$year,$today,$meetingpass);

	# Find out what the time at noon today is.  This is used to determine if a reminder has
	# already been sent out today.
	($day,$month,$year) = (localtime($time))[3,4,5];
	$today = timelocal "0","0","12",$day,$month,$year;

	# First, loop through all the items for the calendar.
	for $calendar (keys %$itemsref) {
		for $itemid (keys %{ $itemsref->{$calendar} }) {
			my (@remindwhen,@remindersent,$remindwhen,$x,$shour,$smin);
			@remindwhen   = split /,/, $itemsref->{$calendar}{$itemid}{'remindwhen'}   || "";
			@remindersent = split /,/, $itemsref->{$calendar}{$itemid}{'remindersent'} || "";
			$x = 0;
			$shour = $smin = 0;
			if ($itemsref->{$calendar}{$itemid}{'starttime'} =~ /(\d+)(\d{2})/) {
				$shour = $1;
				$smin = $2;
			} 
			for $remindwhen (@remindwhen) {
				my ($testtime,$testday,$testmonth,$testyear);
				# Skip to the next one if this reminder is for the immediate message,
				# and it has already been sent.
				if (($remindwhen == 1) && ($remindersent[$x])) {
					$x++;
					next;
				}
				# Skip to the next one if this reminder has already been sent today.
				if (($remindersent[$x]) && ($remindersent[$x] == $today)) {
					$x++;
					next;
				}
				$testtime = $time + $remindwhen;
				($testday,$testmonth,$testyear) = (localtime($testtime))[3,4,5];
				$testmonth++;
				$testyear += 1900;
				if (($remindwhen == 1) || (valid_item($itemsref,$exceptions_ref,$calendar,$itemid,$testday,$testmonth,$testyear))) {
					my $itemtime = timelocal "0",$smin,$shour,$testday,($testmonth-1),($testyear-1900);
					if (($remindwhen == 1) || (($time + $remindwhen) >= $itemtime)) {
						my (@remindwho,$to,$usercal,$email,$success);
						@remindwho = split /,/, $itemsref->{$calendar}{$itemid}{'remindwho'};
						for (@remindwho) {
							if (/(.*?);(.*)/) {
								$to      = $1;
								$usercal = $2;
							} else {
								$to      = $_;
								$usercal = 0;
							}
							# Don't send an invite to the calendar owner who created a meeting.
							next if (($remindwhen == 1) && ($itemsref->{$calendar}{$itemid}{'meetingid'}) && ($usercal eq $calendar));
							next if (! $to);
							# Get the password for this persons meeting invite if need be.
							if (($itemsref->{$calendar}{$itemid}{'meetingid'}) && (! $usercal)) {
								my $ref = webcalng_io::get_meeting_data($itemsref->{$calendar}{$itemid}{'meetingid'},$calendar);
								if ($ref->{$to}{'password'}) {
									$meetingpass = $ref->{$to}{'password'};
								} else {
									$meetingpass = "";
								}
				
							} else {
								$meetingpass = "";
							}
							$email = format_message($itemsref->{$calendar}{$itemid},$itemid,"0",$remindwhen,$to,$itemtime,$calendar,$usercal,$itemsref->{$calendar}{$itemid}{'meetingid'},$meetingpass);
							$success = send_email($email,$to);
						}
						webcalng_io::mark_remindersent($calendar,$itemid,$x,$today) if ($success);
					}
				}
				$x++;
			}
		}
	}

	# If there are no exceptions to look at, return now.
	return 1 if (! $exceptions_ref);

	# Now, loop through the exceptions on the calendar.
	for $calendar (keys %$exceptions_ref) {
		for $itemid (keys %{ $exceptions_ref->{$calendar} }) {
			for $exceptionid (keys %{ $exceptions_ref->{$calendar}{$itemid} }) {
				next if ($exceptions_ref->{$calendar}{$itemid}{$exceptionid}{'description'} eq "");
				my (@remindwhen,@remindersent,$remindwhen,$x,$shour,$smin,$itemtime,$testtime);
				@remindwhen   = split /,/, $itemsref->{$calendar}{$itemid}{'remindwhen'};
				@remindersent = split /,/, $itemsref->{$calendar}{$itemid}{'remindersent'};
				$x = 0;
				$shour = $smin = 0;
				if ($exceptions_ref->{$calendar}{$itemid}{$exceptionid}{'starttime'} =~ /(\d+)(\d{2})/) {
					$shour = $1;
					$smin = $2;
				}
				for $remindwhen (@remindwhen) {
					# Skip immediate reminders for exception items.
					next if ($remindwhen == 1);
					# Skip to the next one if this reminder has already been sent.
					if ($remindersent[$x] == $today) {
						$x++;
						next;
					}
					# See if the exception falls on the right day.
					$testtime = $time + $remindwhen;
					$itemtime = $exceptions_ref->{$calendar}{$itemid}{$exceptionid}{'exceptiondate'};
					my ($testday,$testmonth,$testyear) = (localtime($testtime))[3,4,5];
					my ($itemday,$itemmonth,$itemyear) = (localtime($itemtime))[3,4,5];
					next unless (($testday == $itemday) && ($testmonth == $itemmonth) && ($testyear == $itemyear));
					# Now see if this exception falls in the right time window to send the reminder.
					$itemtime = timelocal "0",$smin,$shour,$itemday,$itemmonth,$itemyear;
					if (($time + $remindwhen) >= $itemtime) {
						my (@remindwho,$to,$usercal,$email,$success);
						@remindwho = split /,/, $itemsref->{$calendar}{$itemid}{'remindwho'};
						for (@remindwho) {
							if (/(.*?);(.*)/) {
								$to      = $1;
								$usercal = $2;
							} else {
								$to      = $_;
								$usercal = 0;
							}
							next if (! $to);
							# Get the password for this persons meeting invite if need be.
							if (($itemsref->{$calendar}{$itemid}{'meetingid'}) && (! $usercal)) {
								my $ref = webcalng_io::get_meeting_data($itemsref->{$calendar}{$itemid}{'meetingid'},$calendar);
								if ($ref->{$to}{'password'}) {
									$meetingpass = $ref->{$to}{'password'};
								} else {
									$meetingpass = "";
								}
				
							} else {
								$meetingpass = "";
							}
							$email = format_message($exceptions_ref->{$calendar}{$itemid}{$exceptionid},$itemid,$exceptionid,$remindwhen,$to,$exceptions_ref->{$calendar}{$itemid}{$exceptionid}{'exceptiondate'},$calendar,$usercal,$itemsref->{$calendar}{$itemid}{'meetingid'},$meetingpass);
							$success = send_email($email,$to);
						}
						webcalng_io::mark_remindersent($calendar,$itemid,$x,$today) if ($success);
					}
					$x++;
				}
			}
		}
	}

	return 1;
}

#
# Determine if a given item really falls on the date in question.
#
sub valid_item {
	my ($itemsref,$exceptions_ref,$calendar,$itemid,$day,$month,$year) = (@_);
	my ($valid,%new_item,$new_itemref,$key,$exceptionid,$itemdate);

	# If they are looping through the immediate checks, there will be no exceptions_ref, so return.
	return 0 if (! $exceptions_ref);
	
	# Get the item date to use when comparing exceptions.
	$itemdate = timelocal "0","0","12",$day,($month-1),($year-1900);

	# Copy the item into a hash by itself, and see if it passes the filter for the date.
	$valid = 0;
	for $key (keys %{ $itemsref->{$calendar}{$itemid} }) {
		$new_item{$calendar}{$itemid}{$key} = $itemsref->{$calendar}{$itemid}{$key};
	}	
	$new_itemref = webcalng_subs::filter_items(\%new_item,$day,$month,$year);
	$valid = 1 if ($new_itemref->{$calendar}{$itemid}{'description'});

	# If there is an exception that has an exception date matching the date we are
	# looking at, then skip this reminder as we loop through exceptions seperately.
	if ($valid) {
		for $exceptionid (keys %{ $exceptions_ref->{$calendar}{$itemid} }) {
			$valid = 0 if ($itemdate == $exceptions_ref->{$calendar}{$itemid}{$exceptionid}{'exceptiondate'});
		}
	}

	return $valid;
}

#
# Format the email message that is going to be sent out.
#
sub format_message {
	my ($ref,$itemid,$exceptionid,$remindwhen,$to,$itemtime,$calendar,$usercal,$meetingid,$meetingpass) = (@_);
	my (@MONTHS,$itemdate,$day,$month,$year,$dow,%remindwhen,$time,$email_template,$email);
	my ($emailmessage,$subject,$meetingurl,$occurs,$messageid,$notes,$vcalstart,$vcalend);

	# Setup the item date.
	@MONTHS = qw (Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
	$itemtime = $ref->{'startdate'} if ($remindwhen == 1);
	($day,$month,$year) = (localtime($itemtime))[3,4,5];
	$month++;
	$year += 1900;
	$dow  = (split /\s+/, localtime($itemtime))[0];
	if ($::preferences{'date_format'}{'value'}) {
		$itemdate = "$::language{$dow} $day $::language{ $MONTHS[$month-1] } $year";
	} else {
		$itemdate = "$::language{$dow} $::language{ $MONTHS[$month-1] } $day, $year";
	}

	# Determine what the message should say.
	if ($meetingid) {
		if ($remindwhen == 1) {
			$emailmessage = $::language{'meetingnotice'};
			$subject      = $::language{'webcalngupdate'},
		} else {
			$emailmessage = $::language{'meetingreminder'};
			$subject      = $::language{'webcalngreminder'},
		}
		my ($cgi_calendar,$cgi_usercal);
		$cgi_calendar = $calendar;
		$cgi_usercal  = $usercal;
		$meetingid    =~ s/;.*//;
		$cgi_calendar =~ s/\s+/+/g;
		$cgi_usercal  =~ s/\s+/+/g;
		$meetingurl   = $::webcalng_conf{'HTTP'} . $::webcalng_conf{'HTTP_HOST'} . $::webcalng_conf{'WEBCAL'};
		if ($cgi_calendar ne $cgi_usercal) {
			$meetingurl  .= "?op=respond&calendar=$cgi_calendar&email=$to&usercal=$cgi_usercal&meetingid=$meetingid&return_link=op--meeting..itemid--$itemid..exceptionid--$exceptionid..calendar--$cgi_calendar..meetingcal--$cgi_calendar..year--$year..month--$month..day--$day";
		} else {
			$meetingurl  .= "?op=meeting&calendar=$cgi_calendar&meetingcal=$cgi_calendar&itemid=$itemid&exceptionid=$exceptionid&year=$year&month=$month&day=$day";
		}
	} else {
		if ($remindwhen == 1) {
			$emailmessage = $::language{'itemnotice'};
			$subject      = $::language{'webcalngupdate'},
		} else {
			$emailmessage = $::language{'emailreminder'};
			$subject      = $::language{'webcalngreminder'},
		}
		$meetingurl = "";
	}

	# Setup strings that describe how soon the item occurs.
	%remindwhen = qw (
		1        immediate
		300      in5min
		600      in10min
		900      in15min
		1800     in30min
		3600     in1hour
		7200     in2hour
		10800    in3hour
		86400    in1day
		172800   in2day
		259200   in3day
		604800   in1week
		1209600  in2week
		2419200  in4week
	);

	# Setup string defining when the event occurs, unless this is an immediate reminder.
	if ($remindwhen{$remindwhen} eq "immediate") {
		$occurs = "";
	} else {
		$occurs = "$::language{'occurs'}: $::language{ $remindwhen{$remindwhen} }";
	}

	# Generate a messageid and update the messagecounter.
	$messageid = "<" . $loopstart . $messagecounter . "@" . $::webcalng_conf{'HTTP_HOST'} . ">";
	$messagecounter++;

	# Clean up the notes, if there are any.
	if ($ref->{'notes'}) {
		$notes = $ref->{'notes'};
		$notes =~ s/<BR>/<NEWLINE>\n/g;
	} else {
		$notes = "";
	}

	# Setup vcal start and end dates.
	if (($ref->{'starttime'}) && ($ref->{'endtime'})) {
		my ($vcalmonth,$vcalday,$vcalstime,$vcaletime);
		$vcalmonth = $month;
		$vcalmonth = "0" . $vcalmonth if ($vcalmonth !~ /\d{2}/);
		$vcalday   = $day;
		$vcalday   = "0" . $vcalday if ($vcalday !~ /\d{2}/);
		$vcalstime = $ref->{'starttime'};
		$vcalstime = "0" . $vcalstime if ($vcalstime !~ /\d{4}/);
		$vcaletime = $ref->{'endtime'};
		$vcaletime = "0" . $vcaletime if ($vcaletime !~ /\d{4}/);
		$vcalstart = $year . $vcalmonth . $vcalday . "T" . $vcalstime . "00";
		$vcalend   = $year . $vcalmonth . $vcalday . "T" . $vcaletime . "00";
	} else {
		$vcalstart = 0;
		$vcalend   = 0;
	}

	# Format the email message template.
	if ($ref->{'starttime'} ne "") {
		$time = webcalng_subs::format_time($ref->{'starttime'});
		if ($ref->{'endtime'} ne "") {
			$time .= " - " . webcalng_subs::format_time($ref->{'endtime'});
		}
	} else {
		$time = $::language{'none'};
	}
	$email_template = HTML::Template->new(
		filename          => 'email.tmpl',
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);
	$email_template->param(
		to           => $to,
		subject      => $subject,
		emailmessage => $emailmessage,
		description  => $ref->{'description'},
		occurs       => $occurs,
		ondate       => $itemdate,
		time         => "$::language{'time'}: $time",
		meetingurl   => $meetingurl,
		calendar     => $calendar,
		meetingpass  => $meetingpass,
		messageid    => $messageid,
		hyperlink    => $ref->{'hyperlink'},
		notes        => $notes,
		http_host    => $::webcalng_conf{'HTTP_HOST'},
		meetingid    => $meetingid,
		vcalstart    => $vcalstart,
		vcalend      => $vcalend,
	);
	$email = $email_template->output;

	# Strip empty lines and format the email.
	$email = strip_email($email);

	return ($email);
}

#
# This subroutine gathers and formats any messages that are in the messagequeue.
#
sub send_messagequeue {
	my ($messages,@indexes,$id,$emailaddress,$email,$success);

	($messages,@indexes) = webcalng_io::read_messagequeue();
	$success             = 0;
	for $id (keys %$messages) {
		$emailaddress = $id;
		$emailaddress =~ s/^\d+;//;    # Get rid of counter.
		$emailaddress =~ s/;.*//;      # Get rid of calendar name from id.
		$email        = format_simple_message($emailaddress,$messages->{$id});	
		$success      = send_email($email,$emailaddress);
	}	
	webcalng_io::empty_messagequeue(@indexes) if ($success);

	return 1;
}

#
# Format a simple message from the messagequeue to be sent.
#
sub format_simple_message {
	my ($emailaddress,$text) = (@_);
	my ($email_template,$email,$messageid);

	# Generate a messageid and update the messagecounter.
	$messageid = "<" . $loopstart . $messagecounter . "@" . $::webcalng_conf{'HTTP_HOST'} . ">";
	$messagecounter++;

	# Setup template and get output from it.
	$email_template = HTML::Template->new(
		filename          => 'email.tmpl',
		filter            => $::language_filter,
		die_on_bad_params => '0',
		cache             => '1',
		global_vars       => '1',
	);
	$email_template->param(
		to           => $emailaddress,
		subject      => $::language{'webcalngupdate'},
		emailmessage => $text,
		messageid    => $messageid,
		http_host    => $::webcalng_conf{'HTTP_HOST'},
	);
	$email = $email_template->output;

	# Strip empty lines and format the email.
	$email = strip_email($email);

	return $email;
}

#
# Subroutine to send out an email reminder.
#
sub send_email {
	my ($email,$to) = (@_);
	my ($mailhost,$from,$smtp,@lines);

	# Untaint $mailhost.
	$mailhost = $::webcalng_conf{'MAILHOST'};
	if ($mailhost =~ /([\w,\-,\.,\@]+)/) {
		$mailhost = $1;
	} else {
		if ($loop) {
			log_message("Could not untaint mailhost: $mailhost");
			return 0;
		} else {
			log_and_die("Could not untaint mailhost: $mailhost");
		}
	}

	# Get the from address out of the email template so that people can 
	# easily modify this.
	@lines = split /\n/, $email;
	for (@lines) {
		chomp;
		if (/^From:\s+(.*)/) {
			$from = $1;
			last;
		}
	}
	$from = "webcalng@" . $::webcalng_conf{'HTTP_HOST'} if (! $from);

	# Use Net::SMTP to send the message.
	$smtp = Net::SMTP->new($mailhost);
	if (! $smtp) {
		if ($loop) {
			log_message("Error connecting to $mailhost");
			return 0;
		} else {
			log_and_die("Error connecting to $mailhost");
		}
	}
	$smtp->mail($from);
	$smtp->to($to);
	$smtp->data();
	$smtp->datasend($email);
	$smtp->datasend("\n");
	$smtp->dataend();
	$smtp->quit();
	log_message("Email sent to $to");

	return 1;
}

#
# Strip blank lines and comments from the email template, and
# turn <NEWLINE> tags into blank lines.
#
sub strip_email {
	my ($text) = (@_);
	my (@lines);

	@lines = split /(\n)|(\r\n)/, $text;
	$text  = "";
	for (@lines) {
		next unless $_;
		chomp;
		s/^<!---.*--->$//;
		next unless $_;
		s/<NEWLINE>//g;
		$text .= "$_\n";
	}

	return $text;
}

#
#
# Subroutine to log a message.
#
sub log_message {
	my ($message) = (@_);
	print FILE "$message\n";
}

#
# Subroutine to log error messages.
#
sub log_and_die {
	my ($message) = (@_);
	print FILE "$message\n";
	close FILE;
	exit 0;
}
