// ==============================================================
//
//  Copyright (c) 2003 by Alex Vinokur.
//
//  For conditions of distribution and use, see
//  copyright notice in version.h
//
// ==============================================================


// ##############################################################
//
//  SOFTWARE : Turing Machine with faults, failures and recovery
//             C++ Simulator
//
//  FILE     : turing-s.h
//
//  DESCRIPTION :
//         Class TuringMachine and auxilary classes (Definition)
//
// ##############################################################


// ################
#ifndef _TURING_S_H
#define _TURING_S_H

// ===============
#include "enums.h"
#include "rules.h"
#include "tape.h"
// ===============


typedef map<CurSituation, NextSituation>	Transitions_t;
typedef map<size_t, Tape>			Tapes_t;

#define	CHECK_POINT_FIELD_IN_TRANSITION_FILE	"check-point"
#define	DAEMON_FIELD_IN_TRANSITION_FILE		"daemon"

// ----------------------------------------
#define	RULE_NO_INFO 		" "
#define	RULE_NEW		"+"
#define	RULE_OLD      		"."
#define	RULE_UPDATED   		"~"
#define	RULE_MULTI_UPDATED	"&"



// ----------------------------------------
#define	UNBELIEVABLE_NUMBER_OF_TAPES	100

// ----------------------------------------

#define	DAEMON_PASSIVE_STATE		"passive" 
#define	DAEMON_ACTIVE_STATE		"active" 
#define	DAEMON_AGGRESSSIVE_STATE	"aggressive"
 
#define	TESTER_TRACKING_STATE		"tracking" 
#define	TESTER_STABILIZING_STATE	"stabilizing" 

#define	APPARATUS_NORMAL_STATE		"normal" 
#define	APPARATUS_EMERGENCY_STATE	"emergency" 


// =========			
class AuxTuringMachine
{
friend class TuringMachine;
  private :
    // ------ Private Date ------
    vector<state_t>	daemon_passive_states_;
    vector<state_t>	daemon_active_states_;
    vector<state_t>	daemon_aggressive_states_;
    vector<state_t>	tester_tracking_states_;
    vector<state_t>	tester_stabilizing_states_;
    vector<state_t>	apparatus_normal_states_;
    vector<state_t>	apparatus_emergency_states_;

    vector<symbol_mt>	embedded_blank_synchronous_tape_alphabet_;
    vector<symbol_mt>	embedded_internal_synchronous_tape_alphabet_;

    vector<string>	descr_of_tape_kinds_;
    size_t		max_size_of_descr_of_tape_kinds_;

    vector<string>	descr_of_program_state_top_kinds_;
    size_t		max_size_of_descr_of_program_state_top_kinds_;

    vector<string>	descr_of_user_defined_program_state_kinds_;
    size_t		max_size_of_descr_of_user_defined_program_state_kinds_;
    vector<vector<string> >	descr_of_user_defined_program_states_;
    vector<size_t>     	max_size_of_descr_of_user_defined_program_states_;


    vector<string>	descr_of_neutral_program_state_kinds_;
    size_t		max_size_of_descr_of_neutral_program_state_kinds_;
    vector<vector<string> >	descr_of_neutral_program_states_;
    vector<size_t>     	max_size_of_descr_of_neutral_program_states_;


    vector<string>	descr_of_pre_initial_program_state_kinds_;
    size_t		max_size_of_descr_of_pre_initial_program_state_kinds_;
    vector<vector<string> >	descr_of_pre_initial_program_states_;
    vector<size_t>     	max_size_of_descr_of_pre_initial_program_states_;


    vector<string>	descr_of_post_halting_program_state_kinds_;
    size_t		max_size_of_descr_of_post_halting_program_state_kinds_;
    vector<vector<string> >	descr_of_post_halting_program_states_;
    vector<size_t>     	max_size_of_descr_of_post_halting_program_states_;

    vector<string>	descr_of_user_required_program_state_kinds_;
    size_t		max_size_of_descr_of_user_required_program_state_kinds_;
    vector<vector<string> >	descr_of_user_required_program_states_;
    vector<size_t>     	max_size_of_descr_of_user_required_program_states_;


    vector<string>	descr_of_embedded_program_state_kinds_;
    size_t		max_size_of_descr_of_embedded_program_state_kinds_;
    vector<vector<string> >	descr_of_embedded_program_states_;
    vector<size_t>     	max_size_of_descr_of_embedded_program_states_;

    size_t		max_size_of_descr_of_program_state_kinds_;

    // ------ Private Methods ------
    size_t 		get_max_size_of_daemon_states();
    size_t 		get_max_size_of_tester_states();
    size_t 		get_max_size_of_apparatus_states();

    void		init ();
    void		init_daemon_states ();
    void		init_tester_states ();
    void		init_apparatus_states ();
    void		init_synchronous_tape_alphabet ();
    void       		init_descr_of_tape_kinds ();
    void       		init_descr_of_program_state_top_kinds ();
    void       		init_descr_of_user_defined_program_state_kinds ();
    void       		init_descr_of_user_defined_program_states ();
    void       		init_descr_of_neutral_program_state_kinds ();
    void       		init_descr_of_neutral_program_states ();
    void       		init_descr_of_pre_initial_program_state_kinds ();
    void       		init_descr_of_pre_initial_program_states ();
    void       		init_descr_of_post_halting_program_state_kinds ();
    void       		init_descr_of_post_halting_program_states ();
    void       		init_descr_of_user_required_program_state_kinds ();
    void       		init_descr_of_user_required_program_states ();
    void       		init_descr_of_embedded_program_state_kinds ();
    void 		init_descr_of_embedded_program_states ();

  public :
    AuxTuringMachine ();
    ~AuxTuringMachine ();
};


// =========
class ExtendedState
{
friend class TuringMachine;
  private :
    state_t	program_state_;
    state_t	program_substate1_;
    state_t	program_substate2_;
    state_t	daemon_state_on_tact1_;
    state_t	daemon_state_on_tact2_;
    state_t	tester_state_;
    state_t	apparatus_state_;

    void	show_extended_state (const string& msf1_i, const string& pref_i = string()) const;

  public :
    ExtendedState (
		const state_t&	program_state_i,
		const state_t&	program_substate1_i,
		const state_t&	program_substate2_i,
		const state_t&	daemon_state_on_tact1_i,
		const state_t&	daemon_state_on_tact2_i,
		const state_t&	tester_state_i,
		const state_t&	apparatus_state_i
		);
    ~ExtendedState ();

};


// =========
class TuringMachine
{
  private :
    // //////////////////////////
    // ////// Private Data //////
    // //////////////////////////

	// --------------------------------
	// ----- DEFINITIONS : BEGIN ------
	// --------------------------------
	// - - -  States - - - 
		// ::: User defined program states :::
    const vector<vector<string>	> descr_;
    const vector<state_t>	user_defined_initial_program_states_;
    const vector<state_t>	user_defined_halting_program_states_;
    const vector<state_t>	user_defined_internal_program_states_;
		// ::: Extra program states :::
    vector<state_t>		neutral_program_states_;
    vector<state_t>		pre_initial_program_states_;
    vector<state_t>		post_halting_program_states_;
    map<CurSituation, state_t>	user_required_check_point_program_states_;
    vector<vector<state_t> >	embedded_program_states_;


		// ::: Daemon states :::
    static const vector<state_t> daemon_passive_states_s;
    static const vector<state_t> daemon_active_states_s;
    static const vector<state_t> daemon_aggressive_states_s;

		// ::: Tester states :::
    static const vector<state_t> tester_tracking_states_s;
    static const vector<state_t> tester_stabilizing_states_s;

		// ::: Apparatus states :::
    static const vector<state_t> apparatus_normal_states_s;
    static const vector<state_t> apparatus_emergency_states_s;


	// - - -  Alphabets - - - 
    const vector<symbol_mt>	user_defined_blank_alphabet_;
    const vector<symbol_mt>	user_defined_internal_alphabet_;
    const vector<symbol_mt>	user_defined_input_alphabet_;

    static const vector<symbol_mt> embedded_blank_synchronous_tape_alphabet_s;
    static const vector<symbol_mt> embedded_internal_synchronous_tape_alphabet_s;

    vector<symbol_mt> 		pseudo_never_mind_symbols_alphabet_;
    vector<symbol_mt> 		pseudo_not_marker_symbols_alphabet_;
    vector<symbol_mt> 		pseudo_not_marker_and_not_blank_symbols_alphabet_;
    vector<symbol_mt> 		pseudo_write_nothing_symbols_alphabet_;

	// - - - Transitions - - -
    const Transitions_t		user_defined_transitions_;
    Transitions_t		daemon_defined_fault_transitions_;
    Transitions_t		built_transitions_;
    map<CurOutsideSituation, NextOutsideSituation>	outside_tact1_transitions_;
    map<CurOutsideSituation, NextOutsideSituation>	outside_tact2_transitions_;
    mutable map<CurSituation, string>	info_user_transitions_;
    mutable map<CurSituation, count_t>	count_daemon_transitions_;


	// - - - Tapes - - -
    Tapes_t			tapes_[NUMBER_OF_TAPE_KINDS];
    const bool			set_marker_flag_;

	// ----- DEFINITIONS : END --------



	// -------------------------------
	// ----- PROCESSING : BEGIN ------
	// -------------------------------
    bool					check_process_results_;

    vector<Transitions_t::const_iterator>	process_transitions_;
    vector<Transitions_t::const_iterator>	process_tester_transitions_;
    vector<ExtendedState>			process_extended_states_;
    vector<state_t>				process_tester_states_;

    vector<uint>       				process_statistics_;

    static size_t				counter_s;
    const size_t				machine_serial_no_;
    size_t					input_set_no_;
    size_t					alert_counter_;

	// ----- PROCESSING : END --------



	// ---------------------------------
	// ----- DESCRIPTIONS : BEGIN ------
	// ---------------------------------
    static const vector<string>	descr_of_tape_kinds_s;
    static const size_t		max_size_of_descr_of_tape_kinds_s;

    static const vector<string>	descr_of_program_state_top_kinds_s;
    static const size_t		max_size_of_descr_of_program_state_top_kinds_s;

    static const vector<string>	descr_of_user_defined_program_state_kinds_s;
    static const size_t		max_size_of_descr_of_user_defined_program_state_kinds_s;
    static const vector<vector<string> >	descr_of_user_defined_program_states_s;
    static const vector<size_t>	max_size_of_descr_of_user_defined_program_states_s;

    static const vector<string>	descr_of_neutral_program_state_kinds_s;
    static const size_t		max_size_of_descr_of_neutral_program_state_kinds_s;
    static const vector<vector<string> >	descr_of_neutral_program_states_s;
    static const vector<size_t>	max_size_of_descr_of_neutral_program_states_s;

    static const vector<string>	descr_of_pre_initial_program_state_kinds_s;
    static const size_t		max_size_of_descr_of_pre_initial_program_state_kinds_s;
    static const vector<vector<string> >	descr_of_pre_initial_program_states_s;
    static const vector<size_t>	max_size_of_descr_of_pre_initial_program_states_s;

    static const vector<string>	descr_of_post_halting_program_state_kinds_s;
    static const size_t		max_size_of_descr_of_post_halting_program_state_kinds_s;
    static const vector<vector<string> >	descr_of_post_halting_program_states_s;
    static const vector<size_t>	max_size_of_descr_of_post_halting_program_states_s;


    static const vector<string>	descr_of_user_required_program_state_kinds_s;
    static const size_t		max_size_of_descr_of_user_required_program_state_kinds_s;
    static const vector<vector<string> >	descr_of_user_required_program_states_s;
    static const vector<size_t>	max_size_of_descr_of_user_required_program_states_s;

    static const vector<string>	descr_of_embedded_program_state_kinds_s;
    static const size_t		max_size_of_descr_of_embedded_program_state_kinds_s;
    static const vector<vector<string> >	descr_of_embedded_program_states_s;
    static const vector<size_t>	max_size_of_descr_of_embedded_program_states_s;

    static const size_t	max_size_of_descr_of_program_state_kinds_s;

    size_t   		max_size_of_user_defined_program_states_;
    size_t   		max_size_of_user_defined_and_required_states_;
    size_t   		max_size_of_extra_program_states_;
    size_t   		max_size_of_program_states_;

    static const size_t	max_size_of_daemon_states_s;
    static const size_t	max_size_of_tester_states_s;
    static const size_t	max_size_of_apparatus_states_s;

    size_t     		max_size_of_user_defined_alphabet_symbol_;


	// ----- DESCRIPTIONS : END --------



    // /////////////////////////////
    // ////// Private Methods //////
    // /////////////////////////////
    void	clear_process_data ();
    void	set_input (const vector<vector<symbol_mt> >& input_words_i);
    void	set_marker ();
    void	build_transition ();

    bool	perform_machine_step (
			state_t&	program_state_o, 
			bool&		controlling_flag_o, 
			string&		str_cur_situation_o,
			string&		str_next_situation_o,
			const string& msg_i = string()
			);

    bool	perform_tester_step (state_t& program_state_o, bool& controlling_flag_o, const string& msg_i = string());


    TesterCalculationResults	execute_tester_checked_interval (
			size_t		call_no_i, 
			size_t&		serial_no_io,
			bool&		succces_tester_flag_o	
			);

    MachineCalculationResults	execute_machine_checked_interval (
			size_t		interval_no_i, 
			size_t&		serial_no_io, 
			bool&		success_flag_o
			);

    vector<symbol_mt> 		get_all_master_tape_alphabet_symbols () const;
    vector<symbol_mt> 		get_all_not_blank_master_tape_alphabet_symbols () const;
    vector<symbol_mt> 		get_all_not_marker_master_tape_alphabet_symbols () const;
    vector<symbol_mt> 		get_all_not_marker_and_not_blank_master_tape_alphabet_symbols () const;
    static vector<symbol_mt> 	get_all_synchronous_tape_alphabet_symbols_s ();
    static vector<symbol_mt> 	get_all_not_blank_synchronous_tape_alphabet_symbols_s ();
    static vector<symbol_mt> 	get_all_not_marker_synchronous_tape_alphabet_symbols_s ();
    static vector<symbol_mt> 	get_all_not_marker_and_not_blank_synchronous_tape_alphabet_symbols_s ();
    vector<vector<symbol_mt> > 	get_all_tapes_all_alphabet_symbols () const;
    vector<vector<symbol_mt> >	get_all_tapes_alphabets_combinations (size_t k_i = NUMBER_OF_TAPE_KINDS) const;
    vector<state_t> 		get_all_user_defined_program_states () const;
    vector<state_t> 		get_all_extra_program_states () const;
    static vector<state_t>	get_all_daemon_states_s ();
    static vector<state_t>	get_all_tester_states_s ();
    static vector<state_t>	get_all_apparatus_states_s ();
    size_t			get_internal_index_for_tape (TapeKinds kind_i, size_t external_index_i) const;
    state_t  			get_neutral_cur_substate_alphabet_symbol () const;
    state_t  			get_neutral_next_the_same_substate_alphabet_symbol () const;
    state_t  			get_neutral_next_not_the_same_substate_alphabet_symbol () const;

    symbol_mt			get_user_defined_marker () const;
    bool			is_user_defined_marker (const symbol_mt& symbol_i) const;
    static symbol_mt		get_embedded_synchronous_tape_marker_s ();
    static bool			is_embedded_synchronous_tape_marker_s (const symbol_mt& symbol_i);
    static symbol_mt		get_embedded_synchronous_tape_right_sign_s ();

    bool	is_pseudo_never_mind_symbol (const symbol_mt& symbol_i) const;
    bool	is_pseudo_not_marker_symbol (const symbol_mt& symbol_i) const;
    bool	is_pseudo_not_marker_and_not_blank_symbol (const symbol_mt& symbol_i) const;
    bool	is_pseudo_write_nothing_symbol (const symbol_mt& symbol_i) const;

    SymbolTypes	get_symbol_mtype (TapeKinds kind_i, const symbol_mt& symbol_i) const;
    vector<SymbolTypes>	get_symbol_mtypes (const vector<symbol_mt>& symbols_i) const;

    bool	is_initial_program_state (const state_t& state_i) const;
    bool	is_halting_program_state (const state_t& state_i) const;
    bool	is_internal_program_state (const state_t& state_i) const;
    bool	is_user_defined_program_state (const state_t& state_i) const;
    bool	is_neutral_program_state (const state_t& state_i) const;
    bool	is_pre_initial_program_state (const state_t& state_i) const;
    bool	is_post_halting_program_state (const state_t& state_i) const;
    bool	is_user_required_check_point_program_pre_state (const state_t& state_i) const;
    bool	is_user_required_check_point_program_state (const state_t& state_i) const;
    bool	is_embedded_program_state_of_specific_kind (EmbeddedProgramStateKinds kind_i, const state_t& state_i) const;
    bool	is_embedded_program_state (const state_t& state_i, EmbeddedProgramStateKinds& kind_o, size_t& index_o) const;
    bool	is_embedded_program_state (const state_t& state_i) const;
    bool	is_embedded_shutting_down_program_state (const state_t& state_i) const;

    void 	set_max_user_defined_program_state_size();
    void 	set_max_size_of_user_defined_and_required_states();
    void 	set_max_size_of_program_states();
    void 	set_max_size_of_extra_program_states_size();
    void	set_max_size_of_user_defined_alphabet_symbol();

    void	set_tape_alphabet (
			size_t		number_of_tapes_i,
			TapeKinds	tape_kind_i,
			const vector<symbol_mt>& user_defined_blank_alphabet_i,
			const vector<symbol_mt>& user_defined_internal_alphabet_i,
			const vector<symbol_mt>& user_defined_input_alphabet_i
			);

    void	set_extra_program_states ();
    void	set_pseudo_symbols_alphabet ();
    void	share_information1 () const;
    void	share_information2 () const;
    vector<state_t>	get_notation_of_extra_program_states (size_t add_size_i) const;
    vector<symbol_mt>	get_notation_of_pseudo_symbols_alphabet () const;
    void	set_never_mind_symbol (TapeKinds kind_i, vector<symbol_mt>& vect_o) const;
    void	set_write_nothing_symbol (TapeKinds kind_i, vector<SymbolAndShift>& vect_o) const;


    void	elongate_user_defined_transitions ();
    void	elongate_daemon_defined_fault_transitions ();
    void	build_check_point_transitions ();
    void	sub_build_check_point_transitions (const Transitions_t::const_iterator& iter_i);
    void	build_basement_transitions ();
    void	build_loft_transitions ();
    void	old_build_basement_transitions ();
    void	build_floor1_transitions ();
    void	build_floor2_transitions ();
    void	build_floor3_transitions ();
    void	build_floor4_transitions ();
    void	build_floor5_transitions ();
    void	build_floor6_transitions ();
    void	build_floor7_transitions ();
    void	build_roof1_transitions ();
    void	build_roof2_transitions ();
    void	build_outside_tact1_transitions ();
    void	build_outside_tact2_transitions ();
    void	set_old_info_user_transitions ();
    bool	find_appropriate_rule (
			const CurSituation&	 inst_i, 
			Transitions_t::iterator& iter_o
			);


    void translate_logical_rule_to_actual_program_triple_state_s (
			const NextSituation& next_rule_situation_i,
			const state_t& last_process_substate1_i,
			const state_t& last_process_substate2_i,
			state_t& process_state_to_be_o,
			state_t& process_substate1_to_be_o,
			state_t& process_substate2_to_be_o
			);

    // --------- floors ---------

    void	build_floor2_1_transitions ();
    void	build_floor2_2_transitions ();

    void	build_floor3_1_transitions ();
    void	build_floor3_2_transitions ();
    void	build_floor3_3_transitions ();
    void	build_floor3_4_transitions ();

    void	build_floor4_1_transitions ();
    void	build_floor4_2_transitions ();
    void	build_floor4_3_transitions ();
    void	build_floor4_4_transitions ();

    void	build_floor5_1_transitions ();
    void	build_floor5_2_transitions ();
    void	build_floor5_3_transitions ();
    void	build_floor5_4_transitions ();

    void	build_floor6_1_transitions ();
    void	build_floor6_2_transitions ();
    void	build_floor6_3_transitions ();
    void	build_floor6_4_transitions ();

    void	build_floor7_1_transitions ();
    void	build_floor7_2_transitions ();


    // --------------------------

    void	remote_master_and_masterbackup_tape (
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_master_markers_i
			);
    
    void	remote_master_and_tester_tape (
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_master_markers_i
			);

    void	remote_synchro_and_synchrobackup_tape (
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_synchro_markers_i
			);
    

    void	remote_master_and_synchro_and_synchrobackup_tape (
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_markers_i
			);


    void	compare_master_and_tester_tape (
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_master_blanks_i,
			const state_t&	right_state_for_not_identical_i
			);


    void	compare_master_and_masterbackup_tape (
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_master_blanks_i,
			const state_t&	right_state_for_not_identical_i
			);


    void	compare_synchro_and_synchrobackup_tape_and_setting_master_tape (
			const state_t&	left_state_i, 
			const vector<state_t>&	right_state_for_all_synchro_blanks_i,
			const state_t&	right_state_for_not_identical_i
			);


    void	backup_master_and_masterbackup_tape (
			TapeKinds	from_tape_i, 
			TapeKinds	to_tape_i, 
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_master_blanks_i
			);
    
    void	backup_synchro_and_synchrobackup_tape (
			TapeKinds	from_tape_i, 
			TapeKinds	to_tape_i, 
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_synchro_blanks_i
			);
    

    // --------------------------

    bool	check_user_defined_program_states () const;
    bool	check_extra_program_states () const;
    static bool	check_daemon_states_s ();
    static bool	check_tester_states_s ();
    static bool	check_apparatus_states_s ();
    bool	check_alphabets () const;
    bool	check_user_defined_transition () const;
    bool	check_input_words (const vector<vector<symbol_mt> >& input_words_i) const;
    void	check1_assert () const;

    void	show_descr (const string& msg_i = string()) const;
    void	show_detail_description_of_program_states (const string& msg_i = string()) const;
    void	show_user_defined_program_states (const string& msg_i = string()) const;
    void	show_extra_program_states (const string& msg_i = string()) const;
    static void	show_daemon_states_s (const string& msg_i = string());
    static void	show_tester_states_s (const string& msg_i = string());
    static void	show_apparatus_states_s (const string& msg_i = string());
    void	show_alphabet (const Tapes_t::const_iterator& iter) const;
    void	show_alphabets () const;
    void	show_pseudo_alphabets () const;
    void	show_transitions (
			bool flag_control_via_i,   
			bool flag_from_to_i,  
			const string& msg1_i = string(),
			const string& msg2_i = string()
			) const;
    void	show_daemon_transitions (
			bool flag_control_via_i,   
			bool flag_from_to_i,  
			const string& msg1_i = string(),
			const string& msg2_i = string()
			) const;
    void	show_outside_tact1_transitions (const string& msg1_i = string()) const;
    void	show_outside_tact2_transitions (const string& msg1_i = string()) const;
    vector<string>	getstr_transition_rule (
				const Transitions_t::const_iterator& iter_begin_i, 
				const Transitions_t::const_iterator& iter_i, 
				bool		flag_control_via_i,
				bool		flag_from_to_i,
				const string&	msg_i
				) const;
    vector<string>	getstr_machine_applied_rule (
		   		const Transitions_t::const_iterator& iter_i, 
				const string&	msg_i
				) const;
    vector<string>	getstr_tester_applied_rule (
		  		const Transitions_t::const_iterator& iter_i, 
		       		const string&	msg_i
		       		) const;
    vector<string>	getstr_daemon_applied_rule (
		  		const Transitions_t::const_iterator& iter_i, 
		       		const string&	msg_i
		       		) const;
    string		get_descr_of_program_state (
				const state_t&	state_i, 
				bool		setw_flag_i = false,
				bool		prefix_flag_i = true, 
				bool		suffix_flag_i = true
				) const;
    static string	get_descr_of_tape_kind_s (TapeKinds kind_i);
    static string	get_descr_of_program_state_top_kind_s (ProgramStateTopKinds kind_i);
    static string	get_descr_of_user_defined_program_state_kind_s (UserDefinedProgramStateKinds kind_i);
    static string	get_descr_of_neutral_program_state_kind_s (NeutralProgramStateKinds kind_i);
    static string	get_descr_of_pre_initial_program_state_kind_s (PreInitialProgramStateKinds kind_i);
    static string	get_descr_of_post_halting_program_state_kind_s (PostHaltingProgramStateKinds kind_i);
    static string	get_descr_of_user_required_program_state_kind_s (UserRequiredProgramStateKinds kind_i);
    static string	get_descr_of_embedded_program_state_kinds_s (
				EmbeddedProgramStateKinds	kind_i
				);
    static string	get_descr_of_embedded_program_states__themselves_s (
				EmbeddedProgramStateKinds	kind_i,
				size_t			index_i
				);
    static size_t	get_max_size_descr_of_embedded_program_states__themselves_s ();
    static void 	set_no_shift_s (vector<SymbolAndShift>& vect_o);

    void	show_tape (const Tapes_t::const_iterator& iter) const;
    void	show_situation (
			const string& info_i, 
			size_t interval_no_i = 0, 
			size_t local_no_i = 0, 
			size_t serial_no_i = 0, 
			const string& msg_i = string()
			) const;

    void	show_tester_situation (
			bool start_flag_i, 
			size_t call_no_i = 0, 
			size_t local_no_i = 0, 
			size_t serial_no_i = 0, 
			const string& msg_i = string()
			) const;

    void	show_statistics (const string& msg_i = string()) const;

    void	show_input (const vector<vector<symbol_mt> >& input_words_i, bool full_flag_i = true) const;
    void	show_all_tapes_alphabets_combinations () const;

    string	getstr_id_info (bool detail_flag_i = false) const;


    OusideRuleTypes	get_daemon_behavior (
			const pair<const CurSituation, NextSituation>& rule_i,
			Transitions_t::const_iterator& daemon_fault_iter_o
			) const;
    Transitions_t::const_iterator	daemon_fault_action (
			const pair<const CurSituation, NextSituation>& rule_i
			) const;


  public :
    // Constructor-1
    TuringMachine (
	size_t	number_of_tapes_i,
	// ---
	const vector<vector<string> >& descr_i, 
	// ---
	const vector<state_t>& user_defined_initial_program_states_i, 
	const vector<state_t>& user_defined_halting_program_states_i, 
	const vector<state_t>& user_defined_internal_program_states_i,
	// ---
	const vector<symbol_mt>& user_defined_blank_alphabet_i, 
	const vector<symbol_mt>& user_defined_internal_alphabet_i,
	const vector<symbol_mt>& user_defined_input_alphabet_i,
	// ---
	const Transitions_t& user_defined_transitions_i,
	const Transitions_t& daemon_defined_fault_transitions_i,
	// ---
	const string& msg_i = string ()
	);
    // Destructor
    ~TuringMachine ();  

    bool	process_input (
			const vector<vector<symbol_mt> >& input_words_i,
			const string& msg_i = string()
			);
    void	show1_env () const;
    void	show2_env () const;
    bool	get_check_process_results () const;

};


// ========================			
// --------------
template<typename T>
vector<vector<T> > get_all_permuts (size_t n, const vector<T>& abt_i)
{
vector<vector<T> >	ret_vect;

  if (n > 1)
  {
    vector<vector<T> >	prev_vect (get_all_permuts (n - 1, abt_i));
    for (size_t i = 0; i < prev_vect.size(); i++)
    {
      for (size_t j = 0; j < abt_i.size(); j++)
      {
        vector<T>	tmp_vect (prev_vect[i]);
        tmp_vect.push_back (abt_i[j]);
        ret_vect.push_back (tmp_vect);
      }
    }
  }
  else
  {
    for (size_t j = 0; j < abt_i.size(); j++)
    {
      vector<T>	tmp_vect (1, abt_i[j]);
      ret_vect.push_back (tmp_vect);
    }
  }

  return ret_vect;
}


// --------------
template<typename T>
vector<vector<T> > get_all_taped_permuts (size_t no_of_tapes_i, size_t factor_i, const vector<T>& abt_i)
{
  assert (no_of_tapes_i > 0);
  assert (factor_i > 0);
  return get_all_permuts (no_of_tapes_i * factor_i, abt_i);
}


// --------------
template<typename T>
bool is_homogeneous_permut (const vector<T>& vect_i, const T& symbol_i)
{
  for (size_t i = 0; i < vect_i.size(); i++)
  {
    if (!(vect_i[i] == symbol_i)) return false;
  } 
  return true;
}


// --------------
template<typename T>
vector<vector<T> > get_one_per_tape_kind_permut (size_t no_of_tapes_i, size_t factor_i, const vector<T>& pline_i)
{
  assert (no_of_tapes_i > 0);
  assert (factor_i > 0);
  assert (pline_i.size() == (no_of_tapes_i * factor_i));

vector<vector<T> > ret_vect (factor_i);
  for (size_t i = 0; i < no_of_tapes_i; i++)
  {
    for (size_t j = 0; j < ret_vect.size(); j++)
    {
      const size_t pos = i + no_of_tapes_i * j;
      assert (pos < pline_i.size());
      ret_vect [j].push_back (pline_i[pos]);
    }
  }
  return ret_vect;
}


#endif	// _TURING_S_H
