#!/bin/bash
#
# RatheDG Cachalot System 1.2.0. System installation script.
#

# -- defaults ----------------------------------------------------------------

DEFAULT_CACHALOT_HOME=/opt/cachalot

DEFAULT_LOG_DIR=/var/opt/cachalot

DEFAULT_JAVA_DIST=/tmp/cachalot-java-linux.zip

DEFAULT_TOMCAT_DIST=/tmp/cachalot-tomcat.zip

DEFAULT_JPCAP_DIST=/tmp/cachalot-jpcap-linux.zip

DEFAULT_PGSQL_VER=7.3.4
DEFAULT_PGSQL_DIST=/tmp/postgresql-base-$DEFAULT_PGSQL_VER.tar.gz
DEFAULT_PGSQL_DIR=/opt/pgsql
DEFAULT_PGSQL_DATA=/opt/pgsql/data
DEFAULT_PGSQL_LOG=/var/opt/postgresql
DEFAULT_PGSQL_JDBC=/tmp/cachalot-jdbc-pgsql.zip


# -- helper functions --------------------------------------------------------

# if CACHALOT_HOME not defined
if [ -z "$CACHALOT_HOME" ]; then
    PRG=$0
    # Resolve symlinks.
    while [ -L "$PRG" ]; do
        ls=`ls -ld "$PRG"`
        link=`expr "$ls" : '.*-> \(.*\)$'`
        if expr "$link" : '/' > /dev/null; then
            PRG="$link"
        else
            PRG="`dirname $PRG`/$link"
        fi
    done
    # define CACHALOT_HOME
    CACHALOT_HOME=`dirname "$PRG"`/..
fi

#
# .
#
out()
{    
    echo
    echo $1. 
}

#
# .
#
check()
{    
    ERR=$?
    if [ $ERR -ne 0 ]; then
        echo "Failed, status code = $ERR."
        echo
        exit $ERR
    fi
}

#
# .
#
chfile()
{
    if [ ! -r "$1" ]; then
        echo
        echo "Can't find file: $1"
        echo "Ensure this file exists and can be read."
        echo
        exit 1
    fi
}


# -- users -------------------------------------------------------------------

#
# Creates users for running Cachalot System. Sets file system permissions.
#
users()
{
    CACHALOT_HOME=$1
    if [ -z "$CACHALOT_HOME" ]; then
        CACHALOT_HOME=$DEFAULT_CACHALOT_HOME
    fi

    echo "Using CACHALOT_HOME:  $CACHALOT_HOME"

    #
    # Creating group 'cachalot' for easy permissions management for different
    # system units.
    #
    out "Creating group 'cachalot'"
    groupadd cachalot
    check

    #
    # Creating user 'cachalot' for running UIU.
    # (See bin/uiu_start.sh, bin/uiu_stop.sh).
    #
    out "Creating user 'cachalot' for running UIU"
    useradd -g cachalot cachalot -c "RatheDG Cachalot System"
    check
    
    #
    # Creating user 'cachalot-eru' for running ERU server.
    # (See bin/eru_start.sh, bin/eru_stop.sh).
    #
    out "Creating user 'cachalot-eru' for running ERU server"
    useradd -g cachalot cachalot-eru -c "RatheDG Cachalot System. ERU/S"
    check
    
    #
    # Changing $CACHALOT_HOME owner to user 'cachalot' and group 'cachalot'.
    #
    out "Changing CACHALOT_HOME owner to user 'root' and group 'cachalot'"
    chown -R root:cachalot $CACHALOT_HOME
    check
    
    out "Done"
}


# -- logs --------------------------------------------------------------------

#
# Creates directory for log files.
#
logs()
{
    DIR=$1
    if [ -z "$DIR" ]; then
        DIR=$DEFAULT_LOG_DIR
    fi

    echo "Using CACHALOT_LOGS:  $DIR"

    #
    # Creating directory for log files.
    #
    out "Creating directory for log files"
    mkdir -p $DIR
    check
    
    #
    # Changing owner to user 'cachalot' and group 'cachalot'.
    #
    out "Changing owner to user 'cachalot' and group 'cachalot'"
    chown cachalot:cachalot $DIR
    check

    #
    # Setting permissions on directory.
    #
    out "Setting permissions on directory"
    chmod 775 $DIR
    check

    out "Done"
}
        

# -- java --------------------------------------------------------------------

#
# Installs and configures Java Runtime Environment 1.3.1_08.
#
java()
{
    FILE=$1
    if [ -z "$FILE" ]; then
        FILE=$DEFAULT_JAVA_DIST
    fi

    echo "Using CACHALOT_HOME:  $CACHALOT_HOME"
    echo "Using JAVA_INSTALL:   $FILE"

    #
    # Checking cachalot-java-linux.zip availability.
    #
    chfile $FILE

    #
    # Extracting Java Runtime Environment 1.3.1_08.
    #
    out "Extracting Java Runtime Environment 1.3.1_08"
    unzip -qq -d $CACHALOT_HOME $FILE
    check

    #
    # Changing $CACHALOT_HOME/external owner to user 'root' and
    # group 'cachalot'.
    #
    out "Changing CACHALOT_HOME/external/java owner to user 'root' and group 'cachalot'"
    chown -R root:cachalot $CACHALOT_HOME/external/java
    check
    
    out "Done"
}
               
# -- tomcat ------------------------------------------------------------------

#
# Installs and configures Jakarta Tomcat 4.1.24.
#
tomcat()
{
    TOMCAT=$1
    if [ -z "$TOMCAT" ]; then
        TOMCAT=$DEFAULT_TOMCAT_DIST
    fi

    echo "Using CACHALOT_HOME:   $CACHALOT_HOME"
    echo "Using TOMCAT_INSTALL:  $TOMCAT"

    #
    # Checking cachalot-tomcat.zip availability.
    #
    chfile $TOMCAT

    #
    # Extracting Jakarta Tomcat 4.1.24.
    #
    out "Extracting Jakarta Tomcat 4.1.24"
    unzip -qq -d $CACHALOT_HOME $TOMCAT
    check

    #
    # Changing $CACHALOT_HOME/external owner to user 'root' and
    # group 'cachalot'.
    #
    out "Changing CACHALOT_HOME/external/tomcat owner to user 'root' and group 'cachalot'"
    chown -R root:cachalot $CACHALOT_HOME/external/tomcat
    check

    out "Done"
}

# -- jpcap -------------------------------------------------------------------

#
# Installs and configures JPcap 0.01.13.
#
jpcap()
{
    JPCAP=$1
    if [ -z "$JPCAP" ]; then
        JPCAP=$DEFAULT_JPCAP_DIST
    fi

    HOME=$2
    if [ -z "$HOME" ]; then
        HOME=$DEFAULT_CACHALOT_HOME
    fi

    echo "Using CACHALOT_HOME:  $HOME"
    echo "Using JPCAP_INSTALL:  $JPCAP"

    #
    # Checking cachalot-jpcap-linux.zip availability.
    #
    chfile $JPCAP

    #
    # Extracting JPcap.
    #
    out "Extracting JPcap"
    unzip -qq -d $CACHALOT_HOME $JPCAP
    check

    out "Done"
}

# -- pgsql_inst --------------------------------------------------------------

#
# Installs PostgreSQL 7.3.4.
#
pgsql_inst()
{
    PGDIST=$1
    if [ -z "$PGDIST" ]; then
        PGDIST=$DEFAULT_PGSQL_DIST
    fi

    PGDIR=$2
    if [ -z "$PGDIR" ]; then
        PGDIR=$DEFAULT_PGSQL_DIR
    fi

    PGDATA=$3
    if [ -z "$PGDATA" ]; then
        PGDATA=$DEFAULT_PGSQL_DATA
    fi

    PGLOG=$4
    if [ -z "$PGLOG" ]; then
        PGLOG=$DEFAULT_PGSQL_LOG
    fi

    echo "Using CACHALOT_HOME:    $CACHALOT_HOME"
    echo "Using PGSQL_INSTALL:    $PGDIST"
    echo "Using PGSQL_DIRECTORY:  $PGDIR"
    echo "Using PGSQL_DATA:       $PGDATA"
    echo "Using PGSQL_LOGFILE:    $PGLOG"

    #
    # Checking postgresql-base-7.3.4.tar.gz availability.
    #
    chfile $PGDIST

    #
    # Extracting PostgreSQL.
    #
    out "Extracting PostgreSQL"

    out "Going to /tmp"
    cd /tmp
    check

    out "Extracting $PGDIST"
    tar xzf $PGDIST
    check

    #
    # Installing PostgreSQL.
    #
    out "Installing PostgreSQL"

    out "Going to postgresql-$DEFAULT_PGSQL_VER"
    cd postgresql-$DEFAULT_PGSQL_VER
    check

    #
    # Following is from postgresql-7.3.4/INSTALL file.
    #

    #
    # Configuring PostgreSQL.
    #
    out "Configuring PostgreSQL"
    ./configure --prefix=$PGDIR
    check

    #
    # Building PostgreSQL.
    #
    out "Building PostgreSQL"
    gmake install
    check

    #
    # Creating user 'postgres' for running PostgreSQL.
    #
    out "Creating user 'postgres'"
    useradd -g cachalot postgres
    check

    #
    # Creating database data file $PGDATA.
    #
    out "Creating database data file $PGDATA"
    
    out "Creating directory $PGDATA"
    mkdir $PGDATA
    check

    out "Changing $PGDATA owner to user 'postgres'"
    chown postgres $PGDATA
    check

    out "Setting permissions on $PGDATA"
    chmod 770 $PGDATA
    check
    
    #
    # Creating database log file $LOG.
    #
    out "Creating database log file $PGLOG"
    touch $PGLOG
    check

    out "Changing $PGLOG owner to user 'postgres'"
    chown postgres $PGLOG
    check

    #
    # Starting PostgreSQL.
    #
    out "Initializing PostgreSQL database file"
    DO="cd `pwd`;$PGDIR/bin/initdb -D $PGDATA"
    \su - postgres -m -c "$DO"
    check

    out "Starting PostgreSQL"
    . "$CACHALOT_HOME/bin/postgres.sh" start $PGDIR $PGDATA $PGLOG
    check

    out "Done"
}

# -- pgsql_db ----------------------------------------------------------------

#
# Creates database 'cachalot' in PostgreSQL.
#
pgsql_db()
{
    PGDIR=$1
    if [ -z "$PGDIR" ]; then
        PGDIR=$DEFAULT_PGSQL_DIR
    fi

    echo "Using CACHALOT_HOME:    $CACHALOT_HOME"
    echo "Using PGSQL_DIRECTORY:  $PGDIR"

    PSQL=$PGDIR/bin/psql
    DB=$CACHALOT_HOME/system/database

    out "Initializing PostgreSQL database 'cachalot'"
    DO="cd `pwd`;$PGDIR/bin/createdb cachalot"
    \su - postgres -m -c "$DO"
    check

    out "Creating database 'cachalot': general"
    DO="cd `pwd`;$PSQL -f $DB/create_general.pgsql.sql cachalot"
    \su - postgres -m -c "$DO"
    check

    out "Creating database 'cachalot': operations"
    DO="cd `pwd`;$PSQL -f $DB/create_operations.pgsql.sql cachalot"
    \su - postgres -m -c "$DO"
    check
    
    out "Creating database 'cachalot': permissions"
    DO="cd `pwd`;$PSQL -f $DB/create_permissions.pgsql.sql cachalot"
    \su - postgres -m -c "$DO"
    check

    out "Done"
}

# -- pgsql_jdbc --------------------------------------------------------------

#
# Installs JDBC driver for PostgreSQL.
#
pgsql_jdbc()
{
    JDBC=$1
    if [ -z "$JDBC" ]; then
        JDBC=$DEFAULT_PGSQL_JDBC
    fi

    echo "Using CACHALOT_HOME:  $CACHALOT_HOME"
    echo "Using PGSQL_JDBC:     $JDBC"

    #
    # Checking /tmp/cachalot-jdbc-pgsql.zip availability.
    #
    chfile $JDBC

    #
    # Installing PostgreSQL JDBC driver.
    #
    out "Installing PostgreSQL JDBC driver"
    unzip -qq -d $CACHALOT_HOME $JDBC
    check

    out "Done"
}

# -- usage -------------------------------------------------------------------

echo

case "$1" in
    'users' | 'logs' | 'java' | 'tomcat' | 'jpcap' | \
    'pgsql_inst' | 'pgsql_db' | 'pgsql_jdbc') $* ;;
    *)
        echo "RatheDG Cachalot System 1.2.0 Installer."
        echo
        echo "Parameters:"
        echo
        echo "    users [path-to-cachalot]"
        echo "        Creates users for running Cachalot System. Sets file system permissions."
        echo "        * 'path-to-cachalot' is directory to which distributive file was extracted."
        echo "           Default value is /opt/cachalot/."
        echo
        echo "    logs [path-to-logs]"
        echo "        Creates directory for log files."
        echo "        * 'path-to-logs' is directory for log files."
        echo "           Default value is /var/opt/cachalot/."
        echo
        echo "    java [cachalot-java]"
        echo "        Installs and configures Java Runtime Environment 1.3.1_08."
        echo "        * 'cachalot-java' is path to cachalot-java.zip file."
        echo "           Default value is /tmp/cachalot-java.zip."
        echo
        echo "    tomcat [cachalot-tomcat]"
        echo "        Installs and configures Jakarta Tomcat 4.1.24."
        echo "        * 'cachalot-tomcat' is path to cachalot-tomcat.zip file."
        echo "           Default value is /tmp/cachalot-tomcat.zip."
        echo
        echo "    jpcap [cachalot-jpcap]"
        echo "        Installs and configures JPcap 0.01.13."
        echo "        * 'cachalot-jpcap' is path to cachalot-jpcap-linux.zip file."
        echo "           Default value is /tmp/cachalot-jpcap-linux.zip."
        echo
        echo "    pgsql_inst [distr [install-dir [data-dir [log-file]]]]"
        echo "        Installs PostgreSQL $DEFAULT_PGSQL_VER."
        echo "        * 'distr' is path to postgresql-base-$DEFAULT_PGSQL_VER.tar.gz file."
        echo "           Default value is /tmp/postgresql-base-$DEFAULT_PGSQL_VER.tar.gz."
        echo "        * 'install-dir' is path to PostgreSQL installation."
        echo "           Default value is /opt/postgresql."
        echo "        * 'data-dir' is path to PostgreSQL data directory."
        echo "           Default value is 'install-dir'/data."
        echo "        * 'log-file' is path to PostgreSQL log file."
        echo "           Default value is /var/opt/postgresql."
        echo
        echo "    pgsql_db [path-to-postgresql]"
        echo "        Creates database 'cachalot' in PostgreSQL."
        echo "        * 'path-to-postgresql' is path to PostgreSQL installation."
        echo "           Default value is /usr/local/pgsql."
        echo
        echo "    pgsql_jdbc [pgsql-jdbc]"
        echo "        Installs JDBC driver for PostgreSQL."
        echo "        * 'pgsql_jdbc' is path to cachalot-jdbc-pgsql.zip file."
        echo "           Default value is /tmp/cachalot-jdbc-pgsql.zip."
esac

echo

