using System;
using System.Runtime.Serialization;
using System.ComponentModel;
using System.ComponentModel.Design.Serialization;
using System.Globalization;
using System.Reflection;
using System.Drawing;
using System.Drawing.Design;
using System.Web.UI.WebControls;
using System.Web.UI;
using System.Web.UI.Design;

namespace ComputerBuff.Controls.XPanel
{
	/// <summary>
	///		TargetType enumeration
	/// </summary>
	public enum TargetType
	{
		/// <summary>
		///		Blank
		/// </summary>
		_blank = 0,
		/// <summary>
		///		Parent
		/// </summary>
		_parent = 1,
		/// <summary>
		///		Self
		/// </summary>
		_self = 2,
		/// <summary>
		///		Top
		/// </summary>
		_top = 3,
	}

	/// <summary>
	///		MenuItem class
	/// </summary>
	[Serializable,
	TypeConverter(typeof(MenuItemTypeConverter)),
	DefaultProperty("ID")]
	public class MenuItem
	{
		private string navigateURL = String.Empty;
		private string text = String.Empty;
		private bool showNewImage = false;
		private TargetType urlTarget = TargetType._top;
		private string cssClass = String.Empty;
		private string id = String.Empty;
		private FontInfo font;
		private Color foreColor = Color.Black;
		private Color backColor = Color.White;
		private string imageURL = String.Empty;

		#region Properties
		/// <summary>
		///		Font property
		/// </summary>
		[Bindable(true),
		Category("Appearance"),
		Description("Font used for the menu item"),
		PersistenceMode(PersistenceMode.Attribute),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
		public FontInfo Font
		{
			get
			{
				return this.font;
			}

			set
			{
				this.font = value;
			}
		}

		/// <summary>
		///		BackColor property
		/// </summary>
		[Bindable(true),
		Category("Appearance"),
		Description("Color used for the menu item background"),
		DefaultValue(typeof(Color), "Color.White")]
		public Color BackColor
		{
			get
			{
				return this.backColor;
			}

			set
			{
				this.backColor = value;
			}
		}

		/// <summary>
		///		ForeColor property
		/// </summary>
		[Bindable(true),
		Category("Appearance"),
		Description("Color used for the menu item text"),
		DefaultValue(typeof(Color), "Color.Black")]
		public Color ForeColor
		{
			get
			{
				return this.foreColor;
			}

			set
			{
				this.foreColor = value;
			}
		}

		/// <summary>
		///		Programmatic name of the menu item
		/// </summary>
		[ParenthesizePropertyName(true),
		DefaultValue("")]
		public string ID
		{
			get
			{
				return this.id;
			}

			set
			{
				this.id = value;
			}
		}

		/// <summary>
		///		URL of the menu item
		/// </summary>
		//Editor(typeof(UrlEditor), typeof(UITypeEditor))
		[Browsable(true),
		Category("Navigation"),
		Description("URL of the menu item")]
		public string NavigateUrl
		{
			get
			{
				return this.navigateURL;
			}

			set
			{
				this.navigateURL = value;
			}
		}

		/// <summary>
		///		URL of the menu item
		/// </summary>
		//Editor(typeof(ImageUrlEditor), typeof(UITypeEditor))
		[Browsable(true),
		Category("Appearance"),
		Description("URL of the image for the menu item")]
		public string ImageUrl
		{
			get
			{
				return this.imageURL;
			}

			set
			{
				this.imageURL = value;
			}
		}

		/// <summary>
		///		Text of the menu item
		/// </summary>
		[Browsable(true),
		Category("Appearance"),
		Description("Text of the menu item")]
		public string Text
		{
			get
			{
				return this.text;
			}

			set
			{
				this.text = value;
			}
		}

		/// <summary>
		///		Text of the menu item
		/// </summary>
		[Browsable(true),
		Category("Appearance"),
		Description("CSS Class name to be applied to the menu item"),
		DefaultValue("")]
		public string CssClass
		{
			get
			{
				return this.cssClass;
			}

			set
			{
				this.cssClass = value;
			}
		}

		/// <summary>
		///		Indicates whether to show the 'New' image next to menu item
		/// </summary>
		[Browsable(true),
		Category("Appearance"),
		Description("Indicates whether to show the 'New' image next to menu item")]
		public bool ShowNewImage
		{
			get
			{
				return this.showNewImage;
			}

			set
			{
				this.showNewImage = value;
			}
		}

		/// <summary>
		///		Window target of the menu item
		/// </summary>
		[Browsable(true),
		Category("Navigation"),
		Description("Window target of the menu item")]
		public TargetType Target
		{
			get
			{
				return this.urlTarget;
			}

			set
			{
				this.urlTarget = value;
			}
		}
		#endregion Properties

		#region Methods
		/// <summary>
		///		Custom ToString
		/// </summary>
		public override string ToString()
		{
			return "Menu Item";
		}

		/// <summary>
		///		Creates a new menu item
		/// </summary>
		public MenuItem()
		{
			this.font = new Style().Font;
		}

		/// <summary>
		///		Creates a new menu item
		/// </summary>
		/// <param name="text">Text of the menu item</param>
		/// <param name="navigateURL">URL of the menu item</param>
		/// <param name="urlTarget">Target of the menu item</param>
		public MenuItem(string text, string navigateURL, TargetType urlTarget)
		{
			this.text = text;
			this.navigateURL = navigateURL;
			this.Target = urlTarget;

			this.font = new Style().Font;
		}
		#endregion Methods
	}

	#region MenuItemTypeConverter
	internal class MenuItemTypeConverter : TypeConverter
	{
		public override bool CanConvertTo(ITypeDescriptorContext context, Type destinationType)
		{
			if (destinationType == typeof(InstanceDescriptor))
			{
				return true;
			} 
			
			return base.CanConvertTo(context, destinationType);
		}
		public override object ConvertTo(ITypeDescriptorContext context, CultureInfo culture, 
											object value, Type destinationType)
		{
			if (destinationType == typeof(InstanceDescriptor))
			{
				ConstructorInfo ci = typeof(MenuItem).GetConstructor(
						new Type[] {typeof(string), 
									typeof(string), 
									typeof(TargetType)});

				MenuItem menuItem = (MenuItem) value;
				return new InstanceDescriptor(ci, 
												new object[] {	menuItem.Text, 
																menuItem.NavigateUrl,
																menuItem.Target}, true);
			} 
			
			return base.ConvertTo(context, culture, value, destinationType);
		}
	}
	#endregion MenuItemTypeConverter
}
