/*
    Watbat.c

    Version 0.2 (March 5, 1998) - added support for command-line options to 
    specify variables to set.  Made string compares case insensitive.

    This program scans a wattcp.cfg file and generates a batch file to set 
    the following environment variables:

       IP (local IP address)
       GATEWAY (gateway IP address)
       NETMASK (netmask)
       DNS (nameserver)
       DOMAIN (domain)

    Actually, we just read from standard input and write to standard 
    output.  Usage would be (in a batch file):

       watbat < wattcp.cfg > setparms.bat
       if errorlevel 2 goto badwtcp
       call setparms.bat

    We will only use the first gateway, first netmask, etc., if there is 
    more than one, and only the first domain in the domainslist.  We need 
    to recurse into include files if there are any.  We'll set ERRORLEVELs 
    as follows:

       0    found all requested variables (by default, my_ip, gateway, 
            netmask, nameserver, and domainslist)

       1    found all "required" variables (by default, my_ip, gateway, 
            netmask, and nameserver), but not all of those "not required" 
            (by default, domainslist)

       2    did not find one or more of the "required" variables
*/

#include <stdio.h>      /* Standard I/O functions header file */
#include <string.h>     /* Standard string functions header file */
#include <ctype.h>      /* Standard character functions header file */


/*************************************************************************/
/*                                                                       */
/* Macro definitions.                                                    */
/*                                                                       */
/*************************************************************************/

#ifndef TRUE
#define TRUE    1
#endif
#ifndef FALSE
#define FALSE   0
#endif
#ifndef NULL
#define NULL    0
#endif

    /* Maximum length of a line, plus one. */
#define MAXLINELEN 256

    /* Maximum number of variables to look for. */
#define MAXFIND 10


/*************************************************************************/
/*                                                                       */
/* Types.                                                                */
/*                                                                       */
/*************************************************************************/

struct findparm {
    char *findstr;      /* string to look for */
    char *varname;      /* name of variable to set */
    int notfound;       /* TRUE if not found yet */
    int required;       /* TRUE if required - my_ip, gateway, netmask, and 
                           nameserver are required */
};


/*************************************************************************/
/*                                                                       */
/* Function prototypes.                                                  */
/*                                                                       */
/*************************************************************************/

int find_var( char *var, char **linestart );
void process_file( FILE *infile );


/*************************************************************************/
/*                                                                       */
/* Global variables.                                                     */
/*                                                                       */
/*************************************************************************/

    /* Variables to look for in the file, and DOS environment variables to 
       set for them. */
struct findparm tofind[MAXFIND] =
{
    "my_ip", "IP", TRUE, TRUE,
    "gateway", "GATEWAY", TRUE, TRUE,
    "netmask", "NETMASK", TRUE, TRUE,
    "nameserver", "DNS", TRUE, TRUE,
    "domainslist", "DOMAIN", TRUE, FALSE
    /* rest set to NULL, NULL, FALSE, FALSE by compiler (if MAXFIND > 5) */
};


/*************************************************************************/
/*                                                                       */
/* find_var() function.  Checks whether configuration variable var is    */
/* set on a line.  If not, returns FALSE (linestart is not modified).    */
/* If so, returns TRUE, linestart is set to the start of the variable    */
/* value, and a null is appended at the end of the value.                */
/*                                                                       */
/*************************************************************************/

int find_var( char *var, char **linestart )
{
    int varlen;         /* length of the variable name */
    char *pos;          /* current position in input line */
    char *endquote;     /* position of closing quotation mark */

    /* We stripped off leading whitespace in process_file(), so the 
       variable name must be at the beginning of the line. */
    varlen = strlen( var );
    if ( strnicmp( var, *linestart, varlen ) )
        return( FALSE );

    /* Skip over any whitespace following the variable name to get to the 
       equals sign. */
    for ( pos = *linestart + varlen; isspace( *pos ); pos++ );

    /* There must be an equals. */
    if ( *pos != '=' )
        return( FALSE );

    /* Skip over any whitespace following the equals sign. */
    for ( pos++; isspace( *pos ); pos++ );

    /* If there is nothing left on the line, the value is not set here. */
    if ( !(*pos) )
        return( FALSE );

    /* If the value begins with a quotation mark, we want to find the 
       closing quotation mark.  The quotation marks need to be discarded. 
       */
    if ( *pos == '\"' )
    {
        if ( (endquote = strchr( ++pos, '\"' )) != NULL )
            *endquote = '\0';
        *linestart = pos;
        return( strlen( pos ) );
    }

    /* The value does not begin with a quotation mark.  Whitespace 
       terminates it. */
    *linestart = pos;
    for ( ; !isspace( *pos ); pos++ );
    *pos = '\0';
    return( TRUE );
}


/*************************************************************************/
/*                                                                       */
/* process_file() function.  This routine scans a Waterloo TCP/IP        */
/* configuration file line by line, looking for desired variables.       */
/*                                                                       */
/*************************************************************************/

void process_file( FILE *infile )
{
        /* Buffer used to process lines from the file.  Since this routine 
           is recursive, we want this buffer to be static to save stack 
           space. */
    static char linebuf[MAXLINELEN];
    char *pos;                  /* position of element in line */
    char *linestart;            /* start of processed part of line */
    FILE *includefile;          /* include file, if any */
    int i;                      /* for looping over the variables */

    /* Read lines until EOF or file error. */
    while ( fgets( linebuf, MAXLINELEN, infile ) != NULL )
    {
        /* Delete leading whitespace on the line. */
        for ( linestart = linebuf; isspace( *linestart ); linestart++ );

        /* Scan for a comment.  If there is a comment, truncate the line. 
           */
        if ( (pos = strchr( linestart, '#' )) != NULL )
            *pos = '\0';

        /* Delete trailing whitespace on the line.  Note that newline is 
           whitespace. */
        for ( pos = linestart + strlen( linestart ) - 1;
                pos >= linestart && isspace( *pos );
                pos-- );
        *(++pos) = '\0';

        /* Check for an include file. */
        if ( find_var( "include", &linestart ) )
        {
            /* Take care of conditional includes. */
            if ( *linestart == '?' )
                linestart++;

            /* Open the include file. */
            if ( (includefile = fopen( linestart, "rt" )) == NULL )
                continue;

            /* Process the include file.  Note recursive call - possibility 
               of stack overflow (unlikely, though). */
            process_file( includefile );

            /* Close the include file. */
            fclose( includefile );

            /* Done with this line. */
            continue;
        }

        /* Not an include line.  Check for the variables we're looking for. 
           */
        for ( i=0; i<MAXFIND; i++ )
            if ( tofind[i].notfound )
                if ( !(tofind[i].notfound =
                        !find_var( tofind[i].findstr, &linestart )) )
                {
                    printf( "SET %s=%s\n", tofind[i].varname, linestart );
                    break;
                }
    }
}


/*************************************************************************/
/*                                                                       */
/* Main program.  We're not actually using command-line arguments, but   */
/* perhaps in some future version we'll have options (say, to set MYIP   */
/* instead of IP), so we're including argc, argv for future refernce.    */
/*                                                                       */
/*************************************************************************/

int main( int argc, char *argv[] )
{
    int i, j;           /* for looping */
    char *equalpos;     /* position of '=' or '~' in option string */
    char *findstr;      /* WATTCP variable name */
    char *varname;      /* DOS environment variable name */
    int retcode = 0;    /* ERRORLEVEL returned */

    /* Process command-line options. */
    for ( i=1; i<argc; i++ )
    {
        /* Valid option must have an '=' or '~' in it. */
        if ( (equalpos = strchr( argv[i], '=' )) == NULL )
            if ( (equalpos = strchr( argv[i], '~' )) == NULL )
                continue;

        /* The '=' terminates the WATTCP configuration item.  Convert the 
           DOS variable name to uppercase. */
        findstr = argv[i];
        *equalpos = '\0';
        varname = equalpos + 1;
        strupr( varname );

        /* No variable name after the '=' means do not set the variable for 
           that item (overrides defaults). */
        if ( *varname == '\0' )
        {
            /* Look for the WATTCP configuration item in our list of things 
               to set. */
            for ( j=0; j<MAXFIND; j++ )
                if ( !stricmp( tofind[j].findstr, findstr ) )
                    break;

            /* If we didn't find it, ignore the option.  Otherwise, do not 
               set the DOS variable for this configuration item. */
            if ( j != MAXFIND )
            {
                tofind[j].findstr = tofind[j].varname = NULL;
                tofind[j].notfound = tofind[j].required = FALSE;
            }
        }

        /* DOS variable name follows the '='. */
        else
        {
            /* Look for the WATTCP configuration item in our list of things 
               to set. */
            for ( j=0; j<MAXFIND; j++ )
                if ( !stricmp( tofind[j].findstr, findstr ) )
                    break;

            /* If we didn't find it already in the list, look for a free 
               spot in the list. */
            if ( j == MAXFIND )
                for ( j=0; j<MAXFIND; j++ )
                    if ( tofind[j].findstr == NULL )
                        break;

            /* If the item wasn't already in the list, and there was no 
               free space available, display an error message (on stderr, 
               since stdout is redirected to the batch file) and ignore the 
               option. */
            if ( j == MAXFIND )
            {
                *equalpos = '=';
                fprintf( stderr, "Warning:  too many options - %s ignored.\n",
                    findstr );
                continue;
            }

            /* Add the item to the list.  '!' at the beginning of the 
               variable name means that that variable is to be considered 
               "required" for the purpose of setting ERRORLEVEL. */
            tofind[j].findstr = findstr;
            tofind[j].notfound = TRUE;
            if ( *varname == '!' )
            {
                tofind[j].varname = ++varname;
                tofind[j].required = TRUE;
            }
            else
            {
                tofind[j].varname = varname;
                tofind[j].required = FALSE;
            }
        }
    }

    /* Process the WATTCP.CFG file, generate batch file. */
    process_file( stdin );

    /* Determine return code and exit. */
    for ( i=0; i<MAXFIND; i++ )
    {
        if ( tofind[i].notfound )
        {
            if ( tofind[i].required )
                retcode = 2;
            else if ( retcode < 1 )
                retcode = 1;
        }
    }
    return( retcode );
}
