/* Sound Off!
/* Written by Christopher Taveres 
/* Copyright (c) January 1992
/* Falsoft, Inc.
/* PCM
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <dos.h>
#include <io.h>
#include <ctype.h>

void play(char *buffer, unsigned int length, int speed, int volume);
void wait_for_sound_done(void);
void reset_sound(void);
void print_help_message(void);
int check_machine(void);

/***********************************************************/
/* SOUNDOFF.C   By Christopher Tavares                     */
/* Play a 'raw' sound sample file.  May include any number */
/* of files on the command line, and two command line      */
/* switches - s<speed> and v<volume>.                      */
/* Added 3/20/91 - /b switch, controls the size of the     */
/* chunks loaded in from disk, to compensate for gaps in   */
/* playback.                                               */
/* Added 3/22/91 - program now autodetects Deskmate .snd   */
/* files and properly sets the playback rate from that     */
/* also added - /d switch.  When used the autodetect will  */
/* be disabled until /d shows up on the command line again.*/
/***********************************************************/

#define BUFFERSIZE 31000

/** Definintion for the header block for Deskmate .snd files **/

struct dmheader {                  /* Structure of the header block      */
       int marker;                 /* Marker bytes - should be 00 1a     */
       char note_count;            /* Number of notes in instrument file */
       char inst_num;              /* Instrument number                  */
       char inst_name[10];         /* Instrument name                    */
       int sample_rate;            /* Sampling rate                      */
       char filler[16];            /* I don't know what this does        */
       unsigned long sample_size;  /* Number of samples in file          */
       char filler2[8];            /* More unknown space                 */
};

#define VALIDMARKER 0x001a      /* All .snd files have this marker byte */

void main(int argc,char *argv[])
{
    int fp, i, autodetect, rates[3];
    unsigned int size;
    int speed, volume;
    int bufflen = BUFFERSIZE;
    char buffer1[BUFFERSIZE];
    char buffer2[BUFFERSIZE];
    struct dmheader header;    /* deskmate header */

/* Check if there are any command line arguments - i.e. filenames */

    if (argc == 1) {
        printf("Error:  Must put filename on command line\n");
		print_help_message();
		exit(1);
	}

/* Now, see if this machine even has the right hardware */
    i = check_machine();

    if (i==0) {
        printf("ERROR!  MAJOR ERROR!  Sorry, but this program will only run on\n");
        printf("Tandy 1000SL, 1000TL, 1000SL/2, 1000TL/2, and 1000TL/3 machines.\n");
        printf("Sorry about that!\n");
        exit(1);
    }

/* Set up default values - maximum volume, playback at 22 kHz */
    volume = 7;
    autodetect = 1;   /* autodetect of Deskmate files is on */
    if (i==1)  { /* then machine was an SL */
        speed = 160;
        rates[0] = 160;
        rates[1] = 320;
        rates[2] = 640;
        }
    else {      /* its a TL               */
        speed = 400;
        rates[0] = 400;
        rates[1] = 800;
        rates[2] = 1600;
        }

/* Step through the command line arguments, one at a time */

	for(i = 1; i < argc; i++) {
        if (argv[i][0] == '/') {

			switch (toupper(argv[i][1]))
			{

/* First switch - playback speed */
				case 'S':
                    speed = atoi(argv[i]+2);
                    if(speed > 4095 || speed < 1) {
                        printf("Error:  speed must be between 1 and 4095\n");
                        print_help_message();
                        exit(1);
                    }
                    break;

/* second command line switch - playback volume */
				case 'V':
					volume = atoi(argv[i]+2);
					if(volume < 1 || volume > 7) {
						printf("Error:  volume must be between 1 and 7\n");
						print_help_message();
						exit(1);
					}
					break;

/* third switch - buffer size.  This changes the variable bufflen, */
/* which sets the size of the piece of data that the program reads */
                case 'B':
                     bufflen = atoi(argv[i]+2);
                     if(bufflen<1000 || bufflen > BUFFERSIZE) {
                         printf("Error:  buffer size must be between 1000 and 31000\n");
                         print_help_message();
                         exit(1);
                     }
                     break;
/* fourth switch - /d. Toggle autodetection of Deskmate files  */
                case 'D':
                    if(autodetect == 1)
                        autodetect = 0;
                    else
                        autodetect = 1;

                    break;

                default:
                    printf("Error:  Unknown command line switch\n");
					print_help_message();
                    exit(1);
					break;

				} /* close off the switch */

            }     /* close off the if */

    else {        /* it's a filename, open it up and play it */

        fp = open(argv[i],O_RDONLY | O_BINARY);

        if (fp == -1) {
            fprintf(stderr,"ERROR:  Cannot open %s\n",argv[i]);
            fprintf(stderr,"%s\n",sys_errlist[errno]);
            exit(1);
        }

/* Check to see if its a Deskmate file */

    size = read(fp, &header, sizeof(header));
    if(header.marker == VALIDMARKER && autodetect) {
        switch(header.sample_rate) {
            case 22000:
                 speed = rates[0];
                 break;
            case 11000:
                 speed = rates[1];
                 break;
            case 5500:
                 speed = rates[2];
                 break;
            default:   /* If it wasn't one of the above, it isn't a .snd file */
                 lseek(fp,0,SEEK_SET);  /* reset to beginning of file */
            }
        }
        else
                lseek(fp,0,SEEK_SET);

        wait_for_sound_done();
        size = read(fp,buffer1,bufflen);

        while (size!=0 && size!=0xffff) {
            play(buffer1, size, speed, volume);
            size = read(fp,buffer2, bufflen);

            if (size != 0 && size !=0xffff) {
                wait_for_sound_done();
                play(buffer2, size, speed, volume);
                size = read(fp,buffer1, bufflen);
                wait_for_sound_done();
                }
			}

			close(fp);
			wait_for_sound_done();

        }    /* close off else */

    }        /* for loop */

}            /* and end the function */

void play(char *buffer, unsigned int length, int speed, int volume)
{

    struct REGPACK regs;

    /* set up registers for play sound call */
        regs.r_ax = 0x8300 + volume;
        regs.r_bx = FP_OFF(buffer);
        regs.r_es = FP_SEG(buffer);
        regs.r_dx = speed;
        regs.r_cx = length;
	/* call the bios interrupt */
        intr(0x1a, &regs);

}

void wait_for_sound_done(void)
{
/* wait for the sound to finish */
	union REGS regs;

    do {
		regs.x.ax = 0x8100;
		int86(0x1a, &regs,&regs);
	} while (regs.x.cflag == 1);

}

void reset_sound(void)
{
  /* stop all sound input and output */
  struct REGPACK regs;

  regs.r_ax = 0x8400;
  intr(0x1a,&regs);
}

void print_help_message(void)
{
	/* Print out a help message for the program */

	printf("Usage:  soundoff <options> filename <options> <filename> ... \n");
    printf("\nValid options are:\n");
    printf("     /s<speed>  - playback speed, 1 to 4095, 1 fastest.\n");
    printf("     /v<volume> - volume, 1 to 7, 7 loudest.\n");
    printf("     /d         - disable autodetection of Deskmate .snd files for the\n");
    printf("                  next file only.\n");
    printf("     /b<size>   - size of piece of sample loaded at one time.\n");
    printf("                  Must be between 1000 and 31000.  Use this switch if\n");
    printf("                  there are gaps in the playback.\n\n");
}

int check_machine(void)
{
    /*
       Call Bios interrupt 0x15, function 0xC0, to see if we're running
       on an SL, TL, or other machine.
       Function returns 1 for an SL, 2 for a TL, and 0 for anything else.
    */

    struct REGPACK regs;

    /* this structure is how the information is arranged in memory for */
    /* the machine identification call */

    struct {
        char model;
        char submodel;
        char biosrev;
    } far *m_info;

    /* call machine id info */
    regs.r_ax = 0xc000;
    intr(0x15,&regs);

    /* If carry flag is set, machine isn't TL or SL */
    if (regs.r_flags & 1)
        return(0);

    /* Otherwise, set m_info to point to ROM i.d. data */
    m_info = MK_FP(regs.r_es,regs.r_bx);

    return ((m_info->submodel) + 1);
}

