; RAW2WAV.ASM
;
; This program converts an 8-bit unsigned headerless PCM sound file to .wav
; format by attaching a header and renaming the file.  8-bit raw sound files
; like this are created by MultiPlayer when "file" is selected as the output
; sound device.  Combined with MultiPlayer, this program permits several .mod
; file formats to be converted to .wav.
;     Since the .raw files produced by MultiPlayer are typically several
; megabytes in size, this program overwrites the original .raw file in order
; to save disk space.
;     Syntax:
;
;     RAW2WAV /n <input raw file>
;
; where /n is an optional sampling-rate specification in samples per second,
; e.g., /11025 or /22050.  22000 samples per second is the default.
;

	JMP	START

;
; Data.
;
; Template for .wav header.
;
WAVHEADER	EQU	$
		DB	"RIFF"		; RIFF signature
WAVLEN		DD	0		; (length of .wav file) - 8
		DB	"WAVEfmt "	; .wav signature & format block header
		DD	16		; length of format block
		DW	1		; format type (Microsoft PCM)
		DW	1		; number of channels (1 = mono)
SAMPLESPERSEC	DD	22000		; samples per second
BYTESPERSEC	DD	22000		; bytes/second (same as samples/sec)
		DW	1		; bytes per sample
		DW	8		; bits per sample
		DB	"data"		; data block header
NSAMPLES	DD	0		; number of samples
		;
		; Buffer pointers.
		;
BUFSIZ		EQU	16384		; size of file buffers
BUFFERPTRS	DW	BUF0,BUF1
INBUFFER	DW	0		; current input buffer (0 or 2)
OUTBUFFER	DW	0		; current output buffer (0 or 2)
		;
		; File handles.
		;
INHANDLE	DW	0		; input file handle
OUTHANDLE	DW	0		; output file handle
		;
		; End-of-file flag, set when reading if end of file is
		; encountered.
		;
EOF		DB	0
		;
		; Number of bytes in last buffer of data.
		;
LASTBYTES	DW	0
		;
		; Input file name, and name with ".wav" extension.
		;
FILENAME	DB	128 DUP (0)
NEWNAME		DB	128 DUP (0)
WAVEXT		DB	"WAV"			; for string copy
		;
		; Strings for display to the user.
		;
NOFILEERR	DB	"Must specify input .raw file.",0Dh,0Ah,"$"
SAMPERR		DB	"Sampling rate must be an integer in the range "
		DB	"875-65535.",0Dh,0Ah,"$"
OPENERR		DB	"Error opening input file for reading.",0Dh,0Ah,"$"
SEEKERR		DB	"Seek failed on input file.",0Dh,0Ah,"$"
OUTOPENERR	DB	"Error opening input file for writing.",0Dh,0Ah,"$"
READERR		DB	"Error reading from file.",0Dh,0Ah,"$"
WRITEERR	DB	"Error writing to file.",0Dh,0Ah,"$"
FULLDISKERR	DB	"Disk full.",0Dh,0Ah,"$"
MEMERR		DB	"Insufficient memory.",0Dh,0Ah,"$"
WARNINGMSG	DB	"The input file will be overwritten.  Continue?$"
NOCONFIRMMSG	DB	"Conversion not done.",0Dh,0Ah,"$"
YHIMSG		DB	" (Y/n)  $"
NHIMSG		DB	" (y/N)  $"
		;
		; Buffer for user input.
		;
USERBUF		DB	2
ANSWERED	DB	0			; = 1 if the user answered
ANSWER		DB	2 DUP (0)		; user's answer in first byte
		;
		; User's confirmation that it's OK to overwrite the input
		; file.
		;
CONFIRM		DB	0

;
; Low-level subroutines.
;
; Subroutine, takes pointer to string in DS:SI and length of string in CX,
; skips over blanks and tabs, returns pointer to first nonblank character
; in the string in DS:SI, length of remaining string in CX.  If end of
; string is reached, return pointer to end of string in DS:SI, zero in CX.
;
SKIPBLANKS:
	PUSH	AX
	CLD
SKIPBLANKS_LOOP:
	JCXZ	SKIPBLANKS_END
	LODSB
	DEC	CX
	CMP	AL,9
	JE	SKIPBLANKS_LOOP
	CMP	AL,20h
	JE	SKIPBLANKS_LOOP
	DEC	SI
	INC	CX
SKIPBLANKS_END:
	POP	AX
	RET
;
; Subroutine, takes pointer to string is DS:SI and length of string in CX,
; skips over nonblank characters, returns pointer to first blank or tab in
; the string in DS:SI, length of remaining string in CX.  If end of string
; is reached, return pointer to end of string in DS:SI, zero in CX.
;
SKIPNONBLANK:
	CLD
SKIPNONBLANK_LOOP:
	JCXZ	SKIPNONBLANK_END
	LODSB
	DEC	CX
	CMP	AL,9
	JE	SKIPNONBLANK_LPEND
	CMP	AL,20h
	JNE	SKIPNONBLANK_LOOP
SKIPNONBLANK_LPEND:
	DEC	SI
	INC	CX
SKIPNONBLANK_END:
	RET
;
; Subroutine, converts the decimal number at DS:SI into an integer and stores
; it in the header template.  Halts the program with an error message if not
; a number followed by space or tab, or if the number is not in range (875-
; 65535 inclusive).  At exit, DS:SI addresses the first character following
; the integer, and CX is reduced by the length of the integer.
;
GETRATE:
	PUSH	AX
	PUSH	BX
	PUSH	DX
	PUSH	DI
	MOV	DI,10
	XOR	AX,AX
GETRATE_LOOP:
	MOV	BL,[SI]
	CMP	BL,'9'
	JA	GETRATE_LOOPEND
	SUB	BL,'0'
	JB	GETRATE_LOOPEND
	MOV	BH,0
	MUL	DI
	JC	GETRATE_INVALID
	ADD	AX,BX
	JC	GETRATE_INVALID
	INC	SI
	DEC	CX
	JNZ	GETRATE_LOOP
GETRATE_LOOPEND:
	CMP	AX,875
	JB	GETRATE_INVALID
	MOV	DI,SI
	CALL	SKIPNONBLANK
	CMP	SI,DI
	JNE	GETRATE_INVALID
	MOV	WORD PTR SAMPLESPERSEC,AX
	MOV	WORD PTR BYTESPERSEC,AX
	POP	DI
	POP	DX
	POP	BX
	POP	AX
	RET
	;
	; Invalid sampling rate specified.
	;
GETRATE_INVALID:
	MOV	DX,OFFSET SAMPERR
	JMP	ERROR
;
; High-level subroutines.
;
; Parse command line.  If the first parameter begins with "/", the sampling
; rate is determined from the number that follows, and the second parameter
; is taken for the input filename.  Otherwise, the first parameter becomes 
; the input filename.  Halts the program with an error message if the sampling 
; rate is invalid.
;
PARSECMD:
	PUSH	CX
	PUSH	SI
	PUSH	DI
	;
	; Get first command line argument.
	;
	MOV	CL,[80h]
	MOV	CH,0
	MOV	SI,81h
	CALL	SKIPBLANKS
	JCXZ	PARSECMD_NOFILE		; return if none
	;
	; Command-line argument found.  Check if sampling rate is specified.
	;
	CMP	BYTE PTR [SI],'/'
	JNE	PARSECMD_NORATE
	INC	SI
	DEC	CX
	CALL	GETRATE
	CALL	SKIPBLANKS
	JCXZ	PARSECMD_NOFILE
	;
	; Copy the filename to the local data area.
	;
PARSECMD_NORATE:
	PUSH	SI
	CALL	SKIPNONBLANK
	MOV	DI,SI
	POP	SI
	SUB	DI,SI
	MOV	CX,DI
	MOV	DI,OFFSET FILENAME
	CLD
	REP	MOVSB
	POP	DI
	POP	SI
	POP	CX
	RET
	;
	; No input file name specified.
	;
PARSECMD_NOFILE:
	MOV	DX,OFFSET NOFILEERR
	JMP	ERROR
;
; Open input file for reading.  Halts the program with an error message if
; unable to open the file, or if the file is a device.  Also determines the 
; size of the input file and sets the appropriate fields in the header 
; template.
;
OPENINPUT:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	;
	; Open file for reading, OK to share (deny none).
	;
	MOV	AX,3D40h
	MOV	DX,OFFSET FILENAME
	INT	21h
	JC	OPENINPUT_OPENFAIL
	MOV	INHANDLE,AX			; save handle
	;
	; Seek to end of file to determine size.
	;
	MOV	BX,AX
	MOV	AX,4202h
	XOR	CX,CX
	MOV	DX,CX
	INT	21h
	JC	OPENINPUT_SEEKFAIL
	;
	; Set header fields.
	;
	MOV	WORD PTR NSAMPLES,AX
	MOV	WORD PTR NSAMPLES+2,DX
	ADD	AX,36
	ADC	DX,0
	MOV	WORD PTR WAVLEN,AX
	MOV	WORD PTR WAVLEN+2,DX
	;
	; Rewind the file.
	;
	MOV	AX,4200h
	XOR	CX,CX
	MOV	DX,CX
	INT	21h
	JC	OPENINPUT_SEEKFAIL
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET
	;
	; Unable to open input file.
	;
OPENINPUT_OPENFAIL:
	MOV	DX,OFFSET OPENERR
	JMP	ERROR
	;
	; Input file not seekable.
	;
OPENINPUT_SEEKFAIL:
	MOV	DX,OFFSET SEEKERR
	JMP	ERROR
;
; Subroutine to display a prompt to the user and get a "yes" or "no" answer.
; Takes 3 parameters.  DS:DX addresses the prompt to be displayed (terminated
; with a dollar sign).  DS:BX addresses the location where the answer will
; be stored (0 = no, 1 = yes).  AL is the default answer (if the user just
; hits return or enters something other than "y" or "n").
;
GETYN:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	;
	; Save default answer in CL.
	;
	MOV	CL,AL
	;
	; Display the prompt.
	;
	MOV	AH,9
	INT	21h
	;
	; Display "(Y/n)" if "yes" is the default, or "(y/N)" if "no" is
	; the default.
	;
	CMP	CL,1
	JNE	>L0
	MOV	DX,OFFSET YHIMSG
	JMP	>L1
L0:
	MOV	DX,OFFSET NHIMSG
L1:
	MOV	AH,9
	INT	21h
	;
	; Get the answer.
	;
	MOV	AH,0Ah
	MOV	DX,OFFSET USERBUF
	INT	21h
	;
	; Set result according to answer.
	;
	MOV	[BX],CL				; assume no (default) answer
	CMP	BYTE PTR ANSWERED,1		; just exit if no answer
	JNE	GETYN_EXIT
	;
	; Get answer in AL and convert to 0 or 1.
	;
	MOV	AL,ANSWER
	AND	AL,0DFh				; convert to uppercase
	CMP	AL,'Y'
	JNE	>L0
	MOV	AL,1
	JMP	>L1
L0:
	CMP	AL,'N'
	JNE	GETYN_EXIT
	MOV	AL,0
L1:
	MOV	[BX],AL
	;
	; Move cursor to next line.
	;
GETYN_EXIT:
	MOV	AH,2
	MOV	DL,0Dh
	INT	21h
	MOV	AH,2
	MOV	DL,0Ah
	INT	21h
	POP	DX
	POP	CX
	POP	AX
	RET
;
; Warn the user that the input file will be overwritten and get confirmation
; that that's OK.
;
WARNUSER:
	PUSH	AX
	PUSH	BX
	PUSH	DX
	MOV	DX,OFFSET WARNINGMSG
	MOV	BX,OFFSET CONFIRM
	MOV	AL,1
	CALL	GETYN
	CMP	CONFIRM,0
	JE	WARNUSER_NOCONFIRM
	POP	DX
	POP	BX
	POP	AX
	RET
WARNUSER_NOCONFIRM:
	MOV	DX,OFFSET NOCONFIRMMSG
	JMP	ERROR
;
; Open input file for writing.  Halts the program with an error message if
; unable to open the file.
;
OPENOUTPUT:
	PUSH	AX
	PUSH	DX
	MOV	AX,3D41h
	MOV	DX,OFFSET FILENAME
	INT	21h
	JC	OPENOUTPUT_OPENFAIL
	MOV	OUTHANDLE,AX			; save handle
	POP	DX
	POP	AX
	RET 
	;
	; Error opening file for writing.
	;
OPENOUTPUT_OPENFAIL:
	MOV	DX,OFFSET OUTOPENERR
	JMP	ERROR
;
; Read BUFSIZ bytes of data into the current input buffer.  Halts the program
; with an error message if there is a file error.  If fewer than BUFSIZ bytes
; are read, indicating end of file, the EOF flag is set and the number of
; bytes read is recorded in LASTBYTES for the benefit of WRITELAST.  Switches
; input buffers after reading the data.
;
READDATA:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	;
	; Read BUFSIZ bytes.
	;
	MOV	AH,3Fh
	MOV	BX,INHANDLE
	MOV	CX,BUFSIZ
	MOV	SI,INBUFFER
	MOV	DX,BUFFERPTRS[SI]
	INT	21h
	JC	READDATA_ERROR
	;
	; If less than BUFSIZ bytes read, end of file.
	;
	CMP	AX,BUFSIZ
	JE	READDATA_EXIT
	MOV	EOF,1
	MOV	LASTBYTES,AX
	;
	; Switch buffers.
	;
READDATA_EXIT:
	XOR	INBUFFER,2
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET
	;
	; Error reading from the file.
	;
READDATA_ERROR:
	MOV	DX,OFFSET READERR
	JMP	ERROR
;
; Write BUFSIZ bytes of data from the current output buffer.  Halts the program
; with an error message if there is a file error.  Switches output buffers
; after writing the data.
;
WRITEDATA:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	;
	; Write BUFSIZ bytes.
	;
	MOV	AH,40h
	MOV	BX,OUTHANDLE
	MOV	CX,BUFSIZ
	MOV	SI,OUTBUFFER
	MOV	DX,BUFFERPTRS[SI]
	INT	21h
	JC	WRITEDATA_ERROR
	;
	; Check for full disk.
	;
	CMP	AX,BUFSIZ
	JE	WRITEDATA_EXIT
	MOV	DX,OFFSET FULLDISKERR
	JMP	ERROR
	;
	; Switch output buffers.
	;
WRITEDATA_EXIT:
	XOR	OUTBUFFER,2
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET
	;
	; Error writing to the file.
	;
WRITEDATA_ERROR:
	MOV	DX,OFFSET WRITEERR
	JMP	ERROR
;
; Write the .wav file header from the template.  Halts the program with an
; error message if there is a file error.
;
WRITEHEADER:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	MOV	AH,40h
	MOV	BX,OUTHANDLE
	MOV	CX,44
	MOV	DX,WAVHEADER
	INT	21h
	JC	WRITEHEADER_ERROR
	;
	; Check for full disk.
	;
	CMP	AX,44
	JE	WRITEHEADER_EXIT
	MOV	DX,OFFSET FULLDISKERR
	JMP	ERROR
WRITEHEADER_EXIT:
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET
	;
	; Error writing to file.
	;
WRITEHEADER_ERROR:
	MOV	DX,OFFSET WRITEERR
	JMP	ERROR
;
; Write the last bit of data from the current (last) output buffer.  Halts
; the program with an error message if there is a file error.
;
WRITELAST:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	MOV	AH,40h
	MOV	BX,OUTHANDLE
	MOV	CX,LASTBYTES
	MOV	SI,OUTBUFFER
	MOV	DX,BUFFERPTRS[SI]
	INT	21h
	JC	WRITELAST_ERROR
	;
	; Check for full disk.
	;
	CMP	AX,LASTBYTES
	JE	WRITELAST_EXIT
	MOV	DX,OFFSET FULLDISKERR
	JMP	ERROR
WRITELAST_EXIT:
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET
	;
	; Error writing to file.
	;
WRITELAST_ERROR:
	MOV	DX,OFFSET WRITEERR
	JMP	ERROR
;
; Close the input and output file handles.  File errors ignored since at this
; point there's nothing you could do about it anyway.
;
CLOSEFILES:
	PUSH	AX
	PUSH	BX
	MOV	AH,3Eh
	MOV	BX,INHANDLE
	INT	21h
	MOV	AH,3Eh
	MOV	BX,OUTHANDLE
	INT	21h
	POP	BX
	POP	AX
	RET
;
; Rename the (converted) input file with a ".wav" extension.  The input file
; name is copied and the current extension, if any, is overwritten, then DOS
; is called to change the file name.
;
RENAME:
	PUSH	AX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
	;
	; Copy file name.
	;
	MOV	SI,OFFSET FILENAME
	MOV	DI,OFFSET NEWNAME
	MOV	CX,128
	CLD
	REP	MOVSB
	;
	; Find the end of the new name.
	;
	DEC	DI
	MOV	AL,0
	MOV	CX,128
	STD
	REPE	SCASB
	INC	DI
	MOV	SI,DI			; save pointer to end in case no period
	;
	; Locate a period, if there is one.
	;
	MOV	AL,'.'
	MOV	CX,4
	REPNE	SCASB
	JNE	RENAME_NOPERIOD
	;
	; Period found.
	;
	ADD	DI,2
	JMP	RENAME_WRITEEXT
	;
	; No period.  Put one in.
	;
RENAME_NOPERIOD:
	MOV	DI,SI
	INC	DI
	MOV	BYTE PTR [DI],'.'
	INC	DI
	;
	; Insert "wav" extension.
	;
RENAME_WRITEEXT:
	MOV	SI,OFFSET WAVEXT
	MOV	CX,3
	CLD
	REP	MOVSB
	;
	; Rename the file.
	;
	MOV	AH,56h
	MOV	DX,OFFSET FILENAME
	MOV	DI,OFFSET NEWNAME
	INT	21h
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	AX
	RET

;
; Main program.
;
; Check whether the program has sufficient memory allocated.
;
START:
	MOV	AX,[2]
	MOV	BX,CS
	SUB	AX,BX
	CMP	AX,NEEDED
	JAE	MEMOK
	MOV	DX,OFFSET MEMERR
	JMP	ERROR
	;
	; Parse command line.
	;
MEMOK:
	CALL	PARSECMD
	;
	; Open input file.
	;
	CALL	OPENINPUT
	;
	; Warn the user that the input file will be overwritten.
	;
	CALL	WARNUSER
	;
	; Read in the first BUFSIZ bytes of samples.
	;
	CALL	READDATA
	;
	; Open same file for output.
	;
	CALL	OPENOUTPUT
	;
	; Write the .wav header.
	;
	CALL	WRITEHEADER
	;
	; Loop over the rest of the file, copying.  Stop at EOF.
	;
MAINLOOP:
	CMP	EOF,1
	JE	MAINLOOP_END
	CALL	READDATA
	CALL	WRITEDATA
	JMP	MAINLOOP
	;
	; Write out the partial buffer remaining.
	;
MAINLOOP_END:
	CALL	WRITELAST
	;
	; Close both file handles.
	;
	CALL	CLOSEFILES
	;
	; Rename the file (change extension to ".wav")
	;
	CALL	RENAME
	;
	; Terminate (successful).
	;
	MOV	AX,4C00h
	INT	21h
	;
	; Terminate (error).  Displays the string addressed by DS:DX and exits
	; with ERRORLEVEL 1.
	;
ERROR:	MOV	AH,9
	INT	21h
	MOV	AX,4C01h
	INT	21h

	EVEN
BUF0	EQU	$
BUF1	EQU	BUF0+BUFSIZ
STBOT	EQU	BUF1+BUFSIZ
STTOP	EQU	STBOT+512
NEEDED	EQU	(STTOP+15)/16