; RECWAV.ASM
;
; Version 1.3 - September 30, 1996.  This is a complete rewrite of the
; Recwav program to record .wav files on the Tandy DAC chip.
;
; This is a program to record .wav files on the Tandy 1000SL/TL/RL and
; later models with the proprietary Tandy DAC chip.  In accordance with
; the chip's capabilities, .wav's are recorded in 8-bit mono.
;
; This version is .com rather than .exe; it manages its own memory.  It
; uses autoinitialize DMA for recording to avoid wavering in the recorded
; sound caused by buffer switching in earlier versions.  The DMA buffer
; is 64k, divided into 32 2k segments.  The program will terminate if unable
; to allocate the needed buffer.
;
; This program is based on information contained in the _Tandy 1000TL
; Technical Reference Manual_; the TLSLSND package by Bruce M. Terry, Jr.;
; the RIFF WAVE file format specification excerpted by Rob Ryan from the
; official Microsoft "Multimedia Programming Interface and Data Specification,
; v1.0"; and Frank Durda's PSSJ Digital Sound Toolkit.
;
; Syntax:
;   RECWAV [/N] [/C] [/nnnn] <filename>
;
; Where:
;   /N, if specified, tells Recwav to bypass its search for the best
; sampling rate.  The Tandy DAC can record at a certain discrete set of
; rates.  Recwav, by default, will search for an available rate close to
; the specified one, use the rate it finds, and adjust the .wav header
; accordingly.  With /N, Recwav will take a stab at finding a rate close
; to the specified rate, but it will not check to see whether the rate
; selected is the best one, and it will not adjust the .wav header.  The
; program will start up faster with /N, but sounds will play back at a
; slightly different pitch than they record.
;   /C, if specified, tells Recwav to calibrate the microphone before
; recording.  The user is prompted to plug in the microphone but remain
; silent, and the program records a short snippet of sound to determine
; the correct baseline sample.  All samples subsequently recorded are
; adjusted for the baseline.
;   /nnnn is the optional sampling rate in Hz; for example, /11025 or
; /22050.  A rate between 875 and 64000 may be specified, with the default
; being 11000.  The DAC can record at rates as low as 88Hz, but there's no
; point in it.  If /N is not specified, the specified rate will be ad-
; justed to match one of the DAC's available recording rates (see above).
;   filename is the name of the output .wav file, including drive and path
; if desired.  If the extension is omitted, it defaults to .wav.  To
; specify a file without an extension, end the name with a period.
;

JMP	START

;
; Local data.
;
; Return code from the program (ERRORLEVEL for batch files).  Possible
; return codes are:
;    0      normal termination
;    1      insufficient memory
;    2      no Tandy DAC present
;    3      invalid command line syntax
;    4      create failed on .wav file
;    5      file or disk error writing .wav file
;    6      out of disk space writing .wav file
;    7      input overflow
;
RETCODE		DB	0	; assume no problems
		;
		; Messages to be displayed to the user, one for each
		; possible return code.
		;
MSGNORMAL	DB	"Done.",0Dh,0Ah,"$"
MSGNOMEM	DB	"Insufficient memory.",0Dh,0Ah,"$"
MSGNODAC	DB	"Tandy DAC not found.",0Dh,0Ah,"$"
MSGSYNTAX	DB	"Syntax:",0Dh,0Ah,0Dh,0Ah
		DB	"  RECWAV [/N] [/C] [/nnnn] <filename>"
		DB	0Dh,0Ah,0Dh,0Ah
		DB	"where:",0Dh,0Ah
		DB	"    /N     do not adjust recording rate",0Dh,0Ah
		DB	"    /C     calibrate microphone",0Dh,0Ah
		DB	"    /nnnn  recording rate in Hz (875-64000)",0Dh,0Ah
		DB	"See TSPAK.DOC for details.",0Dh,0Ah,"$"
MSGCREATFAIL	DB	"Unable to create .wav file.",0Dh,0Ah,"$"
MSGFILEERR	DB	"File or disk error writing .wav file.",0Dh,0Ah,"$"
MSGFULL		DB	"Disk full - recording halted.",0Dh,0Ah,"$"
MSGOVERFLOW	DB	"Input overflow - unable to maintain sampling "
		DB	"rate.  Record to a hard disk",0Dh,0Ah
		DB	"or RAM disk, or specify a lower rate.  "
		DB	"Recording deleted.",0Dh,0Ah,"$"
		;
		; Extra messages unrelated to the return code.
		;
ADJMSG		DB	"Adjusting sampling rate....",0Dh,0Ah,"$"
NOADJMSG	DB	"Sampling rate not adjusted.",0Dh,0Ah,"$"
RATEMSG1	DB	"Recording RIFF WAVE file at $"
RATEMSG2	DB	"Hz.",0Dh,0Ah,"$"
CALIBMSG	DB	"Calibrating for the baseline sample.  If using "
		DB	"a microphone, connect",0Dh,0Ah
		DB	"it and remain silent.  If connecting a keyboard, "
		DB	"radio, or tape deck",0Dh,0Ah
		DB	"via a dubbing cable, connect the cable and turn "
		DB	"the device on with the",0Dh,0Ah
		DB	"volume off.  Press a key when ready.",0Dh,0Ah,"$"
RECMSG1		DB	"Press a key to begin recording.",0Dh,0Ah,"$"
RECMSG2		DB	"Press a key to stop recording.",0Dh,0Ah,"$"
		;
		; Pointers to the termination messages above, conveniently
		; placed in an array for easy access.
		;
MSGS		DW	OFFSET MSGNORMAL
		DW	OFFSET MSGNOMEM
		DW	OFFSET MSGNODAC
		DW	OFFSET MSGSYNTAX
		DW	OFFSET MSGCREATFAIL
		DW	OFFSET MSGFILEERR
		DW	OFFSET MSGFULL
		DW	OFFSET MSGOVERFLOW
		;
		; Variables for memory allocation.
		;
FREESEG		DW	0		; paragraph address of unused RAM
ENDALLOC	EQU	WORD PTR [2]	; end of allocated RAM
DMASEG		DW	0		; segment address of 64k DMA buffer
DMAPAGE		DB	0	; DMA page register value for our buffer
		;
		; Base I/O port address of the Tandy DAC.
		;
DACBASE		DW	0
		;
		; PSSJ chip version.  Values are:
		;    0     1000SL
		;    1     1000TL
		;    2     double-buffered chip
		; This information is needed to program the sampling rate.
		;
CHIPTYPE	DB	0
		;
		; Guesstimated clock rates for different chip types.
		;
RATES		DD	311293		; 1000SL
		DD	357986		; 1000TL
		DD	357955		; double-buffered chip
		;
		; DAC divider value.
		;
DIVIDER		DW	0
		;
		; This "magic" number is 0.4394, with the implied decimal
		; point on the left.  To determine the divider value closest
		; to the sampling rate specified by the user, we're going
		; to measure the number of samples recorded at different
		; dividers in 8 18.2 Hz timer ticks.  Since there are 65536
		; PIT clocks in a timer tick, that makes 524,288 PIT clocks.
		; Since the PIT is clocked at 1,193,181 Hz, we get 524,288
		; divided by 1,193,181 equals 0.4394 seconds.  Thus, when we
		; multiply the sampling rate in samples per second by 0.4394,
		; we get the number of samples that *should* be recorded in
		; 8 timer ticks.  The divider that comes closest to recording
		; that number of samples in 8 ticks is the right one.  When
		; we multiply the sampling rate by the magic number, DX is
		; the integer part of the result, AX is the fractional part.
		;     Similarly, when we get a divider that gets as close
		; as we can come to recording the right number of samples,
		; we divide the number of samples recorded in 8 timer ticks
		; by this magic number to get the actual sampling rate that
		; corresponds to the divider.  To do that, put the number of
		; samples recorded in DX and clear AX; the quotient in AX
		; is the result.
		;     Got all that?
		;
MAGIC		DW	707Dh
		;
		; The number of samples that *should* be recorded in 8 timer
		; ticks, if we were able to record at the desired sampling
		; rate.  For adjusting the sampling rate.  See above.
		;
SHOULDGET	DW	0
		;
		; The best value for the divider we found so far, and the
		; number of samples recorded by that divider in 8 timer ticks.
		;
BESTDIVIDER	DW	0
BESTSAMPLES	DW	0
		;
		; The minimum absolute difference between the number of
		; samples that *should* be recorded in 8 timer ticks, and the
		; number that *are*, for all dividers tested so far.
		;
MINDIFF		DW	0
		;
		; Count of DMA EOP interrupts that have occurred since we
		; started writing out a buffer - for detecting overflow.
		;
IRQCOUNT	DB	0
		;
		; Count of timer ticks that have occurred since we started
		; recording.  Used when timing the chip to adjust the sampling
		; rate.
		;
TICKCOUNT	DB	0
		;
		; Default vectors for hooked interrupts.
		;
INT08DEFAULT	DD	0
INT0FDEFAULT	DD	0
INT1BDEFAULT	DD	0
		;
		; Command-line parameter values.
		;
NOADJUST	DB	0	; 1 if no adjustment of sampling rate
CALIBRATE	DB	0	; 1 if the microphone should be calibrated
SAMPRATE	DW	11000	; sampling rate in Hz
		;
		; Filename from the command line.  Copied here so we can
		; append a .wav extension to it.
		;
FILENAME	DB	128 DUP ('*')
WAVSTR		DB	".WAV",0	; .wav extension, for appending
		;
		; Handle of output file.
		;
HANDLE		DW	0
		;
		; .wav header (to write to file).  We will need to fill in
		; RIFFLEN, WAVRATE, WAVRATE2, and DATALEN.
		;
WAVHDRLEN	EQU	2048		; padded out for efficient disk access
EXTRALEN	EQU	WAVHDRLEN-44	; extra length to add for padding
LISTCHKLEN	EQU	EXTRALEN-8	; length of list chunk for header
ISFTLEN		EQU	LISTCHKLEN-12	; length of ISFT info in list chunk
WAVHEADER	DB	"RIFF"
RIFFLEN		DD	0	; length of RIFF thing
RIFFHDRLEN	EQU	$-OFFSET WAVHEADER
		DB	"WAVE"
		DB	"LIST"	; list chunk tag
		DD	LISTCHKLEN	; length of list chunk
		DB	"INFO"	; info subchunk tag
		DB	"ISFT"	; ISFT info type = info about creator program
		DD	ISFTLEN		; ISFT info length
		DB	ISFTLEN/4 DUP ("PAD*")	; we're just padding the
					 	;   header
		DB	"fmt "	; format chunk tag
		DD	16	; format chunk length, always 16
		DW	1	; format tag, 1 = Microsoft PCM
		DW	1	; number of channels, 1 = mono
WAVRATE		DW	0	; sampling rate in Hz
		DW	0	; (high word of sampling rate)
WAVRATE2	DW	0	; bytes per second = sampling rate for 8-bit
				;   mono
		DW	0	; (high word of bytes per second)
		DW	1	; bytes per sample, always 1
		DW	8	; bits per channel, always 8
		DB	"data"	; data chunk tag
DATALEN		DD	0	; length of data chunk
		;
		; Lookup table for calibrating the microphone (adjusting
		; the baseline).  Initially, this table is set for no
		; calibration (or a perfect sound chip :-) so that it works
		; if the user says not to calibrate.
		;
CALIBTABLE	DB	0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15
		DB	16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31
		DB	32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47
		DB	48,49,50,51,52,53,54,55,56,57,58,59,60,61,62,63
		DB	64,65,66,67,68,69,70,71,72,73,74,75,76,77,78,79
		DB	80,81,82,83,84,85,86,87,88,89,90,91,92,93,94,95
		DB	96,97,98,99,100,101,102,103,104,105,106,107,108
		DB	109,110,111,112,113,114,115,116,117,118,119,120
		DB	121,122,123,124,125,126,127,128,129,130,131,132
		DB	133,134,135,136,137,138,139,140,141,142,143,144
		DB	145,146,147,148,149,150,151,152,153,154,155,156
		DB	157,158,159,160,161,162,163,164,165,166,167,168
		DB	169,170,171,172,173,174,175,176,177,178,179,180
		DB	181,182,183,184,185,186,187,188,189,190,191,192
		DB	193,194,195,196,197,198,199,200,201,202,203,204
		DB	205,206,207,208,209,210,211,212,213,214,215,216
		DB	217,218,219,220,221,222,223,224,225,226,227,228
		DB	229,230,231,232,233,234,235,236,237,238,239,240
		DB	241,242,243,244,245,246,247,248,249,250,251,252
		DB	253,254,255
		;
		; 2k buffer segment to write to the file.  Naturally, we
		; process segments in order, 0-31 and back to 0.
		;
CURRENTSEG	DW	0
		;
		; 2k buffer segment that DMA was in when we started to write
		; the current segment.  This is used to determine whether
		; overflow has occurred.
		;
SEGPRE		DW	0

;
; Handler for <control>-C and <control>-<break>.  Does nothing, disabling
; those keys.
;
INT1BHDLR:
INT23HDLR:
	IRET

;
; Critical error handler.  Fails the system call (DOS 3.1 or later).  This
; will abort the program under earlier DOSes, but if playing on a Tandy
; system, DOS 3.3 or later can be assumed, and PSSJ expansion cards are
; very rare.
;
INT24HDLR:
	MOV	AL,3
	IRET 

;
; Replacement timer tick interrupt handler.  Used when timing the chip to
; adjust the sampling rate.
;
INT08HDLR:	
	PUSH	AX
	INC	CS:TICKCOUNT		; add to count of timer ticks
	MOV	AL,20h
	OUT	20h,AL
	POP	AX
	IRET

;
; Routine to check for a Tandy DAC.  If present, clears carry and returns
; with the DAC base port address in AX.  If not, returns with carry set.
;
CHKDAC:
	PUSH	CX
	PUSH	DX
	;
	; Check for PCMCIA Socket Services.
	;
	MOV	AX,8003h
	XOR	CX,CX
	INT	1Ah
	CMP	CX,5353h
	JE	CHKDAC_NODAC
	;
	; Check for Tandy DAC.
	;
	MOV	AX,8100h
	INT	1Ah
	CMP	AX,8000h
	JAE	CHKDAC_NODAC
	MOV	DX,AX		; DX = base DAC port + 2
	ADD	DX,2
	CLI
	IN	AL,DX		; get current value and save in CL
	MOV	CL,AL
	MOV	CH,1		; set CH=1 (assume no DAC)
	XOR	AL,AL		; clear all the bits
	OUT	DX,AL
	IN	AL,DX		; read them back
	OR	AL,AL			; all clear?
	JNZ	CHKDAC_DACCHKDONE	; if not, no DAC
	NOT	AL			; set all the bits
	OUT	DX,AL
	IN	AL,DX			; read them back
	NOT	AL			; all set?
	JNZ	CHKDAC_DACCHKDONE	; if not, no DAC
	MOV	CH,0		; CH=0 (definitely a DAC here)
CHKDAC_DACCHKDONE:
	MOV	AL,CL		; restore previous value at DAC base + 2
	OUT	DX,AL
	STI
	MOV	AX,DX		; get DAC base in AX
	SUB	AX,2
	RCR	CH,1		; clear carry if DAC was there, set if not
	JMP	CHKDAC_END
CHKDAC_NODAC:
	STC
CHKDAC_END:
	POP	DX
	POP	CX
	RET

;
; Routine to determine the PSSJ chip type.  Returns nothing, sets CHIPTYPE.
; This information is needed to program the recording speed.
;
CHKVER:
	PUSH	AX
	PUSH	BX
	PUSH	DX
	PUSH	ES
	MOV	DX,DACBASE	; address amplitude/frequency MSB register
	ADD	DX,3
	CLI
	IN	AL,DX		; get current value
	MOV	AH,AL		; save in AH
	AND	AL,0EFh		; clear bit 4
	OUT	DX,AL
	OR	AL,10h		; set bit 4
	OUT	DX,AL
	IN	AL,DX		; is it set?
	TEST	AL,10h
	JZ	CHKVER_NEW	; if not, it's a new version
	AND	AL,0EFh		; clear bit 4
	OUT	DX,AL
	IN	AL,DX		; is it clear?
	TEST	AL,10h
	JNZ	CHKVER_NEW	; if not, it's a new version
	MOV	AL,AH		; restore DAC register
	OUT	DX,AL
	STI
	MOV	AH,0C0h		; old chip - get system ID
	INT	15h
	JC	CHKVER_TL	; use TL timing if unknown system
	CMP	WORD PTR ES:[BX+2],00FFh
	JE	CHKVER_SL	; it's a 1000SL
	;
	; 1000TL, or unknown system (probably a PSSJ card in an expansion
	; slot).  If the system type is unknown, I just use TL timings for
	; recording.
	;
CHKVER_TL:
	MOV	CHIPTYPE,1
	JMP	CHKVER_DONE
	;
	; 1000SL.
	;
CHKVER_SL:
	MOV	CHIPTYPE,0
	JMP	CHKVER_DONE
	;
	; New-version DAC with a double-buffer for recording (yay!).
	;
CHKVER_NEW:
	MOV	AL,AH		; restore DAC register
	OUT	DX,AL
	STI
	MOV	CHIPTYPE,2
CHKVER_DONE:
	POP	ES
	POP	DX
	POP	BX
	POP	AX
	RET

;
; Subroutine, takes pointer to string in DS:SI and length of string in CX,
; skips over blanks and tabs, returns pointer to first nonblank character
; in the string in DS:SI, length of remaining string in CX.  If end of
; string is reached, return pointer to end of string in DS:SI, zero in CX.
;
SKIPBLANKS:
	PUSH	AX
SKIPBLANKS_LOOP:
	JCXZ	SKIPBLANKS_END
	LODSB
	DEC	CX
	CMP	AL,9
	JE	SKIPBLANKS_LOOP
	CMP	AL,20h
	JE	SKIPBLANKS_LOOP
	DEC	SI
	INC	CX
SKIPBLANKS_END:
	POP	AX
	RET

;
; Subroutine, takes pointer to string in DS:SI and length of string in CX,
; skips over nonblank characters, returns pointer to first blank or tab in
; the string in DS:SI, length of remaining string in CX.  If end of string
; is reached, return pointer to end of string in DS:SI, zero in CX.
;
SKIPNONBLANK:
	PUSH	AX
SKIPNONBLANK_LOOP:
	JCXZ	SKIPNONBLANK_END
	LODSB
	DEC	CX
	CMP	AL,9
	JE	SKIPNONBLANK_LPEND
	CMP	AL,20h
	JNE	SKIPNONBLANK_LOOP
SKIPNONBLANK_LPEND:
	DEC	SI
	INC	CX
SKIPNONBLANK_END:
	POP	AX
	RET

;
; Subroutine, takes pointer to string of digits in DS:SI and length of string
; in CX.  Returns the number represented by the string in AX, DS:SI pointing
; to the first nonnumeric character, length of the remaining string in CX.
; If the end of the string is reached, returns pointer to the end of the
; string in DS:SI, zero in CX.  Returns carry set if the number is greater
; than 65535, carry clear otherwise.
;
PARSERATE:
	PUSH	BX
	PUSH	DX
	PUSH	DI
	;
	; AX accumulates the number.  BX is 10, for multiplying the current
	; number.
	;
	XOR	AX,AX
	MOV	BX,10
	;
	; Loop over the digits.
	;
PARSERATE_LOOP:
	JCXZ	PARSERATE_STREND	; stop if at end of string
	MOV	DI,AX			; save number so far in DI
	LODSB				; get next digit
	DEC	CX
	SUB	AL,'0'			; convert to number
	JB	PARSERATE_NAN		; not a digit
	CMP	AL,9
	JA	PARSERATE_NAN		; not a digit
	CBW				; AX = digit
	XCHG	AX,DI			; number so far in AX, digit in DI
	MUL	BX			; multiply current number by 10
	OR	DX,DX			; did we overflow 16 bits?
	JNZ	PARSERATE_OVERFLOW
	ADD	AX,DI			; add in the current digit
	JNC	PARSERATE_LOOP		; go to next digit if no overflow
	;
	; The number overflowed 16 bits.  Skip over whatever digits remain,
	; and return with carry set.
	;
PARSERATE_OVERFLOW:
	JCXZ	PARSERATE_OVDONE
	LODSB
	DEC	CX
	CMP	AL,'0'
	JB	PARSERATE_OVLPEND
	CMP	AL,'9'
	JBE	PARSERATE_OVERFLOW
PARSERATE_OVLPEND:
	DEC	SI
	INC	CX
PARSERATE_OVDONE:
	STC
	JMP	PARSERATE_DONE
	;
	; We reached the end of the number, but not the end of the string.
	; Back up to the nonnumeric that ended the scan.
	;
PARSERATE_NAN:
	DEC	SI
	INC	CX
	MOV	AX,DI			; get the number in AX
	;
	; We're at the end of the number, one way or the other, and AX is
	; it.  Clear carry and return.
	;
PARSERATE_STREND:
	CLC
PARSERATE_DONE:
	POP	DI
	POP	DX
	POP	BX
	RET

;
; Subroutine, returns with ZF set if the character in AL is a space, tab,
; or slash.
;
STS:
	CMP	AL,' '
	JE	STS_DONE
	CMP	AL,9
	JE	STS_DONE
	CMP	AL,'/'
STS_DONE:
	RET

;
; Routine to parse the command line.  This routine checks for the command
; line switches /N, /C, and /nnnn, setting variables NOADJUST, CALIBRATE,
; and SAMPRATE according to what it finds.  Returns with carry clear if the
; command line is valid, set if not.  If valid, DS:SI addresses the filename
; on return, with a terminating null appended.
;
PARSECMD:
	PUSH	AX
	PUSH	CX
	;
	; Get pointer to the command line in DS:SI, length in CX.  Clear
	; the direction flag.
	;
	MOV	CL,[80h]
	MOV	CH,0
	MOV	SI,81h
	CLD
	;
	; Loop over the command-line parameters.
	;
PARSECMD_LOOP:
	CALL	SKIPBLANKS
	JCXZ	PARSECMD_INVALID	; error if no filename
	;
	; Command-line argument found.  Check if it's a switch.
	;
	LODSB
	CMP	AL,'/'
	JNE	PARSECMD_FILENAME
	;
	; Go to the letter following the slash.  It's an error if there is
	; none.
	;
	DEC	CX
	JZ	PARSECMD_INVALID
	;
	; Get the letter and convert it to uppercase.
	;
	LODSB
	DEC	CX
	AND	AL,0DFh
	;
	; Is it an "N"?
	;
	CMP	AL,'N'
	JNE	PARSECMD_NOTN
	MOV	NOADJUST,1
	MOV	AL,[SI]
	CALL	STS
	JNZ	PARSECMD_INVALID
	JMP	PARSECMD_LOOP
	;
	; Is it a "C"?
	;
PARSECMD_NOTN:
	CMP	AL,'C'
	JNE	PARSECMD_NOTC
	MOV	CALIBRATE,1
	MOV	AL,[SI]
	CALL	STS
	JNZ	PARSECMD_INVALID
	JMP	PARSECMD_LOOP
	;
	; It must be a sampling rate.  Convert the sampling rate from ASCII
	; and save.
	;
PARSECMD_NOTC:
	DEC	SI			; go back to the start of the rate
	INC	CX
	CALL	PARSERATE
	JC	PARSECMD_INVALID	; rate > 65535
	JCXZ	PARSECMD_INVALID	; no filename on command line
	CMP	AX,875
	JB	PARSECMD_INVALID	; must be 875 or greater
	CMP	AX,64000
	JA	PARSECMD_INVALID	; must be 64,000 or less
	MOV	SAMPRATE,AX		; save the rate
	MOV	AL,[SI]
	CALL	STS
	JNZ	PARSECMD_INVALID
	JMP	PARSECMD_LOOP
	;
	; We have a candidate for the filename.
	;
PARSECMD_FILENAME:
	DEC	SI		; get back to the start of the name
	MOV	AX,SI		; save pointer to the name
	;
	; Skip over nonblank characters to find the end of the name and
	; append a null.
	;
	CALL	SKIPNONBLANK
	MOV	BYTE PTR [SI],0
	;
	; The command line is valid.  Clear carry and return a pointer to
	; the name in SI.
	;
	MOV	SI,AX
	CLC
	JMP	PARSECMD_DONE
	;
	; The command line is invalid.  Set carry and return.
	;
PARSECMD_INVALID:
	STC
PARSECMD_DONE:
	POP	CX
	POP	AX
	RET

;
; Subroutine, copies the output filename from the command line into the
; local buffer FILENAME and appends the ".WAV" extension if the name doesn't
; already have an extension.  Returns nothing.  On entry, DS:SI addresses
; the ASCIIZ name from the command line.
;
WAVEXT:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	SI
	PUSH	DI
	;
	; Get the length of the name from the command line.
	;
	MOV	DI,SI		; ES:DI -> name from command line
	MOV	CX,128
	MOV	AL,0
	CLD
	REPNE	SCASB
	SUB	DI,SI		; DI is length of string
	MOV	CX,DI		; get length in CX for move (including null)
	DEC	DI		; go back to the terminating null
	MOV	BX,DI		; save length in BX for scan (no null)
	;
	; Move the name from the command line to the FILENAME buffer.
	;
	MOV	DI,OFFSET FILENAME
	REP	MOVSB		; copy to FILENAME buffer
	;
	; Scan for a period in the name.
	;
	MOV	DI,OFFSET FILENAME	; DI addresses the name
	MOV	AL,'.'			; scan for a period
	MOV	CX,BX			; get count for scan
	REPNE	SCASB
	JE	WAVEXT_EXTFOUND
	;
	; No extension is present.  Append one.
	;
	MOV	SI,OFFSET WAVSTR	; SI -> ".WAV" (ASCIIZ)
	MOV	CX,5			; move 5 bytes
	REP	MOVSB			; append the extension
	;
	; All done.
	;
WAVEXT_EXTFOUND:
	POP	DI
	POP	SI
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Subroutine, records for 8 timer ticks using the recording divider passed
; in AX, returns the number of samples recorded in that time in AX.
;
TIMEDIV:
	PUSH	BX
	PUSH	DX
	PUSH	ES
	;
	; Save the divider in BX for when we need it.
	;
	MOV	BX,AX
	;
	; Set up the DMA controller for the transfer.
	;
	CLI
	MOV	AL,5		; disable DMA channel 1
	OUT	0Ah,AL
	JMP	$+2
	MOV	AL,45h		; select channel 1, write transfer to memory,
	OUT	0Bh,AL		;   autoinitialization disabled, address
	JMP	$+2		;   increment, single mode
	MOV	AL,DMAPAGE
	OUT	83h,AL		; set DMA channel 1 page register
	JMP	$+2
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL
	JMP	$+2
	OUT	03h,AL		; set DMA channel 1 count to maximum
	JMP	$+2
	OUT	03h,AL
	JMP	$+2
	INC	AL		; (0FFh = -1) + 1 = 0 :-)
	OUT	02h,AL		; set DMA channel 1 base address
	JMP	$+2
	OUT	02h,AL
	;
	; DMA controller is ready to go (when enabled).  Set up timer.
	; These are really the BIOS default settings for the timer, but
	; just to make sure ....
	;
	MOV	AL,36h		; counter 0, LSB then MSB, mode 3 (square
	OUT	43h,AL		;   wave), binary
	JMP	$+2
	MOV	AL,0		; use maximum count = 65536 (18.2 ticks/sec)
	OUT	40h,AL
	JMP	$+2
	OUT	40h,AL
	;
	; Timer is ready.  Set up sound chip.  Modified September 30, 1996
	; to add support for the newer version of the PSSJ.  Since we're not
	; really recording, we don't worry about the first couple samples
	; being wrong here; we're only interested in the timing.
	;
	MOV	DX,DACBASE
	ADD	DX,2
	MOV	AX,BX		; get divider back from BX
	OUT	DX,AL
	INC	DX
	MOV	AL,AH
	OUT	DX,AL
	MOV	DX,DACBASE	; set for record mode, DMA disabled,
	MOV	AL,2		;   DMA interrupt disabled, DMA
	OUT	DX,AL		;   interrupt cleared
	MOV	AL,6		; set for record mode, DMA enabled, DMA
	OUT	DX,AL		;   interrupt disabled, DMA interrupt cleared
	INC	DX		; read a sample to start the
	IN	AL,DX		;   approximator - required on 2500's!
	DEC	DX		; get back to DACBASE
	;
	; Sound chip is ready.  Hook Int 8.
	;
	XOR	AX,AX
	MOV	ES,AX
	MOV	BX,20h
	MOV	AX,ES:[BX]
	MOV	WORD PTR INT08DEFAULT,AX
	MOV	AX,ES:[BX+2]
	MOV	WORD PTR INT08DEFAULT+2,AX
	MOV	WORD PTR ES:[BX],OFFSET INT08HDLR
	MOV	ES:[BX+2],CS
	;
	; Set count of timer tick interrupts to zero and enable interrupts.
	;
	MOV	TICKCOUNT,0
	STI
	;
	; Wait for two timer ticks.
	;
TIMEDIV_WAIT1:
	CMP	TICKCOUNT,2
	JB	TIMEDIV_WAIT1
	;
	; Enable DMA channel 1 and start sound chip recording.  Set the
	; tick count back to zero.
	;
	CLI
	MOV	AL,1
	OUT	0Ah,AL
	MOV	TICKCOUNT,0
	STI
	;
	; Wait for eight timer ticks.
	;
TIMEDIV_WAIT2:
	CMP	TICKCOUNT,8
	JB	TIMEDIV_WAIT2
	;
	; Stop sound DMA at the DMA controller and at the sound chip.  DX
	; is still DACBASE from above.
	;
	CLI
	MOV	AL,5		; disable DMA channel 1
	OUT	0Ah,AL
	MOV	AL,2		; disable DMA at the sound chip
	OUT	DX,AL		;   (approximator is still running)
	;
	; Unhook Int 8.  ES:BX -> Int 8 vector still, from above.
	;
	MOV	AX,WORD PTR INT08DEFAULT
	MOV	ES:[BX],AX
	MOV	AX,WORD PTR INT08DEFAULT+2
	MOV	ES:[BX+2],AX
	;
	; Get DMA channel 1 count.
	;
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL
	JMP	$+2
	IN	AL,03h
	JMP	$+2
	MOV	AH,AL
	IN	AL,03h
	XCHG	AL,AH
	NOT	AX		; return number of bytes transferred in AX
	STI
	POP	ES
	POP	DX
	POP	BX
	RET

;
; Routine to set the DAC divider value to be used for recording.  If /N
; was not specified on the command line (i.e., NOADJUST is 0), this routine
; will refine its initial guess by timing the DAC at various divider values,
; then when it finds the best one, change SAMPRATE to match the actual
; recording speed for the divider chosen.
;
SETRATE:
	PUSH	AX
	PUSH	BX
	PUSH	DX
	;
	; Compute the initial guess based on the system type.
	;
	MOV	BL,CHIPTYPE
	XOR	BH,BH
	SHL	BX,1
	SHL	BX,1
	MOV	AX,WORD PTR RATES[BX]	; get a putative clock rate
	MOV	DX,WORD PTR RATES[BX+2]
	MOV	BX,SAMPRATE		; divide by the sampling rate
	DIV	BX
	SHR	BX,1			; round to nearest
	CMP	BX,DX
	ADC	AX,0
	;
	; AX is our initial divider guess.  If NOADJUST is true, we're
	; done.
	;
SETRATE_GOTGUESS:
	MOV	DIVIDER,AX		; save it
	CMP	NOADJUST,1
	JNE	>L0
	JMP	SETRATE_DONE
	;
	; We need to adjust.  Figure out how many samples *should* be
	; recorded in 8 18.2 Hz timer ticks.
	;
L0:
	MOV	AX,SAMPRATE
	MUL	MAGIC			; see notes in data section above
	SHL	AX,1			; round to nearest
	ADC	DX,0
	MOV	SHOULDGET,DX		; save it
	;
	; See how many samples we actually *do* get when we record using
	; our divider guess.
	;
	MOV	AX,DIVIDER
	MOV	BESTDIVIDER,AX	; the guess is the best divider so far
	CALL	TIMEDIV
	MOV	BESTSAMPLES,AX	; ... best number of samples too
	SUB	AX,SHOULDGET	; AX is the difference
	JE	SETRATE_GOTDIV	; not likely, but great if it happens
	JB	SETRATE_TOOHI	; divider is too high if no. of samples 2 low
	;
	; Our guess for the divider seems to be too low.  Increment it and
	; see if it gets better.
	;
	MOV	MINDIFF,AX	; difference was positive
SETRATE_2LOWLP:
	MOV	AX,DIVIDER	; increment the current divider
	INC	AX		; get in AX for TIMEDIV
	MOV	DIVIDER,AX
	CALL	TIMEDIV
	MOV	BX,AX		; save number of samples in BX
	SUB	AX,SHOULDGET	; calculate the absolute difference
	CWD			; this code from Peter Norton - clever, eh?
	XOR	AX,DX
	SUB	AX,DX
	CMP	AX,MINDIFF	; is it better?
	JAE	SETRATE_GOTDIV	; if not, we're done
	MOV	MINDIFF,AX	; it's better - save this difference
	MOV	AX,DIVIDER	; ... and this divider
	MOV	BESTDIVIDER,AX
	MOV	BESTSAMPLES,BX	; ... and this number of samples
	JMP	SETRATE_2LOWLP	; go again
	;
	; Our guess for the divider seems to be too high.  Decrement it and
	; see if it gets better.
	;
SETRATE_TOOHI:
	NEG	AX		; difference was negative, negate for
	MOV	MINDIFF,AX	;   absolute value
SETRATE_2HILP:
	MOV	AX,DIVIDER	; decrement the current divider
	DEC	AX		; get in AX for TIMEDIV
	MOV	DIVIDER,AX
	CALL	TIMEDIV
	MOV	BX,AX		; save number of samples in BX
	SUB	AX,SHOULDGET	; calculate the absolute difference
	CWD			; this code from Peter Norton - clever, eh?
	XOR	AX,DX
	SUB	AX,DX
	CMP	AX,MINDIFF	; is it better?
	JAE	SETRATE_GOTDIV	; if not, we're done
	MOV	MINDIFF,AX	; it's better - save this difference
	MOV	AX,DIVIDER	; ... and this divider
	MOV	BESTDIVIDER,AX
	MOV	BESTSAMPLES,BX	; ... and this number of samples
	JMP	SETRATE_2HILP	; go again
	;
	; We have the best divider available.  Adjust SAMPRATE to match
	; it.  Make the best divider the divider we will use.
	;
SETRATE_GOTDIV:
	XOR	AX,AX
	MOV	DX,BESTSAMPLES
	MOV	BX,MAGIC		; see notes in data section above
	DIV	BX
	SHR	BX,1			; round to nearest
	CMP	BX,DX
	ADC	AX,0
	MOV	SAMPRATE,AX
	MOV	AX,BESTDIVIDER
	MOV	DIVIDER,AX
SETRATE_DONE:
	POP	DX
	POP	BX
	POP	AX
	RET

;
; Routine to display a number in decimal on the screen.  The number is
; passed in AX.  Returns nothing.
;
SHOWNUM:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	XOR	CX,CX		; CX is digit count
	MOV	BX,10		; 10 for division
	;
	; First loop:  Divide repeatedly by 10, pushing the remainders on
	; the stack, till the quotient is zero.  This is a posttest loop,
	; so zero will be displayed correctly.
	;
SHOWNUM_LOOP1:
	XOR	DX,DX
	DIV	BX
	PUSH	DX
	INC	CX
	OR	AX,AX
	JNZ	SHOWNUM_LOOP1
	;
	; Second loop:  Pop the digits off the stack and display them.
	;
SHOWNUM_LOOP2:
	POP	DX
	ADD	DL,'0'
	MOV	AH,2
	INT	21h
	LOOP	SHOWNUM_LOOP2
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Routine to flush the DOS and BIOS keyboard buffers.
;
KEYFLUSH:
	PUSH	AX
	;
	; Flush the DOS typeahead buffer.
	;
	MOV	AX,0CFFh
	INT	21h
	;
	; Flush the BIOS typeahead buffer.
	;
KEYFLUSH_LOOP:
	MOV	AH,1
	INT	16h
	JZ	KEYFLUSH_DONE
	MOV	AH,0
	INT	16h
	JMP	KEYFLUSH_LOOP
KEYFLUSH_DONE:
	POP	AX
	RET

;
; Routine to wait for a keystroke.
;
WAITKEY:
	PUSH	AX
WAITKEY_LOOP:
	MOV	AH,1
	INT	16h
	JZ	WAITKEY_LOOP
	MOV	AH,0
	INT	16h
	POP	AX
	RET

;
; Routine to calibrate the baseline sample for the input - another PSSJ
; bug that we have to work around.  Here we record 2000 and see which one
; occurs most often, assuming that to be the baseline.  It works if the
; microphone or line audio is giving us silence.  This is based on code
; in the PSSJ Digital Sound Toolkit.
;
CALIB:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
	PUSH	BP
	;
	; Set up the DMA controller for the transfer.
	;
	CLI
	MOV	AL,5		; disable DMA channel 1
	OUT	0Ah,AL
	JMP	$+2
	MOV	AL,45h		; select channel 1, write transfer to memory,
	OUT	0Bh,AL		;   autoinitialization disabled, address
	JMP	$+2		;   increment, single mode
	MOV	AL,DMAPAGE
	OUT	83h,AL		; set DMA channel 1 page register
	JMP	$+2
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL
	JMP	$+2
	MOV	AX,2000		; set DMA channel 1 count to 2000
	OUT	03h,AL
	JMP	$+2
	MOV	AL,AH
	OUT	03h,AL
	JMP	$+2
	MOV	AL,0		; set DMA channel 1 base address
	OUT	02h,AL
	JMP	$+2
	OUT	02h,AL
	;
	; DMA controller is ready to go (when enabled).  Set up sound chip.
	;
	MOV	DX,DACBASE
	ADD	DX,2
	MOV	AL,16		; do this at 22kHz, or thereabouts
	OUT	DX,AL
	INC	DX
	XOR	AL,AL
	OUT	DX,AL
	MOV	DX,DACBASE	; set for record mode, DMA disabled,
	MOV	AL,2		;   DMA interrupt disabled, DMA
	OUT	DX,AL		;   interrupt cleared
	MOV	AL,6		; set for record mode, DMA enabled, DMA
	OUT	DX,AL		;   interrupt disabled, DMA interrupt cleared
	INC	DX		; read a sample to start the
	IN	AL,DX		;   approximator - required on 2500's!
	STI
	;
	; Read the first two samples.  The first couple samples are wrong
	; as the approximator takes a few microseconds to get warmed up.
	;
	DEC	DX		; DX = DACBASE
L1:
	IN	AL,DX		; check whether a sample is ready
	TEST	AL,80h
	JNZ	L1
	INC	DX		; sample is ready - read and throw it away
	IN	AL,DX
	DEC	DX		; get back to DACBASE
L2:
	IN	AL,DX		; wait for another sample
	TEST	AL,80h
	JNZ 	L2
	INC	DX		; throw the second sample away too
	IN	AL,DX
	DEC	DX		; get back to DACBASE
	;
	; Enable DMA channel 1 to get things going.
	;
	CLI
	MOV	AL,1
	OUT	0Ah,AL
	JMP	$+2
	STI
	;
	; Gee, we don't have an interrupt.  Guess we better poll to see
	; when we reach terminal count :-).
	;
CALIB_TCLOOP:
	IN	AL,8
	TEST	AL,2
	JZ	CALIB_TCLOOP
	;
	; Stop sound DMA at the DMA controller and at the sound chip.  DX
	; is still DACBASE from above.
	;
	CLI
	MOV	AL,5		; disable DMA channel 1
	OUT	0Ah,AL
	MOV	AL,2		; disable DMA at the sound chip
	OUT	DX,AL		;   (approximator is still running)
	STI
	;
	; Terminal count is reached; we have 2000 samples.  We're going to
	; make a table of how many times we got each sample.  The table can
	; just go after the samples in the DMA buffer.
	;
	MOV	ES,DMASEG	; DS,ES address the DMA buffer
	MOV	DS,DMASEG	; DS doesn't point at local data now!
	CLD			; DF set for incrementing
	;
	; Clear the table to all zeroes.
	;
	MOV	DI,2000		; DS:DI, ES:DI -> occurrence table
	MOV	CX,256
	MOV	BL,0		; start with sample 0
	XOR	AX,AX		; initialize all sample frequencies to zero
CALIB_TABLELP:
	MOV	[DI],BL
	INC	BL
	INC	DI
	STOSW
	LOOP	CALIB_TABLELP
	;
	; Loop over the samples and construct an occurrence table.
	;
	XOR	SI,SI		; DS:SI -> sound samples
	MOV	BX,2000		; DS:BX -> occurrence table
	MOV	CX,BX		; 2000 samples to look at
CALIB_COUNTLP:
	LODSB			; get a sample
	MOV	AH,0		; convert to a word value
	MOV	DI,AX		; DI = 3*sample = offset of entry in table
	ADD	AX,AX
	ADD	DI,AX
	INC	WORD PTR [BX+DI+1] ; increment count for sample
	LOOP	CALIB_COUNTLP
	;
	; Now sort the occurrence table in decreasing order of frequency.
	; DS:BX -> table still.
	;
	MOV	CX,255		; passes through the outer loop (N-1)
CALIB_SORTOUTER:
	MOV	DI,CX		; save outer loop count in DI
	MOV	SI,BX		; start at the top of the table
CALIB_SORTINNER:
	MOV	AX,[SI+1]	; get count from table entry
	CMP	AX,[SI+4]	; compare against next count
	JAE	CALIB_NOSWAP	; if count[n] < count[n+1]:
	XCHG	AX,[SI+4]	;   swap counts
	MOV	[SI+1],AX
	MOV	AL,[SI]		;   swap sample numbers
	XCHG	AL,[SI+3]
	MOV	[SI],AL
CALIB_NOSWAP:
	ADD	SI,3		; go to next table entry
	LOOP	CALIB_SORTINNER
	MOV	CX,DI
	LOOP	CALIB_SORTOUTER
	;
	; The occurrence table is sorted in decreasing order of frequency.
	; Scan through the table computing the cumulative frequency until
	; we get to 1700, or 85% of all the samples.
	;
	MOV	CX,255		; 256-(no. of samples so far)
	MOV	SI,BX		; SI -> first frequency
	INC	SI
	XOR	DX,DX		; DX accumulates frequency
CALIB_FIND85:
	LODSW			; get this frequency
	ADD	DX,AX		; add it in
	CMP	DX,1700		; to 85% yet?
	JAE	CALIB_FIND85DONE ; if so, exit the loop
	INC	SI		; go to next table entry
	LOOP	CALIB_FIND85
	;
	; We know how many entries at the start of the sorted occurrence
	; table we need to get 85% or more of the samples.  Compute the
	; weighted average of those samples.
	;
CALIB_FIND85DONE:
	NEG	CL		; CX is *never* 0 here, never > 255.
	MOV	SI,BX		; SI -> beginning of table
	MOV	BP,DX		; BP = total frequency
	XOR	DI,DI		; DI:BX = sum of sample*weight
	MOV	BX,DI
CALIB_WEIGHTLP:
	LODSB			; AL = sample value
	MOV	DL,AL		; DL = sample value
	MOV	DH,0		; DX = sample value
	LODSW			; AX = weight (frequency)
	MUL	DX		; DX:AX = sample*weight
	ADD	BX,AX		; add into sum
	ADC	DI,DX
	LOOP	CALIB_WEIGHTLP
	;
	; Divide by the total weight to get the weighted average sample.
	;
	MOV	AX,BX
	MOV	DX,DI
	DIV	BP
	SHR	BP,1
	CMP	BP,DX
	ADC	AX,0
	;
	; AX is the baseline sample.  Get DS and ES back.
	;
	MOV	DX,CS
	MOV	DS,DX
	MOV	ES,DX
	;
	; AX is the presumed baseline sample to use for calibration.  Adjust
	; the sample translation table for the baseline.
	;
	MOV	SI,OFFSET CALIBTABLE
	MOV	DI,SI
	MOV	CX,256
	SUB	AX,128		; AX = -adjust, e.g. -2 for 7Eh, 2 for 82h
	NEG	AX		; AX = adjust
	MOV	BX,AX		; BX = adjust
CALIB_CALIBLP:
	LODSB
	MOV	AH,0
	ADD	AX,BX
	JNB	>L3
	MOV	AL,0
	JMP	>L4
L3:
	OR	AH,AH
	JZ	>L4
	MOV	AL,255
L4:
	STOSB
	LOOP	CALIB_CALIBLP
	;
	; All done.
	;
	POP	BP
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Routine to write out the .wav header.  Assumes that all needed fields
; have been filled in and that the file pointer is at the beginning of the
; file.  Returns carry clear if successful.  If unsuccessful, sets carry
; and sets RETCODE to 5 (file or disk error) or 6 (out of disk space).
;
WRITEHDR:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	MOV	AH,40h		; write file or device
	MOV	BX,HANDLE
	MOV	CX,WAVHDRLEN
	MOV	DX,OFFSET WAVHEADER
	INT	21h
	JC	WRITEHDR_FILEERR
	CMP	AX,CX
	JNE	WRITEHDR_DISKFULL
	;
	; All okay.
	;
	CLC
	JMP	WRITEHDR_DONE
	;
	; File or disk error.
	;
WRITEHDR_FILEERR:
	MOV	RETCODE,5
	STC
	JMP	WRITEHDR_DONE
	;
	; Out of disk space.
	;
WRITEHDR_DISKFULL:
	MOV	RETCODE,6
	STC
WRITEHDR_DONE:
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Routine to close the output file.  Returns carry clear if successful.  If
; not, sets carry and sets RETCODE to 5 (file or disk error).  Note:  since
; we intercept critical errors, "invalid handle" is *not* the only possible
; error here.
;
CLOSEFILE:
	PUSH	AX
	PUSH	BX
	MOV	AH,3Eh
	MOV	BX,HANDLE
	INT	21h
	JNC	CLOSEFILE_DONE		; carry already clear if no error
	MOV	RETCODE,5		; carry already set
CLOSEFILE_DONE:
	POP	BX
	POP	AX
	RET

;
; Routine to delete the output file; assumes it has been closed.  For error
; conditions.  Returns nothing (if we have to use this, we've already had
; at least one error, so we ignore additional ones).
;
DELFILE:
	PUSH	AX
	PUSH	DX
	MOV	AH,41h
	MOV	DX,OFFSET FILENAME
	INT	21h
	POP	DX
	POP	AX
	RET

;
; Interrupt 0Fh (DMA EOP) interrupt handler.  This routine just increments
; a count for overflow detection purposes and issues EOI.
;
INT0FHDLR:
	CLI
	PUSH	AX
	PUSH	DX
        ;
        ; Read the interrupt controller's in-service register to see if an
        ; IRQ 7 has in fact occurred.  If not (electrical noise), just restore
        ; registers and return.
        ;
        MOV	AL,0Bh
        OUT	20h,AL
        JMP	$+2
        IN	AL,20h
        TEST	AL,80h
        JZ	INT0FHDLR_RESTORE
	;
	; Check if it was a DAC interrupt.  If not, just issue EOI and
	; return.
	;
	MOV	DX,CS:DACBASE
	IN	AL,DX
	TEST	AL,8
	JZ	INT0FHDLR_EOI
	;
	; It was our interrupt.  Increment the count of DMA EOP interrupts
	; for the underflow checker.  Clear the DMA interrupt at the sound
	; chip so we will get another interrupt.
	;
	INC	CS:IRQCOUNT
	AND	AL,0F7h
	OUT	DX,AL
	OR	AL,8
	OUT	DX,AL
INT0FHDLR_EOI:
	MOV	AL,20h
	OUT	20h,AL
INT0FHDLR_RESTORE:
	POP	DX
	POP	AX
	IRET			; will set the interrupt bit

;
; Routine to start sound recording.  Programs the DMA controller, sound
; chip, and interrupt controller, and hooks Int 0Fh.  Returns nothing.
;
DMASTART:
	PUSH	AX
	PUSH	BX
	PUSH	DX
	PUSH	ES
	;
	; Set up the DMA controller for the transfer.
	;
	CLI
	MOV	AL,5		; disable DMA channel 1
	OUT	0Ah,AL
	JMP	$+2
	MOV	AL,55h		; select channel 1, write transfer to memory,
	OUT	0Bh,AL		;   autoinitialization enabled, address
	JMP	$+2		;   increment, single mode
	MOV	AL,DMAPAGE
	OUT	83h,AL		; set DMA channel 1 page register
	JMP	$+2
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL
	JMP	$+2
	OUT	03h,AL		; set DMA channel 1 count to maximum (64k)
	JMP	$+2
	OUT	03h,AL
	JMP	$+2
	INC	AL		; AL = 0
	OUT	02h,AL		; set DMA channel 1 base address
	JMP	$+2
	OUT	02h,AL
	;
	; DMA controller is ready to go (when enabled).  Set up sound chip.
	;
	MOV	DX,DACBASE
	ADD	DX,2
	MOV	AX,DIVIDER	; program clock divider
	OUT	DX,AL
	INC	DX
	MOV	AL,AH
	OUT	DX,AL
	MOV	DX,DACBASE	; set for record mode, DMA disabled,
	MOV	AL,12h		;   DMA interrupt enabled, DMA
	OUT	DX,AL		;   interrupt cleared
	MOV	AL,1Ah		; set for record mode, DMA disabled, DMA
	OUT	DX,AL		;   interrupt enabled, DMA interrupt allowed
	INC	DX		; read a sample to start the
	IN	AL,DX		;   approximator - required on 2500's!
	;
	; Hook Int 0Fh.
	;
	XOR	AX,AX
	MOV	ES,AX
	MOV	BX,4*0Fh
	MOV	AX,ES:[BX]
	MOV	WORD PTR INT0FDEFAULT,AX
	MOV	AX,ES:[BX+2]
	MOV	WORD PTR INT0FDEFAULT+2,AX
	MOV	WORD PTR ES:[BX],OFFSET INT0FHDLR
	MOV	ES:[BX+2],CS
	;
	; Enable IRQ7 at the interrupt controller.
	;
	IN	AL,21h
	JMP	$+2
	AND	AL,7Fh
	OUT	21h,AL
	STI
	;
	; Read the first two samples.  The first couple samples are wrong
	; as the approximator takes a few microseconds to get warmed up.
	;
	DEC	DX		; DX = DACBASE
L1:
	IN	AL,DX		; check whether a sample is ready
	TEST	AL,80h
	JNZ	L1
	INC	DX		; sample is ready - read and throw it away
	IN	AL,DX
	DEC	DX		; get back to DACBASE
L2:
	IN	AL,DX		; wait for another sample
	TEST	AL,80h
	JNZ 	L2
	INC	DX		; throw the second sample away too
	IN	AL,DX
	DEC	DX		; get back to DACBASE
	;
	; Enable DMA at the sound chip, then at the DMA controller.
	;
	CLI
	MOV	AL,1Eh		; set for record mode, DMA enabled, DMA
	OUT	DX,AL		;   interrupt enabled, DMA interrupt allowed
	MOV	AL,1		; enable DMA channel 1
	OUT	0Ah,AL
	STI
	POP	ES
	POP	DX
	POP	BX
	POP	AX
	RET

;
; Routine to get the DMA current address and convert it to a 2k buffer
; segment number.  Returns the segment number in AX.
;
GETSEG:
	PUSH	CX
	CLI
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL
	JMP	$+2
	IN	AL,2		; read low byte of count
	JMP	$+2
	IN	AL,2		; read high byte of count
	STI
	MOV	CL,3		; convert to segment number
	SHR	AL,CL
	CBW
	POP	CX
	RET

;
; Routine to poll the DMA current address registers to see when it is
; safe to write sound to disk from the current 2k buffer segment, CURRENTSEG.
; It is safe when the current address is not in CURRENTSEG.  Saves the segment
; DMA is in in SEGPRE and sets IRQCOUNT to 0 for later overflow testing.
;
WAITSEG:
	PUSH	AX
WAITSEG_LOOP:
	CALL	GETSEG
	CMP	AX,CURRENTSEG
	JE	WAITSEG_LOOP
	MOV	SEGPRE,AX
	MOV	IRQCOUNT,AH	; segments are 0-31, so AH is 0
	POP	AX
	RET

;
; File writer function.  Writes samples from the 2k buffer segment numbered
; CURRENTSEG (0-31).  Writes out 2k samples and clears carry if successful.
; If unsuccessful, sets carry and sets RETCODE to 5 (file or disk error) or
; 6 (out of disk space).
;
WRITESEG:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	DI
	PUSH	ES
	PUSH	DS
	;
	; Get the offset of CURRENTSEG in the DMA buffer in DI.  Save a
	; copy in DX for later when we write to the file.
	;
	MOV	DX,CURRENTSEG
	XCHG	DL,DH
	MOV	CL,3
	SHL	DH,CL
	MOV	DI,DX
	;
	; Get the offset of the translation table used to adjust the
	; baseline in BX for the XLAT instruction.  ES addresses the DMA
	; buffer.  CX is the number of bytes to adjust (2k).  Clear DF
	; for incrementing.
	;
	MOV	BX,OFFSET CALIBTABLE
	MOV	ES,DMASEG
	MOV	CX,2048
	CLD
	;
	; Adjust the input samples for the baseline.
	;
WRITESEG_ADJLOOP:
	MOV	AL,ES:[DI]	; get a sample
	XLATB			; adjust it
	STOSB			; put it back
	LOOP	WRITESEG_ADJLOOP
	;
	; Write the samples to the file.
	;
	MOV	AH,40h		; write file or device
	MOV	BX,HANDLE	; BX = handle of .wav file
	MOV	CX,2048		; write 2048 bytes
	MOV	DS,DMASEG	; DS:DX -> sound samples
	INT	21h
	JC	WRITESEG_FILEERR
	CMP	AX,CX
	JNE	WRITESEG_DISKFULL
	;
	; All okay.
	;
	CLC
	JMP	WRITESEG_DONE
	;
	; File or disk error.
	;
WRITESEG_FILEERR:
	MOV	CS:RETCODE,5
	STC
	JMP	WRITESEG_DONE
	;
	; Out of disk space.
	;
WRITESEG_DISKFULL:
	MOV	CS:RETCODE,6
	STC
WRITESEG_DONE:
	POP	DS
	POP	ES
	POP	DI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Routine to check for overflow during file writing.  Reads the DMA
; current address registers and the DMA interrupt count to determine
; whether overflow occurred during file writing.  Returns with carry clear
; if no overflow; if overflow, sets RETCODE to 7 ("input overflow") and
; returns with carry set.
;
CHKSEG:
	PUSH	AX
	PUSH	BX
	;
	; Get the current DMA segment (SEGPOST) in AX, segment we loaded
	; into in BX.
	;
	CALL	GETSEG
	MOV	BX,CURRENTSEG
	;
	; Case 1:  previous DMA segment < CURRENTSEG.
	;
	CMP	SEGPRE,BX
	JA	CHKSEG_ABOVE
	CMP	IRQCOUNT,0		; overflow if DMA EOP happened
	JNE	CHKSEG_OVERFLOW
	CMP	AX,BX			; overflow if DMA not still below
	JAE	CHKSEG_OVERFLOW		;   segment just written
	JMP	CHKSEG_OK
	;
	; Case 2:  previous DMA segment > CURRENTSEG.
	;
CHKSEG_ABOVE:
	CMP	IRQCOUNT,1		; no overflow if no DMA EOP
	JB	CHKSEG_OK		; ... but definitely overflow if
	JA	CHKSEG_OVERFLOW		;   2 or more DMA EOP's
	CMP	AX,BX			; overflow if 1 DMA EOP and DMA is
	JAE	CHKSEG_OVERFLOW		;   not below segment just written
	;
	; No overflow.
	;
CHKSEG_OK:
	CLC
	JMP	CHKSEG_DONE
	;
	; Overflow occurred.
	;
CHKSEG_OVERFLOW:
	MOV	RETCODE,7
	STC
CHKSEG_DONE:
	POP	BX
	POP	AX
	RET

;
; Routine to clean up after completing sound playback.  Unhooks Int 0Fh
; and puts the sound chip back in joystick mode.
;
DMASTOP:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	ES
	;
	; Disable DMA channel 1.
	;
	CLI
	MOV	AL,5
	OUT	0Ah,AL
	;
	; Stop recording at the sound chip.
	;
	MOV	DX,DACBASE
	IN	AL,DX
	AND	AL,0FCh		; switch to playback mode temporarily
	OR	AL,3
	OUT	DX,AL
	INC	DX		; read a sample to clear the approximator
	IN	AL,DX
	DEC	DX		; set DAC for joystick mode, DMA interrupt
	MOV	AL,10h		;   enabled still, DMA interrupt cleared
	OUT	DX,AL		;   DMA disabled
	MOV	AL,0		; set DAC for joystick mode, DMA interrupt
	OUT	DX,AL		;   disabled, DMA interrupt cleared, DMA
	INC	DX		;   disabled
	IN	AL,DX		; read another sample just to make sure
	MOV	AL,IRQCOUNT	; get the IRQ count, might need it later
	STI
	;
	; Disabling DMA interrupts at the sound chip may cause one.  Wait
	; for the interrupt, but not forever, since the old version of the
	; chip won't produce one.
	;
	MOV	CX,2000
	LOOP	$
	;
	; The DAC is in joystick mode.  Disable further interrupts on IRQ 7.
	;
	CLI
	MOV	IRQCOUNT,AL	; restore IRQ count
	IN	AL,21h
	JMP	$+2
	OR	AL,80h
	OUT	21h,AL
	JMP	$+2
	;
	; Unhook Int 0Fh.
	;
	XOR	AX,AX
	MOV	ES,AX
	MOV	BX,4*0Fh
	MOV	AX,WORD PTR INT0FDEFAULT
	MOV	ES:[BX],AX
	MOV	AX,WORD PTR INT0FDEFAULT+2
	MOV	ES:[BX+2],AX
	STI
	POP	ES
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Routine to write out the last bit of sound when recording has been
; stopped by a keystroke.  This may be anywhere from 1 to 64k samples,
; starting at the beginning of the 2k buffer segment numbered CURRENTSEG
; (0-31) and possibly wrapping around to the beginning of the DMA buffer.
; When this routine is called, DMA has already stopped.  The DMA channel
; 1 current address registers are used to determine *where* we stopped,
; hence how many bytes need to be written to disk and which.
;     Clears carry if successful.  If unsuccessful, sets carry and sets
; RETCODE to 5 (file or disk error) or 6 (out of disk space).
;
WRITELAST:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	PUSH	SI
	PUSH	DI
	PUSH	ES
	PUSH	DS
	;
	; First, let's find out what segment we stopped in.  If there are
	; whole segments to be written out, we might as well use WRITESEG
	; to do that.
	;
	CALL	GETSEG		; DX = segment we stopped in
	MOV	DX,AX
WRITELAST_WHOLELP:
	CMP	DX,CURRENTSEG	; CURRENTSEG = next segment to be written
	JE	WRITELAST_DOPART ; last segment? go do the partial segment
	CALL	WRITESEG	; write out the whole segment
	LAHF			; save the carry flag
	INC	CURRENTSEG	; go to next segment
	AND	CURRENTSEG,31
	SAHF			; restore the carry flag
	JNC	WRITELAST_WHOLELP ; loop again if no error
	JMP	WRITELAST_DONE	; if file error or full disk, stop
	;
	; OK, all whole segments that needed to be written to disk have
	; been, and DX = CURRENTSEG is the last partial segment.  Get the
	; offset of CURRENTSEG in the DMA buffer in DI.  Save a copy in DX
	; for later when we write to the file.
	;
WRITELAST_DOPART:
	XCHG	DL,DH
	MOV	CL,3
	SHL	DH,CL
	MOV	DI,DX
	;
	; Get the number of samples actually recorded into the current
	; buffer segment in CX.  Save a copy in SI.
	;
	MOV	AL,0FFh		; clear byte pointer flip/flop
	OUT	0Ch,AL
	JMP	$+2
	IN	AL,2		; get DMA channel 1 current address
	MOV	CL,AL		; save low byte in CL
	JMP	$+2
	IN	AL,2		; get high byte and save in CH
	MOV	CH,AL
	SUB	CX,DX		; get number in this segment (omit last one)
	JBE	WRITELAST_OK	; unlikely, but possible
	MOV	SI,CX
	;
	; Get the offset of the translation table used to adjust the
	; baseline in BX for the XLAT instruction.  ES addresses the DMA
	; buffer.  CX is already the number of bytes to adjust.  Clear DF
	; for incrementing.
	;
	MOV	BX,OFFSET CALIBTABLE
	MOV	ES,DMASEG
	CLD
	;
	; Adjust the input samples for the baseline.
	;
WRITELAST_ADJLOOP:
	MOV	AL,ES:[DI]	; get a sample
	XLATB			; adjust it
	STOSB			; put it back
	LOOP	WRITELAST_ADJLOOP
	;
	; Write the samples to the file.
	;
	MOV	AH,40h		; write file or device
	MOV	BX,HANDLE	; BX = handle of .wav file
	MOV	CX,SI		; get back number of bytes
	MOV	DS,DMASEG	; DS:DX -> sound samples
	INT	21h		; DS does not address local data any more!
	JC	WRITELAST_FILEERR
	CMP	AX,CX
	JNE	WRITELAST_DISKFULL
	;
	; All okay.
	;
WRITELAST_OK:
	CLC
	JMP	WRITELAST_DONE
	;
	; File or disk error.
	;
WRITELAST_FILEERR:
	MOV	CS:RETCODE,5
	STC
	JMP	WRITELAST_DONE
	;
	; Out of disk space.
	;
WRITELAST_DISKFULL:
	MOV	CS:RETCODE,6
	STC
WRITELAST_DONE:
	POP	DS
	POP	ES
	POP	DI
	POP	SI
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Routine to add the length fields to the .wav header.  This is done by
; seeking to the end of the file to determine its length (the kludgy way
; you have to do it under DOS), then subtracting the length of the header.
; The results are filled in in the WAVHEADER structure above, then we
; seek to the beginning of the file and write out the adjusted header.
;
FIXHDR:
	PUSH	AX
	PUSH	BX
	PUSH	CX
	PUSH	DX
	;
	; Seek to the end of the file to get its length.  In reality, we're
	; already there when this routine is called, but this is the way to
	; get the file pointer, which is the length.
	;
	MOV	AX,4202h	; seek relative to end of file
	MOV	BX,HANDLE
	XOR	CX,CX		; seek to offset 0
	MOV	DX,CX
	INT	21h
	JC	FIXHDR_FILEERR
	;
	; DX:AX is the length of the file.  Fill in the length of the
	; data block.
	;
	SUB	AX,WAVHDRLEN
	SBB	DX,0
	MOV	WORD PTR DATALEN,AX
	MOV	WORD PTR DATALEN+2,DX
	;
	; Fill in the length of the RIFF "thing" (yes, that's what MS calls
	; it).
	;
	ADD	AX,WAVHDRLEN-RIFFHDRLEN
	ADC	DX,0
	MOV	WORD PTR RIFFLEN,AX
	MOV	WORD PTR RIFFLEN+2,DX
	;
	; Seek back to the beginning of the file.
	;
	MOV	AX,4200h	; seek relative to beginning of file
	XOR	DX,DX		; seek to offset 0 - BX = HANDLE still,
	INT	21h		;   CX is still 0 from above
	JC	FIXHDR_FILEERR
	;
	; Write out the adjusted header.
	;
	CALL	WRITEHDR
	JC	FIXHDR_FILEERR	; "full disk" is an impossible error here
	;
	; Seek back to the end of the file.  This isn't really necessary,
	; but I feel strange closing a file with the pointer in the middle.
	;
	MOV	AX,4202h	; seek relative to end of file
	INT	21h		; BX = HANDLE still, CX never changed, DX
	JNC	FIXHDR_DONE	;   set to 0 by previous seek
	;
	; File or disk error.
	;
FIXHDR_FILEERR:
	MOV	RETCODE,5
	STC
FIXHDR_DONE:
	POP	DX
	POP	CX
	POP	BX
	POP	AX
	RET

;
; Main program.
;
; Save the default Int 1Bh vector.  We don't need to save the 23h or 24h
; vectors since DOS will restore them for us.
;
START:
	MOV	AX,351Bh
	INT	21h
	MOV	WORD PTR INT1BDEFAULT,BX
	MOV	WORD PTR INT1BDEFAULT+2,ES
	MOV	AX,DS			; restore ES (ES = DS = CS always)
	MOV	ES,AX
	;
	; Hook Int 1Bh to disable <control>-<break>.
	;
	MOV	AX,251Bh
	MOV	DX,OFFSET INT1BHDLR
	INT	21h
	;
	; Hook Int 23h to disable <control>-C.
	;
	MOV	AX,2523h
	MOV	DX,OFFSET INT23HDLR
	INT	21h
	;
	; Hook the critical error interrupt.  We don't want to get aborted
	; if there's a disk error since the DAC will need to be reset and
	; interrupts unhooked.
	;
	MOV	AX,2524h
	MOV	DX,OFFSET INT24HDLR
	INT	21h
	;
	; Shrink the stack to 2 kilobytes.
	;
	MOV	SP,OFFSET STACKEND
	;
	; Find the start of unused RAM.
	;
	MOV	AX,SP		; AX = paragraphs for data, code and stack
	ADD	AX,15
	MOV	CL,4
	SHR	AX,CL
	MOV	BX,CS		; add in paragraph address of program start
	ADD	AX,BX		; AX = paragraph address of unused RAM
	MOV	FREESEG,AX	; save it
	;
	; Determine a candidate for DMA buffer.
	;
	ADD	AX,0FFFh	; round AX up to a 64k boundary
	AND	AX,0F000h
	MOV	DMASEG,AX	; save it
	;
	; Verify the DMA buffer.
	;
	MOV	RETCODE,1	; "insufficient RAM" if we can't get buffer
	CMP	AX,ENDALLOC	; if past the end of RAM, skip it
	JB	>L1
	JMP	TERMINATE
L1:
	ADD	AX,1000h	; make sure we own the whole thing
	CMP	AX,ENDALLOC
	JBE	>L2
	JMP	TERMINATE
	;
	; Compute the DMA page register value for our buffer.
	;
L2:
	MOV	AX,DMASEG
	MOV	CL,4
	SHR	AH,CL
	MOV	DMAPAGE,AH
	;
	; We have a buffer allocated.  Check for the presence of a PSSJ
	; chip.
	;
	MOV	RETCODE,2	; "no DAC" if we don't find it
	CALL	CHKDAC
	JNC	>L3
	JMP	TERMINATE
L3:
	MOV	DACBASE,AX	; save DAC base I/O port address
	CALL	CHKVER		; get the DAC version, old or new
	;
	; Parse the command line.
	;
	MOV	RETCODE,3	; syntax message if invalid
	CALL	PARSECMD
	JNC	>L4
	JMP	TERMINATE
	;
	; The command line was valid, and DS:SI addresses the filename.
	; Copy the filename to the local buffer and append the .wav extension
	; if it doesn't already have one.
	;
L4:
	CALL	WAVEXT
	;
	; The filename is in the FILENAME buffer, with extension appended.
	; Create (and open) the file.
	;
	MOV	RETCODE,3	; "unable to create file" if unsuccessful
	MOV	AH,3Ch		; create file with handle
	XOR	CX,CX		; normal file
	MOV	DX,OFFSET FILENAME
	INT	21h
	JNC	>L5
	JMP	TERMINATE
L5:
	MOV	HANDLE,AX	; save file handle returned
	;
	; The command line was valid and we created an output file.  We set
	; RETCODE to 0 - it doesn't change again unless we get an error of
	; some sort.
	;
	MOV	RETCODE,0
	;
	; Tell the user what we're doing.
	;
	MOV	DX,OFFSET ADJMSG
	CMP	NOADJUST,0
	JE	>L6
	MOV	DX,OFFSET NOADJMSG
L6:
	MOV	AH,9
	INT	21h
	;
	; Set the recording speed (may take a little time).
	;
	CALL	SETRATE
	;
	; Display the sampling rate we're using to the user.
	;
	MOV	DX,OFFSET RATEMSG1
	MOV	AH,9
	INT	21h
	MOV	AX,SAMPRATE
	CALL	SHOWNUM
	MOV	DX,OFFSET RATEMSG2
	MOV	AH,9
	INT	21h
	;
	; Check if we're supposed to calibrate the microphone.
	;
	CMP	CALIBRATE,1
	JNE	NOCALIBRATE
	;
	; The microphone should be calibrated.  Do so.  Tell the user what
	; to do.
	;
	MOV	DX,OFFSET CALIBMSG
	MOV	AH,9
	INT	21h
	CALL	KEYFLUSH	; flush the keyboard buffer
	CALL	WAITKEY		; wait for a keystroke
	CALL	CALIB		; do the calibration
	;
	; Now we should be ready to start recording.  Write out a dummy
	; .wav header to reserve space for it in the file; we'll fill in
	; the length fields later.
	;
NOCALIBRATE:
	MOV	AX,SAMPRATE	; fill in sampling rate fields
	MOV	WAVRATE,AX
	MOV	WAVRATE2,AX
	CALL	WRITEHDR	; write the dummy .wav header
	JNC	RECSTART	; if unsuccessful, then:
	CMP	RETCODE,6	;   if disk is full (i.e., not a file error)
	JNE	>L7		;       then:
	CALL	CLOSEFILE	;     close the file
	JC	>L7
	CALL	DELFILE		;     delete the file if able to close it
L7:
	JMP	TERMINATE
	;
	; We're ready to start recording.  Tell the user so.
	;
RECSTART:
	MOV	DX,OFFSET RECMSG1
	MOV	AH,9
	INT	21h
	CALL	KEYFLUSH	; flush the keyboard buffer
	CALL	WAITKEY		; wait for a keystroke
	CALL	DMASTART	; start DMA
	;
	; Tell the user to hit a key when he wants to stop.
	;
	MOV	DX,OFFSET RECMSG2
	MOV	AH,9
	INT	21h
	CALL	KEYFLUSH	; flush the keyboard buffer
	;
	; Write buffer segments to disk until the user hits a key or until
	; there is an error of some sort.
	;
	MOV	CURRENTSEG,0	; not really necessary, for clarity
RECLOOP:
	MOV	AH,1		; has a key been pressed?
	INT	16h
	JNZ	RECSTOP		; stop if so
	CALL	WAITSEG		; wait until a segment is ready
	CALL	WRITESEG	; write a 2k buffer segment to disk
	JC	RECSTOP		; stop if error writing
	CALL	CHKSEG		; check for overflow
	JC	RECSTOP
	INC	CURRENTSEG	; next buffer segment for the reader
	AND	CURRENTSEG,31
	JMP	RECLOOP		; go again (wrap around to the beginning)
	;
	; It's time to stop recording, hopefully because the user pressed
	; a key, but possibly for a less pleasant reason.
	;
RECSTOP:
	CALL	DMASTOP		; unhook Int 0Fh, finalize sound chip
	;
	; If things have gone well so far, write out the last partial
	; buffer.
	;
	CMP	RETCODE,0
	JNE	>L8
	CALL	WRITELAST
	;
	; If we got a file or disk error, stop right now.  The output file,
	; if any, is junk, but we can't reliably delete it.
	;
L8:
	CMP	RETCODE,5
	JE	TERMINATE
	;
	; If we got input overflow, try to close and delete the file, then
	; exit.
	;
	CMP	RETCODE,7
	JNE	>L9
	CALL	CLOSEFILE
	JC	TERMINATE
	CALL	DELFILE
	JMP	TERMINATE
	;
	; At this point, we either have a full disk, which is OK (as long
	; as there is room for the .wav header), or we're headed towards
	; normal termination.  Go fix up the .wav header.  The only possible
	; error is a file or disk error, so if we get one, just terminate.
	;
L9:
	CALL	FIXHDR
	JC	TERMINATE
	;
	; All done.  Close the file.  There's nothing left to do if we get
	; an error here.
	;
	CALL	CLOSEFILE
	;
	; Jump here when we're done.  RETCODE is the return code for the
	; program.
	;
TERMINATE:
	CALL	KEYFLUSH	; flush the keyboard buffer
	LDS	DX,INT1BDEFAULT	; unhook Int 1Bh
	MOV	AX,251Bh
	INT	21h
	MOV	AX,CS		; get DS back
	MOV	DS,AX
	MOV	BL,RETCODE	; display the message for the return code
	MOV	BH,0
	SHL	BX,1
	MOV	DX,MSGS[BX]
	MOV	AH,9
	INT	21h
	MOV	AL,RETCODE	; terminate with return code
	MOV	AH,4Ch
	INT	21h
	EVEN

;
; Reserve 2k for the stack.
;
STACKSTART	DW	1024 DUP (0)
STACKEND	EQU	$
