; VIDEO.ASM
;
; This assembler module contains routines for handling the video display,
; as well as miscellaneous junk that doesn't fit elsewhere.
;

;
; Globally-visible stuff.
;
PUBLIC	INITVID		; video initialization routine
PUBLIC	ENDVID		; video finalization routine
PUBLIC	VIDPUTENTRY	; routine to put pattern table entry on screen
PUBLIC	VIDPUTNENTRIES	; routine to put size of pattern table on screen
PUBLIC	VIDPUTPATNUM	; routine to put pattern number on screen
PUBLIC	VIDPUTNPATTERNS	; routine to put number of patterns on screen
PUBLIC	VIDPUTNOTENUM	; routine to put a note number on screen
PUBLIC	VIDPUTNUMS	; routine to set all numbers on the screen
PUBLIC	VIDUPDATENUMS	; routine to update numbers during song playback
PUBLIC	VIDPUTVOLUME	; routine to put the DAC volume on screen
PUBLIC	VIDPUTRATE	; routine to put output sampling rate on screen
PUBLIC	VIDPUTLOOPEN	; routine to put loop disable setting on screen
PUBLIC	VIDPUTTICKSPER	; routine to put ticks/division on screen
PUBLIC	VIDPUTBPM	; routine to put beats/minute on screen
PUBLIC	VIDPUTNCHANS	; routine to put number of channels on screen
PUBLIC	VIDPUTNINSTRS	; routine to put number of instruments on screen
PUBLIC	VIDPUTGLOBALVOL	; routine to put global volume on screen
PUBLIC	VIDPUTEMSEN	; routine to put EMS enable setting on screen
PUBLIC	VIDPUTSAVEMEM	; routine to put save memory setting on screen
PUBLIC	VIDPUTLOADOPT	; routine to put loading optimization setting on screen
PUBLIC	VIDPUTMSG	; routine to put a message/prompt on screen
PUBLIC	VIDPUTERRMSG	; routine to put an error message on screen
PUBLIC	VIDSTARTINPUT	; routine to clear input line on screen
PUBLIC	VIDBLOCKCURS	; routine to make the cursor a block
PUBLIC	VIDLINECURS	; routine to make the cursor an underline
PUBLIC	VIDGETCHAR	; routine to get a character from the BIOS
PUBLIC	VIDGETNCHAR	; nonblocking routine to get a character
PUBLIC	VIDPOLLCHAR	; routine to check for a particular character
PUBLIC	VIDFLUSHKEY	; routine to flush the keyboard buffer
PUBLIC	VIDENDINPUT	; routine to hide input line on screen
PUBLIC	VIDMOVECURS	; routine to move the cursor
PUBLIC	VIDPUTINPUT	; routine to put a string on the input line
PUBLIC	VIDPUTMSG2	; routine to put a message on the input line
PUBLIC	VIDPUTFILNAM	; routine to put the mod file name on screen
PUBLIC	VIDPUTCMDS	; routine to redisplay the commands
PUBLIC	VIDSTARTSEL	; routine to clear the screen for the file selector
PUBLIC	VIDPUTDIRENT	; routine to display a directory entry for selector
PUBLIC	VIDHILIGHT	; routine to highlight a directory entry
PUBLIC	VIDUNHILIGHT	; routine to unhighlight a directory entry
PUBLIC	DEBUGSTRING	; routine to send a string out COM2:
PUBLIC	GETSYSTIME	; routine to get the system time

;
; Establish the segment ordering.  Code comes first.  This is the C
; standard segment ordering for Small model.
;
_TEXT	SEGMENT WORD PUBLIC 'CODE'
_TEXT	ENDS

_DATA	SEGMENT PARA PUBLIC 'DATA'
_DATA	ENDS

CONST	SEGMENT WORD PUBLIC 'CONST'
CONST	ENDS

_BSS	SEGMENT WORD PUBLIC 'BSS'
_BSS	ENDS

STACK	SEGMENT PARA STACK 'STACK'
STACK	ENDS

DGROUP	GROUP	_DATA,CONST,_BSS,STACK

_DATA	SEGMENT PARA PUBLIC 'DATA'
OLDMODE		DB	0	; active video mode at entry
MONO		DB	0	; 1 if monochrome display
		EVEN
VIDSEG		DW	0B800h	; video buffer segment address
		;
		; Screen color equates - bright foreground and blink bits.
		;
BRIGHT		EQU	8
BLINK		EQU	80h
		;
		; Screen color equates for color - low-intensity foreground
		; colors.
		;
BLACKFG		EQU	0
BLUEFG		EQU	1
GREENFG		EQU	2
CYANFG		EQU	BLUEFG+GREENFG
REDFG		EQU	4
MAGENTAFG	EQU	BLUEFG+REDFG
BROWNFG		EQU	GREENFG+REDFG
LIGHTGRAYFG	EQU	BLUEFG+GREENFG+REDFG
		;
		; Screen color equates for color - high-intensity foreground
		; colors.
		;
DARKGRAYFG	EQU	BLACKFG+BRIGHT
LIGHTBLUEFG	EQU	BLUEFG+BRIGHT
LIGHTGREENFG	EQU	GREENFG+BRIGHT
LIGHTCYANFG	EQU	CYANFG+BRIGHT
LIGHTREDFG	EQU	REDFG+BRIGHT
LIGHTMAGENTAFG	EQU	MAGENTAFG+BRIGHT
YELLOWFG	EQU	BROWNFG+BRIGHT
WHITEFG		EQU	LIGHTGRAYFG+BRIGHT
		;
		; Screen color equates for color - background colors.
		;
BLACKBG		EQU	BLACKFG SHL 4
BLUEBG		EQU	BLUEFG SHL 4
GREENBG		EQU	GREENFG SHL 4
CYANBG		EQU	CYANFG SHL 4
REDBG		EQU	REDFG SHL 4
MAGENTABG	EQU	MAGENTAFG SHL 4
BROWNBG		EQU	BROWNFG SHL 4
LIGHTGRAYBG	EQU	LIGHTGRAYFG SHL 4
		;
		; Screen color equates for monochrome.
		;
MONOBLACKFG	EQU	0
MONOUNDERFG	EQU	1
MONOWHITEFG	EQU	7
MONOBRIGHTFG	EQU	MONOWHITEFG+BRIGHT
MONOBRIGHTULFG	EQU	MONOUNDERFG+BRIGHT
MONOBLACKBG	EQU	MONOBLACKFG SHL 4
MONOWHITEBG	EQU	MONOWHITEFG SHL 4
		;
		; Color index equates for the color tables below.
		;
MAINCOLOR	EQU	0	; color of program name, prompts
STATUSCOLOR	EQU	1	; color of pattern position, volume, etc.
INFOCOLOR	EQU	2	; color of command descriptions, etc.
KEYCOLOR	EQU	3	; color of command keystroke names
ERRORCOLOR	EQU	4	; color of error messages
NOCOLOR		EQU	5	; color of input area when no input pending
INPUTCOLOR	EQU	6	; color of input area when input pending
FILECOLOR	EQU	7	; color of regular filename in file selector
DIRCOLOR	EQU	8	; color of directory or drive in file selector
HFILECOLOR	EQU	9	; color of highlighted filename in selector
HDIRCOLOR	EQU	10	; color of highlighted directory in selector
		;
		; Colors for color displays.
		;
CGACOLORS	LABEL	BYTE
		DB	BLACKFG+GREENBG			; MAINCOLOR
		DB	REDFG+GREENBG			; STATUSCOLOR
		DB	BLUEFG+GREENBG			; INFOCOLOR
		DB	WHITEFG+GREENBG			; KEYCOLOR
		DB	WHITEFG+REDBG			; ERRORCOLOR
		DB	GREENFG+GREENBG			; NOCOLOR
		DB	LIGHTGRAYFG+BLACKBG		; INPUTCOLOR
		DB	LIGHTGRAYFG+BLACKBG		; FILECOLOR
		DB	LIGHTBLUEFG+BLACKBG		; DIRCOLOR
		DB	BLACKFG+LIGHTGRAYBG		; HFILECOLOR
		DB	BLUEFG+LIGHTGRAYBG		; HDIRCOLOR
		;
		; Colors for monochrome displays.
		;
MDACOLORS	LABEL	BYTE
		DB	MONOBLACKFG+MONOWHITEBG		; MAINCOLOR
		DB	MONOBRIGHTFG+MONOBLACKBG	; STATUSCOLOR
		DB	MONOWHITEFG+MONOBLACKBG		; INFOCOLOR
		DB	MONOBRIGHTFG+MONOBLACKBG	; KEYCOLOR
		DB	MONOBRIGHTFG+MONOBLACKBG+BLINK	; ERRORCOLOR
		DB	MONOBLACKFG+MONOBLACKBG		; NOCOLOR
		DB	MONOWHITEFG+MONOBLACKBG		; INPUTCOLOR
		DB	MONOWHITEFG+MONOBLACKBG		; FILECOLOR
		DB	MONOUNDERFG+MONOBLACKBG		; DIRCOLOR
		DB	MONOBRIGHTFG+MONOBLACKBG	; HFILECOLOR
		DB	MONOBRIGHTULFG+MONOBLACKBG	; HDIRCOLOR
		;
		; Color table in effect.
		;
		EVEN
COLORTBL	DW	OFFSET CGACOLORS
		;
		; Parts of the screen for the various colors.  These are
		; records of the following format:
		;	WORD	number of characters this color
		;	WORD	color to use (MAINCOLOR, STATUSCOLOR, etc.)
		;
		EVEN
SCREENPARTS	DW	160,MAINCOLOR	; program name, filename
		DW	9,INFOCOLOR	; "pattern" string
		DW	7,STATUSCOLOR	; pattern position/entries in table
		DW	18,INFOCOLOR	; pattern #/# patterns, "note" string
		DW	5,STATUSCOLOR	; note number/"64"
		DW	10,INFOCOLOR	; "volume" string
		DW	3,STATUSCOLOR	; volume/"7"
		DW	3,INFOCOLOR	; three spaces
		DW	5,STATUSCOLOR	; output sampling rate
		DW	11,INFOCOLOR	; "Hz" and "loop" strings
		DW	9,STATUSCOLOR	; "disabled"/"enabled" string
		DW	3,STATUSCOLOR	; ticks per division
		DW	14,INFOCOLOR	; "ticks/note" string
		DW	3,STATUSCOLOR	; beats per minute
		DW	7,INFOCOLOR	; "bpm" string
		DW	2,STATUSCOLOR	; number of channels
		DW	12,INFOCOLOR	; "channels" string
		DW	2,STATUSCOLOR	; number of instruments
		DW	29,INFOCOLOR	; "instruments" and "global volume"
					;   strings
		DW	8,STATUSCOLOR	; global volume
		DW	4,INFOCOLOR	; "EMS" string
		DW	8,STATUSCOLOR	; "disabled"/"enabled" string
		DW	15,INFOCOLOR	; "save memory" string
		DW	8,STATUSCOLOR	; "disabled"/"enabled" string
		DW	24,INFOCOLOR	; "loading optimization" string
		DW	8,STATUSCOLOR	; "disabled"/"enabled" string
		DW	13,INFOCOLOR	; reserved for status
		; Command keystrokes and descriptions, 18 lines, 2 columns.
CMDPARTS	DW	36 DUP (12,KEYCOLOR,28,INFOCOLOR)
NCMDPARTS	EQU	($-OFFSET CMDPARTS) / 4
		DW	80,MAINCOLOR	; message/prompt area
		DW	80,NOCOLOR	; input area, no input pending
		;
		; Number of records in the above table.
		;
NSCREENPARTS	EQU	($-OFFSET SCREENPARTS) / 4
		;
		; Main screen text.  80 columns x 25 rows = 2000 characters.
		;
#IF M_I286
MAINTEXT	DB	"Tantrakr Music Module Player for the Tandy DAC                v"
		DB	VERSIONMAJOR,'.',VERSIONMINOR
		DB	" (286 version)"
#ELSE
MAINTEXT	DB	"Tantrakr Music Module Player for the Tandy DAC               v"
		DB	VERSIONMAJOR,'.',VERSIONMINOR
		DB	" (8086 version)"
#ENDIF
FILENAMEOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"<no file loaded>                                                                "
		DB	"position "
PATENTRYOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"000/"
NPATENTRIESOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"000 ("
PATNUMOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"000/"
NPATTERNSOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"000)   note "
NOTENUMOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"00/64   volume "
VOLUMEOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"7/7   "
RATEOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"22000 Hz   loop "
LOOPSTROFFS	EQU	($-OFFSET MAINTEXT) * 2
DISABLESTR	DB	"disabled "
TICKSPEROFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"006 ticks/note   "
BPMOFFS		EQU	($-OFFSET MAINTEXT) * 2
		DB	"125 bpm   "
NCHANSOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"00 channels   "
NINSTRSOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"00 instruments   global volume "
GLOBALVOLOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"00/64   "
		DB	"EMS "
EMSSTROFFS	EQU	($-OFFSET MAINTEXT) * 2
ENABLESTR	DB	"enabled    save memory "
SAVEMEMSTROFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"enabled    loading optimization "
LOADOPTSTROFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"enabled              "
CMDOFFS		EQU	($-OFFSET MAINTEXT) * 2
CMDSTART	LABEL	BYTE
		DB	" l          load file (by name)          t          decrease ticks/note         "
		DB	" p          play file                    T          increase ticks/note         "
		DB	" j          jump to new position         b          decrease bpm                "
		DB	" <          rewind                       B          increase bpm                "
		DB	" >          fast forward                 f          select file (from list)     "
		DB	" P          pause playback               g          decrease global volume      "
		DB	" R          resume playback              G          increase global volume      "
		DB	" s          stop playback                <alt>-X    EMS enable on/off           "
		DB	" u          unload file                  <alt>-S    save memory on/off          "
		DB	" r          rip sample to file           <alt>-O    loading optimization on/off "
		DB	" <ctrl>-R   rip all samples to files                                            "
		DB	" S          play sample                                                         "
		DB	" <ctrl>-D   dump module data to file                                            "
		DB	" v          decrease volume                                                     "
		DB	" V          increase volume                                                     "
		DB	" L          loop enable on/off                                                  "
		DB	" <ctrl>-O   set output sampling rate                                            "
		DB	" <esc>      exit program                                                        "
CMDSIZE		EQU	$-OFFSET CMDSTART
MESSAGEOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"Prompts and error messages will appear on this line; you enter input below.     "
INPUTOFFS	EQU	($-OFFSET MAINTEXT) * 2
		DB	"                                                                                "
_DATA	ENDS

_TEXT	SEGMENT WORD PUBLIC 'CODE'
;
; INITVID routine, to set up the initial screen.  Saves the current video
; mode to restore later, detects the video type (mono/color), and draws the
; screen.  Takes no parameters.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, BX, CX, DX, ES.
;
	EVEN
INITVID:
	PUSH	SI		; save SI,DI for C
	PUSH	DI
	MOV	AH,0Fh		; get current video mode
	INT	10h
	MOV	OLDMODE,AL	; save it
	CMP	AL,7		; is it mono?
	JE	INITVID_MONO
	;
	; Set up for color.
	;
	MOV	MONO,0		; not mono - clear flag (for clarity)
	MOV	VIDSEG,0B800h	; CGA/EGA/VGA buffer is at 0B800h (for clarity)
	MOV	COLORTBL,OFFSET CGACOLORS	; use CGA colors (for clarity)
	MOV	AX,2		; change to mode 2 - 80x25 color text, color
	INT	10h		;   suppressed w/composite, and clear screen
	MOV	AH,0Bh		; set the border color to green
	MOV	BH,0
	MOV	BL,GREENFG
	INT	10h
	JMP	SHORT INITVID_SETCOLORS
	;
	; Set up for monochrome.
	;
	EVEN
INITVID_MONO:
	MOV	MONO,1		; mono - set flag
	MOV	VIDSEG,0B000h	; MDA/Hercules buffer is at 0B000h
	MOV	COLORTBL,OFFSET MDACOLORS	; use MDA "colors"
	MOV	AX,7		; "change" to mode 7 (actually already in that
	INT	10h		;   mode), clearing the screen
	;
	; Set the colors of the various screen regions.
	;
INITVID_SETCOLORS:
	MOV	CX,NSCREENPARTS		; CX = number of screen regions
	MOV	ES,VIDSEG		; ES:DI -> video buffer
	XOR	DI,DI
	MOV	SI,OFFSET SCREENPARTS	; DS:SI -> color data for regions
	CLD				; DF clear for incrementing
INITVID_COLORLOOP:
	PUSH	CX
	LODSW				; CX = number of characters in region
	MOV	CX,AX
	LODSW				; AL = color attribute
	ADD	AX,COLORTBL
	MOV	BX,AX
	MOV	AL,[BX]
INITVID_REGIONLOOP:			; loop over characters in the region
	INC	DI			; skip over character
	STOSB				; set color attribute
	LOOP	INITVID_REGIONLOOP
	POP	CX
	LOOP	INITVID_COLORLOOP
	;
	; Display the initial text.
	;
	XOR	DI,DI			; ES:DI -> video buffer
	MOV	SI,OFFSET MAINTEXT	; DS:SI -> initial screen text
	MOV	CX,2000			; 80x25 = 2000 characters
INITVID_TEXTLOOP:
	MOVSB				; set the character
	INC	DI			; skip over the attribute
	LOOP	INITVID_TEXTLOOP
	;
	; Move the cursor to the beginning of the bottom row.
	;
	MOV	AH,2
	MOV	BH,0		; display page 0 is the default when mode set
	MOV	DH,24
	MOV	DL,0
	INT	10h
	;
	; Set the volume, output sampling rate, and loop enable on the screen
	; to match the actual settings.
	;
	CALL	VIDPUTVOLUME
	CALL	VIDPUTRATE
	CALL	VIDPUTLOOPEN
	POP	DI		; restore DI,SI for C
	POP	SI
	RET
;
; ENDVID routine, to clear the screen and restore the video at program
; termination.  Takes no parameters.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, BX.
;
	EVEN
ENDVID:
	CMP	MONO,1		; if not mono, set the border color to black
	JE	ENDVID_RESET
	MOV	AH,0Bh
	XOR	BX,BX
	INT	10h
ENDVID_RESET:
	XOR	AH,AH		; set video mode back to what it was before
	MOV	AL,OLDMODE
	INT	10h
	RET
;
; VIDPUTENTRY routine, to put the current pattern entry number on the screen.
; Takes one parameter, the entry number.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, CX, DX, ES.
;
	EVEN
VIDPUTENTRY:
	POP	DX		; DX is return address
	POP	AX		; AX is the pattern entry number (0-127)
	MOV	CL,100		; divide by 100
	DIV	CL
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES,VIDSEG	; ES addresses video buffer
	MOV	ES:PATENTRYOFFS,CL	; put most significant digit in buffer
	MOV	ES:PATENTRYOFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:PATENTRYOFFS+4,AL	; put 3rd digit in buffer
	JMP	DX		; jump to return address
;
; VIDPUTNENTRIES routine, to put the number of pattern table entries on the
; screen.  This is a constant for the loaded mod file, so we will just take
; the global variable from the Modstuf module and display it, i.e., no
; parameters here.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, CX, ES.
;
	EVEN
VIDPUTNENTRIES:
	MOV	AX,_pattablesize ; AX is the number of pattern table entries
	MOV	CL,100		; divide by 100
	DIV	CL
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES,VIDSEG	; ES addresses video buffer
	MOV	ES:NPATENTRIESOFFS,CL	; put most significant digit in buffer
	MOV	ES:NPATENTRIESOFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:NPATENTRIESOFFS+4,AL	; put 3rd digit in buffer
	RET
;
; VIDPUTPATNUM routine, to put a pattern number on the screen.  Takes one
; parameter, the pattern number.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, CX, DX, ES.
;
	EVEN
VIDPUTPATNUM:
	POP	DX		; DX is return address
	POP	AX		; AX is the pattern number (0-255)
	MOV	CL,100		; divide by 100
	DIV	CL
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES,VIDSEG	; ES addresses video buffer
	MOV	ES:PATNUMOFFS,CL	; put most significant digit in buffer
	MOV	ES:PATNUMOFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:PATNUMOFFS+4,AL	; put 3rd digit in buffer
	JMP	DX		; jump to return address
;
; VIDPUTNPATTERNS routine, to put the number of patterns on the screen.
; This is a constant for the loaded mod file, so we will just take the global
; variable from the Modstuf module and display it, i.e., no parameters here.
; Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, CX, ES.
;
	EVEN
VIDPUTNPATTERNS:
	MOV	AX,_nfilepatterns ; AX is the number of patterns in the file
	MOV	CL,100		; divide by 100
	DIV	CL
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES,VIDSEG	; ES addresses video buffer
	MOV	ES:NPATTERNSOFFS,CL	; put most significant digit in buffer
	MOV	ES:NPATTERNSOFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:NPATTERNSOFFS+4,AL	; put 3rd digit in buffer
	RET
;
; VIDPUTNOTENUM routine, to put a note number on the screen.  Takes one
; parameter, the note number.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, DX, ES.
;
	EVEN
VIDPUTNOTENUM:
	POP	DX		; DX is return address
	POP	AX		; AX is the note number (0-63)
	AAM			; AH = 1st digit, AL = 2nd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES,VIDSEG	; ES addresses video buffer
	MOV	ES:NOTENUMOFFS,AH	; put first digit in buffer
	MOV	ES:NOTENUMOFFS+2,AL	; put 2nd digit in buffer
	JMP	DX		; jump to return address
;
; VIDPUTNUMS routine, to set all numbers on the screen at the start of song
; playback.  Takes one parameter, the 2k buffer segment number where playback
; is beginning.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, BX, CX, DX, ES.
;
	EVEN
VIDPUTNUMS:
	POP	DX		; get return address in DX
	POP	AX		; AL = segment number
	MOV	AH,TYPE BUFSEGREC ; BX addresses data for segment
	MUL	AH
	MOV	BX,AX
	MOV	ES,VIDSEG	; ES addresses video buffer
	MOV	CH,100		; CH is divider to convert to ASCII
	MOV	AL,_bufsegrecs[BX].BUFSEGROW ; get note number
	AAM			; AH = 1st digit, AL = 2nd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES:NOTENUMOFFS,AH	; put first digit in buffer
	MOV	ES:NOTENUMOFFS+2,AL	; put 2nd digit in buffer
	MOV	AL,_bufsegrecs[BX].BUFSEGBPM ; get beats/minute
	XOR	AH,AH
	DIV	CH		; divide by 100
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES:BPMOFFS,CL	; put most significant digit in buffer
	MOV	ES:BPMOFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:BPMOFFS+4,AL	; put 3rd digit in buffer
	MOV	AX,_bufsegrecs[BX].BUFSEGTICKSPER ; get ticks/division
	DIV	CH		; divide by 100
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES:TICKSPEROFFS,CL	; put 1st digit in buffer
	MOV	ES:TICKSPEROFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:TICKSPEROFFS+4,AL	; put 3rd digit in buffer
	MOV	AL,_bufsegrecs[BX].BUFSEGGVOL ; get global volume
	AAM			; AH = 1st digit, AL = 2nd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES:GLOBALVOLOFFS,AH	; put first digit in buffer
	MOV	ES:GLOBALVOLOFFS+2,AL	; put 2nd digit in buffer
	MOV	AX,_bufsegrecs[BX].BUFSEGENTRY ; get pattern table entry
	DIV	CH		; entries different - display new one
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES:PATENTRYOFFS,CL	; put most significant digit in buffer
	MOV	ES:PATENTRYOFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:PATENTRYOFFS+4,AL	; put 3rd digit in buffer
	MOV	AX,_bufsegrecs[BX].BUFSEGPATNUM ; get pattern number
	DIV	CH		; entries different - display new one
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES:PATNUMOFFS,CL	; put most significant digit in buffer
	MOV	ES:PATNUMOFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:PATNUMOFFS+4,AL	; put 3rd digit in buffer
	JMP	DX		; jump to return address
;
; VIDUPDATENUMS routine, to update numbers on the screen during song
; playback.  Takes two parameters, the previous 2k buffer segment number
; and the current 2k buffer segment number.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, BX, CX, ES.
;
	EVEN
VIDUPDATENUMS:
	PUSH	BP		; save caller's BP
	MOV	BP,SP		; BP addresses the parameters
	PUSH	SI		; save caller's SI
	MOV	AL,[BP+6]	; AL = old 2k segment
	CMP	AL,[BP+4]	; if old = new, nothing to do
	JNE	VIDUPDATENUMS_GETSEGS
	JMP	VIDUPDATENUMS_DONE
	EVEN
VIDUPDATENUMS_GETSEGS:
	MOV	AH,TYPE BUFSEGREC ; BX addresses data for old segment
	MUL	AH
	MOV	BX,AX
	MOV	AL,TYPE BUFSEGREC ; SI addresses data for new segment
	MUL	BYTE PTR [BP+4]
	MOV	SI,AX
	MOV	ES,VIDSEG	; ES addresses video buffer
	MOV	CH,100		; CH is divider to convert to ASCII
	MOV	AL,_bufsegrecs[SI].BUFSEGROW ; see if note numbers are the same
	CMP	AL,_bufsegrecs[BX].BUFSEGROW
	JE	VIDUPDATENUMS_CHKBPM ; if same, go check beats/minute
	AAM			; AH = 1st digit, AL = 2nd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES:NOTENUMOFFS,AH	; put first digit in buffer
	MOV	ES:NOTENUMOFFS+2,AL	; put 2nd digit in buffer
VIDUPDATENUMS_CHKBPM:
	MOV	AL,_bufsegrecs[SI].BUFSEGBPM ; see if beats/minute is the same
	CMP	AL,_bufsegrecs[BX].BUFSEGBPM
	JE	VIDUPDATENUMS_CHKTICKS ; if same, check ticks/division
	XOR	AH,AH
	DIV	CH		; divide by 100
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES:BPMOFFS,CL	; put most significant digit in buffer
	MOV	ES:BPMOFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:BPMOFFS+4,AL	; put 3rd digit in buffer
VIDUPDATENUMS_CHKTICKS:
	MOV	AX,_bufsegrecs[SI].BUFSEGTICKSPER ; see if ticks/division same
	CMP	AX,_bufsegrecs[BX].BUFSEGTICKSPER
	JE	VIDUPDATENUMS_CHKGVOL ; if same, go check global volume
	DIV	CH		; divide by 100
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES:TICKSPEROFFS,CL	; put 1st digit in buffer
	MOV	ES:TICKSPEROFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:TICKSPEROFFS+4,AL	; put 3rd digit in buffer
VIDUPDATENUMS_CHKGVOL:
	MOV	AL,_bufsegrecs[SI].BUFSEGGVOL ; see if global volume is same
	CMP	AL,_bufsegrecs[BX].BUFSEGGVOL
	JE	VIDUPDATENUMS_CHKENTRY ; if same, go check pattern entry
	AAM			; AH = 1st digit, AL = 2nd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES:GLOBALVOLOFFS,AH	; put first digit in buffer
	MOV	ES:GLOBALVOLOFFS+2,AL	; put 2nd digit in buffer
VIDUPDATENUMS_CHKENTRY:
	MOV	AX,_bufsegrecs[SI].BUFSEGENTRY ; see if entries are the same
	CMP	AX,_bufsegrecs[BX].BUFSEGENTRY
	JE	VIDUPDATENUMS_DONE ; if same, all done
	DIV	CH		; entries different - display new one
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES:PATENTRYOFFS,CL	; put most significant digit in buffer
	MOV	ES:PATENTRYOFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:PATENTRYOFFS+4,AL	; put 3rd digit in buffer
	MOV	AX,_bufsegrecs[SI].BUFSEGPATNUM ; see if patterns are the same
	CMP	AX,_bufsegrecs[BX].BUFSEGPATNUM
	JE	VIDUPDATENUMS_DONE ; if same, all done
	DIV	CH		; entries different - display new one
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES:PATNUMOFFS,CL	; put most significant digit in buffer
	MOV	ES:PATNUMOFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:PATNUMOFFS+4,AL	; put 3rd digit in buffer
VIDUPDATENUMS_DONE:
	POP	SI		; restore caller's SI
	POP	BP		; restore caller's BP
	RET	4		; remove parameters from the stack
;
; VIDPUTVOLUME routine, to put the current DAC volume setting on the screen.
; This is global parameter, so we will just take the global variable from
; the Dacstuf module and display it, i.e., no parameters here.  Uses Pascal
; linkage conventions.
;
; Returns nothing.  Destroys AX, ES.
;
	EVEN
VIDPUTVOLUME:
	MOV	AL,_volume 		; AL is the current DAC volume setting
	ADD	AL,'0'			; convert to ASCII
	MOV	ES,VIDSEG		; ES addresses video buffer
	MOV	ES:VOLUMEOFFS,AL	; put in buffer
	RET
;
; VIDPUTRATE routine, to display the current output sampling rate on the
; screen.  This is global parameter, so we will just take the global
; variable from the Dacstuf module and display it, i.e., no parameters here.
; Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, BX, CX, DX, ES.
;
	EVEN
VIDPUTRATE:
	MOV	AX,_samprate	; AX is the output sampling rate
	XOR	DX,DX		; divide by 100
	MOV	CX,100
	DIV	CX
	MOV	BL,DL		; BL is 4th and 5th digits
	XOR	DX,DX
	DIV	CX		; DL is 2nd and 3rd digits, AL is 1st digit
	XCHG	AL,DL		; AL is 2nd and 3rd digits, DL is 1st digit
	AAM			; AH is 3rd digit, AL is 2nd digit
	XCHG	AX,BX		; AL is 4th and 5th digits, 3rd & 2nd in BX
	AAM			; DL=1st, BH=2nd, BL=3rd, AH=4th, AL=5th
	OR	DL,DL		; is the 1st digit 0?
	JNZ	VIDPUTRATE_1STNONZERO
	MOV	DL,' '		; 1st digit is 0 - make it a blank
	JMP	SHORT VIDPUTRATE_DOOTHERS
	EVEN
VIDPUTRATE_1STNONZERO:
	ADD	DL,'0'		; 1st digit nonzero - make it ASCII
VIDPUTRATE_DOOTHERS:
	ADD	BX,3030h	; make the 2nd and 3rd digits ASCII
	ADD	AX,3030h	; make the 4th and 5th digits ASCII
	MOV	ES,VIDSEG	; ES addresses video buffer
	MOV	ES:RATEOFFS,DL		; put 1st digit in buffer
	MOV	ES:RATEOFFS+2,BH	; put 2nd digit in buffer
	MOV	ES:RATEOFFS+4,BL	; put 3rd digit in buffer
	MOV	ES:RATEOFFS+6,AH	; put 4th digit in buffer
	MOV	ES:RATEOFFS+8,AL	; put 5th digit in buffer
	RET
;
; VIDPUTLOOPEN routine, to put the current loop enable setting on the screen.
; This is global parameter, so we will just take the global variable from
; the Modstuf module and display it, i.e., no parameters here.  Uses Pascal
; linkage conventions.
;
; Returns nothing.  Destroys ES.
;
	EVEN
VIDPUTLOOPEN:
	PUSH	SI			; save SI,DI for C
	PUSH	DI
	MOV	SI,OFFSET ENABLESTR	; assume enabled - use "enabled" string
	CMP	_loopdisable,1		; is looping disabled?
	JNE	VIDPUTLOOPEN_ENABLED
	MOV	SI,OFFSET DISABLESTR	; disabled - use "disabled" string
VIDPUTLOOPEN_ENABLED:
	MOV	ES,VIDSEG		; ES:DI -> string on screen
	MOV	DI,LOOPSTROFFS
	CLD				; DF set for incrementing
	MOVSB				; copy character #1
	INC	DI			; skip over attribute
	MOVSB				; copy character #2
	INC	DI			; skip over attribute
	MOVSB				; copy character #3
	INC	DI			; skip over attribute
	MOVSB				; copy character #4
	INC	DI			; skip over attribute
	MOVSB				; copy character #5
	INC	DI			; skip over attribute
	MOVSB				; copy character #6
	INC	DI			; skip over attribute
	MOVSB				; copy character #7
	INC	DI			; skip over attribute
	MOVSB				; copy character #8
	POP	DI			; restore DI,SI for C
	POP	SI
	RET
;
; VIDPUTTICKSPER routine, to put the number of ticks per division on the
; screen.  Takes one parameter, the number of ticks per division to display.
; Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, CX, DX, ES.
;
	EVEN
VIDPUTTICKSPER:
	POP	DX		; DX is return address
	POP	AX		; AX is the number of ticks per division
	MOV	CL,100		; divide by 100
	DIV	CL
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES,VIDSEG	; ES addresses video buffer
	MOV	ES:TICKSPEROFFS,CL	; put 1st digit in buffer
	MOV	ES:TICKSPEROFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:TICKSPEROFFS+4,AL	; put 3rd digit in buffer
	JMP	DX		; jump to return address
;
; VIDPUTBPM routine, to put the number of beats per minute on the screen.
; Takes one parameter, the number of beats per minute to display.  Uses
; Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, CX, DX, ES.
;
	EVEN
VIDPUTBPM:
	POP	DX		; DX is return address
	POP	AX		; AX is the number of beats per minute (33-255)
	XOR	AH,AH
	MOV	CL,100		; divide by 100
	DIV	CL
	ADD	AL,'0'		; convert most significant digit to ASCII
	MOV	CL,AL		; CL = most significant digit
	MOV	AL,AH		; AL = 2 least significant digits
	AAM			; AH = 2nd digit, AL = 3rd digit
	ADD	AX,3030h	; convert both digits to ASCII
	MOV	ES,VIDSEG	; ES addresses video buffer
	MOV	ES:BPMOFFS,CL	; put most significant digit in buffer
	MOV	ES:BPMOFFS+2,AH	; put 2nd digit in buffer
	MOV	ES:BPMOFFS+4,AL	; put 3rd digit in buffer
	JMP	DX		; jump to return address
;
; VIDPUTNCHANS routine, to put the number of channels on the screen.  Takes
; no parameters.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, ES.
;
	EVEN
VIDPUTNCHANS:
	MOV	AX,_nchannels		; AX is the number of channels
	AAM				; AH = 1st digit, AL = 2nd digit
	ADD	AX,3030h		; convert both digits to ASCII
	MOV	ES,VIDSEG		; ES addresses video buffer
	MOV	ES:NCHANSOFFS,AH	; put first digit in buffer
	MOV	ES:NCHANSOFFS+2,AL	; put 2nd digit in buffer
	RET
;
; VIDPUTNINSTRS routine, to put the number of instruments on the screen.
; Takes no parameters.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, ES.
;
	EVEN
VIDPUTNINSTRS:
	MOV	AX,_ninstruments	; get number of instruments
	AAM				; AH = 1st digit, AL = 2nd digit
	ADD	AX,3030h		; convert both digits to ASCII
	MOV	ES,VIDSEG		; ES addresses video buffer
	MOV	ES:NINSTRSOFFS,AH	; put first digit in buffer
	MOV	ES:NINSTRSOFFS+2,AL	; put 2nd digit in buffer
	RET
;
; VIDPUTGLOBALVOL routine, to put the global volume on the screen.  Takes
; no one parameter, the global volume to display.  Uses Pascal linkage
; conventions.
;
; Returns nothing.  Destroys AX, DX, ES.
;
	EVEN
VIDPUTGLOBALVOL:
	POP	DX			; DX is return address
	POP	AX			; AL is the global volume (0-64)
	AAM				; AH = 1st digit, AL = 2nd digit
	ADD	AX,3030h		; convert both digits to ASCII
	MOV	ES,VIDSEG		; ES addresses video buffer
	MOV	ES:GLOBALVOLOFFS,AH	; put first digit in buffer
	MOV	ES:GLOBALVOLOFFS+2,AL	; put 2nd digit in buffer
	JMP	DX			; jump to return address
;
; VIDPUTEMSEN routine, to put the current EMS enable setting on the screen.
; This is global parameter, so we will just take the global variable from
; the Memmgr module and display it, i.e., no parameters here.  Uses Pascal
; linkage conventions.
;
; Returns nothing.  Destroys ES.
;
	EVEN
VIDPUTEMSEN:
	PUSH	SI			; save SI,DI for C
	PUSH	DI
	MOV	SI,OFFSET ENABLESTR	; assume enabled - use "enabled" string
	CMP	_emsenable,0		; is looping disabled?
	JNE	VIDPUTEMSEN_ENABLED
	MOV	SI,OFFSET DISABLESTR	; disabled - use "disabled" string
VIDPUTEMSEN_ENABLED:
	MOV	ES,VIDSEG		; ES:DI -> string on screen
	MOV	DI,EMSSTROFFS
	CLD				; DF set for incrementing
	MOVSB				; copy character #1
	INC	DI			; skip over attribute
	MOVSB				; copy character #2
	INC	DI			; skip over attribute
	MOVSB				; copy character #3
	INC	DI			; skip over attribute
	MOVSB				; copy character #4
	INC	DI			; skip over attribute
	MOVSB				; copy character #5
	INC	DI			; skip over attribute
	MOVSB				; copy character #6
	INC	DI			; skip over attribute
	MOVSB				; copy character #7
	INC	DI			; skip over attribute
	MOVSB				; copy character #8
	POP	DI			; restore DI,SI for C
	POP	SI
	RET
;
; VIDPUTSAVEMEM routine, to put the current save memory enable setting on
; the screen.  This is global parameter, so we will just take the global
; variable from the Modstuf module and display it, i.e., no parameters here.
; Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys ES.
;
	EVEN
VIDPUTSAVEMEM:
	PUSH	SI			; save SI,DI for C
	PUSH	DI
	MOV	SI,OFFSET ENABLESTR	; assume enabled - use "enabled" string
	CMP	_savemem,0		; is save memory disabled?
	JNE	VIDPUTSAVEMEM_ENABLED
	MOV	SI,OFFSET DISABLESTR	; disabled - use "disabled" string
VIDPUTSAVEMEM_ENABLED:
	MOV	ES,VIDSEG		; ES:DI -> string on screen
	MOV	DI,SAVEMEMSTROFFS
	CLD				; DF set for incrementing
	MOVSB				; copy character #1
	INC	DI			; skip over attribute
	MOVSB				; copy character #2
	INC	DI			; skip over attribute
	MOVSB				; copy character #3
	INC	DI			; skip over attribute
	MOVSB				; copy character #4
	INC	DI			; skip over attribute
	MOVSB				; copy character #5
	INC	DI			; skip over attribute
	MOVSB				; copy character #6
	INC	DI			; skip over attribute
	MOVSB				; copy character #7
	INC	DI			; skip over attribute
	MOVSB				; copy character #8
	POP	DI			; restore DI,SI for C
	POP	SI
	RET
;
; VIDPUTLOADOPT routine, to put the current loading optimization enable
; setting on the screen.  This is global parameter, so we will just take
; the global variable from the Modstuf module and display it, i.e., no
; parameters here.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys ES.
;
VIDPUTLOADOPT:
	PUSH	SI			; save SI,DI for C
	PUSH	DI
	MOV	SI,OFFSET ENABLESTR	; assume enabled - use "enabled" string
	CMP	_optimize,0		; is loading optimization disabled?
	JNE	VIDPUTLOADOPT_ENABLED
	MOV	SI,OFFSET DISABLESTR	; disabled - use "disabled" string
VIDPUTLOADOPT_ENABLED:
	MOV	ES,VIDSEG		; ES:DI -> string on screen
	MOV	DI,LOADOPTSTROFFS
	CLD				; DF set for incrementing
	MOVSB				; copy character #1
	INC	DI			; skip over attribute
	MOVSB				; copy character #2
	INC	DI			; skip over attribute
	MOVSB				; copy character #3
	INC	DI			; skip over attribute
	MOVSB				; copy character #4
	INC	DI			; skip over attribute
	MOVSB				; copy character #5
	INC	DI			; skip over attribute
	MOVSB				; copy character #6
	INC	DI			; skip over attribute
	MOVSB				; copy character #7
	INC	DI			; skip over attribute
	MOVSB				; copy character #8
	POP	DI			; restore DI,SI for C
	POP	SI
	RET
;
; VIDPUTMSG routine, to put a normal message or prompt on the screen.  Takes
; one parameter, a pointer to the null-terminated message string.  Uses
; Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, CX, ES.
;
	EVEN
VIDPUTMSG:
	PUSH	BP
	MOV	BP,SP
	PUSH	SI			; save SI,DI for C
	PUSH	DI
	MOV	ES,VIDSEG		; ES:DI -> string on screen
	MOV	DI,MESSAGEOFFS+1	; (we do attributes first)
	MOV	SI,COLORTBL		; AL is attribute to use
	MOV	AL,[SI].MAINCOLOR
	MOV	CX,80			; put 80 attributes on screen
	CLD				; DF set for incrementing
VIDPUTMSG_ATTRLOOP:
	STOSB				; put the attribute up
	INC	DI			; skip over the character
	LOOP	VIDPUTMSG_ATTRLOOP
	MOV	SI,[BP+4]		; SI addresses string now
	MOV	DI,MESSAGEOFFS		; ES:DI -> string on screen again
	MOV	CX,80			; put max 80 characters on screen
VIDPUTMSG_CHARLOOP:
	LODSB				; get character
	OR	AL,AL			; is it a null?
	JZ	VIDPUTMSG_FILLBLANKS	; jump out of the loop if so
	STOSB				; put the character in the string
	INC	DI			; skip over the attribute
	LOOP	VIDPUTMSG_CHARLOOP
	JMP	SHORT VIDPUTMSG_ALLDONE	; if 80 characters, all done
	EVEN
VIDPUTMSG_FILLBLANKS:
	MOV	AL,' '			; fill the rest of the line w/blanks
VIDPUTMSG_BLANKLOOP:
	STOSB				; store the blank
	INC	DI			; skip over the attribute
	LOOP	VIDPUTMSG_BLANKLOOP
VIDPUTMSG_ALLDONE:
	POP	DI			; restore DI,SI for C
	POP	SI
	POP	BP
	RET	2
;
; VIDPUTERRMSG routine, to put an error message on the screen.  Takes one
; parameter, a pointer to the null-terminated message string.  Uses Pascal
; linkage conventions.
;
; Returns nothing.  Destroys AX, CX, ES.
;
	EVEN
VIDPUTERRMSG:
	PUSH	BP
	MOV	BP,SP
	PUSH	SI			; save SI,DI for C
	PUSH	DI
	MOV	ES,VIDSEG		; ES:DI -> string on screen
	MOV	DI,MESSAGEOFFS+1	; (we do attributes first)
	MOV	SI,COLORTBL		; AL is attribute to use
	MOV	AL,[SI].ERRORCOLOR
	MOV	CX,80			; put 80 attributes on screen
	CLD				; DF set for incrementing
VIDPUTERRMSG_ATTRLOOP:
	STOSB				; put the attribute up
	INC	DI			; skip over the character
	LOOP	VIDPUTERRMSG_ATTRLOOP
	MOV	SI,[BP+4]		; SI addresses string now
	MOV	DI,MESSAGEOFFS		; ES:DI -> string on screen again
	MOV	CX,80			; put max 80 characters on screen
VIDPUTERRMSG_CHARLOOP:
	LODSB				; get character
	OR	AL,AL			; is it a null?
	JZ	VIDPUTERRMSG_FILLBLANKS	; jump out of the loop if so
	STOSB				; put the character in the string
	INC	DI			; skip over the attribute
	LOOP	VIDPUTERRMSG_CHARLOOP
	JMP	SHORT VIDPUTERRMSG_ALLDONE ; if 80 characters, all done
	EVEN
VIDPUTERRMSG_FILLBLANKS:
	MOV	AL,' '			; fill the rest of the line w/blanks
VIDPUTERRMSG_BLANKLOOP:
	STOSB				; store the blank
	INC	DI			; skip over the attribute
	LOOP	VIDPUTERRMSG_BLANKLOOP
VIDPUTERRMSG_ALLDONE:
	POP	DI			; restore DI,SI for C
	POP	SI
	POP	BP
	RET	2
;
; VIDSTARTINPUT routine, to clear the input line on the screen and set its
; attribute for user input.  Takes no parameters.  Uses Pascal calling
; conventions.
;
; Returns nothing.  Destroys AX, BX, CX, DX, ES.
;
	EVEN
VIDSTARTINPUT:
	PUSH	DI		; save DI for C
	MOV	AL,' '		; set characters to all blanks
	MOV	DI,COLORTBL	; set attribute
	MOV	AH,[DI].INPUTCOLOR
	MOV	ES,VIDSEG	; ES:DI -> input line on screen
	MOV	DI,INPUTOFFS
	MOV	CX,80		; 80 characters on the input line
	CLD			; DF set for incrementing
	REP	STOSW		; clear the line
	MOV	AH,2		; move cursor to beginning of input line
	MOV	BH,0
	MOV	DH,24
	MOV	DL,0
	INT	10h
	POP	DI		; restore DI for C
	RET
;
; VIDBLOCKCURS routine, to make the cursor a block cursor.  Takes no
; parameters.  Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX, CX.
;
	EVEN
VIDBLOCKCURS:
	MOV	CX,7			; assume not mono - 8 scan lines/char
	CMP	MONO,1			; is it mono?
	JNE	VIDBLOCKCURS_SET
	MOV	CX,13			; mono - 14 scan lines/char
VIDBLOCKCURS_SET:
	MOV	AH,1			; set cursor size
	INT	10h
	RET
;
; VIDLINECURS routine, to make the cursor an underline.  Takes no parameters.
; Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX, CX.
;
	EVEN
VIDLINECURS:
	MOV	CX,607h			; assume not mono - 8 scan lines/char
	CMP	MONO,1			; is it mono?
	JNE	VIDLINECURS_SET
	MOV	CX,0C0Dh		; mono - 14 scan lines/char
VIDLINECURS_SET:
	MOV	AH,1			; set cursor size
	INT	10h
	RET
;
; VIDGETCHAR routine, to get a character from the BIOS.  Takes no parameters.
; Uses Pascal calling conventions.
;
; Returns the scan code+ASCII code in AX (scan code is most significant 8
; bits).  Destroys nothing.
;
	EVEN
VIDGETCHAR:
	XOR	AH,AH
	INT	16h
	RET
;
; VIDGETNCHAR routine, to get a character from the BIOS.  If no character
; is available, 0 is returned; this routine does not block.  Takes no
; parameters.  Uses Pascal calling conventions.
;
; Returns the scan code+ASCII code in AX (scan code is most significant 8
; bits), or 0 if no character.  Destroys nothing.
;
	EVEN
VIDGETNCHAR:
	MOV	AH,1
	INT	16h
	JNZ	VIDGETNCHAR_HAVECHAR
	XOR	AX,AX
	RET
	EVEN
VIDGETNCHAR_HAVECHAR:
	XOR	AH,AH
	INT	16h
	RET
;
; VIDPOLLCHAR routine, to check whether a given character has been typed
; and is waiting in the BIOS typeahead buffer.  If no character has been
; typed, or a different character has been typed, 0 is returned, and the
; typeahead buffer is left alone.  If the desired character is there, all
; occurrences of it are removed, and the number of occurrences of the
; character is returned.  Takes one parameter, the scan code + ASCII code
; of the character to look for.  Uses Pascal calling conventions.
;
; Returns the number of times the character occurred in the typeahead buffer
; in AX, 0 if not there.  Destroys BX, CX, DX.
;
	EVEN
VIDPOLLCHAR:
	POP	DX		; DX is return address
	POP	CX		; CX is character to look for
	XOR	BX,BX		; BX is the count of occurrences found
VIDPOLLCHAR_LOOP:
	MOV	AH,1		; scan keyboard
	INT	16h
	JZ	VIDPOLLCHAR_DONE
	CMP	AX,CX		; is it the character we want?
	JNE	VIDPOLLCHAR_DONE
	XOR	AH,AH		; read keyboard (remove from typeahead buffer)
	INT	16h
	INC	BX		; increment count of occurrences found
	JMP	SHORT VIDPOLLCHAR_LOOP
	EVEN
VIDPOLLCHAR_DONE:
	MOV	AX,BX		; return number of occurrences found
	JMP	DX		; jump to return address
;
; VIDFLUSHKEY routine, to flush the keyboard typeahead buffer.  Takes no
; parameters.  Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX.
;
	EVEN
VIDFLUSHKEY:
	MOV	AH,1		; scan keyboard
	INT	16h
	JZ	VIDFLUSHKEY_DONE
	XOR	AH,AH		; read keyboard
	INT	16h
	JMP	SHORT VIDFLUSHKEY
	EVEN
VIDFLUSHKEY_DONE:
	RET
;
; VIDENDINPUT routine, to clear the input line on the screen and set its
; attribute to indicate that no input is pending.  Takes no parameters.
; Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX, CX, ES.
;
	EVEN
VIDENDINPUT:
	PUSH	DI		; save DI for C
	MOV	AL,' '		; set characters to all blanks
	MOV	DI,COLORTBL	; set attribute
	MOV	AH,[DI].NOCOLOR
	MOV	ES,VIDSEG	; ES:DI -> input line on screen
	MOV	DI,INPUTOFFS
	MOV	CX,80		; 80 characters on the input line
	CLD			; DF set for incrementing
	REP	STOSW		; clear the line
	POP	DI		; restore DI for C
	RET
;
; VIDMOVECURS routine, to move the cursor to a desired location on the input
; line.  Takes one parameter, the new cursor position on the line (0-79).
; Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX, BX, CX, DX.
;
	EVEN
VIDMOVECURS:
	POP	CX		; CX is return address
	POP	DX		; DL is new cursor position
	MOV	AH,2		; move cursor to new position on input line
	MOV	BH,0
	MOV	DH,24
	INT	10h
	JMP	CX		; jump to return address
;
; VIDPUTINPUT routine, to copy a null-terminated string into the input line.
; Takes one paramter, the string to copy.  Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX, CX, ES.
;
	EVEN
VIDPUTINPUT:
	PUSH	BP
	MOV	BP,SP
	PUSH	SI			; save SI,DI for C
	PUSH	DI
	MOV	ES,VIDSEG		; ES:DI -> string on screen
	MOV	DI,INPUTOFFS
	MOV	SI,[BP+4]		; SI addresses string
	MOV	CX,80			; put max 80 characters on screen
	CLD				; DF set for incrementing
VIDPUTINPUT_CHARLOOP:
	LODSB				; get character
	OR	AL,AL			; is it a null?
	JZ	VIDPUTINPUT_FILLBLANKS	; jump out of the loop if so
	STOSB				; put the character in the string
	INC	DI			; skip over the attribute
	LOOP	VIDPUTINPUT_CHARLOOP
	JMP	SHORT VIDPUTINPUT_ALLDONE ; if 80 characters, all done
	EVEN
VIDPUTINPUT_FILLBLANKS:
	MOV	AL,' '			; fill the rest of the line w/blanks
VIDPUTINPUT_BLANKLOOP:
	STOSB				; store the blank
	INC	DI			; skip over the attribute
	LOOP	VIDPUTINPUT_BLANKLOOP
VIDPUTINPUT_ALLDONE:
	POP	DI			; restore DI,SI for C
	POP	SI
	POP	BP
	RET	2
;
; VIDPUTMSG2 routine, to put a second message line in the input area.  Takes
; one parameter, the null-terminated string to display.  Uses Pascal calling
; conventions.
;
; Returns nothing.  Destroys AX, BX, CX, DX, ES.
;
	EVEN
VIDPUTMSG2:
	PUSH	DI		; save DI for C
	MOV	DI,COLORTBL	; get attribute in AL
	MOV	AL,[DI].MAINCOLOR
	MOV	ES,VIDSEG	; ES:DI -> input line on screen
	MOV	DI,INPUTOFFS+1
	MOV	CX,80		; put 80 attributes on screen
	CLD			; DF set for incrementing
VIDPUTMSG2_ATTRLOOP:
	STOSB			; put the attribute up
	INC	DI		; skip over the character
	LOOP	VIDPUTMSG2_ATTRLOOP
	POP	DI		; restore DI for C
#IF M_I286
	PUSH	80		; move the cursor offscreen
#ELSE
	MOV	AX,80
	PUSH	AX
#ENDIF
	CALL	VIDMOVECURS	; (AX, BX, CX, DX destroyed)
	JMP	SHORT VIDPUTINPUT ; use VIDPUTINPUT to put the string up
				  ; (AX, CX, ES destroyed)
;
; VIDPUTFILNAM routine, to put the name of the currently-loaded mod file on
; the screen.  If the name is longer than 80 characters, only the first 80
; are displayed.  Takes one parameter, a pointer to the null-terminated
; filename.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, CX, ES.
;
	EVEN
VIDPUTFILNAM:
	PUSH	BP
	MOV	BP,SP
	PUSH	SI			; save SI,DI for C
	PUSH	DI
	MOV	ES,VIDSEG		; ES:DI -> string on screen
	MOV	DI,FILENAMEOFFS
	MOV	SI,[BP+4]		; SI addresses filename
	MOV	CX,80			; put max 80 characters on screen
	CLD				; DF set for incrementing
VIDPUTFILNAM_CHARLOOP:
	LODSB				; get character
	OR	AL,AL			; is it a null?
	JZ	VIDPUTFILNAM_FILLBLANKS	; jump out of the loop if so
	STOSB				; put the character in the string
	INC	DI			; skip over the attribute
	LOOP	VIDPUTFILNAM_CHARLOOP
	JMP	SHORT VIDPUTFILNAM_ALLDONE ; if 80 characters, all done
	EVEN
VIDPUTFILNAM_FILLBLANKS:
	MOV	AL,' '			; fill the rest of the line w/blanks
VIDPUTFILNAM_BLANKLOOP:
	STOSB				; store the blank
	INC	DI			; skip over the attribute
	LOOP	VIDPUTFILNAM_BLANKLOOP
VIDPUTFILNAM_ALLDONE:
	POP	DI			; restore DI,SI for C
	POP	SI
	POP	BP
	RET	2
;
; VIDPUTCMDS routine.  Redisplays the command screen after it has been
; overwritten, say by the file list.  Takes no parameters.  Uses Pascal
; calling conventions.
;
; Returns nothing.  Destroys AX, BX, CX, ES.
;
	EVEN
VIDPUTCMDS:
	PUSH	SI			; save SI, DI for C
	PUSH	DI
	;
	; Set the colors of the various screen regions.
	;
	MOV	CX,NCMDPARTS		; CX = number of screen regions
	MOV	ES,VIDSEG		; ES:DI -> location of commands in
	MOV	DI,CMDOFFS		;   video buffer
	MOV	SI,OFFSET CMDPARTS	; DS:SI -> color data for regions
	CLD				; DF clear for incrementing
VIDPUTCMDS_COLORLOOP:
	PUSH	CX
	LODSW				; CX = number of characters in region
	MOV	CX,AX
	LODSW				; AL = color attribute
	ADD	AX,COLORTBL
	MOV	BX,AX
	MOV	AL,[BX]
VIDPUTCMDS_REGIONLOOP:			; loop over characters in the region
	INC	DI			; skip over character
	STOSB				; set color attribute
	LOOP	VIDPUTCMDS_REGIONLOOP
	POP	CX
	LOOP	VIDPUTCMDS_COLORLOOP
	;
	; Display the command text.
	;
	MOV	DI,CMDOFFS		; ES:DI -> commands in video buffer
	MOV	SI,OFFSET CMDSTART	; DS:SI -> command text
	MOV	CX,CMDSIZE		; CX = number of characters
VIDPUTCMDS_TEXTLOOP:
	MOVSB				; set the character
	INC	DI			; skip over the attribute
	LOOP	VIDPUTCMDS_TEXTLOOP
	POP	DI			; restore DI, SI
	POP	SI
	RET
;
; VIDSTARTSEL routine.  Clears the command area for the file selector.
; Takes no parameters.  Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX, BX, CX, ES.
;
	EVEN
VIDSTARTSEL:
	PUSH	DI			; save DI for C
	MOV	ES,VIDSEG		; ES:DI -> location of commands in
	MOV	DI,CMDOFFS		;   video buffer
	MOV	CX,CMDSIZE		; CX = number of characters
	MOV	BX,COLORTBL		; AH = attribute
	MOV	AH,[BX].FILECOLOR
	MOV	AL,20h			; AL = space character
	CLD
	REP	STOSW			; clear the command area
	POP	DI			; restore DI
	RET
;
; VIDPUTDIRENT routine.  Displays a drive, directory or file on the screen
; for the file selector.  Takes 4 paramters:  the name of the drive,
; directory, or file; the type (0 for drive, 1 for directory, 2 for file);
; the row where the the entry should be displayed on the screen; and the
; column where the entry should be displayed.  Uses Pascal calling
; conventions.
;
; Returns nothing.  Destroys AX, BX, CX, DX, ES.
;
		;
		; Parameters to the function.
		;
VD_PARM		STRUC	[BP]
		DB	4 DUP (?)	; caller's BP and return address
VD_COLUMN	DW	?		; column to display at (0-5)
VD_ROW		DW	?		; row to display at (0-17)
VD_TYPE		DW	?		; 0 = drive, 1 = directory, 2 = file
VD_NAME		DW	?		; name of file, directory or drive
		ENDS
	EVEN
VIDPUTDIRENT:
	PUSH	BP
	MOV	BP,SP
	PUSH	SI			; save SI, DI for C
	PUSH	DI
	MOV	ES,VIDSEG		; ES addresses video buffer
	MOV	AL,160			; get row address
	MUL	BYTE PTR VD_ROW
	ADD	AX,CMDOFFS
	MOV	DX,AX
	MOV	AL,26			; add in column address
	MUL	BYTE PTR VD_COLUMN
	ADD	DX,AX			; (save in DX)
	MOV	DI,DX			; ES:DI is where to put the string
	MOV	CX,12			; CX is number of characters
	MOV	BX,COLORTBL		; get attribute in AL
	CLD
	MOV	AL,[BX].DIRCOLOR	; (assume directory or drive)
	CMP	BYTE PTR VD_TYPE,FILETYPE
	JNE	VIDPUTDIRENT_ATTRLOOP
	MOV	AL,[BX].FILECOLOR
VIDPUTDIRENT_ATTRLOOP:
	INC	DI			; skip over character
	STOSB				; put in attribute
	LOOP	VIDPUTDIRENT_ATTRLOOP
	MOV	DI,DX			; ES:DI addresses string again
	MOV	SI,VD_NAME		; SI addresses source string
	MOV	CX,12			; CX is number of characters
VIDPUTDIRENT_CHARLOOP:
	LODSB				; get character
	OR	AL,AL			; is it a null?
	JZ	VIDPUTDIRENT_FILLBLANKS	; jump out of the loop if so
	STOSB				; put the character in the string
	INC	DI			; skip over the attribute
	LOOP	VIDPUTDIRENT_CHARLOOP
	JMP	SHORT VIDPUTDIRENT_ALLDONE ; if 80 characters, all done
	EVEN
VIDPUTDIRENT_FILLBLANKS:
	MOV	AL,' '			; fill the rest of the line w/blanks
VIDPUTDIRENT_BLANKLOOP:
	STOSB				; store the blank
	INC	DI			; skip over the attribute
	LOOP	VIDPUTDIRENT_BLANKLOOP
VIDPUTDIRENT_ALLDONE:
	POP	DI			; restore DI, SI
	POP	SI
	POP	BP
	RET	8
;
; VIDHILIGHT routine.  Highlights a directory entry on the selector screen.
; Takes two parameters, the row and column where the highlight should be
; placed.  Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, BX, CX, ES.
;
		;
		; Parameters to the function.
		;
VH_PARM		STRUC	[BP]
		DB	4 DUP (?)	; caller's BP and return address
VH_COLUMN	DW	?		; column to highlight (0-5)
VH_ROW		DW	?		; row to highlight (0-17)
		ENDS
	EVEN
VIDHILIGHT:
	PUSH	BP
	MOV	BP,SP
	PUSH	DI			; save DI for C
	MOV	ES,VIDSEG		; ES addresses video buffer
	MOV	AL,160			; get row address
	MUL	BYTE PTR VH_ROW
	ADD	AX,CMDOFFS
	MOV	DI,AX
	MOV	AL,26			; add in column address
	MUL	BYTE PTR VH_COLUMN
	ADD	DI,AX			; ES:DI is where to highlight
	INC	DI			; (skip over character)
	MOV	CX,12			; CX is number of characters
	CLD
	MOV	AH,ES:[DI]		; get current attribute
	MOV	BX,COLORTBL		; see if it's a file
	MOV	AL,[BX].HDIRCOLOR	; (assume drive or directory)
	CMP	AH,[BX].FILECOLOR
	JNE	VIDHILIGHT_ATTRLOOP
	MOV	AL,[BX].HFILECOLOR
VIDHILIGHT_ATTRLOOP:
	STOSB				; set attribute
	INC	DI			; skip over character
	LOOP	VIDHILIGHT_ATTRLOOP
	POP	DI			; restore DI
	POP	BP
	RET	4
;
; VIDUNHILIGHT routine.  Unhighlights a directory entry on the selector
; screen.  Takes two parameters, the row and column where the highlight is.
; Uses Pascal linkage conventions.
;
; Returns nothing.  Destroys AX, BX, CX, ES.
;
	EVEN
VIDUNHILIGHT:
	PUSH	BP
	MOV	BP,SP
	PUSH	DI			; save DI for C
	MOV	ES,VIDSEG		; ES addresses video buffer
	MOV	AL,160			; get row address
	MUL	BYTE PTR VH_ROW
	ADD	AX,CMDOFFS
	MOV	DI,AX
	MOV	AL,26			; add in column address
	MUL	BYTE PTR VH_COLUMN
	ADD	DI,AX			; ES:DI is where to unhighlight
	INC	DI			; (skip over character)
	MOV	CX,12			; CX is number of characters
	CLD
	MOV	AH,ES:[DI]		; get current attribute
	MOV	BX,COLORTBL		; see if it's a file
	MOV	AL,[BX].DIRCOLOR	; (assume drive or directory)
	CMP	AH,[BX].HFILECOLOR
	JNE	VIDUNHILIGHT_ATTRLOOP
	MOV	AL,[BX].FILECOLOR
VIDUNHILIGHT_ATTRLOOP:
	STOSB				; set attribute
	INC	DI			; skip over character
	LOOP	VIDUNHILIGHT_ATTRLOOP
	POP	DI			; restore DI
	POP	BP
	RET	4
;
; DEBUGSTRING routine.  Writes a null-terminated string to COM2: at 9600
; baud, 8 data bits, 1 stop bit, no parity.  A carriage return/linefeed
; pair is appended to the string.  For debugging.  Takes one parameter, a
; pointer to the string.  Uses Pascal calling conventions.
;
; Returns nothing.  Destroys nothing.
;
	EVEN
DEBUGSTRING:
	PUSH	BP		; save caller's BP
	MOV	BP,SP		; BP addresses parameters on stack
	PUSH	AX
	PUSH	DX
	PUSH	SI
	MOV	SI,[BP+4]	; DS:SI -> caller's string
	CLD			; DF set for incrementing
	MOV	AX,0E3h		; set serial port for 9600 8 N 1
	MOV	DX,1		; set COM2:
	INT	14h
DEBUGSTRING_LOOP:
	MOV	AH,1		; write character to serial port
	LODSB			; get character from string in AL
	OR	AL,AL		; if null, go write CR/LF
	JZ	DEBUGSTRING_CRLF
	INT	14h		; send the character out the serial port
	JMP	SHORT DEBUGSTRING_LOOP
	EVEN
DEBUGSTRING_CRLF:
	MOV	AL,0Dh		; write carriage return
	INT	14h
	MOV	AX,10Ah		; write line feed
	INT	14h
	POP	SI
	POP	DX
	POP	AX
	POP	BP		; restore caller's BP
	RET	2		; discard parameter on stack
;
; GETSYSTIME routine, to get the current system time from low memory.  Uses
; Pascal calling conventions.
;
; Returns the system time in DX:AX.  Destroys ES.
;
	EVEN
GETSYSTIME:
	MOV	AX,40h
	MOV	ES,AX
	MOV	AX,ES:[6Ch]
	MOV	DX,ES:[6Eh]
	RET
_TEXT	ENDS
	END
