/*	Compress.c - Compression/Decompression functions for the PSSJ
	Digital Sound Toolkit
	Copyright 1994, Frank Durda IV. 
	This module originally written by Gordon Burditt
	Commercial use is restricted.  See intro(PSSJ) for more information.
*/
#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include "sound.h"

# ifdef __STDC__
extern void far * far _normalize(void far *pointer);
# else
extern char far * far _normalize();
# endif

typedef struct {
	u_long sound_addr;
	u_long length;
	unsigned int num_codes, RLE_count;
	int dt_deltas[14];
	u_short threshhold_length;
	u_char codes[14];
	int last_sample;
	int current_sample;
	u_char in_RLE;
	u_char threshhold;
	u_char precision;
} REALCOMPINFO;
long far snd_compress_desk88(SOUND far *psound, u_char far *dest,
	u_short dest_size, COMPPARAM far *pParam);

#define TRUE	1
#define FALSE	0

#define UPPER	1

#define MP	3	/*The maximum value we are allowing for precision */
#define NO_CACHE_ENTRY	0xff
#define THRESHHOLD_MIN_LEN	25
#define THRESHHOLD_QUANTUM	51

/*	These are the constants used to run the finite-state automata that
	decompresses compressed data fed to it
*/
#define GET_NEW		0
#define GET_RLE		1
#define GET_RLE_MSN	2
#define GET_RLE_LSN	3
#define GET_NS_MSN	4
#define GET_NS_LSN	5
#define GET_TH_MSN	6
#define GET_TH_LSN	7

/* This macro grabs either the upper or lower nibble of a byte and returns it */
#define getcode(src, ul)	(ul ? ((*src >> 4) & 0xf) : (*src & 0xf))

typedef unsigned long physaddr_t;

/*	compress_part -- This routine is called repeatedly in a loop in order
	to compress a sound into a fixed size buffer.  The current
	implementation is in C but an assembly language version will be
	forthcoming for the sound libraries.

	Input:	pSound -- Pointer to a structure containing the sound.
		start -- Where within the sound to begin the compression.
		end -- Where within the sound the compression will end.
		pInfo -- Pointer to struct where information will be kept
		 	between calls to the routine.
		dest -- Pointer to the buffer where the uncompressed data
			will be stored.
		dest_size -- The length of the storage buffer.
		precision -- How close the actual (calculated) delta's must
		     	be to the delta's in the table before a match
		     	is made.  Zero is an exact match and means that
		     	no data will be lost in the compression process.

	Returns:The number of bytes of data that have been placed into the
		dest buffer.
*/

/*	These really should be auto variables */
static	u_char	far *current, in_RLE, far *thresh;
static	u_long	length, comparison_addr;
static	physaddr_t	sound_addr;
static	unsigned int	processed, chunk;
static	int	delta, max_index, loopvar2, dt_deltas[14];
static	unsigned int	dt_table[512 + (MP * 2)];
static	unsigned int	space_remaining, num_codes, RLE_count, max_hits;
static	u_char	codes[14], entry;
static	int	last_sample;
static	unsigned int	loopvar;
static	u_char	current_sample, top, bottom, start_bottom, start_top;
static	u_long	end;
REALCOMPINFO	far *pinfo;

long	far	snd_compress_part(SOUND far *psound, 
				  u_char far *dest,
				  u_short dest_size,
				  u_short type,
				  COMPPARAM far *pParam)
{
	switch(type)
	{
		case CTYPE_DESKMATE88:
			return snd_compress_desk88(psound, dest, dest_size, 
				pParam);
		default:	/*Unsupported type*/
			return BADPARM;
	}
}

long far snd_compress_desk88(SOUND far *psound,
				u_char far *dest,
				u_short dest_size,
				COMPPARAM far *pParam)
{
/*	normalize pointers */
	pinfo = _normalize(pParam->pinfo);
	dest = _normalize(dest);

/*	Set the amount of space remaining to the buffer size. */
	space_remaining = dest_size;


/*	Check to see if we are just beginning a compression or continuing */
/*	one already started. */
	if (psound != NULL) {
/*	end = 0 indicates use whole sound. */
		if (pParam->end == 0)
		{
			pinfo->length = psound->sndlen;
		}
		else
		{
			pinfo->length = pParam->end - pParam->start;
		}

		current = psound->buffer;
		sound_addr = (((long)FP_SEG(current)) << 4) +
			FP_OFF(current) + pParam->start;
		last_sample = 128;
		in_RLE = FALSE;
		switch(pParam->compress_mode)
		{
			case DESKMATE88_MUSIC:
				pinfo->precision = SND_PRECIS;
				pinfo->threshhold = SND_MUSIC_THR;
				pinfo->threshhold_length = SND_THRESH_LEN
					* THRESHHOLD_QUANTUM;
				break;
			case DESKMATE88_SPEECH:
				pinfo->precision = SND_PRECIS;
				pinfo->threshhold = SND_SPEECH_THR;
				pinfo->threshhold_length = SND_THRESH_LEN
					* THRESHHOLD_QUANTUM;
				break;
			case DESKMATE88_ADJUSTABLE:
				pinfo->precision = pParam->precision;
				pinfo->threshhold = pParam->threshhold;
				pinfo->threshhold_length = 
						pParam->threshhold_length;

				if (pinfo->threshhold_length == 0) {
		    			pinfo->threshhold_length = THRESHHOLD_MIN_LEN;
				} else {
		    			pinfo->threshhold_length *= THRESHHOLD_QUANTUM;
				}
				break;
			default:
				return BADPARM;
		}


/*	Set all entries in the counting table to 0 */
		for (loopvar = 0; loopvar < (512 + (MP * 2)); loopvar++) {
			dt_table[loopvar] = 0;
		}

/*	Get a count of the number of times each delta value occurs. */
		length = pinfo->length;
		while (length > 0) {
			FP_SEG(current) = sound_addr >> 4;
			FP_OFF(current) = sound_addr & 0xf;

			if (length < 65000L) {
				loopvar = length;
				sound_addr += length;
				length = 0;
			} else {
				loopvar = 65000L;
				sound_addr += 65000L;
				length -= 65000L;
 			}

			for (; loopvar > 0; loopvar--, current++) {
				current_sample = *current;
				delta = current_sample - last_sample;
				last_sample = current_sample;
				dt_table[delta + (256 + MP)]++;
			}
		}

		dt_deltas[0] = 0;
		for (loopvar = (256 + MP - pinfo->precision);
		     loopvar <= (256 + MP + pinfo->precision); loopvar++) {
			dt_table[loopvar] = 0;
		}

/*	We need to search through the table and identify the entry with
	the most hits given not only it but also its neighbors within
	precision distance.  At the moment we ignore the contributions
	of any neighbors. 
*/
		for (loopvar = 1; loopvar < 14; loopvar++) {
			max_hits = dt_table[MP];
			max_index = MP;
			for (loopvar2 = MP+1; loopvar2<512+MP; loopvar2++) {
				if (dt_table[loopvar2] > max_hits) {
					max_index = loopvar2;
					max_hits = dt_table[loopvar2];
				}
			}
			dt_deltas[loopvar] = (max_index - 256) - MP;

			for (loopvar2 = (max_index - pinfo->precision);
			     loopvar2 <= (max_index + pinfo->precision);
			     loopvar2++) {
				dt_table[loopvar2] = 0;
			}
		}

		current = psound->buffer;
		sound_addr = (((long)FP_SEG(current)) << 4) + 
			FP_OFF(current) + pParam->start;

		for (loopvar = 0; loopvar < 14; loopvar++) {
			pinfo->dt_deltas[loopvar] = dt_deltas[loopvar];
		}

		pinfo->in_RLE = FALSE;
		pinfo->RLE_count = RLE_count;
		pinfo->last_sample = 128;
		pinfo->num_codes = 0;
		pinfo->sound_addr = sound_addr;

		for (loopvar = 0; loopvar < 14; loopvar++, dest++) {
/*	We need to write out the entries in the delta table.  */
			*dest = (dt_deltas[loopvar] >> 8) & 0xff;
			dest++;
			*dest = dt_deltas[loopvar] & 0xff;
		}
		*dest = (pinfo->threshhold_length >> 8) & 0xff;
		dest++;
		*dest = pinfo->threshhold_length & 0xff;
		dest++;

		return (30);
	} else {
		sound_addr = pinfo->sound_addr;

		length = pinfo->length;
		num_codes = pinfo->num_codes;
		last_sample = pinfo->last_sample;
		in_RLE = pinfo->in_RLE;
		RLE_count = pinfo->RLE_count;

		for (loopvar = 0; loopvar < (512 + (MP * 2)); loopvar++) {
			dt_table[loopvar] = NO_CACHE_ENTRY;
		}

		for (loopvar = 0; loopvar < 14; loopvar++) {
			codes[loopvar] = pinfo->codes[loopvar];
			dt_deltas[loopvar] = pinfo->dt_deltas[loopvar];

			for (loopvar2=(dt_deltas[loopvar]+256+MP-pinfo->precision);
			     loopvar2<=(dt_deltas[loopvar]+256+MP+pinfo->precision);
			     loopvar2++) {
				dt_table[loopvar2] = loopvar;
			}
		}
		for (loopvar = (256 + MP - pinfo->precision);
		     loopvar <= (256 + MP + pinfo->precision); loopvar++) {
			dt_table[loopvar] = 0;
		}
	}

/*	No bytes of data remain to be compressed so we need to return 0 
	to let the calling program know that we are finished compressing 
	the sound.
*/
	if (length == 0) {
		if (num_codes) {

/*	Write out any leftover codes we might have that couldn't be
	written out during the last call.
 */
			if (num_codes % 2) {
				/* There are an odd number of codes */
				for (loopvar = 0; loopvar < num_codes / 2; 
					loopvar++, dest++) {
					*dest = (codes[loopvar * 2] << 4)
						 | (codes[(loopvar*2)+1]&0xf);
				}
				*dest = codes[loopvar * 2] << 4;
 				pinfo->num_codes = 0;
				return ((num_codes / 2) + 1);
			} else {
				/* There are an even number of codes */
				for (loopvar = 0; loopvar < num_codes / 2; 
					loopvar++, dest++) {
					*dest = (codes[loopvar*2] << 4)
						 | (codes[(loopvar*2)+1]&0xf);
				}
				pinfo->num_codes = 0;
				return (num_codes / 2);
			}
		}
		return (0);
	}

	top = 0x80 + pinfo->threshhold;
	bottom = 0x80 - pinfo->threshhold;
	start_top = 0x80 + 2;
	start_bottom = 0x80 - 2;

/*	Set the segment and offset that current will point to. */
	FP_SEG(current) = sound_addr >> 4;
	FP_OFF(current) = sound_addr & 0xf;

/*	Determine how many bytes of data we will attempt to compress */
	if (length < 65000L) {
		chunk = length;
	} else {
		chunk = 65000L;
	}

	comparison_addr = sound_addr;
/*	We can only process 64K - (sound_addr & 0xf) before we overflow 
	the segment value.  Thus processed will always be kept at 65000 
	 or less to prevent overflow of the segment boundary
 */
	for (processed = 0; processed < chunk;
	     processed++, current++, sound_addr++) {
		current_sample = *current;

/*	Compare the sample value against the top and bottom of the
	threshhold range.
 */
		if ((current_sample >= start_bottom)
		    && (current_sample <= start_top)) {
			if (sound_addr >= comparison_addr) {
				/* Begin checking forward */
				thresh = current + 1;
				
				loopvar2 = 255 + pinfo->threshhold_length;

				if ((chunk - processed) < loopvar2) {
					loopvar2 = chunk - processed;
				}

				/* Count forwards until a value is found that is
				 * outside the range of the threshhold box.
				 */
				for (loopvar = 0; loopvar < loopvar2;
				     loopvar++, thresh++) {
					current_sample = *thresh;
					if ((current_sample < bottom)
					    || (current_sample > top)) {
						break;
					}
				}

				/* Backup until we encounter a value that is 
				 * small enough so that we don't chop off the
				 * start of a sound.
				 */
				for (loopvar2 = loopvar;
				     loopvar2 > (loopvar-THRESHHOLD_QUANTUM);
				     loopvar2--, thresh--) {
					current_sample = *thresh;
					if ((current_sample >= start_bottom) && 
						(current_sample <= start_top)) {
						break;
					}
				}
				loopvar = loopvar2;				

				comparison_addr += loopvar;

				if (loopvar >= pinfo->threshhold_length) {
					if (in_RLE) {
						if (RLE_count > 0xd) {
							RLE_count -= 0xd;
							codes[num_codes++]=0xf;
							codes[num_codes++]=0xf;
							codes[num_codes++]=RLE_count >> 4;
							codes[num_codes++]=RLE_count;
						} else if (RLE_count == 0) {
							codes[num_codes++]= 0x0;
						} else {
							codes[num_codes++]=0xf;
							codes[num_codes++]=RLE_count;
						}
						in_RLE = FALSE;
					}

					codes[num_codes++]=0xf;
					codes[num_codes++]=0xe;
					codes[num_codes++]=(loopvar - pinfo->threshhold_length) >> 4;
					codes[num_codes++]=loopvar - pinfo->threshhold_length;

					processed += loopvar - 1;
					current = thresh + 1;
					sound_addr = comparison_addr;
					last_sample = 0x80;
				}
				current_sample = *current;
			}
		}

		delta = current_sample - last_sample;

		if ((delta > pinfo->precision) || (delta < -pinfo->precision)) {
			if (in_RLE) {
				if (RLE_count > 0xd) {
					RLE_count -= 0xd;
					codes[num_codes++] = 0xf;
					codes[num_codes++] = 0xf;
					codes[num_codes++] = RLE_count >> 4;
					codes[num_codes++] = RLE_count;
				} else if (RLE_count == 0) {
					codes[num_codes++] = 0x0;
				} else {
					codes[num_codes++] = 0xf;
					codes[num_codes++] = RLE_count;
				}
				in_RLE = FALSE;
			}

			if ((entry = dt_table[delta+256+MP]) != NO_CACHE_ENTRY) {
				codes[num_codes++] = entry;
				last_sample = last_sample + dt_deltas[entry];

				if (last_sample < 0) {
					last_sample = 0;
				} else if (last_sample > 255) {
					last_sample = 255;
				}
			} else {
				codes[num_codes++] = 0xe;
				codes[num_codes++] = current_sample >> 4;
				codes[num_codes++] = current_sample;
				last_sample = current_sample;
			}
		} else {

/*	The sample value stayed the same so we need to start a RLE or
	continue an existing one.
*/

			if (in_RLE) {
				RLE_count++;

				if (RLE_count == (255 + 0xd)) {
					codes[num_codes++] = 0xf;
					codes[num_codes++] = 0xf;
					codes[num_codes++] = 0xf;
					codes[num_codes++] = 0xf;
					in_RLE = FALSE;
				}
			} else {
				in_RLE = TRUE;
				RLE_count = 0;
			}
		}

		if ((num_codes / 2) <= space_remaining) {
			if (num_codes > 1) {
				if (num_codes % 2) {
					/* There are an odd number of codes */
					for (loopvar = 0; loopvar < (num_codes / 2); 
						loopvar++, dest++, space_remaining--) {
						*dest = (codes[loopvar * 2] << 4) | (codes[(loopvar * 2) + 1] & 0xf);
					}
					codes[0] = codes[loopvar * 2];
					num_codes = 1;
				} else {
					/* There are an even number of codes */
					for (loopvar = 0; loopvar < (num_codes / 2); 
						loopvar++, dest++, space_remaining--) {
						*dest = (codes[loopvar * 2] << 4) | (codes[(loopvar * 2) + 1] & 0xf);
					}
					num_codes = 0;
				}
			}
		} else {
			processed++;
			sound_addr++;
			break;
		}
	}

/*	Set the variables in the info structure for the next pass
*/
	for (loopvar = 0; loopvar < 14; loopvar++) {
		pinfo->codes[loopvar] = codes[loopvar];
	}
	pinfo->in_RLE = in_RLE;
	pinfo->RLE_count = RLE_count;
	pinfo->last_sample = last_sample;
	pinfo->num_codes = num_codes;
	pinfo->sound_addr = sound_addr;
	length -= processed;
	pinfo->length = length;

	return (dest_size - space_remaining);
}

/*	decompress_part -- Decompresses the src into the buffer area
	pointed to by the SOUND structure.  This function is called
	repeatedly in a loop and fed compressed src info a chunk at a time
	until it is completely decompressed.

	Input:	psound --
		pinfo --
		src --
 		src_len --

	Returns:0 if there were no problems and -1 if it was unable to finish
		the decompressing.
 */

/*	commented out static variables duplicated from snd_compress_part 
	these really should be auto variables 
*/
/*	static u_char far *current, current_sample; */
/*	static int last_sample; */
/*	static unsigned int loopvar; */
/*	static u_long sound_addr; */
/*	static unsigned int RLE_count; */
/*	static int dt_deltas[14]; */

	static unsigned int state;
	static unsigned int upper_lower;
	static u_char compr_code;

/*	ATTENTION: THE PARAMETER MISMATCH THE COMPILER REPORTS IS
	INTENTIONAL - IT ALLOWS FOR PASSING VARIABLE PARAMETERS FOR
	DIFFERENT COMPRESSION SCHEMES.     LINE #557
*/
short far snd_decompress_part(SOUND far *psound, 
			      REALCOMPINFO far *pinfo, 
			      u_char far *src, 
			      u_short src_len)
{
	upper_lower = UPPER;

/*	Normalize the pinfo pointer */
	pinfo = _normalize(pinfo);
	src = _normalize(src);

/*	Check to see if we are just beginning a decompression or */
/*	continuing one already started */
	if (psound != NULL) {
		current = psound->buffer;
		if (*current != 0 && (*current) & 0xff != 0xff)
			return BADFMT;
		sound_addr = (((long)FP_SEG(current)) << 4) + 
			FP_OFF(current);
		FP_SEG(current) = sound_addr >> 4;
		FP_OFF(current) = sound_addr & 0xf;

		state = GET_NEW;
		last_sample = 128;

/*	We have to read the first 15 words out of the source
	in order to get the table of deltas and the threshhold_len.
*/

		/* Save these values for future loading in. */
		for (loopvar = 0; loopvar < 14; loopvar++, src++) {
			dt_deltas[loopvar] = *src << 8;
			src++;
			dt_deltas[loopvar] |= *src;
			pinfo->dt_deltas[loopvar] = dt_deltas[loopvar];
		}
		pinfo->threshhold_length = *src << 8;
		src++;
		pinfo->threshhold_length |= *src;
		src++;
		src_len -= 30;
	} else {
		state = pinfo->num_codes;
		current_sample = pinfo->current_sample;
		RLE_count = pinfo->RLE_count;
		sound_addr = pinfo->sound_addr;
		last_sample = pinfo->last_sample;

		for (loopvar = 0; loopvar < 14; loopvar++) {
			dt_deltas[loopvar] = pinfo->dt_deltas[loopvar];
		}
	}

	if (src_len == 0) {
		return(0);
	}

/*	Set the segment and offset that current will point to. */
	FP_SEG(current) = sound_addr >> 4;
	FP_OFF(current) = sound_addr & 0xf;

	for (loopvar = src_len * 2; loopvar > 0; loopvar--) {
		compr_code = getcode(src, upper_lower);
		upper_lower = !upper_lower;

		/* Since we started on UPPER we won't advance immediately */

		if (upper_lower) {
			src++;
		}

		switch (state) {
			case GET_NEW:
				switch (compr_code) {
					case 0xe:
						state = GET_NS_MSN;
						break;
					case 0xf:
						state = GET_RLE;
						break;
					default:
						last_sample += dt_deltas[compr_code];
						if (last_sample < 0) {
							last_sample = 0;
						} else if (last_sample > 255) {
							last_sample = 255;
						}

						*current = last_sample;
						current++;
						sound_addr++;
				}
				break;
			case GET_RLE:
				switch (compr_code) {
					case 0xe:
						last_sample = 0x80;
						state = GET_TH_MSN;
						break;
					case 0xf:
						state = GET_RLE_MSN;
						break;
					default:
						/* This is a short RLE */
						RLE_count = compr_code;
						RLE_count++;
						for (; RLE_count > 0;
						     RLE_count--, current++,
						     sound_addr++) {
							*current = last_sample;
						}
						state = GET_NEW;
				}
				break;
			case GET_RLE_MSN:
				RLE_count = compr_code << 4;
				state = GET_RLE_LSN;
				break;
			case GET_RLE_LSN:
				RLE_count |= compr_code;
				RLE_count += 0xd + 1;
				/* Replicate the last sample decoded RLE
				 * count times.
				 */
				for (; RLE_count > 0; RLE_count--, 
				     current++, sound_addr++) {
					*current = last_sample;
				}
				state = GET_NEW;
				break;
			case GET_TH_MSN:
				RLE_count = compr_code << 4;
				state = GET_TH_LSN;
				break;
			case GET_TH_LSN:
				RLE_count |= compr_code;
				RLE_count += pinfo->threshhold_length + 1;
				/* Replicate the last sample decoded RLE
				 * count times.
				 */
				for (; RLE_count > 0; RLE_count--, 
				     current++, sound_addr++) {
					*current = last_sample;
				}
				state = GET_NEW;
				break;
			case GET_NS_MSN:
				last_sample = compr_code << 4;
				state = GET_NS_LSN;
				break;
			case GET_NS_LSN:
				last_sample |= compr_code;
				*current = last_sample;

				current++;
				sound_addr++;
				state = GET_NEW;
				break;
		}
	}

	pinfo->sound_addr = sound_addr;
	pinfo->current_sample = current_sample;
	pinfo->RLE_count = RLE_count;
	pinfo->last_sample = last_sample;
	pinfo->num_codes = state;

	return (0);
}

