/*-----------------------------------------------------------------------------
 * TV_1200.C
 *
 * Version 1.20 - February 9, 2000
 *
 * This file contains routines to control the SC1200 TV encoder.
 *
 * History:
 *    Initial version by Eyal Cohen.
 *
 * Copyright (c) 1999-2000 National Semiconductor.
 *-----------------------------------------------------------------------------
 */

/*-----------------------------------------------------------------------------
 * gfx_set_tv_format
 *
 * This routine sets the TV encoder registers to the specified format
 * and resolution.
 * Currently only NTSC 640x480 is supported.
 *-----------------------------------------------------------------------------
 */
#if GFX_TV_DYNAMIC
int sc1200_set_tv_format(int format, int resolution)
#else
int gfx_set_tv_format(int format, int resolution)
#endif
{
	switch (format) {
		case TV_FORMAT_NTSC :
			switch (resolution) {
				case TV_RES_640_480 :
					/* Horizontal Display start is 108
					  Total number of pixels per line is 857 */
					WRITE_VID32(SC1200_TVOUT_HORZ_TIM, 0x006c0359);
					/* Horizontal Sync Start is 848
					   Horizontal Sync End is 856   */
					WRITE_VID32(SC1200_TVOUT_HORZ_SYNC, 0x03580350);
					/* Vertical Sync Start is 0
					// Vertical Sync End is 1
					// Vertical Display Start Skew is 1
					// Vertical Display End Skew is 1 */
					WRITE_VID32(SC1200_TVOUT_VERT_SYNC, 0x05001000);
					/* Horizontal display end is 911
					// Total number of display lines per field is 240 */
					WRITE_VID32(SC1200_TVOUT_LINE_END, 0x038f00f0);
					/* Disable vertical down scaling, take all lines */
					WRITE_VID32(SC1200_TVOUT_VERT_DOWNSCALE, 0xffffffff);
					/* HSYNC generated in the TV Encoder module
					// Interval between resets of TV Encoder is once every 2 odd fields
					// Flicker Filter takes 1/8 of skipped line
					// Enable Horizontal interpolation
					// Enable Horizontal up scaling 9/8
					// Disable Horizontal downscale  */
					WRITE_VID32(SC1200_TVOUT_HORZ_SCALING, 0x18221700);
					/* Enable video timing
					// Reset Sub Carrier every two frames
					// Disable BLANK
					// Enable color burst
					// Add the IRE offset
					// NTSC color encoding
					// video generator timing is 525 lines / 60Hz
					// Enable FrameRef
					// VPHASE is 2, HPHASE is 0 */
					WRITE_VID32(SC1200_TVENC_TIM_CTRL_1, 0xa2e01000);
					/* Subcarrier Frequency is 3.579809MHz */
					WRITE_VID32(SC1200_TVENC_SUB_FREQ, 0x21f07c1f);
					/* VSTART is 3, HSTART is 113 */
					WRITE_VID32(SC1200_TVENC_DISP_POS, 0x00030071);
					/* Display size: HEIGHT is 239, WIDTH is 719 */
					WRITE_VID32(SC1200_TVENC_DISP_SIZE, 0x00ef02cf);
					break;
				case TV_RES_800_600 :
					return(GFX_STATUS_UNSUPPORTED);
				default :
					return(GFX_STATUS_BAD_PARAMETER);
			}
			break;
		case TV_FORMAT_PAL :
			return(GFX_STATUS_UNSUPPORTED);
		default :
			return(GFX_STATUS_BAD_PARAMETER);
	}
	return(GFX_STATUS_OK);
}

/*-----------------------------------------------------------------------------
 * gfx_set_tv_output
 *
 * This routine sets the TV encoder registers to the specified output type.
 * Supported output types are : S-VIDEO and Composite.
 *-----------------------------------------------------------------------------
 */
#if GFX_TV_DYNAMIC
int sc1200_set_tv_output(int output)
#else
int gfx_set_tv_output(int output)
#endif
{
	unsigned long tmp;

	switch (output) {
		case TV_OUTPUT_COMPOSITE :
			/* Increase horizontal blanking interval
			// Analog outputs provide Y, C and CVBS
			// Low Water Mark for Y is 0x1F
			// Low Water Mark for Cb is 0xF
			// Chrominance Lowpass filter is 1.3MHz
			// HUE is 0
			// SCPHASE is 0xF9 */
			WRITE_VID32(SC1200_TVENC_TIM_CTRL_2, 0x9ff200f9);
			break;
		case TV_OUTPUT_S_VIDEO :
			/* Increase horizontal blanking interval
			// Analog outputs provide Y, C and CVBS
			// Low Water Mark for Y is 0x1F
			// Low Water Mark for Cb is 0xF
			// Chrominance Lowpass filter is 1.8MHz
			// HUE is 0
			// SCPHASE is 0xF9 */
			WRITE_VID32(SC1200_TVENC_TIM_CTRL_2, 0x9ff300f9);
			break;
		default :
			return(GFX_STATUS_BAD_PARAMETER);
	}

	/* Adjusts the internal voltage reference */
	tmp = READ_VID32(SC1200_TVENC_DAC_CONTROL);
	tmp &= ~SC1200_TVENC_TRIM_MASK;
	tmp |= 0x5;
	WRITE_VID32(SC1200_TVENC_DAC_CONTROL, tmp);

	/* Disable 4:2:2 to 4:4:4 converter interpolation */
	WRITE_VID32(SC1200_TVOUT_DEBUG, SC1200_TVOUT_CONVERTER_INTERPOLATION);

	return(GFX_STATUS_OK);
}

/*-----------------------------------------------------------------------------
 * gfx_set_tv_enable
 *
 * This routine enables or disables the TV output.
 *-----------------------------------------------------------------------------
 */
#if GFX_TV_DYNAMIC
int sc1200_set_tv_enable(int enable)
#else
int gfx_set_tv_enable(int enable)
#endif
{
	unsigned long value;
	value = READ_VID32(SC1200_TVENC_DAC_CONTROL);
	if (enable)
		value &= ~(SC1200_TVENC_POWER_DOWN);
	else
		value |= SC1200_TVENC_POWER_DOWN;
	WRITE_VID32(SC1200_TVENC_DAC_CONTROL, value);
	return(GFX_STATUS_OK);
}

/*-----------------------------------------------------------------------------
 * gfx_set_tv_flicker_filter
 *
 * This routine configures the TV out flicker filter.
 *-----------------------------------------------------------------------------
 */
#if GFX_TV_DYNAMIC
int sc1200_set_tv_flicker_filter(int ff)
#else
int gfx_set_tv_flicker_filter(int ff)
#endif
{
	unsigned long mode;
	mode = READ_VID32(SC1200_TVOUT_HORZ_SCALING);
	mode &= ~SC1200_TVOUT_FLICKER_FILTER_MASK;
	switch (ff)
	{
	case TV_FLICKER_FILTER_NONE:
		WRITE_VID32(SC1200_TVOUT_HORZ_SCALING, mode | SC1200_TVOUT_FLICKER_FILTER_DISABLED);
		break;
	case TV_FLICKER_FILTER_NORMAL:
		WRITE_VID32(SC1200_TVOUT_HORZ_SCALING, mode | SC1200_TVOUT_FLICKER_FILTER_FOURTH_HALF_FOURTH);
		break;
	case TV_FLICKER_FILTER_INTERLACED:
		WRITE_VID32(SC1200_TVOUT_HORZ_SCALING, mode | SC1200_TVOUT_FLICKER_FILTER_HALF_ONE_HALF);
		break;
	default:
		return GFX_STATUS_BAD_PARAMETER;
	}
	return(GFX_STATUS_OK);
}

/*-----------------------------------------------------------------------------
 * gfx_set_tv_sub_carrier_reset
 *
 * This routine configures the TV encoder sub carrier reset interval.
 *-----------------------------------------------------------------------------
 */
#if GFX_TV_DYNAMIC
int sc1200_set_tv_sub_carrier_reset(int screset)
#else
int gfx_set_tv_sub_carrier_reset(int screset)
#endif
{
	unsigned long mode;
	mode = READ_VID32(SC1200_TVENC_TIM_CTRL_1);
	mode &= ~SC1200_TVENC_SUB_CARRIER_RESET_MASK;
	switch (screset)
	{
	case TV_SUB_CARRIER_RESET_NEVER:
		WRITE_VID32(SC1200_TVENC_TIM_CTRL_1, mode | SC1200_TVENC_SUB_CARRIER_RESET_NEVER);
		break;
	case TV_SUB_CARRIER_RESET_EVERY_TWO_LINES:
		WRITE_VID32(SC1200_TVENC_TIM_CTRL_1, mode | SC1200_TVENC_SUB_CARRIER_RESET_EVERY_TWO_LINES);
		break;
	case TV_SUB_CARRIER_RESET_EVERY_TWO_FRAMES:
		WRITE_VID32(SC1200_TVENC_TIM_CTRL_1, mode | SC1200_TVENC_SUB_CARRIER_RESET_EVERY_TWO_FRAMES);
		break;
	case TV_SUB_CARRIER_RESET_EVERY_FOUR_FRAMES:
		WRITE_VID32(SC1200_TVENC_TIM_CTRL_1, mode | SC1200_TVENC_SUB_CARRIER_RESET_EVERY_FOUR_FRAMES);
		break;
	default:
		return GFX_STATUS_BAD_PARAMETER;
	}
	return(GFX_STATUS_OK);
}

/*-----------------------------------------------------------------------------
 * gfx_set_tv_cc_enable
 *
 * This routine enables or disables the use of the hardware CC registers 
 * in the TV encoder.
 *-----------------------------------------------------------------------------
 */
#if GFX_TV_DYNAMIC
int sc1200_set_tv_cc_enable(int enable)
#else
int gfx_set_tv_cc_enable(int enable)
#endif
{
	unsigned long value;
	value = READ_VID32(SC1200_TVENC_CC_CONTROL);
	value &= ~(0x0005F);
	if (enable) value |= 0x51;
	WRITE_VID32(SC1200_TVENC_CC_CONTROL, value);
	return(0);
}

/*-----------------------------------------------------------------------------
 * cc_add_parity_bit
 *
 * This routine adds the (odd) parity bit to the data character.
 *-----------------------------------------------------------------------------
 */
unsigned char cc_add_parity_bit(unsigned char data)
{
	int i, num = 0;
	unsigned char d = data;

	for (i = 0; i < 7; i++)
	{
		if (d & 0x1)
			num++;
		d >>= 1;
	}
	if (num & 0x1)
		return data;
	else
		return (data | 0x80);
}

/*-----------------------------------------------------------------------------
 * gfx_set_tv_cc_data
 *
 * This routine writes the two specified characters to the CC data register 
 * of the TV encoder.
 *-----------------------------------------------------------------------------
 */
#if GFX_TV_DYNAMIC
int sc1200_set_tv_cc_data(unsigned char data1, unsigned char data2)
#else
int gfx_set_tv_cc_data(unsigned char data1, unsigned char data2)
#endif
{
	unsigned long value;
	value = cc_add_parity_bit(data1) | (cc_add_parity_bit(data2) << 8);
	WRITE_VID32(SC1200_TVENC_CC_DATA, value);
	return(0);
}

/*---------------------------------------------------------------------------
 * gfx_test_tvout_odd_field
 * 
 * This routine returns 1 if the current TVout field is odd. Otherwise returns 0.
 *---------------------------------------------------------------------------
 */
#if GFX_VIDEO_DYNAMIC
int sc1200_test_tvout_odd_field(void)
#else
int gfx_test_tvout_odd_field(void)
#endif
{
	unsigned long debug = READ_VID32(SC1200_TVOUT_DEBUG);
	WRITE_VID32(SC1200_TVOUT_DEBUG, debug | SC1200_TVOUT_FIELD_STATUS_TV);
	if (READ_VID32(SC1200_TVOUT_DEBUG) & SC1200_TVOUT_FIELD_STATUS_ODD)
		return(1);
	else return(0);
}

/*---------------------------------------------------------------------------
 * gfx_test_tv_encoder_odd_field
 * 
 * This routine returns 1 if the current TVout field is odd. Otherwise returns 0.
 *---------------------------------------------------------------------------
 */
#if GFX_VIDEO_DYNAMIC
int sc1200_test_tv_encoder_odd_field(void)
#else
int gfx_test_tv_encoder_odd_field(void)
#endif
{
	unsigned long debug = READ_VID32(SC1200_TVOUT_DEBUG);
	WRITE_VID32(SC1200_TVOUT_DEBUG, debug & ~SC1200_TVOUT_FIELD_STATUS_TV);
	if (READ_VID32(SC1200_TVOUT_DEBUG) & SC1200_TVOUT_FIELD_STATUS_ODD)
		return(1);
	else return(0);
}

/* END OF FILE */
