/*
 * linux/drivers/video/geodeioctl.c -- Geode frame buffer IOCTLs
 *
 *
 *    Copyright (C) 2000
 *    National semiconductor corporation
 *
 *
 */

#define GEODEFBDEBUG

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/malloc.h>
#include <linux/delay.h>
#include <linux/fb.h>
#include <linux/console.h>
#include <linux/selection.h>
#include <linux/ioport.h>
#include <linux/init.h>

#include <asm/io.h>
#include <asm/mtrr.h>
#include <asm/uaccess.h>

#include "gfx/gfx_rtns.h"
#include "gfx/gfx_defs.h"
#include "gfx/gfx_regs.h"

#include "/usr/src/nsm/galapi.h"
#include "geodeioctl.h"

#define GALFN_LASTFUNCTION_SUPPORTED\
        GALFN_VGASETMODE
/*
 * Look for our signature and if subfunction is
 * within range
 */

#define IO_PACKET_ISVALID(x)\
   ((x.dwSubfunction <= GALFN_LASTFUNCTION_SUPPORTED) &&\
    (x.dwSignature == FBGAL_SIGNATURE))

typedef int (*PIOCTLFN)(void *);

static char bDispatchTableInitialized = 0;
static PIOCTLFN pIoctlFns[GALFN_LASTFUNCTION_SUPPORTED];

int Nscgfxfn_Unsupported(void *pvarg)
{
    return -EBADRQC;
}

int Nscgfxfn_getadapterinfo(void *pvarg)
{
    int retval;
    GAL_GETADAPTERINFO sAdapterInfo;

    if (copy_from_user(&sAdapterInfo, 
                       (void *) pvarg, sizeof(sAdapterInfo)))
    {
        retval = -EFAULT;
    }
    else
    {
        sAdapterInfo.dwReturnValue = 0;
        sAdapterInfo.sAppInfo.dwCPUVersion = 
            gfx_detect_cpu();
        sAdapterInfo.sAppInfo.dwVideoVersion = 
            gfx_detect_video();
        sAdapterInfo.sAppInfo.dwFrameBufferBase = 
            gfx_get_frame_buffer_base();
        sAdapterInfo.sAppInfo.dwFrameBufferSize =
            gfx_get_frame_buffer_size();
        sAdapterInfo.sAppInfo.dwGfxRegisterBase = 
            gfx_get_cpu_register_base();
        sAdapterInfo.sAppInfo.dwVidRegisterSize = 
            gfx_get_vid_register_base();
        sAdapterInfo.sAppInfo.dwVipRegisterBase = 
            gfx_get_vip_register_base();

        copy_to_user((void *) pvarg,
                     &sAdapterInfo, sizeof(sAdapterInfo));
        
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setsoftvgastate(void *pvarg)
{
    int retval;
    GAL_SOFTVGASTATE sSoftVgaState;
    
    if (copy_from_user(&sSoftVgaState, 
                       (void *) pvarg, sizeof(sSoftVgaState)))
    {
        retval = -EFAULT;
    }
    else
    {
        if (sSoftVgaState.bSoftVgaEnable)
            gfx_enable_softvga();
        else
            gfx_disable_softvga();

        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_waituntilidle(void *pvarg)
{
    int retval;
    GAL_WAITUNTILIDLE sWaitIdle;

    if (copy_from_user(&sWaitIdle, 
                       (void *) pvarg, sizeof(sWaitIdle)))
    {
        retval = -EFAULT;
    }
    else
    {
        gfx_wait_until_idle();
        sWaitIdle.dwReturnValue = 0;

        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_waitverticalblank(void *pvarg)
{
    int retval;
    GAL_WAITVERTICALBLANK sWaitVBlank;

    if (copy_from_user(&sWaitVBlank, 
                       (void *) pvarg, sizeof(sWaitVBlank)))
    {
        retval = -EFAULT;
    }
    else
    {
        gfx_wait_vertical_blank();
        sWaitVBlank.dwReturnValue = 0;

        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_ismodesupported(void *pvarg)
{
    int retval;
    GAL_DISPLAYMODE sDisplayMode;
    
    if (copy_from_user(&sDisplayMode, 
                       (void *) pvarg, sizeof(sDisplayMode)))
    {
        retval = -EFAULT;
    }
    else
    {
        sDisplayMode.dwSupported = 
            gfx_is_display_mode_supported(sDisplayMode.wXres, 
                                          sDisplayMode.wYres, 
                                          sDisplayMode.wBpp,
                                          sDisplayMode.wRefresh);
        sDisplayMode.dwReturnValue = 1;
        copy_to_user((void *) pvarg, 
                     &sDisplayMode, sizeof(sDisplayMode));
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setdisplaymode(void *pvarg)
{
    int retval;
    GAL_DISPLAYMODE sDisplayMode;

    if (copy_from_user(&sDisplayMode, 
                       (void *) pvarg, sizeof(sDisplayMode)))
    {
        retval = -EFAULT;
    }
    else
    {
        sDisplayMode.dwReturnValue = 
            gfx_set_display_mode(sDisplayMode.wXres, 
                                 sDisplayMode.wYres, 
                                 sDisplayMode.wBpp,
                                 sDisplayMode.wRefresh);
        copy_to_user((void *) pvarg, 
                     &sDisplayMode, sizeof(sDisplayMode));
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_getdisplaymode(void *pvarg)
{
    int retval;
    GAL_DISPLAYMODE sDisplayMode;
    int xres, yres, bpp, hz;

    if (copy_from_user(&sDisplayMode, 
                       (void *) pvarg, sizeof(sDisplayMode)))
    {
        retval = -EFAULT;
    }
    else
    {
        sDisplayMode.dwReturnValue = 
            gfx_get_display_mode(&xres, &yres, &bpp, &hz);
        
        sDisplayMode.wXres = xres;
        sDisplayMode.wYres = yres;
        sDisplayMode.wBpp = bpp;
        sDisplayMode.wRefresh = hz;
        copy_to_user((void *) pvarg, 
                     &sDisplayMode, sizeof(sDisplayMode));
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setdisplaybpp(void *pvarg)
{
    int retval;
    GAL_DISPLAYPARAMS sDisplayParams;

    if (copy_from_user(&sDisplayParams, 
                       (void *) pvarg, sizeof(sDisplayParams)))
    {
        retval = -EFAULT;
    }
    else
    {
        gfx_set_bpp(sDisplayParams.wBpp);
        
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_getdisplaybpp(void *pvarg)
{
    int retval;
    GAL_DISPLAYPARAMS sDisplayParams;

    if (copy_from_user(&sDisplayParams, 
                       (void *) pvarg, sizeof(sDisplayParams)))
    {
        retval = -EFAULT;
    }
    else
    {
        sDisplayParams.wBpp = 
            gfx_get_display_bpp();
        copy_to_user((void *) pvarg, 
                     &sDisplayParams, sizeof(sDisplayParams));
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setdisplaypitch(void *pvarg)
{
    int retval;
    GAL_DISPLAYPARAMS sDisplayParams;

    if (copy_from_user(&sDisplayParams, 
                       (void *) pvarg, sizeof(sDisplayParams)))
    {
        retval = -EFAULT;
    }
    else
    {
        gfx_set_display_pitch(sDisplayParams.wPitch);
        
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_getdisplaypitch(void *pvarg)
{
    int retval;
    GAL_DISPLAYPARAMS sDisplayParams;

    if (copy_from_user(&sDisplayParams, 
                       (void *) pvarg, sizeof(sDisplayParams)))
    {
        retval = -EFAULT;
    }
    else
    {
        sDisplayParams.wPitch = 
            gfx_get_display_pitch();
        copy_to_user((void *) pvarg, 
                     &sDisplayParams, sizeof(sDisplayParams));
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setdisplayoffset(void *pvarg)
{
    int retval;
    GAL_DISPLAYPARAMS sDisplayParams;

    if (copy_from_user(&sDisplayParams, 
                       (void *) pvarg, sizeof(sDisplayParams)))
    {
        retval = -EFAULT;
    }
    else
    {
        gfx_set_display_offset(sDisplayParams.dwOffset);
        
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_getdisplayoffset(void *pvarg)
{
    int retval;
    GAL_DISPLAYPARAMS sDisplayParams;

    if (copy_from_user(&sDisplayParams, 
                       (void *) pvarg, sizeof(sDisplayParams)))
    {
        retval = -EFAULT;
    }
    else
    {
        sDisplayParams.dwOffset = 
            gfx_get_display_offset();
        copy_to_user((void *) pvarg, 
                     &sDisplayParams, sizeof(sDisplayParams));
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_getrefreshfromdotclock(void *pvarg)
{
    int retval;
    GAL_DOTCLKTOREFRESH sDclk2refresh;
    int hz;

    if (copy_from_user(&sDclk2refresh, 
                       (void *) pvarg, sizeof(sDclk2refresh)))
    {
        retval = -EFAULT;
    }
    else
    {
        sDclk2refresh.dwReturnValue = 
            gfx_get_refreshrate_from_frequency(sDclk2refresh.wXres,
                                               sDclk2refresh.wYres,
                                               sDclk2refresh.wBpp,
                                               &hz,
                                               sDclk2refresh.dwDotClock);
        sDclk2refresh.wRefreshRate = hz;
        copy_to_user((void *) pvarg, 
                     &sDclk2refresh, sizeof(sDclk2refresh));
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_getdisplaytiming(void *pvarg)
{
    int retval;
    GAL_DISPLAYTIMING sDisplayTiming;
    PGALAPP_DISPLAYTIMING pTiming;

    if (copy_from_user(&sDisplayTiming, 
                       (void *) pvarg, sizeof(sDisplayTiming)))
    {
        retval = -EFAULT;
    }
    else
    {
        sDisplayTiming.dwReturnValue = 0;
        pTiming = &sDisplayTiming.sAppTiming;

        pTiming->dwDotClock = gfx_get_clock_frequency();
        pTiming->wPitch = gfx_get_display_pitch();
        pTiming->wBpp = gfx_get_display_bpp();
        pTiming->wHTotal = gfx_get_htotal();
        pTiming->wHActive = gfx_get_hactive();
        pTiming->wHSyncStart = gfx_get_hsync_start();
        pTiming->wHSyncEnd = gfx_get_hsync_end();
        pTiming->wHBlankStart = gfx_get_hblank_start();
        pTiming->wHBlankEnd = gfx_get_hblank_end();
        pTiming->wVTotal = gfx_get_vtotal();
        pTiming->wVActive = gfx_get_vactive();
        pTiming->wVSyncStart = gfx_get_vsync_start();
        pTiming->wVSyncEnd = gfx_get_vsync_end();
        pTiming->wVBlankStart = gfx_get_vblank_start();
        pTiming->wVBlankEnd = gfx_get_vblank_end();
        pTiming->wPolarity = gfx_get_sync_polarities();

        copy_to_user((void *) pvarg, 
                     &sDisplayTiming, sizeof(sDisplayTiming));
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setdisplaytiming(void *pvarg)
{
    int retval;
    GAL_DISPLAYTIMING sDisplayTiming;
    PGALAPP_DISPLAYTIMING pTiming;

    if (copy_from_user(&sDisplayTiming, 
                       (void *) pvarg, sizeof(sDisplayTiming)))
    {
        retval = -EFAULT;
    }
    else
    {
        sDisplayTiming.dwReturnValue = 0;
        pTiming = &sDisplayTiming.sAppTiming;

        gfx_set_display_timings(pTiming->wBpp,
                                pTiming->wPolarity,
                                pTiming->wHActive,            
                                pTiming->wHBlankStart,
                                pTiming->wHSyncStart,
                                pTiming->wHSyncEnd,
                                pTiming->wHBlankEnd,
                                pTiming->wHTotal,
                                pTiming->wVActive,
                                pTiming->wVBlankStart,
                                pTiming->wVSyncStart ,
                                pTiming->wVSyncEnd , 
                                pTiming->wVBlankEnd ,
                                pTiming->wVTotal,
                                pTiming->dwDotClock);

        gfx_set_display_pitch(pTiming->wPitch);
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setdisplaypalette(void *pvarg)
{
    int retval;
    GAL_PALETTE sPalette;
    
    if (copy_from_user(&sPalette, 
                       (void *) pvarg, sizeof(sPalette)))
    {
        retval = -EFAULT;
    }
    else
    {
        sPalette.dwReturnValue = 0;

        gfx_set_display_palette((unsigned long *)&sPalette.sAppData);
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setcompressionstate(void *pvarg)
{
    int retval;
    GAL_SETCOMPRESSIONSTATE sCompState;
    
    if (copy_from_user(&sCompState, 
                       (void *) pvarg, sizeof(sCompState)))
    {
        retval = -EFAULT;
    }
    else
    {
        sCompState.dwReturnValue = 0;
        
        gfx_set_compression_enable(sCompState.bCompressionState);
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setcompressionparams(void *pvarg)
{
    int retval;
    GAL_SETCOMPRESSIONPARAMS sCompParams;
    
    if (copy_from_user(&sCompParams, 
                       (void *) pvarg, sizeof(sCompParams)))
    {
        retval = -EFAULT;
    }
    else
    {
        sCompParams.dwReturnValue = 0;
        
        if (sCompParams.dwFlags & GAL_COMPRESSION_OFFSET)
            gfx_set_compression_offset(sCompParams.dwCompOffset);
        
        if (sCompParams.dwFlags & GAL_COMPRESSION_PITCH)
            gfx_set_compression_pitch(sCompParams.dwCompPitch);
        
        if (sCompParams.dwFlags & GAL_COMPRESSION_SIZE)
            gfx_set_compression_size(sCompParams.dwCompSize);
        
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setcursorenable(void *pvarg)
{
    int retval;
    GAL_SETCURSORENABLE sCursorEnable;
    
    if (copy_from_user(&sCursorEnable, 
                       (void *) pvarg, sizeof(sCursorEnable)))
    {
        retval = -EFAULT;
    }
    else
    {
        gfx_set_cursor_enable(sCursorEnable.bCursorEnable);
        sCursorEnable.dwReturnValue = 0;
        
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setcursorcolors(void *pvarg)
{
    int retval;
    GAL_SETCURSORCOLORS sCursorCol;
    
    if (copy_from_user(&sCursorCol, 
                       (void *) pvarg, sizeof(sCursorCol)))
    {
        retval = -EFAULT;
    }
    else
    {
        gfx_set_cursor_colors(sCursorCol.dwBgColor,
                              sCursorCol.dwFgColor);
        sCursorCol.dwReturnValue = 0;
        
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setcursorposition(void *pvarg)
{
    int retval;
    GAL_SETCURSORPOSITION sCursorPos;
    
    if (copy_from_user(&sCursorPos, 
                       (void *) pvarg, sizeof(sCursorPos)))
    {
        retval = -EFAULT;
    }
    else
    {
        gfx_set_cursor_position(sCursorPos.dwMemOffset,
                                sCursorPos.wXPos,
                                sCursorPos.wYPos,
                                sCursorPos.wXHot,
                                sCursorPos.wYHot);
        sCursorPos.dwReturnValue = 0;
        
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setcursorshape32(void *pvarg)
{
    int retval;
    GAL_SETCURSORSHAPE sCursorShape;
    
    if (copy_from_user(&sCursorShape, 
                       (void *) pvarg, sizeof(sCursorShape)))
    {
        retval = -EFAULT;
    }
    else
    {
        gfx_set_cursor_shape32(sCursorShape.dwMemOffset,
                               (unsigned long *)sCursorShape.dwAndMask,
                               (unsigned long *)sCursorShape.dwXorMask);
        sCursorShape.dwReturnValue = 0;
        
        retval = 0;
    }
    
    return retval;    
}

int Nscgfxfn_setsolidpattern(void *pvarg)
{
    int retval;
    GAL_SETSOLIDPATTERN sSetSolidPat;

    if (copy_from_user(&sSetSolidPat, 
                       (void *) pvarg, sizeof(sSetSolidPat)))
    {
        retval = -EFAULT;
    }
    else
    {
        sSetSolidPat.dwReturnValue = 0;
        gfx_set_solid_pattern(sSetSolidPat.dwColor);
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setsolidsource(void *pvarg)
{
    int retval;
    GAL_SETSOLIDSOURCE sSetSolidSrc;

    if (copy_from_user(&sSetSolidSrc, 
                       (void *) pvarg, sizeof(sSetSolidSrc)))
    {
        retval = -EFAULT;
    }
    else
    {
        sSetSolidSrc.dwReturnValue = 0;
        gfx_set_solid_source(sSetSolidSrc.dwColor);
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setmonopattern(void *pvarg)
{
    int retval;
    GAL_SETMONOPATTERN sSetMonoPat;

    if (copy_from_user(&sSetMonoPat, 
                       (void *) pvarg, sizeof(sSetMonoPat)))
    {
        retval = -EFAULT;
    }
    else
    {
        sSetMonoPat.dwReturnValue = 0;
        gfx_set_mono_pattern(sSetMonoPat.dwBgColor,
                             sSetMonoPat.dwFgColor,
                             sSetMonoPat.dwData0,
                             sSetMonoPat.dwData1,
                             sSetMonoPat.cTransparency);
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_setrasteroperation(void *pvarg)
{
    int retval;
    GAL_SETRASTEROPERATION sSetRop;

    if (copy_from_user(&sSetRop, 
                       (void *) pvarg, sizeof(sSetRop)))
    {
        retval = -EFAULT;
    }
    else
    {
        sSetRop.dwReturnValue = 0;
        gfx_set_raster_operation(sSetRop.cRop);
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_patternfill(void *pvarg)
{
    int retval;
    GAL_PATTERNFILL sPatternFill;

    if (copy_from_user(&sPatternFill, 
                       (void *) pvarg, sizeof(sPatternFill)))
    {
        retval = -EFAULT;
    }
    else
    {
        sPatternFill.dwReturnValue = 0;
        gfx_pattern_fill(sPatternFill.wXPos,
                         sPatternFill.wYPos,
                         sPatternFill.wWidth,
                         sPatternFill.wHeight);
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_screentoscreenblt(void *pvarg)
{
    int retval;
    GAL_SCREENTOSCREENBLT sScreen2ScreenBlt;

    if (copy_from_user(&sScreen2ScreenBlt, 
                       (void *) pvarg, sizeof(sScreen2ScreenBlt)))
    {
        retval = -EFAULT;
    }
    else
    {
        sScreen2ScreenBlt.dwReturnValue = 0;
        gfx_screen_to_screen_blt(sScreen2ScreenBlt.wXStart,
                                 sScreen2ScreenBlt.wYStart,
                                 sScreen2ScreenBlt.wXEnd,
                                 sScreen2ScreenBlt.wYEnd,
                                 sScreen2ScreenBlt.wWidth,
                                 sScreen2ScreenBlt.wHeight);
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_screentoscreenxblt(void *pvarg)
{
    int retval;
    GAL_SCREENTOSCREENXBLT sScreen2ScreenXBlt;

    if (copy_from_user(&sScreen2ScreenXBlt, 
                       (void *) pvarg, sizeof(sScreen2ScreenXBlt)))
    {
        retval = -EFAULT;
    }
    else
    {
        sScreen2ScreenXBlt.dwReturnValue = 0;
        gfx_screen_to_screen_xblt(sScreen2ScreenXBlt.wXStart,
                                 sScreen2ScreenXBlt.wYStart,
                                 sScreen2ScreenXBlt.wXEnd,
                                 sScreen2ScreenXBlt.wYEnd,
                                 sScreen2ScreenXBlt.wWidth,
                                 sScreen2ScreenXBlt.wHeight,
                                 sScreen2ScreenXBlt.wColor);
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_bresenhamline(void *pvarg)
{
    int retval;
    GAL_BRESENHAMLINE sBresenhamLine;
    
    if (copy_from_user(&sBresenhamLine, 
                       (void *) pvarg, sizeof(sBresenhamLine)))
    {
        retval = -EFAULT;
    }
    else
    {
        sBresenhamLine.dwReturnValue = 0;
        gfx_bresenham_line(sBresenhamLine.wX1,
                                  sBresenhamLine.wY1,
                                  sBresenhamLine.wLength,
                                  sBresenhamLine.wErr,
                                  sBresenhamLine.wE1,
                                  sBresenhamLine.wE2,
                                  sBresenhamLine.wFlags);
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_vgamodeswitch(void *pvarg)
{
    int retval;
    GAL_VGAMODEDATA sVgaData;
    
    if (copy_from_user(&sVgaData, 
                       (void *) pvarg, sizeof(sVgaData)))
    {
        retval = -EFAULT;
    }
    else
    {
        sVgaData.dwReturnValue = 0;
        gfx_vga_mode_switch(sVgaData.dwFlags);
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_vgaclearextended(void *pvarg)
{
    int retval;
    GAL_VGAMODEDATA sVgaData;
    
    if (copy_from_user(&sVgaData, 
                       (void *) pvarg, sizeof(sVgaData)))
    {
        retval = -EFAULT;
    }
    else
    {
        sVgaData.dwReturnValue = 0;
        gfx_vga_clear_extended();
        retval = 0;
    }
    
    return retval;
}

int Nscgfxfn_vgapitch(void *pvarg)
{
    int retval;
    GAL_VGAMODEDATA sVgaData;
    
    if (copy_from_user(&sVgaData, 
                       (void *) pvarg, sizeof(sVgaData)))
    {
        retval = -EFAULT;
    }
    else
    {
        sVgaData.dwReturnValue = 0;
        gfx_vga_pitch((gfx_vga_struct *)&sVgaData.sVgaRegs, 
                      sVgaData.dwFlags);
        retval = 0;
        copy_to_user((void *) pvarg,
                     &sVgaData, 
                     sizeof(sVgaData));
    }
    
    return retval;
}

int Nscgfxfn_vgarestore(void *pvarg)
{
    int retval;
    GAL_VGAMODEDATA sVgaData;

    if (copy_from_user(&sVgaData, 
                       (void *) pvarg, sizeof(sVgaData)))
    {
        retval = -EFAULT;
    }
    else
    {
        sVgaData.dwReturnValue = 0;
        gfx_vga_restore((gfx_vga_struct *)&sVgaData.sVgaRegs, 
                        sVgaData.dwFlags);
        retval = 0;
        copy_to_user((void *) pvarg,
                     &sVgaData, 
                     sizeof(sVgaData));
    }
    
    return retval;
}

int Nscgfxfn_vgasave(void *pvarg)
{
    int retval;
    GAL_VGAMODEDATA sVgaData;

    if (copy_from_user(&sVgaData, 
                       (void *) pvarg, sizeof(sVgaData)))
    {
        retval = -EFAULT;
    }
    else
    {
        sVgaData.dwReturnValue = 0;
        gfx_vga_save((gfx_vga_struct *)&sVgaData.sVgaRegs,
                     sVgaData.dwFlags);
        retval = 0;
        copy_to_user((void *) pvarg,
                     &sVgaData, 
                     sizeof(sVgaData));
    }
    
    return retval;
}

int Nscgfxfn_vgamode(void *pvarg)
{
    int retval;
    GAL_VGAMODEDATA sVgaData;

    if (copy_from_user(&sVgaData, 
                       (void *) pvarg, sizeof(sVgaData)))
    {
        retval = -EFAULT;
    }
    else
    {
        sVgaData.dwReturnValue = 0;
        gfx_vga_mode((gfx_vga_struct *)&sVgaData.sVgaRegs,
                     sVgaData.wXres,
                     sVgaData.wYres,
                     sVgaData.wBpp,
                     sVgaData.wRefresh);
        retval = 0;
        copy_to_user((void *) pvarg,
                     &sVgaData, 
                     sizeof(sVgaData));
    }
    
    return retval;    
}

void Init_Dispatch_Table()
{
    int iFnIndex = 0;
        
    while(iFnIndex <= GALFN_LASTFUNCTION_SUPPORTED)
        pIoctlFns[iFnIndex++] = Nscgfxfn_Unsupported;

    pIoctlFns[GALFN_GETADAPTERINFO] =
        Nscgfxfn_getadapterinfo;
    pIoctlFns[GALFN_SETSOFTVGASTATE] =
        Nscgfxfn_setsoftvgastate;
    pIoctlFns[GALFN_WAITUNTILIDLE] =
        Nscgfxfn_waituntilidle;
    pIoctlFns[GALFN_WAITVERTICALBLANK] =
        Nscgfxfn_waitverticalblank;

    pIoctlFns[GALFN_ISDISPLAYMODESUPPORTED] =
        Nscgfxfn_ismodesupported;   
    pIoctlFns[GALFN_SETDISPLAYMODE] =
        Nscgfxfn_setdisplaymode;    
    pIoctlFns[GALFN_GETDISPLAYMODE] =
        Nscgfxfn_getdisplaymode;
    pIoctlFns[GALFN_SETDISPLAYBPP] =
        Nscgfxfn_setdisplaybpp;    
    pIoctlFns[GALFN_GETDISPLAYBPP] =        
        Nscgfxfn_getdisplaybpp;    
    pIoctlFns[GALFN_SETDISPLAYPITCH] =
        Nscgfxfn_setdisplaypitch;    
    pIoctlFns[GALFN_GETDISPLAYPITCH] =        
        Nscgfxfn_getdisplaypitch;    
    pIoctlFns[GALFN_SETDISPLAYOFFSET] =
        Nscgfxfn_setdisplayoffset;    
    pIoctlFns[GALFN_GETDISPLAYOFFSET] =        
        Nscgfxfn_getdisplayoffset;    
    pIoctlFns[GALFN_DOTCLKTOREFRESH] =        
        Nscgfxfn_getrefreshfromdotclock;    
    pIoctlFns[GALFN_GETDISPLAYTIMINGS] =        
        Nscgfxfn_getdisplaytiming;    
    pIoctlFns[GALFN_SETDISPLAYTIMINGS] =        
        Nscgfxfn_setdisplaytiming;
    pIoctlFns[GALFN_SETPALETTE] =      
        Nscgfxfn_setdisplaypalette;
    pIoctlFns[GALFN_SETCOMPRESSIONSTATE] =        
        Nscgfxfn_setcompressionstate;
    pIoctlFns[GALFN_SETCOMPRESSIONPARAMS] =
        Nscgfxfn_setcompressionparams;
    
    pIoctlFns[GALFN_SETCURSORENABLE] =
        Nscgfxfn_setcursorenable;
    pIoctlFns[GALFN_SETCURSORCOLORS] =
        Nscgfxfn_setcursorcolors;
    pIoctlFns[GALFN_SETCURSORPOSITION] =
        Nscgfxfn_setcursorposition;
    pIoctlFns[GALFN_SETCURSORSHAPE] =
        Nscgfxfn_setcursorshape32;
    
    pIoctlFns[GALFN_SETSOLIDPATTERN] =
        Nscgfxfn_setsolidpattern;   
    pIoctlFns[GALFN_SETRASTEROPERATION] =
        Nscgfxfn_setrasteroperation;
    pIoctlFns[GALFN_SETSOLIDSOURCE] =
        Nscgfxfn_setsolidsource;    
    pIoctlFns[GALFN_PATTERNFILL] =
        Nscgfxfn_patternfill;       
    pIoctlFns[GALFN_SETMONOPATTERN] =
        Nscgfxfn_setmonopattern;    
    pIoctlFns[GALFN_SCREENTOSCREENBLT] =
        Nscgfxfn_screentoscreenblt; 
    pIoctlFns[GALFN_SCREENTOSCREENXBLT] =
        Nscgfxfn_screentoscreenxblt;
    pIoctlFns[GALFN_BRESENHAMLINE] =
        Nscgfxfn_bresenhamline;

    pIoctlFns[GALFN_VGAMODESWITCH] =
        Nscgfxfn_vgamodeswitch;
    pIoctlFns[GALFN_VGACLEARCRTEXT] =
        Nscgfxfn_vgaclearextended;
    pIoctlFns[GALFN_VGASETPITCH] =
        Nscgfxfn_vgapitch;
    pIoctlFns[GALFN_VGARESTORE] =
        Nscgfxfn_vgarestore;
    pIoctlFns[GALFN_VGASAVE] =
        Nscgfxfn_vgasave;
    pIoctlFns[GALFN_VGASETMODE] =
        Nscgfxfn_vgamode;

    bDispatchTableInitialized = 1;
}

int geodefb_ioctl_ext(struct inode *inode, struct file *file, u_int cmd,
                      u_long arg, int con, struct fb_info *info)
{
    int retval = -EINVAL;
    GAL_BASE sNscgfxbase;

    if (!bDispatchTableInitialized)
    {
        Init_Dispatch_Table();
    }

    if (copy_from_user(&sNscgfxbase, 
                       (void *) arg, sizeof(sNscgfxbase)))
    {
        retval = -EFAULT;
    }
    else
    {
        if (IO_PACKET_ISVALID(sNscgfxbase))
        {
            retval = (pIoctlFns[sNscgfxbase.dwSubfunction])((void *)arg);            
        }
    }

    return retval;
}
