/*---------------------------------------------------------------------------
 * geode_init.c
 *
 * This file contains the driver initialization code.  It makes use of the 
 * Durango graphics software support package from National.
 *
 * It is ported from the geode driver developed by Annius Groenink.
 *
 * Copyright (c) 1999-2000 National Semicondictor.
 *---------------------------------------------------------------------------
 */

/* Xfree86 header files */

#include "xf86.h"
#include "xf86Priv.h"
#include "xf86_OSlib.h"
#include "xf86_HWlib.h"
#include "vga.h"
#include "vga256.h"
#define XCONFIG_FLAGS_ONLY
#include "xf86_Config.h"

/* National Semiconductor header files. */
#include "/usr/src/nsm/galproto.h"

/* controls the inclusion of compression code */
#define COMPRESSION 0

#define DEBUGMSG(cond, printf_exp)\
	((cond)?(ErrorF printf_exp),1:0)

extern vgaHWCursorRec vgaHWCursor;

extern Bool GeodeCursorInit();
extern void GeodeRestoreCursor();
extern void GeodeWarpCursor();
extern void GeodeQueryBestSize();

/* MEMORY ORGANIZATION */

unsigned long geodeCursorOffset;
unsigned long geodeOffscreenOffset;
unsigned long geodeOffscreenSize;

typedef struct {
	vgaHWRec std;			/* IBM VGA */
	GALAPP_VGAREGS durangovga;	/* Extended VGA */
	int BPP;			/* bits per pixel */
} vgaGeodeRec, *vgaGeodePtr;

#define newstate ((vgaGeodePtr)vgaNewVideoState)
#define GFX_CPU_PYRAMID		0x20801	 

static unsigned Geode_ExtPorts[] = { 0x22,0x23,0x80, };  /* I/O ports used by gfx_gxm_config_read */
static int Num_Geode_ExtPorts =
       (sizeof(Geode_ExtPorts)/sizeof(Geode_ExtPorts[0]));
static unsigned long geode_cpu_version;

typedef struct _MemOffset
{
unsigned long xres;
unsigned long yres;
unsigned long bpp;
unsigned long CBOffset;
unsigned long CBPitch;
unsigned long CBSize;
}MemOffset;

MemOffset GeodeMemOffset[]={
{640, 480, 8, 640, 1024, 256},
{640, 480, 16, 1280, 2048, 256},
{800, 600, 8, 800, 1024, 208},
{800, 600, 16, 1600, 2048, 256},
{1024, 768, 8, 0xc0000, 1024, 256},
{1024, 768, 16, 0x180000, 2048, 256},
{1280, 1024, 8, 1280, 2048, 256},
{1280, 1024, 16, 2560, 4096, 256}
};
/* ROUTINE DECLARATIONS */

static Bool	GeodeProbe();
static char 	*GeodeIdent();
static void	GeodeEnterLeave();
static Bool	GeodeInit();
static int 	GeodeValidMode();
static void 	*GeodeSave();
static void	GeodeRestore();
static void	GeodeAdjust();
static void	GeodeFbInit();
static Bool	GeodePitchAdjust();

/* STILL NAMED geode UNTIL PROPER MAKEFILE TO CHANGE IS FOUND */

vgaVideoChipRec GEODE = {
	GeodeProbe,
	GeodeIdent,
	GeodeEnterLeave,
	GeodeInit,
	GeodeValidMode,
	GeodeSave,
	GeodeRestore,
	GeodeAdjust,
	vgaHWSaveScreen,
	(void (*)())NoopDDA,     /* GeodeGetMode not implemented */
	GeodeFbInit,
	(void (*)())NoopDDA,     /* GeodeSetRead not implemented */
	(void (*)())NoopDDA,     /* GeodeSetWrite not implemented */
	(void (*)())NoopDDA,     /* GeodeSetReadWrite not implemented */
	0x10000,                 /* ChipMapSize */
	0x10000,                 /* ChipSegmentSize */
	16,                      /* ChipSegmentShift */
	0xFFFF,                  /* ChipSegmentMask */
	0x00000, 0x10000,        /* ChipReadBottom/Top */
	0x00000, 0x10000,        /* ChipWriteBottom/Top */
	TRUE,                    /* ChipUse2Banks */
	VGA_NO_DIVIDE_VERT,      /* ChipInterlaceType */
	{0,},                    /* ChipOptionFlags */
	8,                       /* ChipRounding */
	TRUE,                    /* ChipUseLinearAddressing */
	0x40800000,              /* ChipLinearBase */
	0x001FFFFF,              /* ChipLinearSize */
	TRUE,                    /* ChipHas16bpp */
	FALSE,                   /* ChipHas24bpp */
	FALSE,                   /* ChipHas32bpp */
	NULL,                    /* ChipBuiltinModes */
	1,                       /* ChipClockMulFactor */
	1                        /* ChipCLockDivFactor */
};

/*---------------------------------------------------------------------------
 * GeodeIdent
 *
 * The server will call this function repeatedly, until a NULL is returned, 
 * when printing out the list of configured drivers.
 *---------------------------------------------------------------------------
 */

#define NUM_CHIPSETS 4
static char chipsets[NUM_CHIPSETS][8] = {
	"mediagx", "GXLV", "SC1400A", "SC1200"
};

static char *GeodeIdent(n)
int n;
{
	if (n >= NUM_CHIPSETS) return(NULL);
	else return((char *)&chipsets[n][0]);
}

/*---------------------------------------------------------------------------
 * GeodeProbe
 *
 * This function detects the presence of the hardware and initializes 
 * the Durango register pointers.
 *---------------------------------------------------------------------------
 */
static Bool GeodeProbe()
{
	unsigned long cpu_version;
  GALAPP_ADAPTERINFO sAdapterInfo;
 
	/* SETUP IO PERMISSIONS */
	/* Works without this for FreeBSD, needed for Linux. */

    xf86ClearIOPortList(vga256InfoRec.scrnIndex);
    xf86AddIOPorts(vga256InfoRec.scrnIndex, Num_VGA_IOPorts, VGA_IOPorts);
    xf86AddIOPorts(vga256InfoRec.scrnIndex, Num_Geode_ExtPorts, Geode_ExtPorts);
	
	/* ENABLE IO PORTS */

    GeodeEnterLeave(ENTER);

    /* Kumar:
     * Initialize our interface to kernel mode gfx support
     */
    Gal_initialize_interface();

	/* USE DURANGO ROUTINE TO DETECT THE CPU */
    Gal_get_adapter_info(&sAdapterInfo);
    geode_cpu_version = cpu_version = sAdapterInfo.dwCPUVersion;

	if (!cpu_version)
	{
		/* NO GEODE PROCESSOR DETECTED */

		if (xf86Verbose > 1)
		{
			DEBUGMSG(1, ("%s %s: No Geode Processor Detected.\n",
				XCONFIG_PROBED, vga256InfoRec.name));
		}
		GeodeEnterLeave(LEAVE);
		return(FALSE);
	}

	/* DISPLAY DETECTION MESSAGE */

	DEBUGMSG(1,("%s %s: Geode Processor Detected (Type = %d, Version = %d.%d).\n",
	       XCONFIG_PROBED, vga256InfoRec.name, cpu_version & 0xFF, 
		   (cpu_version >> 8) & 0xFF, (cpu_version >> 16) & 0xFF));

	/* GET AMOUNT OF VIDEO MEMORY */
	/* Determine value if user has not specified it in XF86Config. */

	if (!vga256InfoRec.videoRam)
      vga256InfoRec.videoRam = sAdapterInfo.dwFrameBufferSize >> 10;

	/* FILL IN SOME INFORMATION */

	vga256InfoRec.maxClock = 157500; /* 1280x1024 @ 85*/
	vga256InfoRec.chipset = GeodeIdent((cpu_version & 0xFF) - 1);
	vga256InfoRec.bankedMono = TRUE;
	OFLG_SET(CLOCK_OPTION_PROGRAMABLE, &vga256InfoRec.clockOptions);

	/* define option set valid for the driver */
	OFLG_SET(OPTION_SW_CURSOR, &GEODE.ChipOptionFlags);
	OFLG_SET(OPTION_HW_CURSOR, &GEODE.ChipOptionFlags);
	OFLG_SET(OPTION_NOACCEL, &GEODE.ChipOptionFlags);

	/* switch on hardware cursor option */
	if( !OFLG_ISSET(OPTION_SW_CURSOR, &vga256InfoRec.options) )
	    OFLG_SET(OPTION_HW_CURSOR, &vga256InfoRec.options);

	/* HOOK ROUTINE TO ADJUST PITCH (NEED 1K OR 2K FOR ACCELERATION) */

	vgaSetPitchAdjustHook(GeodePitchAdjust);

	/* USE LINEAR ADDRESSING */

	GEODE.ChipLinearBase = sAdapterInfo.dwFrameBufferBase;
	GEODE.ChipLinearSize = (1024 * vga256InfoRec.videoRam);
	GEODE.ChipUseLinearAddressing = TRUE;

	/* SET DURANGO REGISTER POINTERS */
	/* The method of mapping from a physical address to a linear address */
	/* is operating system independent.  Set variables to linear address. */

	xf86MapVidMem(vga256InfoRec.scrnIndex,
                EXTENDED_REGION, (void*)sAdapterInfo.dwGfxRegisterBase, 0x20000);
	xf86MapVidMem(vga256InfoRec.scrnIndex,
                EXTENDED_REGION, (void*)sAdapterInfo.dwVidRegisterSize, 0x1000);
	xf86MapVidMem(vga256InfoRec.scrnIndex,
                EXTENDED_REGION, (void*)sAdapterInfo.dwFrameBufferBase, 
                sAdapterInfo.dwFrameBufferSize);

	/* CHECK IF REGISTERS WERE MAPPED SUCCESSFULLY */
#if 0
	if ((!gfx_regptr) || (!gfx_vidptr) || (!gfx_fbptr))
	{	
		DEBUGMSG(1, ("%s %s: Could not map hardware registers.\n",
			XCONFIG_PROBED, vga256InfoRec.name));
		GeodeEnterLeave(LEAVE);
		return(FALSE);
	}	
#endif
	return(TRUE);
}

/*---------------------------------------------------------------------------
 * GeodeCalculatePitchBytes
 * 
 * Local routine to return pitch of the screen, in bytes.
 *---------------------------------------------------------------------------
 */
static int GeodeCalculatePitchBytes()
{
	int lineDelta = vga256InfoRec.virtualX * (vgaBitsPerPixel >> 3);
	DEBUGMSG(0, ("GeodeCalculatePitchBytes = %d %d %d \n",
		vga256InfoRec.virtualX, vgaBitsPerPixel, lineDelta));

	if( !OFLG_ISSET(OPTION_NOACCEL, &vga256InfoRec.options )) {	
		/* FORCE TO 1K, 2K OR 4K (Pyramid only) IF ACCELERATION IS ENABLED */
		if( (geode_cpu_version == GFX_CPU_PYRAMID) && ( lineDelta > 2048 )) { 
			lineDelta = 4096;
		} else if (lineDelta > 1024) { 
			lineDelta = 2048;
		} else {
			lineDelta = 1024; 
		}
	} else {
		/* FORCE TO MULTIPLE OF 16 IF NO ACCELERATION */
		lineDelta = (lineDelta + 15) & 0xFFF0; 
	}
	DEBUGMSG(0, ("GeodeCalculatePitchBytes = %d \n", lineDelta));
	return( lineDelta );
}

/*---------------------------------------------------------------------------
 * GeodeFbInit
 * 
 * From README file: "The FbInit() function is required for drivers with
 * accelerated graphics support.  It is used to replace default cfb.banked
 * functions with accelerated chip-specific versions.
 *
 * This routine is also used to allocate graphics memory. 
 *---------------------------------------------------------------------------
 */
static void GeodeFbInit()
{
	unsigned long total_size, fb_size;
	GALAPP_ADAPTERINFO sAdapterInfo;

  
	DEBUGMSG(0, ("%s %s: GeodeFbInit\n", 
		XCONFIG_PROBED, vga256InfoRec.name));

	/* ALWAYS PUT THE CURSOR AT THE END OF VIDEO MEMORY */
	Gal_get_adapter_info(&sAdapterInfo);
	total_size = sAdapterInfo.dwFrameBufferSize;
	geodeCursorOffset = total_size - 512;
	fb_size = vga256InfoRec.virtualY * GeodeCalculatePitchBytes();
	geodeOffscreenOffset = fb_size;
	geodeOffscreenSize = total_size - fb_size - 512;

	/* PRINT RESULTS OF OFFSCREEN MEMORY CONFIGURATION */

	if (geodeOffscreenSize <= 0)
	{
		DEBUGMSG(1, ("%s %s: No offscreen memory available.\n", 
			XCONFIG_PROBED, vga256InfoRec.name));
	}
	else
	{
		DEBUGMSG(1, ("%s %s: Offscreen memory from 0x%8.8X-0x%8.8X.\n",
			XCONFIG_PROBED, vga256InfoRec.name, 
			geodeOffscreenOffset, 
			geodeOffscreenOffset+geodeOffscreenSize-1));
	}

    /* CALL "GeodeAccelInit" ROUTINE TO SET UP XAA ACCELERATED FUNCTIONS */
	/* Local routine to hook desired rendering operations to accelerate. */

	if (!OFLG_ISSET(OPTION_NOACCEL, &vga256InfoRec.options))
		GeodeAccelInit();

   	/* INSTALL HARDWARE CURSOR ROUTINES */

	if (OFLG_ISSET(OPTION_HW_CURSOR, &vga256InfoRec.options))
	{
		vgaHWCursor.Initialized = TRUE;
		vgaHWCursor.Init = GeodeCursorInit;
		vgaHWCursor.Restore = GeodeRestoreCursor;
		vgaHWCursor.Warp = GeodeWarpCursor;
		vgaHWCursor.QueryBestSize = GeodeQueryBestSize;
		DEBUGMSG(1, ("%s %s: Using hardware cursor at offset 0x%8.8X.\n",
			XCONFIG_PROBED, vga256InfoRec.name, geodeCursorOffset));
	}
}

/*---------------------------------------------------------------------------
 * GeodePitchAdjust
 * 
 * This routine is hooked to allow the driver to enforce pitch requirements
 * that the hardware has.  For acceleration to be used, the pitch must be
 * 1K , 2K or 4K (Pyramid only) bytes.
 *---------------------------------------------------------------------------
 */
static int GeodePitchAdjust()
{
	int pitch, lineDelta = GeodeCalculatePitchBytes();

	/* PYRAMID HAS A MAX OF 4K, OTHER HARDWARE HAS A MAX OF 2K */

	if( geode_cpu_version != GFX_CPU_PYRAMID) { 
		if( lineDelta > 2048 ) {
			FatalError("virtual display width requires a line size of more than 2048 bytes\n");
		}
	} else {
		if( lineDelta > 4096 ) {
			FatalError("virtual display width requires a line size of more than 4096 bytes\n");
		}
	}

	/* CONVERT FROM BYTES TO PIXELS */
	pitch = lineDelta / (vgaBitsPerPixel >> 3);

	/* DISPLAY MESSAGE IF PITCH WAS CHANGED */

	if (pitch != vga256InfoRec.virtualX)
		DEBUGMSG(1, ("%s %s: Display pitch set to %d %d pixels.\n",
		       XCONFIG_PROBED, vga256InfoRec.name, pitch,
			vga256InfoRec.virtualX));
	return pitch;
}

/*---------------------------------------------------------------------------
 * GeodeEnterLeave
 * 
 * This routine is called before and after accessing VGA registers.
 *---------------------------------------------------------------------------
 */
static void GeodeEnterLeave(enter)
Bool enter;
{
	if (enter) xf86EnableIOPorts(vga256InfoRec.scrnIndex);
	else xf86DisableIOPorts(vga256InfoRec.scrnIndex);
}

#define READ_REG_COUNT 0x78

static void print_gxm_reg(unsigned int count)
{
#if 0
	int i, base = 0x8300;
	for(i=0; i<count; i+=4){
		DEBUGMSG(1, ("%x = %08x \n", (base + i), READ_REG32(base + i)));
	}
#endif
}

static void print_crtc(crtc)
vgaGeodePtr crtc;
{
	int i;
	DEBUGMSG(1, ("gfx_print_crtc: %d %d %d %d\n", 
		crtc->durangovga.xsize,
		crtc->durangovga.ysize,
		crtc->durangovga.hz,
		crtc->BPP));

	DEBUGMSG(1, ("%x\n", crtc->durangovga.miscOutput));

	for(i=0; i<25; i++){
		DEBUGMSG(1, ("%x ", crtc->durangovga.stdCRTCregs[i]));
	}
	DEBUGMSG(1, ("\n"));
	for(i=0; i<16; i++){
		DEBUGMSG(1, ("%x ", crtc->durangovga.extCRTCregs[i]));
	}
	DEBUGMSG(1, ("\n"));
}

static unsigned long GraphicsModeSaved = 0;
/*---------------------------------------------------------------------------
 * GeodeRestore
 * 
 * This routine restores (or sets for the first time) a display mode 
 * using the VGA registers.
 *---------------------------------------------------------------------------
 */
static void GeodeRestore(restore)
vgaGeodePtr restore;
{
	unsigned long value;
	int graphicsMode;
	unsigned short bpp;
  
	DEBUGMSG(0, ("%s %s: GeodeRestore\n", 
		XCONFIG_PROBED, vga256InfoRec.name));

	Gal_is_display_mode_supported( restore->durangovga.xsize,
		restore->durangovga.ysize,
		restore->BPP,
		restore->durangovga.hz, &graphicsMode);

#if 0
	print_crtc(restore);
#endif

	/* text mode and comming from graphics mode */
	if((graphicsMode == -1) || (GraphicsModeSaved == 0)){
			Gal_set_softvga_state(TRUE);

#if 0
		print_gxm_reg(READ_REG_COUNT); 
#endif

		if(graphicsMode == -1)
			DEBUGMSG(0, ("GeodeRestore text mode\n"));
		else
			DEBUGMSG(0, ("GeodeRestore text - graphics mode\n"));

#if COMPRESSION
		/* reset the compression buffer */
		Gal_set_compression_state(GAL_COMPRESSION_DISABLE);
#endif

		/* RESET START ADDRESS BACK TO ZERO */
		/* SoftVGA never knows it changed if panning the desktop. */
		Gal_set_display_offset(0);
		Gal_vga_mode_switch(1);

		/* CLEAR EXTENDED VGA REGISTERS */
		/* Need to do before restoring standard VGA state so that the text */
		/* mode really gets set properly when doing the font restore. */

		Gal_vga_clear_extended();

		/* RESTORE STANDARD VGA PORTION */
		vgaHWRestore((vgaHWPtr)restore);

		/* RESTORE EXTENDED VGA PORTION */
		/* Also restore standard CRTC values since we're overwriting values. */
		Gal_vga_restore(&restore->durangovga,
                    VGA_FLAG_MISC_OUTPUT | 
                    VGA_FLAG_STD_CRTC | 
                    VGA_FLAG_EXT_CRTC);

		/* SIGNAL THE END OF A MODE SWITCH */
		Gal_vga_mode_switch(0);

#ifdef DEBUG
		print_gxm_reg(READ_REG_COUNT);
#endif

		/* put no graphics mode */
		if(graphicsMode == -1)
			GraphicsModeSaved = 0;
	}

	/* TELL GRAPHICS UNIT THE BITS PER PIXEL */
	/* Use the BPP from display controller that was just set by SoftVGA. */
	/* Need to do this, instead of using vgaBitsPerPixel, so that it is */
	/* properly set to 8BPP when returning to text mode from 16 BPP. */

	Gal_get_display_bpp(&bpp);
	Gal_set_display_bpp(bpp);

	/* graphics mode */
	if(graphicsMode != -1){
		int i, index=0;

		DEBUGMSG(0, ("GeodeRestore graphics mode\n"));
		DEBUGMSG(0, ("gfx_set_display_mode %d\n", GraphicsModeSaved));

		for(i=0; i<(sizeof(GeodeMemOffset)/sizeof(MemOffset)) ; i++){
			if((vga256InfoRec.virtualX == GeodeMemOffset[i].xres) &&
			(vga256InfoRec.virtualY == GeodeMemOffset[i].yres) &&
			(restore->BPP == GeodeMemOffset[i].bpp)) {
				index = i;
				break;
			}
		}
		if(GraphicsModeSaved == 0){
			/* Disable SoftVGA, as we talk to H/W register
			to set the mode directly
			*/
			Gal_set_softvga_state(FALSE);
			GraphicsModeSaved=1;
		}
		Gal_set_display_bpp(restore->BPP);
		Gal_set_display_mode( restore->durangovga.xsize,
				restore->durangovga.ysize,
				restore->BPP,
				restore->durangovga.hz);
		/* adjust the pitch */
		Gal_set_display_pitch(GeodeCalculatePitchBytes());
		Gal_set_cursor_position(geodeCursorOffset, 0, 0, 0, 0);
		Gal_set_display_offset(0);
#if COMPRESSION
		Gal_set_compression_parameters(GAL_COMPRESSION_ALL,
			GeodeMemOffset[index].CBOffset,
			GeodeMemOffset[index].CBPitch,
			GeodeMemOffset[index].CBSize);
		Gal_set_compression_state(GAL_COMPRESSION_ENABLE); 
#endif

#ifdef DEBUG
		print_gxm_reg(READ_REG_COUNT);
#endif

	}
}

/*---------------------------------------------------------------------------
 * GeodeSave
 * 
 * This routine saves the current VGA register settings.
 *---------------------------------------------------------------------------
 */
static void *GeodeSave(save)
vgaGeodePtr save;
{
	GALAPP_VGAREGS vga;
	int bpp, is_graphics;
	
	/* SAVE EXTENDED VGA REGISTERS */
	/* Also save standard CRTC and misc output since we overwrite */
	/* those values. */

	Gal_vga_save(&vga,
               VGA_FLAG_MISC_OUTPUT | 
               VGA_FLAG_STD_CRTC | 
               VGA_FLAG_EXT_CRTC);

	Gal_get_display_mode(&vga.xsize, &vga.ysize, &bpp, &vga.hz);

	DEBUGMSG(0, ("GeodeSave %d %d %d %d\n", vga.xsize, vga.ysize, bpp, vga.hz));
	/* SAVE STANDARD VGA PORTION */

	save = (vgaGeodePtr)vgaHWSave((vgaHWPtr)save, sizeof(vgaGeodeRec));
	save->durangovga = vga;
	save->BPP = bpp;

#ifdef DEBUG
	print_crtc(save);
#endif

	return ((void *) save);
}

static unsigned long fixedformat_frequency(frequency)
int frequency;
{
	int clock = frequency / 1000;
	unsigned long pll;
	pll = (clock << 16) | ((int)
		((double)(frequency - (clock * 1000)) * (double)65.536));
	DEBUGMSG(0, ("fixedformat_frequency: %d - %d %d %08x\n",
		frequency,
		clock,
		((int)((double)(frequency - (clock * 1000)) * (double)65.536)),
		pll));

	return(pll);
}

/*---------------------------------------------------------------------------
 * GeodeInit
 * 
 * This routine initializes the VGA structure for the specified mode. 
 * It only sets the appropriate register values in the structure.  It 
 * does not load the hardware (that is done in GeodeRestore).
 *---------------------------------------------------------------------------
 */
static Bool GeodeInit(mode)
DisplayModePtr mode;
{
	int frequency = vga256InfoRec.clock[mode->Clock];
	int clock = frequency / 1000;
	int i, hz = 60;
	unsigned long pll;

	/* INITIALIZE STANDARD VGA PORTION */

	if (!vgaHWInit(mode,sizeof(vgaGeodeRec)))
		return(FALSE);

	pll = fixedformat_frequency(frequency);

	DEBUGMSG(1, ("GeodeInit: %dx%d %d @%d=", 
		mode->CrtcHDisplay,
		mode->CrtcVDisplay, 
		vgaBitsPerPixel, frequency));

	Gal_get_refreshrate_from_dotclock(mode->CrtcHDisplay,
		mode->CrtcVDisplay, 
		vgaBitsPerPixel, &hz, pll);

	DEBUGMSG(1, ("%dHz\n", hz));

	/* INITIALIZE DURANGO VGA STRUCTURE FOR THE MODE */
	/* This does not set the hardware.  That is done later in "restore". */

	/* set the VGA reg's with default 60 Hz and later the 
	* correct freq set by durango 
	*/
	Gal_vga_mode(&newstate->durangovga, 
               mode->CrtcHDisplay, 
               mode->CrtcVDisplay, vgaBitsPerPixel, 60);

	/* UPDATE THE PITCH VALUE */
	/* This is in case no acceleration is used and it is continuous. */

	Gal_vga_pitch(&newstate->durangovga, 
                GeodeCalculatePitchBytes());

	newstate->durangovga.xsize = mode->CrtcHDisplay;
	newstate->durangovga.ysize = mode->CrtcVDisplay;
	newstate->durangovga.hz = hz;
	newstate->BPP = vgaBitsPerPixel;

	return(TRUE);
}

/*---------------------------------------------------------------------------
 * GeodeAdjust
 * 
 * This function is called when the server needs to change the start
 * address of the frame buffer.  This is done when panning across a 
 * virtual desktop that is larger than the displayed frame buffer.
 *---------------------------------------------------------------------------
 */
static void GeodeAdjust(x, y)
int x, y;
{	
	unsigned long offset;
  unsigned short pitch;
  
	/* CALCULATE BYTE OFFSET FROM (X,Y) COORDINATE */
  Gal_get_display_pitch(&pitch);
	offset = (unsigned long) y * (unsigned long) pitch;
	offset += x;
	if (vgaBitsPerPixel > 8) offset += x;

	/* CALL DURANGO ROUTINES TO SET THE START ADDRESS */
	/* Waiting until vertical blank prevents some visual anomalies */
	/* such as "jumping" and "tearing" when panning. */

	Gal_wait_vertical_blank();
	Gal_set_display_offset(offset);
}


/* Calculate the refresh rate for the given mode */
static int GeodeGetRefreshRate(pMode)
DisplayModePtr pMode;
{
#define THRESHOLD 2
	int i;
	static int validRates[] = { 60, 70, 72, 75, 85 }; /* Hz */
	unsigned long dotClock;
	int refreshRate;
	int selectedRate;
	int diff;
	dotClock = pMode->Clock * 1000;

	refreshRate = dotClock / (pMode->CrtcHTotal * pMode->CrtcVTotal);

	selectedRate = refreshRate;
	for(i = 0; i < (sizeof(validRates)/sizeof(validRates[0])); i++)
	{
		/* take the difference in refresh rates calculated */
		diff = refreshRate - validRates[i];
		/* make it absolute */
		if(diff < 0) diff = -diff;
		if(diff <= THRESHOLD)
		{
			selectedRate = validRates[i];
		}
	}

	DEBUGMSG(0, ("GeodeGetRefreshRate = %ld/(%dx%d)  %dHz = %dHz\n",
		dotClock, pMode->CrtcHTotal, pMode->CrtcVTotal, 
		refreshRate, selectedRate));

	return selectedRate;
}

/* Set a graphics mode */
/*---------------------------------------------------------------------------
 * GeodeValidMode()
 * 
 * From README file: "The ValidMode() function is required.  It is used to 
 * check for any chipset dependent reasons why a graphics mode might not be
 * valid.  It gets called by higher levels of the code after the Probe()
 * stage.  In many cases no special checking will be required and this 
 * function will simply return TRUE always."
 *---------------------------------------------------------------------------
 */
static int GeodeValidMode(mode, verbose, flag)
DisplayModePtr mode;
Bool verbose;
int flag;
{
	int ret, hz=60;
#if 0
	/* curently the Pixel freq. in Linux do not comply with VESA, 
	* the code below is commented for this purpose. When we have a 
	* better king of approach then this can be either deleted/re-written
	*/
	unsigned long pll = fixedformat_frequency(mode->Clock);
	DEBUGMSG(0, ("GeodeValidMode: %s %d %d %d %d %X", 
		mode->name,
		mode->CrtcHDisplay,
		mode->CrtcVDisplay, 
		vgaBitsPerPixel,
		mode->Clock,
		pll));

	/* check if the frequency provided can be supported by durango */
	hz = gfx_mode_frequency_supported(
		mode->CrtcHDisplay, mode->CrtcVDisplay, 
		vgaBitsPerPixel,
		pll);

	DEBUGMSG(1, (" %d \n",  hz));

	if(hz == -1)
	{
		DEBUGMSG(1, ("%s %s: %s: Mode Not Supported (%dx%dx%d @ %x)\n",
			XCONFIG_PROBED, vga256InfoRec.name,
			vga256InfoRec.chipset,
			mode->CrtcHDisplay, mode->CrtcVDisplay, 
			vgaBitsPerPixel,
			pll));
	    return MODE_BAD;
	}
#endif
	
	hz = GeodeGetRefreshRate(mode);

	DEBUGMSG(0, ("%s %s: %s: Mode Details (%dx%dx%d @ %d)",
		XCONFIG_PROBED, vga256InfoRec.name,
		vga256InfoRec.chipset,
		mode->CrtcHDisplay, mode->CrtcVDisplay, 
		vgaBitsPerPixel,
		hz));

	/* can our h/w really support this */
	Gal_is_display_mode_supported( 
		mode->CrtcHDisplay, mode->CrtcVDisplay, 
		vgaBitsPerPixel, hz, &ret);

	DEBUGMSG(0, (" ret = %d\n", ret));

	if(ret == -1)
	{
		DEBUGMSG(1, ("%s %s: %s: Mode Not Supported By Processor (%dx%dx%d @ %d)\n",
			XCONFIG_PROBED, vga256InfoRec.name,
			vga256InfoRec.chipset,
			mode->CrtcHDisplay, mode->CrtcVDisplay, 
			vgaBitsPerPixel, hz));
	    return MODE_BAD;
	}

	return MODE_OK;
}

/* END OF FILE */
