//Copyright 2014,2015 MCbx, All rights reserved.
//http://mcbx.netne.net/ictester
//This file is part of ICTester.
//ICTester is free software; you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation; either version 2 of the License, or
//(at your option) any later version.
//ICTester is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//You should have received a copy of the GNU General Public License
//along with ICTester; if not, write to the Free Software Foundation,
//Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

#include "romdumper.h"
#include "ui_romdumper.h"
#include <QString>
#include <QByteArray>
#include <QFile>
#include <QDebug>
#include <QFont>
#include <QFileDialog>
#include <QMessageBox>
#include "../devicedriver.h"
#include "powervisualizer.h"
#include "../powertable.h"
#include <QTemporaryFile>
#include <QFileInfo>
#include <QProcess>

QByteArray buffer;
bool save;
QString execFile;

ROMDumper::ROMDumper(QString port, QString pLUTPath, QWidget *parent) :
    QDialog(parent),
    ui(new Ui::ROMDumper)
{
    ui->setupUi(this);
    ROMDumper::setWindowTitle("(E/P)ROM reader");
    QFont font("Monospace");
    font.setStyleHint(QFont::TypeWriter);
    ui->pBufferView->setFont(font);
    ui->pBufferView->setReadOnly(1);
    on_cbType_currentIndexChanged(ui->cbType->currentText());
    this->port=port;
    this->pLUTPath=pLUTPath;
    save=0;

    QFileInfo getDirFromPath(QApplication::applicationFilePath());
    execFile=getDirFromPath.path()+"/icromdump";
    if (!QFile(execFile).exists())
    {
       execFile="icromdump";
       QProcess proc;
       proc.start(execFile);
       if (!proc.waitForFinished())
       {
           QMessageBox::critical(this,"Cannot continue","Put icromdump executable to program's directory or system path directory.");
           ui->btnRead->setEnabled(0);
           ui->btnVerify->setEnabled(0);
       }
    }
}

ROMDumper::~ROMDumper()
{
    delete ui;
}

void ROMDumper::repaintBuffer()
{
    ui->pBufferView->clear();
    QString linia;
    ui->pBufferView->appendPlainText("       00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F");
    for (int i=0;i<buffer.size();i=i+16)
    {
        linia=QString::number(i,16).rightJustified(5,'0');
        linia+=":";
        for (int k=0;k<16;k++)
        {
            linia=linia+" "+QString::number((unsigned char)buffer[k+i],16).rightJustified(2,'0');
        }
        linia+=" | ";
        for (int k=0;k<16;k++)
        {
            if ((buffer[k+i]<char(32))||(buffer[k+i]>char(126)))
                linia=linia+".";
            else
                linia=linia+buffer[k+i];
        }

        linia+"\n";
        ui->pBufferView->appendPlainText(linia);
    }
    ui->pBufferView->moveCursor(QTextCursor::Start);
}

void ROMDumper::on_cbType_currentIndexChanged(const QString &arg1)
{
    QStringList parts=arg1.split(" ");
  //  parm="."+parts[0];
    parts=parts[1].split("x");
    buffer.resize(parts[0].right(parts[0].length()-1).toInt());
    buffer.fill(0,parts[0].right(parts[0].length()-1).toInt());
    ui->pbProgress->setMaximum(parts[0].right(parts[0].length()-1).toInt());
    repaintBuffer();
    save=0;
}

void ROMDumper::on_btnLoad_clicked()
{
    QString fileName=QFileDialog::getOpenFileName(this,"Open file","","BIN(*.BIN);;ROM (*.ROM);;All Files(*.*)");
    if (fileName=="") return;
    QFile file(fileName);
    if (!file.open(QIODevice::ReadOnly)) return;
    QByteArray bytes = file.readAll();
    file.close();
    int max=buffer.size();
    if (bytes.length()<max) max=bytes.length();
    for (int i=0;i<max;i++)
    {
        buffer[i]=bytes[i];
    }
    repaintBuffer();
    save=1;
}

void ROMDumper::on_btnSave_clicked()
{
    QString fileName=QFileDialog::getSaveFileName(this,"Save file","","BIN(*.BIN);;ROM (*.ROM);;All Files(*.*)");
    if (fileName=="") return;
    QFile file(fileName);
    if (!file.open(QIODevice::WriteOnly)) return;
    file.write(buffer);
    file.close();
    save=0;
}

void ROMDumper::on_btnClear_clicked()
{
    buffer.fill(0,buffer.length());
    repaintBuffer();
    save=0;
}

void ROMDumper::on_btnExit_clicked()
{
    if (save)
    {
        QMessageBox::StandardButton reply;
         reply = QMessageBox::question(this, "Close window?", "File not saved. Close window?",
                                       QMessageBox::Yes|QMessageBox::No);
         if (reply == QMessageBox::No) return;
    }
    this->close();
}

bool ROMDumper::powerDisplayer()
{
    PowerTable pLUT(this->pLUTPath);

    //get power and gnd pin no:
    QStringList aa=ui->cbType->currentText().split(" ");
    aa=aa[2].split("/");
    int gnd=aa[0].toInt();
    int vcc=aa[1].toInt();

    //evaluate PLUT
    QString DIPPos="";
    for (int i=0;i<pLUT.getCount();i++) DIPPos+="0";
    QString VccPos="000000000000000000000000";
    QString GNDPos="000000000000000000000000";
    for (int i=0;i<25;i++)
    {
        if (i==gnd)
        {
            bool foundInDIP=0;
            //GND found on pin i+1
            for (int j=0;j<pLUT.getCount();j++) //check switches
            {
                if ((pLUT.getEntryFunc(j)==0)&&(pLUT.getEntryPin(j)==gnd))
                {
                    DIPPos[j]='1';
                    foundInDIP=1;
                }
            }
            //if foundInDip==0 then install the proper link in GNDPos
            if (!foundInDIP)
            {
                GNDPos[i]='1';
            }
        }
        if (i==vcc)
        {
            bool foundInDIP=0;
            //Vcc found on pin i+1
            for (int j=0;j<pLUT.getCount();j++) //check switches
            {
                if ((pLUT.getEntryFunc(j)==1)&&(pLUT.getEntryPin(j)==vcc))
                {
                    DIPPos[j]='1';
                    foundInDIP=1;
                }

            }                //if foundInDIP==0 then install the proper link in VccPos
            if (!foundInDIP)
            {
                VccPos[i]='1';
            }
        }
    }
    PowerVisualizer pvi(DIPPos+","+VccPos+","+GNDPos, aa[2].mid(0,aa[2].length()-1).toInt(),this);
    return (pvi.exec());
}


void ROMDumper::onFinished(int e, QProcess::ExitStatus es)
{
    if (e!=0)
    {
        QString a="Code: "+QString::number(e);
        if (e==1) a="Cannot connect to device.";
        if (e==2) a="Wrong arguments (Unsupported type?).";
        if (e==3) a="Nothing to save (result length is zero).";
       // QMessageBox::critical(this,"Error","There was an error running program: "+a);
        ui->lbStatus->setText("ERROR: "+a);
        ui->lbStatus->setStyleSheet("QLabel {color : red; }");
    }
    else
    {
        ui->lbStatus->setStyleSheet("");
        ui->lbStatus->setText("Dump finished");
    }
}

void ROMDumper::controls(bool state)
{
    ui->btnRead->setEnabled(state);
    ui->btnVerify->setEnabled(state);
    ui->btnClear->setEnabled(state);
    ui->btnLoad->setEnabled(state);
    ui->btnSave->setEnabled(state);
    ui->cbType->setEnabled(state);
    ui->btnExit->setEnabled(state);
}

void ROMDumper::on_btnRead_clicked()
{
    if (!powerDisplayer()) return;

    ui->lbStatus->setStyleSheet("");
    ui->lbStatus->setText("Preparing...");

    QStringList paramStr;
    paramStr.append(this->port);
    //temp file
    QTemporaryFile tempDump;
    tempDump.open();
    QString tempFile=tempDump.fileName(); //reserve file only
    tempDump.close();

    paramStr.append(tempFile);
    paramStr.append("."+ui->cbType->currentText().split(" ")[0]);

    controls(0);

    this->process = new QProcess(this);
   // this->process->setReadChannel();
    connect( this->process, SIGNAL(finished(int,QProcess::ExitStatus)), SLOT(onFinished(int,QProcess::ExitStatus)) );

    this->process->start(execFile, paramStr,  QIODevice::ReadWrite | QIODevice::Text);
    if (!this->process->waitForStarted())
    {
        ui->lbStatus->setText("ERROR: Error running program.");
        ui->lbStatus->setStyleSheet("QLabel {color : red; }");
        return;
    }

    ui->lbStatus->setStyleSheet("");
    ui->lbStatus->setText("Reading chip...");

    //Bad practice - GUI waits for text mode application to finish refreshing progress bar.
    while (ui->lbStatus->text()=="Reading chip...")
    {
        QString mesg=QString(process->readAllStandardError());
        if (mesg.trimmed()!="")
        {
            int k=mesg.trimmed().toInt();
            ui->pbProgress->setValue(k);
       }
        QApplication::processEvents();
    }

    QFile file(tempFile);
    if (!file.open(QIODevice::ReadOnly))
    {
        ui->lbStatus->setText("ERROR: Result cannot be read to buffer.");
        ui->lbStatus->setStyleSheet("QLabel {color : red; }");
        return;
    }
    QByteArray bytes = file.readAll();
    file.close();
    int max=buffer.size();
    if (bytes.length()<max) max=bytes.length();
    for (int i=0;i<max;i++)
    {
        buffer[i]=bytes[i];
    }
     save=1;
     controls(1);
     repaintBuffer();
}

void ROMDumper::on_btnVerify_clicked()
{
    if (!powerDisplayer()) return;

    ui->lbStatus->setStyleSheet("");
    ui->lbStatus->setText("Preparing...");

    QStringList paramStr;
    paramStr.append(this->port);
    //temp file
    QTemporaryFile tempDump;
    tempDump.open();
    QString tempFile=tempDump.fileName(); //reserve file only
    tempDump.close();

    paramStr.append(tempFile);
    paramStr.append("."+ui->cbType->currentText().split(" ")[0]);

    controls(0);

    this->process = new QProcess(this);
   // this->process->setReadChannel();
    connect( this->process, SIGNAL(finished(int,QProcess::ExitStatus)), SLOT(onFinished(int,QProcess::ExitStatus)) );

    this->process->start(execFile, paramStr,  QIODevice::ReadWrite | QIODevice::Text);
    if (!this->process->waitForStarted())
    {
        ui->lbStatus->setText("ERROR: Error running program.");
        ui->lbStatus->setStyleSheet("QLabel {color : red; }");
        return;
    }

    ui->lbStatus->setStyleSheet("");
    ui->lbStatus->setText("Reading chip...");

    //Bad practice - GUI waits for text mode application to finish refreshing progress bar.
    while (ui->lbStatus->text()=="Reading chip...")
    {
        QString mesg=QString(process->readAllStandardError());
        if (mesg.trimmed()!="")
        {
            int k=mesg.trimmed().toInt();
            ui->pbProgress->setValue(k);
       }
        QApplication::processEvents();
    }

    QFile file(tempFile);
    if (!file.open(QIODevice::ReadOnly))
    {
        ui->lbStatus->setText("ERROR: Result cannot be read to compare.");
        ui->lbStatus->setStyleSheet("QLabel {color : red; }");
        return;
    }
    ui->lbStatus->setStyleSheet("");
    ui->lbStatus->setText("Comparing...");
    QApplication::processEvents();

    QByteArray bytes = file.readAll();
    file.close();
    int max=buffer.size();
    if (bytes.length()<max) max=bytes.length();
    for (int i=0;i<max;i++)
    {
        if (buffer[i]!=bytes[i])
        {
            ui->lbStatus->setText("Compare error at 0x"+QString::number(i,16).rightJustified(4,'0')+" - Buffer: 0x"+
                                  QString::number((unsigned char)buffer[i],16).rightJustified(2,'0')+
                                  " ("+QString::number((unsigned char)buffer[i],2).rightJustified(8,'0')+"), "+
                                  +" Chip: 0x"+
                                  QString::number((unsigned char)bytes[i],16).rightJustified(2,'0')+
                                  " ("+QString::number((unsigned char)bytes[i],2).rightJustified(8,'0')+").");
            ui->lbStatus->setStyleSheet("QLabel {color : red; }");
            controls(1);
            return;
        }
    }
     ui->lbStatus->setStyleSheet("");
     ui->lbStatus->setText("Compare OK.");
     controls(1);
}
