/*
 * target_STM32F4.c      STM32F4 Discovery board support routines for TinyBasicLike
 * 
 * This file contains all the target-specific routines to bring up TinyBasicLike on a
 * stock STM32F4 Discovery board (STM32F407).
 *
 * 14 Aug 2-23
 * Added four channels of ADC (ADC0 through ADC3).
 */
 
#include  "stdio.h"
#include  "target_STM32F4.h"
#include  "stm32f4xx_flash.h"
#include  "TinyBasicLike.h"


#define  STM32F4DISC_VERSION_INFO		"v0.01"

const unsigned char					t_TargetIDMsg[] =
		"\nSTM32F4 Discovery TinyBasicLike target (32-bit) " STM32F4DISC_VERSION_INFO "  20 Jul 2023  KEL";


		
#define USER_PB_PIN_NUM						0
#define USER_PB_PIN_MASK                	(1<<USER_PB_PIN_NUM)
#define USER_PB_PORT						GPIOA
#define USER_PB_CLK							RCC_AHB1Periph_GPIOA

#define TONE_PIN_NUM						1
#define TONE_PIN_MASK						(1<<TONE_PIN_NUM)
#define TONE_PORT							GPIOA
#define TONE_CLK							RCC_AHB1Periph_GPIOA
#define TONE_TIMER_CLK						RCC_APB1Periph_TIM2		

#define ORG_LED_PIN_NUM						13
#define ORG_LED_PIN_MASK                	(1<<ORG_LED_PIN_NUM)
#define ORG_LED_PORT						GPIOD
#define ORG_LED_CLK							RCC_AHB1Periph_GPIOD

#define RED_LED_PIN_NUM						14
#define RED_LED_PIN_MASK                	(1<<RED_LED_PIN_NUM)
#define RED_LED_PORT						GPIOD
#define RED_LED_CLK							RCC_AHB1Periph_GPIOD

#define GREEN_LED_PIN_NUM					12
#define GREEN_LED_PIN_MASK                	(1<<GREEN_LED_PIN_NUM)
#define GREEN_LED_PORT						GPIOD
#define GREEN_LED_CLK						RCC_AHB1Periph_GPIOD

#define BLUE_LED_PIN_NUM					15
#define BLUE_LED_PIN_MASK                	(1<<BLUE_LED_PIN_NUM)
#define BLUE_LED_PORT						GPIOD
#define BLUE_LED_CLK						RCC_AHB1Periph_GPIOD

#define LEDS_GPIO_PORT  (GPIOD)

#define DOUT0_PIN_NUM						8
#define DOUT0_PIN_MASK                		(1<<DOUT0_PIN_NUM)
#define DOUT0_PORT							GPIOE
#define DOUT0_CLK							RCC_AHB1Periph_GPIOE

#define DOUT1_PIN_NUM						9
#define DOUT1_PIN_MASK                		(1<<DOUT1_PIN_NUM)
#define DOUT1_PORT							GPIOE
#define DOUT1_CLK							RCC_AHB1Periph_GPIOE

#define DOUT2_PIN_NUM						10
#define DOUT2_PIN_MASK                		(1<<DOUT2_PIN_NUM)
#define DOUT2_PORT							GPIOE
#define DOUT2_CLK							RCC_AHB1Periph_GPIOE

#define DOUT3_PIN_NUM						11
#define DOUT3_PIN_MASK                		(1<<DOUT3_PIN_NUM)
#define DOUT3_PORT							GPIOE
#define DOUT3_CLK							RCC_AHB1Periph_GPIOE

#define DOUT_GPIO_PORT  (GPIOE)

#define DIN0_PIN_NUM						12
#define DIN0_PIN_MASK                		(1<<DIN0_PIN_NUM)
#define DIN0_PORT							GPIOE
#define DIN0_CLK							RCC_AHB1Periph_GPIOE

#define DIN1_PIN_NUM						12
#define DIN1_PIN_MASK                		(1<<DIN1_PIN_NUM)
#define DIN1_PORT							GPIOE
#define DIN1_CLK							RCC_AHB1Periph_GPIOE

#define DIN2_PIN_NUM						12
#define DIN2_PIN_MASK                		(1<<DIN2_PIN_NUM)
#define DIN2_PORT							GPIOE
#define DIN2_CLK							RCC_AHB1Periph_GPIOE

#define DIN3_PIN_NUM						12
#define DIN3_PIN_MASK                		(1<<DIN3_PIN_NUM)
#define DIN3_PORT							GPIOE
#define DIN3_CLK							RCC_AHB1Periph_GPIOE

#define DIN_GPIO_PORT  (GPIOE)

#define ADC0_PIN_NUM						4
#define ADC0_PIN_MASK						(1<<ADC0_PIN_NUM)
#define ADC0_PORT							GPIOA
#define ADC0_CLK							RCC_APB2Periph_ADC1
#define ADC0_CHNL							ADC_Channel_4

#define ADC1_PIN_NUM						5
#define ADC1_PIN_MASK						(1<<ADC1_PIN_NUM)
#define ADC1_PORT							GPIOA
#define ADC1_CLK							RCC_APB2Periph_ADC1
#define ADC1_CHNL							ADC_Channel_5

#define ADC2_PIN_NUM						6
#define ADC2_PIN_MASK						(1<<ADC2_PIN_NUM)
#define ADC2_PORT							GPIOA
#define ADC2_CLK							RCC_APB2Periph_ADC1
#define ADC2_CHNL							ADC_Channel_6

#define ADC3_PIN_NUM						7
#define ADC3_PIN_MASK						(1<<ADC3_PIN_NUM)
#define ADC3_PORT							GPIOA
#define ADC3_CLK							RCC_APB2Periph_ADC1
#define ADC3_CHNL							ADC_Channel_7

#define ADC_GPIO_PORT  (GPIOA)
#define ADC_MODULE		ADC1


struct  flashconfig {
	unsigned char					sector;
	DATA_SIZE						startaddr;
};



/*
 * This code reserves flash sector 4 (64K) for use as a flash file system.
 * The area is divided into fixed-sized blocks, each of which is a single
 * file.
 * 
 * Each file block starts with a header (including name); remaining bytes
 * in the block are for data.
 */
 
/*
 * Create a RAM buffer large enough to hold the entire flash file buffer
 */
uint8_t					flashbuffer[FLASH_SECTOR_STORAGE_SIZE];	// holding buffer for flash image

extern FILE					*rfp;			// use core's version of read file pointer
extern FILE					*wfp;			// use core's version of write file pointer

 

 
/*
 * Define the USART to use as the TBL console serial port.
 */
#define ACTIVE_USART			USART1
#define ACTIVE_USART_BAUD		9600


/*
 * Define the USART pins to use for the TBL console serial port (format is USART_TX_RX).
 * Uncomment only one of the following #defines; the associated USART pins will be used
 * as the console serial port.
 */
//#define ACTIVE_USART_PINS		USART1_PA9_PA10		/* doesn't work because of board H/W issues */
#define ACTIVE_USART_PINS		USART1_PB6_PB7		/* tested, works */
//#define ACTIVE_USART_PINS		USART2_PA2_PA3		/* tested, works */
//#define ACTIVE_USART_PINS		USART2_PD5_PD6		/* tested, works */
//#define ACTIVE_USART_PINS		USART3_PB10_PB11	/* tested, works */	
//#define ACTIVE_USART_PINS		USART3_PC10_PC11	/* tested, works */
//#define ACTIVE_USART_PINS		USART3_PD8_PD9		/* tested, works */
//#define ACTIVE_USART_PINS		UART4_PA0_PA1		/* tested, works (not defined in user manual!) */
//#define ACTIVE_USART_PINS		UART4_PC10_PC11		/* tested, works */
//#define ACTIVE_USART_PINS		UART5_PC12_PD2		/* tested, works */
//#define ACTIVE_USART_PINS		USART6_PC6_PC7		/* tested, works */

static GPIO_InitTypeDef             	GPIO_InitStructure;

static ADC_InitTypeDef					ADC_InitStructure;
static ADC_CommonInitTypeDef			ADC_CommonInitStructure;

static TIM_TimeBaseInitTypeDef			TIM_BaseStruct;
static TIM_OCInitTypeDef				TIM_OCStruct;


unsigned char	t_program[TOTAL_RAM_REQUIRED];  // define RAM block holding entire program, stacks, and vars

char		*rfileptr;
char		*rfilestart;
char		*rfiledatastart;
char		*rfiledataend;

char		*wfileptr;
char		*wfilestart;
char		*wfiledatastart;
char		*wfiledataend;

int					filebufferchanged;


/*
 * Core variables that the target can access (usually only for debug)
 */
extern unsigned char *program_start;			// address of program start
extern unsigned char *program_end;				// address of program end



struct  Port
{
	DATA_SIZE				*portaddr;					// addr of the port
	unsigned int			size;						// size of data cell (bits)
	unsigned char			writeable;					// TRUE if port can be modified
};

#define  WRITEABLE			TRUE
#define  READONLY			FALSE



int							OutputStream;
int							InputStream;

DATA_SIZE					*timeraddrs[T_NUM_TIMERS] = {0, 0, 0, 0};


/*
 * Defines relevant to setting up the timer for tone generation
 */
#define  TIMER_PRESCALER			200
#define  TIMER_CLK_RATE				(168000000 / 2 / TIMER_PRESCALER)


/*
 * Define internal timers dedicated to the hardware and not exposed to the core.
 */
#define  NUM_HWTIMERS			1
#define  HWTIMER_TONE			0			/* hardware timer dedicated to tones */
DATA_SIZE					hwtimers[NUM_HWTIMERS] = {0};




static GPIO_InitTypeDef             GPIO_InitStructure;
static USART_InitTypeDef            USART_InitStructure;


DATA_SIZE							SystemCounter_Msecs;


/*
 * Define a range of virtual ports.  These appear as addresses in the port table, but
 * actually identify individual I/O lines or dedicated variables.
 * 
 * For example, a virtual port might be bit 13 of a 16-bit output port.  Writes to this
 * virtual port would manipulate only bit 13 of the corresponding port.
 */
 enum  VirtualPort
 {
	 RED_LED = 0,
	 ORANGE_LED,
	 BLUE_LED,
	 GREEN_LED,
	 SYSTEM_TIME_MSECS,
	 DOUT0,
	 DOUT1,
	 DOUT2,
	 DOUT3,
	 DIN0,
	 DIN1,
	 DIN2,
	 DIN3,
	 VP_ADC0,
	 VP_ADC1,
	 VP_ADC2,
	 VP_ADC3
 };

 
/*
 * Define a table showing the names of all supported ports.  Structure of this table
 * matches those of the similar tables in the core program.
 * 
 * NOTE: The strings for each port name are NOT followed by a comma!  This is a single,
 * large string of space-separated names, it is not an array of strings.
 * 
 * For example:
 *	const unsigned char			ports_tab[] =
 *  {
 *     "PORTA "					// <-- see, no comma!
 *     "PORTB "
 *     "PORTC "
 *     "\0"
 *   }
 */
const unsigned char			ports_tab[] =
{
	"RED_LED "
	"ORANGE_LED "
	"BLUE_LED "
	"GREEN_LED "

	"SYSTEM_TIME_MSECS "
	
	"DOUT0 "					// PE-8  (P1-26)
	"DOUT1 "					// PE-9  (P1-27)
	"DOUT2 "					// PE-10 (P1-28)
	"DOUT3 "					// PE-11 (P1-29)
	
	"DIN0 "						// PE-12 (P1-30)
	"DIN1 "						// PE-13 (P1-31)
	"DIN2 "						// PE-14 (P1-32)
	"DIN3 "						// PE-15 (P1-33)
	
	"ADC0 "						// PA-4 (P1-16)
	"ADC1 "						// PA-5 (P1-15)
	"ADC2 "						// PA-6 (P1-18)
	"ADC3 "						// PA-7 (P1-17)
	
	"GPIOA_ORESET "
	"GPIOA_OSET "
	"GPIOA_MODE "
	"GPIOA_OTYPE "
	"GPIOA_OSPEED "
	"GPIOA_PUPD "
	"GPIOA_IDATA "
	"GPIOA_ODATA "
	"GPIOA_AFRL "
	"GPIOA_AFRH "

	"GPIOD_ORESET "
	"GPIOD_OSET "
	"GPIOD_MODE "
	"GPIOD_OTYPE "
	"GPIOD_OSPEED "
	"GPIOD_PUPD "
	"GPIOD_IDATA "
	"GPIOD_ODATA "
	"GPIOD_AFRL "
	"GPIOD_AFRH "
	"\0"
};


/*
 * This is a table containing one entry for each supported port.  Each entry
 * contains the address of the corresponding port register and the size (in bits)
 * of the port.
 * 
 * Order is IMPORTANT!  The entries in this table must match exactly the order
 * of the port names in ports_tab[] above.
 */
struct Port					porttable[] =
{
	{(DATA_SIZE *) RED_LED, 		1, WRITEABLE},
	{(DATA_SIZE *) ORANGE_LED, 		1, WRITEABLE},
	{(DATA_SIZE *) BLUE_LED, 		1, WRITEABLE},
	{(DATA_SIZE *) GREEN_LED, 		1, WRITEABLE},
	
	{(DATA_SIZE *) &SystemCounter_Msecs, 32, READONLY},
	
	{(DATA_SIZE *) DOUT0,	 		1, WRITEABLE},
	{(DATA_SIZE *) DOUT1,	 		1, WRITEABLE},
	{(DATA_SIZE *) DOUT2,	 		1, WRITEABLE},
	{(DATA_SIZE *) DOUT3,	 		1, WRITEABLE},

	{(DATA_SIZE *) DIN0,	 		1, READONLY},
	{(DATA_SIZE *) DIN1,	 		1, READONLY},
	{(DATA_SIZE *) DIN2,	 		1, READONLY},
	{(DATA_SIZE *) DIN3,	 		1, READONLY},
	
	{(DATA_SIZE *) VP_ADC0,	 		12, READONLY},
	{(DATA_SIZE *) VP_ADC1,	 		12, READONLY},
	{(DATA_SIZE *) VP_ADC2,	 		12, READONLY},
	{(DATA_SIZE *) VP_ADC3,	 		12, READONLY},
	
	{(DATA_SIZE *) &(GPIOA->BSRRH), 16, WRITEABLE},			// GPIOA bit reset reg
	{(DATA_SIZE *) &(GPIOA->BSRRL), 16, WRITEABLE},			// GPIOA bit set reg
	{(DATA_SIZE *) &(GPIOA->MODER), 32, WRITEABLE},
	{(DATA_SIZE *) &(GPIOA->OTYPER), 16, WRITEABLE},
	{(DATA_SIZE *) &(GPIOA->OSPEEDR), 32, WRITEABLE},
	{(DATA_SIZE *) &(GPIOA->PUPDR), 32, WRITEABLE},
	{(DATA_SIZE *) &(GPIOA->IDR), 	16, WRITEABLE},
	{(DATA_SIZE *) &(GPIOA->ODR), 	16, WRITEABLE},
	{(DATA_SIZE *) &(GPIOA->AFR[0]), 32, WRITEABLE},
	{(DATA_SIZE *) &(GPIOA->AFR[1]), 32, WRITEABLE},

	{(DATA_SIZE *) &(GPIOD->BSRRH), 16, WRITEABLE},			// GPIOD bit reset reg
	{(DATA_SIZE *) &(GPIOD->BSRRL), 16, WRITEABLE},			// GPIOD bit set reg
	{(DATA_SIZE *) &(GPIOD->MODER), 32, WRITEABLE},
	{(DATA_SIZE *) &(GPIOD->OTYPER), 16, WRITEABLE},
	{(DATA_SIZE *) &(GPIOD->OSPEEDR), 32, WRITEABLE},
	{(DATA_SIZE *) &(GPIOD->PUPDR), 32, WRITEABLE},
	{(DATA_SIZE *) &(GPIOD->IDR), 	16, WRITEABLE},
	{(DATA_SIZE *) &(GPIOD->ODR), 	16, WRITEABLE},
	{(DATA_SIZE *) &(GPIOD->AFR[0]), 32, WRITEABLE},
	{(DATA_SIZE *) &(GPIOD->AFR[1]), 32, WRITEABLE},
};

#define  NUM_PORTS			(sizeof(porttable)/sizeof(porttable[0]))


/*
 * Flash write/erase support routines
 * 
 * These routines allow writing and erasing a specific flash sector.
 * The Flash_UnlockCmdReg() routine unlocks the flash Cmd register if
 * it is already locked.
 */
static void  Flash_UnlockCmdReg(void)
{
	if (FLASH->CR & FLASH_CR_LOCK)
	{
		FLASH->KEYR = 0x45670123;
		FLASH->KEYR = 0xcdef89ab;
	}
}

	

/*
 * Flash_EraseSector      erase the flash sector dedicated to program storage
 */
static void  Flash_EraseSector(void)
{
	Flash_UnlockCmdReg();

	while (FLASH->SR & FLASH_SR_BSY)  ;				// block until flash is not busy
	FLASH->CR = FLASH->CR & 0xfff8;					// clear MER, SER, and PG
	FLASH->CR = FLASH->CR | (2 << 8);				// PSIZE = 0b10 = x32 parallelism
	FLASH->CR = FLASH->CR | (FLASH_SECTOR_STORAGE << 3) | FLASH_CR_SER;
	FLASH->CR = FLASH->CR | FLASH_CR_STRT;			// start erasing selected sector
	while (FLASH->SR & FLASH_SR_BSY)  ;				// block until flash is not busy
	FLASH->CR = FLASH->CR & 0xfff8;					// turn off sector erase bit
};


static void  Flash_WriteSector(void)
{
	uint32_t					*rptr;
	uint32_t					*wptr;
	
	Flash_UnlockCmdReg();
	
	while (FLASH->SR & FLASH_SR_BSY)  ;				// block until flash is not busy
	FLASH->CR = FLASH->CR & 0xfff8;					// clear MER, SER, and PG
	FLASH->CR = FLASH->CR | (2 << 8);				// PSIZE = 0b10 = x32 parallelism
	FLASH->CR = FLASH->CR | (FLASH_SECTOR_STORAGE << 3);			// select sector
	FLASH->CR = FLASH->CR | FLASH_CR_PG;			// activate programming

	rptr = (uint32_t *)flashbuffer;
	wptr = (uint32_t *)FLASH_SECTOR_STORAGE_ADDR;
	
	while (rptr != ((uint32_t *)(flashbuffer + FLASH_SECTOR_STORAGE_SIZE)))
	{
		*wptr++ = *rptr++;
		while (FLASH->SR & FLASH_SR_BSY)  ;			// block until flash is not busy
	}
	FLASH->CR = FLASH->CR & 0xfff8;					// deactivate programming
}



/*
 * File support routines
 * 
 * These are NOT FAT_Fs routines!  These are simplified versions of file
 * support for using the on-chip flash and a large block of RAM as a file
 * system.
 */

/*
 * FindFileByName      search flash buffer for file name
 * 
 * If this routine finds the requested file, it writes the address of
 * the file block to argument ptr.  Note that this is the address of the
 * first char of the file header!  To access the first byte of file data,
 * you must adjust the address by adding the size of the file header,
 * SIZEOF_FLASH_FILE_HEADER.
 */
static int  FindFileByName(char  *name, char  **ptr)
{
	char			*fptr;
	
	for (int n = 0; n<NUM_FLASH_FILES; n++)
	{
		 fptr = (char *)(flashbuffer + (n*SIZEOF_FLASH_FILE));

		 for (int i=0; i<MAX_LEN_FILENAME+1; i++)
		 {
			 if (fptr[i] == name[i])
			 {
				 if (fptr[i] == 0)		// end of file name is marked with null byte
				 {
					 *ptr = fptr;
					 return  S_OK;
				 }
			 }
			 else  break;
		 }
	 }
	 *ptr = 0;
	 return  E_FILE_NOT_FOUND;
 }
 
 
 static int  FindNextOpenFile(char  **ptr)
 {
	 int			n;
	 
	 *ptr = (char *)flashbuffer;
	 for (n=0; n<NUM_FLASH_FILES; n++)
	 {
		 if (**ptr == 0xff)  return  S_OK;			// empty area is all FFs
		 *ptr = *ptr + SIZEOF_FLASH_FILE;
	 }
	 return  E_NO_ROOM_IN_FLASH;
 }
 
	 
 static char  *CreateNewFile(char  *name)
 {
	 char						*ptr;
	 int						i;
	 int						s;
	 
	 ptr = (char *)flashbuffer;
	 s = FindNextOpenFile(&ptr);
	 if (s != S_OK)  return  NULL;
	
	 i = 0;
	 while (i<MAX_LEN_FILENAME)
	 {
		 ptr[i] = name[i];
		 if (name[i] == '\0')  break;
		 i++;
	 }
	 if (i == MAX_LEN_FILENAME)
	 {
		 for (i=0; i<MAX_LEN_FILENAME; i++)		// didn't work, need to erase header
		 {
			 ptr[i] = 0xFF;						// empty area is all FFs
		 }
		 return  NULL;
	 }
	 return  ptr;
 }
 




 
	

static void  USART_Configure(USART_TypeDef  *USARTx, uint32_t  baud)
{
/*
 * Configure USARTx for 8N1, no flow control, baud rate supplied by calling routine.
 */
    USART_InitStructure.USART_BaudRate = baud;
    USART_InitStructure.USART_WordLength = USART_WordLength_8b;
    USART_InitStructure.USART_StopBits = USART_StopBits_1;
    USART_InitStructure.USART_Parity = USART_Parity_No;
    USART_InitStructure.USART_HardwareFlowControl = USART_HardwareFlowControl_None;
    USART_InitStructure.USART_Mode = USART_Mode_Rx | USART_Mode_Tx;
    USART_Init(USARTx, &USART_InitStructure);
}



static uint32_t  USART_Initialize(USART_TypeDef  *USARTx, enum usart_txrx_pins  pins, uint32_t  baud)
{
	if (USARTx == USART1)						// USART routed through microUSB connector
	{
/*
 * Force reinit of the RCC register, just in case it was previously in use.
 */
        RCC_APB2PeriphResetCmd(RCC_APB2Periph_USART1, ENABLE);
        RCC_APB2PeriphResetCmd(RCC_APB2Periph_USART1, DISABLE);

        RCC_APB2PeriphClockCmd(RCC_APB2Periph_USART1, ENABLE);
        if      (pins == USART1_PA9_PA10)  	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOA, ENABLE);
        else if (pins == USART1_PB6_PB7)  	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOB, ENABLE);
        else  return  -1;				// illegal or unknown pin assignment, fail  
        
/*
 * Configure USART1 GPIO pins.
 */
        GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
        GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
        GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
        GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
        if (pins == USART1_PA9_PA10)
        {
			GPIO_InitStructure.GPIO_Pin = GPIO_Pin_9 | GPIO_Pin_10;
			GPIO_Init(GPIOA, &GPIO_InitStructure);
		}
		else if (pins == USART1_PB6_PB7)
		{
			GPIO_InitStructure.GPIO_Pin = GPIO_Pin_6 | GPIO_Pin_7;
			GPIO_Init(GPIOB, &GPIO_InitStructure);
		}
		else  return  -1;				// should have caught this failure above...

/*
 * Configure GPIO pins for Alternate Functions.
 */
        if (pins == USART1_PA9_PA10)
        {
			GPIO_PinAFConfig(GPIOA, GPIO_PinSource9, GPIO_AF_USART1); // USART1_TX
			GPIO_PinAFConfig(GPIOA, GPIO_PinSource10, GPIO_AF_USART1); // USART1_RX
		}
		else if (pins == USART1_PB6_PB7)
		{
			GPIO_PinAFConfig(GPIOB, GPIO_PinSource6, GPIO_AF_USART1); // USART1_TX
			GPIO_PinAFConfig(GPIOB, GPIO_PinSource7, GPIO_AF_USART1); // USART1_RX
		}
		else  return  -1;						// should have caught this above...

        USART_Configure(USART1, baud);          // set up the serial comm parameters, including baud
/*
 * Enable USART1.
 */
        USART_Cmd(USART1, ENABLE);
        return  0;
	}
    else if (USARTx == USART2)               // set up USART2 with PD5 as transmit, PD6 as receive
    {
/*
 * Force reinit of the RCC register, just in case it was previously in use.
 */
        RCC_APB1PeriphResetCmd(RCC_APB1Periph_USART2, ENABLE);
        RCC_APB1PeriphResetCmd(RCC_APB1Periph_USART2, DISABLE);

        RCC_APB1PeriphClockCmd(RCC_APB1Periph_USART2, ENABLE);
        if      (pins == USART2_PA2_PA3)  RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOA, ENABLE);
        else if (pins == USART2_PD5_PD6)  RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOD, ENABLE);
        else  return  -1;					// unknown pin configuration, fail

/*
 * Configure USART2 GPIO pins.
 */
        GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
        GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
        GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
        GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
        if (pins == USART2_PA2_PA3)
        {
			GPIO_InitStructure.GPIO_Pin = GPIO_Pin_2 | GPIO_Pin_3;
			GPIO_Init(GPIOA, &GPIO_InitStructure);
		}
		else if (pins == USART2_PD5_PD6)
		{
			GPIO_InitStructure.GPIO_Pin = GPIO_Pin_5 | GPIO_Pin_6;
			GPIO_Init(GPIOD, &GPIO_InitStructure);
		}
		else  return  -1;				// should have been handled by code above...

/*
 * Configure pins for Alternate Functions.
 */
        if (pins == USART2_PA2_PA3)
        {
			GPIO_PinAFConfig(GPIOA, GPIO_PinSource2, GPIO_AF_USART2); // USART2_TX
			GPIO_PinAFConfig(GPIOA, GPIO_PinSource3, GPIO_AF_USART2); // USART2_RX
		}
		else if (pins == USART2_PD5_PD6)
		{
			GPIO_PinAFConfig(GPIOD, GPIO_PinSource5, GPIO_AF_USART2); // USART2_TX
			GPIO_PinAFConfig(GPIOD, GPIO_PinSource6, GPIO_AF_USART2); // USART2_RX
		}
		else  return  -1;				// should have been handled by code above...

        USART_Configure(USART2, baud);          // set up the serial comm parameters, including baud
/*
 * Enable USART2.
 */
        USART_Cmd(USART2, ENABLE);
        return  0;
    }
    else if (USARTx == USART3)              // set up USART3 with PB10 as transmit, PB11 as receive
    {
/*
 * Force reinit of the RCC register, just in case it was previously in use.
 */
        RCC_APB1PeriphResetCmd(RCC_APB1Periph_USART3, ENABLE);
        RCC_APB1PeriphResetCmd(RCC_APB1Periph_USART3, DISABLE);

        RCC_APB1PeriphClockCmd(RCC_APB1Periph_USART3, ENABLE);
        if      (pins == USART3_PB10_PB11)	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOB, ENABLE);
        else if (pins == USART3_PC10_PC11)	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOC, ENABLE);
        else if (pins == USART3_PD8_PD9)	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOD, ENABLE);
        else  return  -1;					// invalid pins argument, fail

/*
 * Configure GPIO pins.
 */
        GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
        GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
        GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
        GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
		if (pins == USART3_PB10_PB11)
		{
			GPIO_InitStructure.GPIO_Pin = GPIO_Pin_10 | GPIO_Pin_11;
			GPIO_Init(GPIOB, &GPIO_InitStructure);
		}
		else if (pins == USART3_PC10_PC11)
		{
			GPIO_InitStructure.GPIO_Pin = GPIO_Pin_10 | GPIO_Pin_11;
			GPIO_Init(GPIOC, &GPIO_InitStructure);
		}
		else if (pins == USART3_PD8_PD9)
		{
			GPIO_InitStructure.GPIO_Pin = GPIO_Pin_8 | GPIO_Pin_9;
			GPIO_Init(GPIOD, &GPIO_InitStructure);
		}
		else  return  -1;					// above code should have caught this failure
		

/*
 * Configure pins for Alternate Functions.
 */
		if (pins == USART3_PB10_PB11)
		{
			GPIO_PinAFConfig(GPIOB, GPIO_PinSource10, GPIO_AF_USART3);
			GPIO_PinAFConfig(GPIOB, GPIO_PinSource11, GPIO_AF_USART3);
		}
		else if (pins == USART3_PC10_PC11)
		{
			GPIO_PinAFConfig(GPIOC, GPIO_PinSource10, GPIO_AF_USART3);
			GPIO_PinAFConfig(GPIOC, GPIO_PinSource11, GPIO_AF_USART3);
		}
		else if (pins == USART3_PD8_PD9)
		{
			GPIO_PinAFConfig(GPIOD, GPIO_PinSource8, GPIO_AF_USART3);
			GPIO_PinAFConfig(GPIOD, GPIO_PinSource9, GPIO_AF_USART3);
		}
		else  return  -1;					// above code should have caught this failure

        USART_Configure(USART3, baud);          // set up the serial comm parameters, including baud
/*
 * Enable USART3.
 */
        USART_Cmd(USART3, ENABLE);
        return  0;
    }
    else if (USARTx == UART4)              // set up UART4  (note: UART, not USART!)
    {
/*
 * Force reinit of the RCC register, just in case it was previously in use.
 */
        RCC_APB1PeriphResetCmd(RCC_APB1Periph_UART4, ENABLE);
        RCC_APB1PeriphResetCmd(RCC_APB1Periph_UART4, DISABLE);

        RCC_APB1PeriphClockCmd(RCC_APB1Periph_UART4, ENABLE);
        if      (pins == UART4_PA0_PA1)		RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOA, ENABLE);
        else if (pins == UART4_PC10_PC11)	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOC, ENABLE);
        else  return  -1;					// invalid pins argument, fail

/*
 * Configure GPIO pins.
 */
        GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
        GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
        GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
        GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
		if (pins == UART4_PA0_PA1)
		{
			GPIO_InitStructure.GPIO_Pin = GPIO_Pin_0 | GPIO_Pin_1;
			GPIO_Init(GPIOA, &GPIO_InitStructure);
		}
		else if (pins == UART4_PC10_PC11)
		{
			GPIO_InitStructure.GPIO_Pin = GPIO_Pin_10 | GPIO_Pin_11;
			GPIO_Init(GPIOC, &GPIO_InitStructure);
		}
		else  return  -1;					// above code should have caught this failure

/*
 * Configure pins for Alternate Functions.
 */
		if (pins == UART4_PA0_PA1)
		{
			GPIO_PinAFConfig(GPIOA, GPIO_PinSource0, GPIO_AF_UART4);
			GPIO_PinAFConfig(GPIOA, GPIO_PinSource1, GPIO_AF_UART4);
		}
		else if (pins == UART4_PC10_PC11)
		{
			GPIO_PinAFConfig(GPIOC, GPIO_PinSource10, GPIO_AF_UART4);
			GPIO_PinAFConfig(GPIOC, GPIO_PinSource11, GPIO_AF_UART4);
		}
		else  return  -1;					// above code should have caught this failure

        USART_Configure(UART4, baud);          // set up the serial comm parameters, including baud
/*
 * Enable USART4.
 */
        USART_Cmd(UART4, ENABLE);
        return  0;
    }
    else if (USARTx == UART5)              // set up UART5  (note: UART, not USART!)
    {
/*
 * Force reinit of the RCC register, just in case it was previously in use.
 */
        RCC_APB1PeriphResetCmd(RCC_APB1Periph_UART5, ENABLE);
        RCC_APB1PeriphResetCmd(RCC_APB1Periph_UART5, DISABLE);

        RCC_APB1PeriphClockCmd(RCC_APB1Periph_UART5, ENABLE);
        if  (pins == UART5_PC12_PD2)
        {
			RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOC, ENABLE);
			RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOD, ENABLE);
		}
        else  return  -1;					// invalid pins argument, fail

/*
 * Configure GPIO pins.
 */
        GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
        GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
        GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
        GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
		if (pins == UART5_PC12_PD2)
		{
			GPIO_InitStructure.GPIO_Pin = GPIO_Pin_12;
			GPIO_Init(GPIOC, &GPIO_InitStructure);
			GPIO_InitStructure.GPIO_Pin = GPIO_Pin_2;
			GPIO_Init(GPIOD, &GPIO_InitStructure);
		}
		else  return  -1;					// above code should have caught this failure

/*
 * Configure pins for Alternate Functions.
 */
		if (pins == UART5_PC12_PD2)
		{
			GPIO_PinAFConfig(GPIOC, GPIO_PinSource12, GPIO_AF_UART5);
			GPIO_PinAFConfig(GPIOD, GPIO_PinSource2, GPIO_AF_UART5);
		}
		else  return  -1;					// above code should have caught this failure

        USART_Configure(UART5, baud);          // set up the serial comm parameters, including baud
/*
 * Enable UART5.
 */
        USART_Cmd(UART5, ENABLE);
        return  0;
    }
    else if (USARTx == USART6)              // set up USART6
    {
/*
 * Force reinit of the RCC register, just in case it was previously in use.
 */
        RCC_APB2PeriphResetCmd(RCC_APB2Periph_USART6, ENABLE);
        RCC_APB2PeriphResetCmd(RCC_APB2Periph_USART6, DISABLE);

        RCC_APB2PeriphClockCmd(RCC_APB2Periph_USART6, ENABLE);
        if  (pins == USART6_PC6_PC7)	RCC_AHB1PeriphClockCmd(RCC_AHB1Periph_GPIOC, ENABLE);
        else  return  -1;					// invalid pins argument, fail

/*
 * Configure GPIO pins.
 */
        GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;
        GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
        GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
        GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
		if (pins == USART6_PC6_PC7)
		{
			GPIO_InitStructure.GPIO_Pin = GPIO_Pin_6 | GPIO_Pin_7;
			GPIO_Init(GPIOC, &GPIO_InitStructure);
		}
		else  return  -1;					// above code should have caught this failure

/*
 * Configure pins for Alternate Functions.
 */
		if (pins == USART6_PC6_PC7)
		{
			GPIO_PinAFConfig(GPIOC, GPIO_PinSource6, GPIO_AF_USART6);
			GPIO_PinAFConfig(GPIOC, GPIO_PinSource7, GPIO_AF_USART6);
		}
		else  return  -1;					// above code should have caught this failure

        USART_Configure(USART6, baud);          // set up the serial comm parameters, including baud
/*
 * Enable USART6.
 */
        USART_Cmd(USART6, ENABLE);
        return  0;
    }
    else                                // unknown USART, return non-zero as failure
    {
        return  -1;
    }
}



static int			CopyFlashToBuffer(unsigned int  snum)
{
	int					n;
	unsigned char		*srcptr;
	unsigned char		*dstptr;
	
	srcptr = (unsigned char *)FLASH_SECTOR_STORAGE_ADDR;
	dstptr = flashbuffer;
	
	for (n=0; n<FLASH_SECTOR_STORAGE_SIZE; n++)
	{
		*dstptr++ = *srcptr++;
	}
	return  0;
}




/*
 *  USART_ReadyToSend      check USART to see if it can accept a new char to send
 *
 *  This routine checks the TXE flag of the selected USART and returns
 *  non-zero if the USART is ready to accept a new char to send, else it
 *  returns zero.
 *
 *  This routine does not send data, it only checks to see if the USART can accept
 *  a new data byte.  This routine does not poll, it only tests the USART once, then
 *  returns the result of that check.
 */
static uint16_t  USART_ReadyToSend(USART_TypeDef  *USARTx)
{
    if (USARTx->SR & USART_FLAG_TXE)  return  TRUE;
    else                              return  FALSE;
}



/*
 *  USART_CharAvail      check USART to see if data has been received
 *
 *  This routine checks the RXNE flag of the selected USART and returns
 *  non-zero if a byte has been recieved and is ready to read, else it
 *  returns zero.
 *
 *  This routine does NOT return the recieved data, only a flag that data
 *  is or is not available.
 */
static uint16_t  USART_CharAvail(USART_TypeDef  *USARTx)
{
    if (USARTx->SR & USART_FLAG_RXNE)  return  TRUE;
    else                               return  FALSE;
}



/*
 *  getchar_USARTx      receive single char from selected USART
 *
 *  Poll USART named in argument USARTx until it reports a character is available.
 *  Return that charcter to the caller.
 *
 *  Note that this routine locks until a char is available.  Use USART_CharAvail() if
 *  you want to poll before calling this routine.
 */
static uint32_t  getchar_USARTx(USART_TypeDef  *USARTx)
{
    while  (USART_CharAvail(USARTx) == FALSE)  ;
    return  (uint32_t)(USARTx->DR & (uint16_t)0x01FF);
}


/*
 *  putchar_USARTx      send single char to selected USART
 *
 *  Send the char in argument ch to the USART selected by argument USARTx.
 *  This routine polls the selected USART until it is ready to accept data.
 *  Upon exit, this routine returns the character sent.
 */
static uint32_t  putchar_USARTx(USART_TypeDef  *USARTx, uint32_t  ch)
{
	while (USART_ReadyToSend(USARTx) == FALSE)  ;
    USARTx->DR = (ch & (uint16_t)0x01FF);
    return  ch;
}



void  SysTick_Handler(void)
{
	SystemCounter_Msecs++;
	
	for (int  n=0; n<T_NUM_TIMERS; n++)
	{
		if (timeraddrs[n])
		{
			if (*timeraddrs[n])  *timeraddrs[n] = *timeraddrs[n] - 1;
		}
	}
	
	if (hwtimers[HWTIMER_TONE])					// check the tone timer
	{
		hwtimers[HWTIMER_TONE]--;
		if (hwtimers[HWTIMER_TONE] == 0)		// if tone timer has elapsed...
		{
			TIM_Cmd(TIM2, DISABLE);				// shut off tone PWM output
		}
	}
}



/*
 * t_ColdBoot      perform coldboot operations; usually called once, following power-on or reset
 */ 
void			t_ColdBoot(void)
{
	SystemInit();
	FLASH_SetLatency(FLASH_Latency_5);
	
    RCC_AHB1PeriphClockCmd(RED_LED_CLK, ENABLE);		// route the LED clock (works for all LEDs)
    RCC_AHB1PeriphClockCmd(USER_PB_CLK, ENABLE);		// route the USER pushbutton clock
    RCC_AHB1PeriphClockCmd(TONE_CLK, ENABLE);			// route the tone output clock
    RCC_APB1PeriphClockCmd(TONE_TIMER_CLK, ENABLE);		// route the tone timer clock
    RCC_AHB1PeriphClockCmd(DOUT0_CLK, ENABLE);			// route the data out clock (works for all DOUTx)
    RCC_APB2PeriphClockCmd(ADC0_CLK, ENABLE);			// route the ADC clock (works for all ADCx)
	RCC_APB2PeriphClockCmd(RCC_APB2Periph_ADC1, ENABLE);

    	
    GPIO_InitStructure.GPIO_Pin = RED_LED_PIN_MASK | ORG_LED_PIN_MASK | GREEN_LED_PIN_MASK | BLUE_LED_PIN_MASK;     // set the pin mask for the LEDs
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;		// set the mode to output
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_100MHz;	// set the I/O speed to 100 MHz
    GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;		// set the output type to push-pull
    GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;	// set the pull-up/down to none
    GPIO_Init(RED_LED_PORT, &GPIO_InitStructure);		// do the init (works for all LEDs)

    GPIO_InitStructure.GPIO_Pin = USER_PB_PIN_MASK;     // set the pin mask for the User PB
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN;		// set the mode to input
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_100MHz;	// set the I/O speed to 100 MHz
    GPIO_InitStructure.GPIO_OType = GPIO_OType_OD;		// set the output type to open-drain
    GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;	// set the pull-up/down to none
    GPIO_Init(USER_PB_PORT, &GPIO_InitStructure);		// do the init

	GPIO_InitStructure.GPIO_Pin = TONE_PIN_MASK;     	// set the pin mask for the tone output
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF;		// set the mode to alt func
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_100MHz;	// set the I/O speed to 100 MHz
    GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;		// set the output type to push-pull
    GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_DOWN;		// set the pull-up/down to weak pulldown
    GPIO_Init(TONE_PORT, &GPIO_InitStructure);			// do the init
 	GPIO_PinAFConfig(TONE_PORT, GPIO_PinSource1, GPIO_AF_TIM2);		// GPIOA-1 is timer 2 PWM output

    GPIO_InitStructure.GPIO_Pin = DOUT0_PIN_MASK | DOUT1_PIN_MASK | DOUT2_PIN_MASK | DOUT3_PIN_MASK;     // set the pin mask for the LEDs
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;		// set the mode to output
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_100MHz;	// set the I/O speed to 100 MHz
    GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;		// set the output type to push-pull
    GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_DOWN;		// set the pull-up/down to weak pulldown
    GPIO_Init(DOUT0_PORT, &GPIO_InitStructure);			// do the init (works for all DOUT outputs)

    GPIO_InitStructure.GPIO_Pin = DIN0_PIN_MASK | DIN1_PIN_MASK | DIN2_PIN_MASK | DIN3_PIN_MASK;     // set the pin mask for the LEDs
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN;		// set the mode to input
    GPIO_InitStructure.GPIO_Speed = GPIO_Speed_100MHz;	// set the I/O speed to 100 MHz
    GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;		// set the output type to push-pull
    GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_DOWN;		// set the pull-up/down to weak pulldown
    GPIO_Init(DIN0_PORT, &GPIO_InitStructure);			// do the init (works for all DIN inputs)

    USART_Initialize(ACTIVE_USART, ACTIVE_USART_PINS, ACTIVE_USART_BAUD);   // USART indentifiers are defined in stm32f4xx.h

    t_WritePort(RED_LED, 1);							// turn on red LED to show we got this far

	CopyFlashToBuffer(0);						// load up the working buffer from flash
	filebufferchanged = FALSE;					// show now changes yet
	
	SystemCounter_Msecs = 0;					// always clear the system counter on cold boot
	t_SetTimerRate(1000);						// default to 1 msec system timer tic rate

    // Init GPIO port for ADC input
    GPIO_InitStructure.GPIO_Pin = ADC0_PIN_MASK | ADC1_PIN_MASK | ADC2_PIN_MASK | ADC3_PIN_MASK;
    GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AN;
    GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
    GPIO_Init(ADC_GPIO_PORT, &GPIO_InitStructure);

   // Init ADC for selected module
    ADC_CommonInitStructure.ADC_Mode = ADC_Mode_Independent;
    ADC_CommonInitStructure.ADC_Prescaler = ADC_Prescaler_Div2;
    ADC_CommonInitStructure.ADC_DMAAccessMode = ADC_DMAAccessMode_Disabled;
    ADC_CommonInitStructure.ADC_TwoSamplingDelay = ADC_TwoSamplingDelay_5Cycles;
    ADC_CommonInit(&ADC_CommonInitStructure);

	ADC_StructInit(&ADC_InitStructure);
    ADC_InitStructure.ADC_ContinuousConvMode = DISABLE;
    ADC_InitStructure.ADC_DataAlign = ADC_DataAlign_Right;
    ADC_InitStructure.ADC_ExternalTrigConv = DISABLE;
    ADC_InitStructure.ADC_ExternalTrigConvEdge = ADC_ExternalTrigConvEdge_None;
    ADC_InitStructure.ADC_NbrOfConversion = 4;
    ADC_InitStructure.ADC_Resolution = ADC_Resolution_12b;
    ADC_InitStructure.ADC_ScanConvMode = ENABLE;
    ADC_Init(ADC_MODULE, &ADC_InitStructure);

	ADC_MODULE->CR2 |= (uint32_t)ADC_CR2_EOCS;			// EOC set after each reg conv

    ADC_RegularChannelConfig(ADC_MODULE, ADC0_CHNL, 1, ADC_SampleTime_144Cycles);
    ADC_RegularChannelConfig(ADC_MODULE, ADC1_CHNL, 2, ADC_SampleTime_144Cycles);
    ADC_RegularChannelConfig(ADC_MODULE, ADC2_CHNL, 3, ADC_SampleTime_144Cycles);
    ADC_RegularChannelConfig(ADC_MODULE, ADC3_CHNL, 4, ADC_SampleTime_144Cycles);
    ADC_MODULE->CR2 |= (uint32_t)ADC_CR2_ADON;			// turn on ADC
}



/*
 * t_WarmBoot      perform warmboot operations; can be called after program error or crash
 */
void			t_WarmBoot(void)
{
	OutputStream = T_STREAM_SERIAL;			// set output stream to serial port
	InputStream = T_STREAM_SERIAL;			// set input stream to serial port
	t_WritePort(BLUE_LED, 1);				// turn on blue LED to show we got this far
}





/*
 * t_Shutdown      performs target-specific actions to shut down TBL
 */
void			t_Shutdown(void)
{
	int			c;
	
	if (filebufferchanged)
	{
		t_SetOutputStream(T_STREAM_SERIAL);
		printstr("\n\nWARNING!  You have made program changes.  Do you want to save");
		printstr("              your changes to flash (y/n)? ");
		while (1)
		{
			c = t_GetChar();
			t_OutChar(c);
			if (c == 'y' || c == 'Y')
			{
				printstr("  erasing flash...");
				Flash_EraseSector();
				printstr("  saving changes...");
				Flash_WriteSector();
				printstr(" done.");
				break;
			}
			else if (c == 'n' || c == 'N')
			{
				break;
			}
		}
	}
	
	printstr("\n\nShutting down; reset the system to restart TinyBasicLike... ");
}



/*
 * t_OutChar      writes (blocks) char to active stream, based on OutputStream.
 */
void			t_OutChar(int  c)
{
	switch  (OutputStream)
	{
		case  T_STREAM_SERIAL:
		c = putchar_USARTx(ACTIVE_USART, c);
		break;
		
		case  T_STREAM_FILE:
		if (wfileptr)
		{
			if (wfileptr < wfiledataend)
			{
				*wfileptr++ = c;
				filebufferchanged = TRUE;
			}
		}
	}
}



/*
 * t_GetChar      read (blocks) a char from active stream, based on InputStream.
 */
int				t_GetChar(void)
{
	int			c;
	
	switch  (InputStream)
	{
		case  T_STREAM_SERIAL:
		return  getchar_USARTx(ACTIVE_USART);
		
		case  T_STREAM_FILE:
		if (rfileptr)
		{
			if (rfileptr < rfiledataend)
			{
				c = *rfileptr++;
				if (c == 0xff)	c = EOF;		// empty byte from flash (0xff) means EOF
				return  c;
			}
		}
		return  0;
		
		default:
		return  0;
	}
}



/*
 * t_ConsoleCharAvailable      returns TRUE if char is available from console stream,
 * ignoring InputStream.
 */
int				t_ConsoleCharAvailable(void)
{
	return  USART_CharAvail(ACTIVE_USART);
}



/*
 * t_GetCharFromConole      read (blocks) a char from console stream, ignoring InputStream.
 */
int				t_GetCharFromConsole(void)
{
	return  getchar_USARTx(ACTIVE_USART);
}


/*
 * t_SetTimerRate      select timer tic rate in usecs
 */
void			t_SetTimerRate(unsigned int  usecs)
{
	SysTick_Config((SystemCoreClock / 1000000) * usecs);
}



/*
 * t_AddTimer      add address to timer list
 * 
 * If an open address is available in the timer table, record the address
 * passed as argument t for later use as a down-counting timer.
 * 
 * If the address in argument t already exists in the timer table, pretend
 * the address was added but don't change the table.
 */
int				t_AddTimer(DATA_SIZE  *t)
{
	int				n;
	int				open;
	
	open = T_NUM_TIMERS+1;
	for (n=0; n<T_NUM_TIMERS; n++)
	{
		if (timeraddrs[n] == 0)
		{
			open = n;
		}
		else
		{
			if (t == timeraddrs[n])		// if same address in already in table...
				return  S_OK;			// pretend we added it
		}
	}
	if (open < T_NUM_TIMERS+1)
	{
		timeraddrs[open] = t;			// record addr
		return  S_OK;					// show it worked
	}
	return  E_COULD_NOT_ADD_TIMER;	// could not add timer
}



/*
 * t_DeleteTimer      remove address from timer list
 */
int				t_DeleteTimer(DATA_SIZE  *t)
{
	int				n;
	
	for (n=0; n<T_NUM_TIMERS; n++)
	{
		if (timeraddrs[n] == t)
		{
			timeraddrs[n] = 0;
			return  0;
		}
	}
	return  -1;							// could not remove timer, not in list
}


/*
 * t_SetOutputStream      change character output stream
 */
int				t_SetOutputStream(int  s)
{
	if (s != T_STREAM_SERIAL && s != T_STREAM_FILE)  return  -1;
	
	OutputStream = s;
	return  0;
}



/*
 * t_SetInputStream      change character input stream
 */
int				t_SetInputStream(int  s)
{
	if (s != T_STREAM_SERIAL && s != T_STREAM_FILE)  return  -1;
	
	InputStream = s;
	return  0;
}




/*
 * t_FileExistsQ      check if named file exists in file buffer
 * 
 * This routine checks the file buffer (NOT flash!) to see if the named
 * file already exists.  Returns TRUE if file exists, else FALSE.
 */
int				t_FileExistsQ(char  *name)
{
	int					r;
	char				*ptr;			// unused, but needed for function call
	
	r = FindFileByName(name, &ptr);
	if (r == S_OK)  return  TRUE;
	else 			return  FALSE;
}



FILE			*t_OpenFile(char  *name, char  *mode)
{
	int				r;
	char			*fptr;
	
	r = FindFileByName(name, &fptr);

	if (*mode == 'r' || *mode == 'R')
	{
		if (r != S_OK)  return  NULL;
		rfilestart = fptr;
		rfiledatastart = rfilestart + SIZEOF_FLASH_FILE_HEADER;
		rfiledataend = rfilestart + SIZEOF_FLASH_FILE;
		rfileptr = rfiledatastart;
		rfp = (FILE *)1;
		return  rfp;
	}
	else if (*mode == 'w' || *mode == 'W')
	{
		if (r == S_OK)						// if file already exists
		{
			t_DeleteFile(name);				// delete the file
		}
		fptr = CreateNewFile(name);					// try to create the file
		if (fptr == NULL)  return  NULL;		// oops, could not write header
		wfilestart = fptr;
		wfiledatastart = wfilestart + SIZEOF_FLASH_FILE_HEADER;
		wfiledataend = wfilestart + SIZEOF_FLASH_FILE;
		wfileptr = wfiledatastart;
		wfp = (FILE *)1;
		return  wfp;
	}
	else	return  NULL;
}


int				t_CloseFile(FILE  *fp)
{
	return  0;
}


int				t_DeleteFile(char  *name)
{
	int				r;
	char			*fptr;
	int				i;
	
	r = FindFileByName(name, &fptr);
	if (r == S_OK)					// if found the file...
	{
//		printstr("  t_DeleteFile:deleting  ");
		for (i=0; i<SIZEOF_FLASH_FILE; i++)
		{
			*(fptr + i) = 0xff;		// erase a file by writing 0xff to all RAM buffer bytes
		}
		filebufferchanged = TRUE;	// let Shutdown know something happened
	}
	return  r;
}


	

char      *t_GetFirstFileName(void)
{
	int				i;
	char			*ptr;
	
	for (i=0; i<NUM_FLASH_FILES; i++)
	{
		ptr = &flashbuffer[i*SIZEOF_FLASH_FILE];
		if (*ptr != 0xff)
		{
			return  ptr;
		}
	}
	return  NULL;
}
		
#if 0

	for (i=0; i<SIZEOF_FLASH_SECTOR; i++)
	{
		if (flashbuffer[i] != 0xff)
		{
			return  (char *)&flashbuffer[i];
		}
	}
	return  NULL;
}
#endif


char		*t_GetNextFileName(char  *ptr)
{
	if (!ptr)  return NULL;					// stupid check

	while (1)
	{
		ptr = ptr + SIZEOF_FLASH_FILE;
		if (ptr >= (char *)(flashbuffer + FLASH_SECTOR_STORAGE_SIZE))
			break;
		if (*ptr != 0xff)
		{
			return  ptr;
		}
	}
	return  NULL;
}

			

/*
 * Support for I/O ports.
 */
DATA_SIZE		t_ReadPort(unsigned int  index)
{
	DATA_SIZE				t;
	DATA_SIZE				adcvalues[4];				// array holds one value per ADC channel
	
	if (index >= NUM_PORTS)
	{
		return  0;
	}
	switch  (index)
	{
		case  BLUE_LED:
		t = BLUE_LED_PORT->ODR & BLUE_LED_PIN_MASK;
		if (t) t = 1;
		return  t;
		
		case  GREEN_LED:
		t = GREEN_LED_PORT->ODR & GREEN_LED_PIN_MASK;
		if (t) t = 1;
		return  t;
		
		case  ORANGE_LED:
		t = ORG_LED_PORT->ODR & ORG_LED_PIN_MASK;
		if (t) t = 1;
		return  t;
		
		case  RED_LED:
		t = RED_LED_PORT->ODR & RED_LED_PIN_MASK;
		if (t) t = 1;
		return  t;

		case  DOUT0:
		t = DOUT0_PORT->ODR & DOUT0_PIN_MASK;
		if (t) t = 1;
		return  t;

		case  DOUT1:
		t = DOUT1_PORT->ODR & DOUT1_PIN_MASK;
		if (t) t = 1;
		return  t;

		case  DOUT2:
		t = DOUT2_PORT->ODR & DOUT2_PIN_MASK;
		if (t) t = 1;
		return  t;

		case  DOUT3:
		t = DOUT3_PORT->ODR & DOUT3_PIN_MASK;
		if (t) t = 1;
		return  t;

		case  DIN0:
		t = DIN0_PORT->IDR & DIN0_PIN_MASK;
		if (t) t = 1;
		return  t;

		case  DIN1:
		t = DIN1_PORT->IDR & DIN1_PIN_MASK;
		if (t) t = 1;
		return  t;

		case  DIN2:
		t = DIN2_PORT->IDR & DIN2_PIN_MASK;
		if (t) t = 1;
		return  t;

		case  DIN3:
		t = DIN3_PORT->IDR & DIN3_PIN_MASK;
		if (t) t = 1;
		return  t;

		case  VP_ADC0:
		case  VP_ADC1:
		case  VP_ADC2:
		case  VP_ADC3:
		for (int n=0; n<4; n++)						// need to read all ADC values
		{
			ADC_MODULE->CR2 |= (uint32_t)ADC_CR2_SWSTART;		// start conversions
			while ((ADC_MODULE->SR & ADC_FLAG_EOC) == 0)  ;	// blocks until end-of-conversion bit is low
			adcvalues[n] = (DATA_SIZE)ADC_MODULE->DR;
		}
		if (index == VP_ADC0)  		return  adcvalues[0];
		else if (index == VP_ADC1)  return  adcvalues[1];
		else if (index == VP_ADC2)  return  adcvalues[2];
		else 					 	return  adcvalues[3];

		default:
		return  (*porttable[index].portaddr) & ((1 << porttable[index].size) -1);
	}
}


void			t_WritePort(unsigned int  index, DATA_SIZE  value)
{
	if (index >= NUM_PORTS) 	return;
	if (porttable[index].writeable == READONLY)  return;		// cannot change this port, ignore

	switch  (index)
	{
		case  BLUE_LED:
		if (value)  BLUE_LED_PORT->BSRRL = BLUE_LED_PIN_MASK;
		else 		BLUE_LED_PORT->BSRRH = BLUE_LED_PIN_MASK;
		break;
		
		case  GREEN_LED:
		if (value)  GREEN_LED_PORT->BSRRL = GREEN_LED_PIN_MASK;
		else 		GREEN_LED_PORT->BSRRH = GREEN_LED_PIN_MASK;
		break;
		
		case  ORANGE_LED:
		if (value)  ORG_LED_PORT->BSRRL = ORG_LED_PIN_MASK;
		else 		ORG_LED_PORT->BSRRH = ORG_LED_PIN_MASK;
		break;
		
		case  RED_LED:
		if (value)  RED_LED_PORT->BSRRL = RED_LED_PIN_MASK;
		else 		RED_LED_PORT->BSRRH = RED_LED_PIN_MASK;
		break;
		
		case  DOUT0:
		if (value)  DOUT0_PORT->BSRRL = DOUT0_PIN_MASK;
		else 		DOUT0_PORT->BSRRH = DOUT0_PIN_MASK;
		break;
		
		case  DOUT1:
		if (value)  DOUT1_PORT->BSRRL = DOUT1_PIN_MASK;
		else 		DOUT1_PORT->BSRRH = DOUT1_PIN_MASK;
		break;
		
		case  DOUT2:
		if (value)  DOUT2_PORT->BSRRL = DOUT2_PIN_MASK;
		else 		DOUT2_PORT->BSRRH = DOUT2_PIN_MASK;
		break;
		
		case  DOUT3:
		if (value)  DOUT3_PORT->BSRRL = DOUT3_PIN_MASK;
		else 		DOUT3_PORT->BSRRH = DOUT3_PIN_MASK;
		break;
		
		default:
		*porttable[index].portaddr = value & ((1 << porttable[index].size) -1);
	}
}



/*
 * t_CheckForHWBreak      check USER pushbutton; if pressed, return TRUE
 */
int				t_CheckForHWBreak(void)
{
	if (GPIOA->IDR & USER_PB_PIN_MASK)  return  TRUE;		// 1 means PB is pressed
	else 								return  FALSE;		// 0 means PB is not pressed
}



/*
 * t_SetTone			start tone of requested freq (Hz) and duration (msecs)
 */
 void			t_SetTone(DATA_SIZE  freq, DATA_SIZE  duration)
 {
	DATA_SIZE						t;
	 
    TIM_Cmd(TIM2, DISABLE);
	if (freq == 0 || duration == 0)
	{
		hwtimers[HWTIMER_TONE] = 0;
		if (hwtimers[HWTIMER_TONE] != 0)  hwtimers[HWTIMER_TONE] = 0;
		return;
	}
	 
	/*
	 * Calc period in tics based on system frequency and requested frequency
	 * 
	 * TIM_Period = (timer_tick_frequency / desired_frequency) - 1
	 */
	t = TIMER_CLK_RATE / freq - 1;
	TIM_BaseStruct.TIM_Period = t;
	//printstr("\nt_SetTone  TIM_Period=");
	//printnum(t, 10);
	//printstr("\n");
	
    TIM_OCStruct.TIM_OCMode = TIM_OCMode_PWM2;
    TIM_OCStruct.TIM_OutputState = TIM_OutputState_Enable;
    TIM_OCStruct.TIM_OCPolarity = TIM_OCPolarity_Low;
    TIM_OCStruct.TIM_Pulse = t/2;		// always use 50% duty cycle
    TIM_OC2Init(TIM2, &TIM_OCStruct);
    TIM_OC2PreloadConfig(TIM2, TIM_OCPreload_Enable);

	TIM_ARRPreloadConfig(TIM2, ENABLE);

	TIM_BaseStruct.TIM_Prescaler = TIMER_PRESCALER;			// timer clock = system clock / prescaler
	TIM_BaseStruct.TIM_CounterMode = TIM_CounterMode_Up;
	TIM_BaseStruct.TIM_ClockDivision = TIM_CKD_DIV1;
    TIM_BaseStruct.TIM_RepetitionCounter = 0;
    TIM_TimeBaseInit(TIM2, &TIM_BaseStruct);
    TIM_Cmd(TIM2, ENABLE);

    hwtimers[HWTIMER_TONE] = duration;
    if (hwtimers[HWTIMER_TONE] != duration)  hwtimers[HWTIMER_TONE] = duration;
}
 

/*
 * t_CheckToneStatus      returns TRUE if tone is on, else FALSE
 */
int				t_CheckToneStatus(void)
{
	 if (hwtimers[HWTIMER_TONE])  return  TRUE;
	 else 						  return  FALSE;
}



/*
 * t_Test      generic test function
 * 
 * This function is called anytime the user executes the TEST keyword in the
 * core program.  This function provides a way to run any dedicated test you
 * code; very handy when adding and testing new features to the target code.
 * 
 * If you don't need any custom test code, just make this an empty function.
 */	

#if 0
void			t_Test(void)
{
	int				c;
	
	printstr("\n\nThis will erase all files in the flash sector!  Are you sure? ");
	c = t_GetCharFromConsole();
	if (c == 'y' || c == 'Y')
	{
		Flash_EraseSector(0);
		printstr("\nDone!\n");
	}
	else
	{
		printstr("\nNot erased.\n");
	}
}
#endif
	

#if 0
void			t_Test(void)
{
	unsigned char		*ptr;
	int					n;
	
	ptr = flashbuffer;
	for (n=0; n<NUM_FLASH_FILES; n++)
	{
		printstr("\n");
		printnum((DATA_SIZE)ptr, RADIX_HEX);
		printstr(": ");
		for (int j=0; j<8; j++)
		{
			printnum((DATA_SIZE)ptr[j], RADIX_HEX);
			printstr(" ");
		}
		ptr = ptr + SIZEOF_FLASH_FILE;
	}
	printstr("\n");
}
#endif
	
#if 1
/* print first few lines of program as hex chars */
void			t_Test(void)
{
	unsigned char		*ptr;
	int					n;
	
	ptr = program_start;
	for (n=0; n<160; n++)
	{
		if ((n % 16) == 0)  printstr("\n");
		if (ptr[n] < 16)  printstr("0");
		printnum(ptr[n], RADIX_HEX);
		printstr(" ");
	}
	printstr("\n");
}

#endif

