/*
 *  sdlocker1k      lock/unlock an SD card, uses ATmega328P, code size < 1K bytes
 *
 *  Provides write-lock and/or password-lock protection.
 *
 *  This code was written as an entry in the Hackaday 1K code challenge, which
 *  ended 5 Jan 2017.  The goal was to write a program that could provide
 *  both write-lock and password-lock in just 1024 bytes.
 *
 *  Unfortunately, I could not get both features in that small footprint.  So
 *  this code lets you select at compile-time one feature or the other, using
 *  #defines.  Note that the code will enable both features if you #define both
 *  macros, but the code will be about 1130 bytes, beyond the 1K limit.
 *
 *  Fair warning, this is ugly code!  I've cut quite a few corners to
 *  fit this into the 1K space.  But what really matters is the code that
 *  reports the state of the SD card, and I think that code is good.
 *
 *  To use this code, burn it into a atmega328P wired into the correct circuitry.
 *  If you push the PWD switch and the code has been built to password-lock,
 *  the code will toggle the password-lock on and off using a fixed password.
 *  Password-lock state will be reflected in the PWD LED (on = locked).
 *
 *  If you push the WRT switch and the code has been built to write-lock, the
 *  code will toggle the write-protect lock on and off.  Write-lock state
 *  will be reflected in the WRT LED (on = locked).
 *
 *  This code was designed to use as few I/O pins as possible.  It only
 *  uses MISO, MOSI, SCK, one chip-select line, and one A/D input for reading
 *  the two pushbutton switches.  In theory, this code could fit in an
 *  8-pin attiny device.  However, you would need to use a device with
 *  built-in SPI or synchronous serial support.  I built a version with
 *  bit-banged SPI, but the transfer speeds were too slow to be usable.
 *
 *  To fit this code into such a small footprint, I removed the interrupt
 *  vectors that normally start at address 0x00.  Doing this requires a
 *  custom linker script and a custom crt0.s.  Refer to my website (below)
 *  for details on these files.  If you are rebuilding this code and don't
 *  care about fitting into a restricted space, just use the stock Atmel
 *  crt0.s and linker scripts.
 *
 *  Karl Lunt  28 Dec 2016
 */

#include  <avr/io.h>
#include  <avr/pgmspace.h>
#include  <inttypes.h>
#include  <ctype.h>
#include  <util/delay.h>


/*
 *  If you build this in Visual Studio, be sure to define the symbol
 *  __AVR_ATmega328P__ so IntelliSense can properly resolve your I/O
 *  register names.  Do this by adding __AVR_ATtiny328P__ to the list
 *  in project Properties/NMake/Preprocessor definitions.
 */


/*
 *  Define the functionality this code is to provide.
 *
 *  If you define BUILD_WRT_LOCKER, this program will write-lock
 *  or write-enable the card, using TMP_WRITE_PROTECT bit in the
 *  CSD register.
 *
 *  If you define BUILD_PWD_LOCKER, this program will password-lock
 *  or password-unlock the card, using CMD42 (LOCK_UNLOCK).
 *
 *  You can define both of the above macros, if you choose, but
 *  the resulting executable will exceed the 1024-byte limit for
 *  the Hackaday 1K code contest.
 */
//#define  BUILD_WRT_LOCKER
#define  BUILD_PWD_LOCKER


#ifndef  FALSE
#define  FALSE		0
#define  TRUE		!FALSE
#endif



/*
 *  Define commands for the SD card
 */
#define  SD_GO_IDLE			(0x40 + 0)			/* CMD0 - go to idle state */
#define  SD_INIT			(0x40 + 1)			/* CMD1 - start initialization */
#define  SD_SEND_IF_COND	(0x40 + 8)			/* CMD8 - send interface (conditional), works for SDHC only */
#define  SD_SEND_CSD		(0x40 + 9)			/* CMD9 - send CSD block (16 bytes) */
#define  SD_SEND_STATUS		(0x40 + 13)			/* CMD13 - send card status */
#define  SD_SET_BLK_LEN		(0x40 + 16)			/* CMD16 - set length of block in bytes */
#define  SD_LOCK_UNLOCK		(0x40 + 42)			/* CMD42 - lock/unlock card */
#define  CMD55				(0x40 + 55)			/* multi-byte preface command */
#define  SD_READ_OCR		(0x40 + 58)			/* read OCR */
#define  SD_ADV_INIT		(0xc0 + 41)			/* ACMD41, for SDHC cards - advanced start initialization */
#define  SD_PROGRAM_CSD		(0x40 + 27)			/* CMD27 - get CSD block (15 bytes data + CRC) */



/*
 *  Define error codes that can be returned by local functions
 */
#define  SDCARD_OK					0			/* success */
#define  SDCARD_NO_DETECT			1			/* unable to detect SD card */
#define  SDCARD_TIMEOUT				2			/* last operation timed out */
#define  SDCARD_RWFAIL				-1			/* read/write command failed */
#define  SDCARD_LOCK_FAILED			-2			/* pwd or write lock failed */
#define  SDCARD_UNLOCK_FAILED		-3			/* pwd or write unlock failed */


/*
 *  Define options for accessing the SD card's PWD (CMD42)
 */
#define  MASK_ERASE					0x08		/* erase the entire card */
#define  MASK_LOCK_UNLOCK			0x04		/* lock or unlock the card with password */
#define  MASK_CLR_PWD				0x02		/* clear password */
#define  MASK_SET_PWD				0x01		/* set password */


#if 0
/*
 *  Define card types that could be reported by the SD card during probe
 */
#define  SDTYPE_UNKNOWN			0				/* card type not determined */
#define  SDTYPE_SD				1				/* SD v1 (1 MB to 2 GB) */
#define  SDTYPE_SDHC			2				/* SDHC (4 GB to 32 GB) */
#endif


/*
 *  Define debounce delays for reading the switches.
 */
#define  DEBOUNCE_DELAY_MS	20
#define  DEBOUNCE_COUNT		10


/*
 *  Define literals for the possible switch states
 */
#define  SW_NONE			0
#define  SW_PWDPROT			1
#define  SW_WRTPROT			2
#define  SW_BOTH			3


/*
 *  Define the port and DDR used by the SPI.
 */
#define  SPI_PORT		PORTB
#define  SPI_DDR		DDRB


/*
 *  Define bits used by the SPI port.
 */
#define  MOSI_BIT		3
#define  MISO_BIT		4
#define  SCK_BIT		5


/*
 *  Define the port, DDR, and bit used as chip-select for the
 *  SD card.
 */
#define  SD_CS_PORT		PORTB
#define  SD_CS_DDR		DDRB
#define  SD_CS_BIT		2
#define  SD_CS_MASK		(1<<SD_CS_BIT)


/*
 *  Define macros for enabling/disabling SPI.  The SPI system
 *  takes control of the SPI port lines if enabled.  If we
 *  want to control the port lines for LEDs, we need to disable
 *  SPI first.
 */
#define  SPI_ENABLE			SPCR=(1<<SPE)|(1<<MSTR)|(1<<SPR1)|(1<<SPR0)
#define  SPI_DISABLE		SPCR=0


/*
 *  Define the port and bit used for the write-protect LED.
 *  To conserve pins, reuse PB5 (SCK).
 */
#define  WRTPROT_LED_PORT	PORTB
#define  WRTPROT_LED_DDR	DDRB
#define  WRTPROT_LED_BIT	5		
#define  WRTPROT_LED_MASK	(1<<WRTPROT_LED_BIT)
#define  WRTPROT_LED_OFF	(WRTPROT_LED_PORT=WRTPROT_LED_PORT&(~WRTPROT_LED_MASK))
#define  WRTPROT_LED_ON		(WRTPROT_LED_PORT=WRTPROT_LED_PORT|WRTPROT_LED_MASK)


/*
 *  Define the port and bit used for the password-protect LED.
 *  To conserve pins, reuse PB3 (MOSI).
 */
#define  PWDPROT_LED_PORT	PORTB
#define  PWDPROT_LED_DDR	DDRB
#define  PWDPROT_LED_BIT	3		
#define  PWDPROT_LED_MASK	(1<<PWDPROT_LED_BIT)
#define  PWDPROT_LED_OFF	(PWDPROT_LED_PORT=PWDPROT_LED_PORT&(~PWDPROT_LED_MASK))
#define  PWDPROT_LED_ON		(PWDPROT_LED_PORT=PWDPROT_LED_PORT|PWDPROT_LED_MASK)




/*
 *  Define the port and bit used for the switches.
 */
#define  SW_PORT			PORTC
#define  SW_DDR				DDRC
#define  SW_PIN				PINC

#define  SW_LOCK_BIT		0
#define  SW_UNLOCK_BIT		1
#define  SW_PWD_BIT			2

#define  SW_LOCK_MASK		(1<<SW_LOCK_BIT)
#define  SW_UNLOCK_MASK		(1<<SW_UNLOCK_BIT)
#define  SW_PWD_MASK		(1<<SW_PWD_BIT)
#define  SW_ALL_MASK		(SW_LOCK_MASK | SW_UNLOCK_MASK | SW_PWD_MASK)


/*
 *  Define macros for selecting/deselecting the SD card.
 *  Using macros saves two bytes per invocation.
 */
#define  SELECT()			SD_CS_PORT=SD_CS_PORT&~SD_CS_MASK
#define  DESELECT()			SD_CS_PORT=SD_CS_PORT|SD_CS_MASK
	

/*
 *  Define bit masks for fields in the lock/unlock command (CMD42) data structure
 */
#define  SET_PWD_MASK		(1<<0)
#define  CLR_PWD_MASK		(1<<1)
#define  LOCK_UNLOCK_MASK	(1<<2)
#define  ERASE_MASK			(1<<3)



/*
 *  Global variables
 *
 *  Assign all global variables to the .noinit section.  This means the linker
 *  will not add initialization code to zero these variables.  This also means
 *  your code can NOT assume a global variable has been cleared before use!
 */
uint8_t					csd[16]			__attribute__ ((section(".noninit")));
uint8_t					cardstatus[2]	__attribute__ ((section(".noninit")));		// updated by ReadLockStatus
uint8_t					pwdswcount		__attribute__ ((section(".noninit")));
uint8_t					wrtswcount		__attribute__ ((section(".noninit")));

/*
 *  Define the password.  This can be up to 16 bytes long, and it may be binary
 *  (not necessarily printable ASCII).  To define it as binary, use something
 *  like:
 *
 *  const char       GlobalPWDStr[] = {0x12, 0x34, 0x56, ... 0xff};
 */
const char						GlobalPWDStr[] PROGMEM = " 3Dw4rd 5n0wdEN";
#define  GLOBAL_PWD_LEN			(sizeof(GlobalPWDStr))

/*
 *  For reference, here is the password I used for the original sdlocker
 *  project; see my webpage: www.seanet.com/~karllunt
 *
 *  char						GlobalPWDStr[16] PROGMEM = 
 *								{'F', 'o', 'u', 'r', 't', 'h', ' ', 'A',
 *								 'm', 'e', 'n', 'd', 'm', 'e', 'n', 't'};
 */



/*
 *  Local functions
 */
static uint8_t					xchg(uint8_t  c);
static void						SDInit(void);
static void  					ReadCSD(void);
static void						WriteCSD(void);
static void						ModifyPWD(uint8_t  mask);
static void						ReadSwitches(void);
static void						ShowStatus(void);
static int8_t					CheckPWDState(void);
static int8_t					CheckWRTState(void);
static  int8_t					sd_send_command(uint8_t  command, uint32_t  arg);
static uint8_t 					AddByteToCRC(uint8_t  crc, uint8_t  b);



int  main(void)
{
	uint8_t			swval;

/*
 *  Set up the hardware lines and ports associated with accessing the SD card.
 */
	SD_CS_DDR = SD_CS_DDR | SD_CS_MASK;		// make CS line an output
	DESELECT();								// always start with SD card deselected
	_delay_ms(250);							// let card power up

	SPI_PORT = SPI_PORT | ((1<<MOSI_BIT) | (1<<SCK_BIT));	// drive outputs to the SPI port
	SPI_DDR = SPI_DDR | ((1<<MOSI_BIT) | (1<<SCK_BIT));		// make the proper lines outputs
	SPI_PORT = SPI_PORT | (1<<MISO_BIT);					// turn on pull-up for DI

/*
 *  Set up the hardware line and port for accessing the LEDs.
 */
	PWDPROT_LED_OFF;										// start with output line low
	PWDPROT_LED_DDR = PWDPROT_LED_DDR | PWDPROT_LED_MASK;	// make the LED line an output
	WRTPROT_LED_OFF;									// start with output line low
	WRTPROT_LED_DDR = WRTPROT_LED_DDR | WRTPROT_LED_MASK;	// make the LED line an output

/*
 *  Set up the switch lines for input.
 */
	ADMUX = ((1<<REFS0) | (1<<ADLAR));		// Avcc as ref, read ADC0 (PC0), left-shift results
	DIDR0 = (1<<ADC0D);						// disable digital input buffer on PC0
	ADCSRA = ((1<<ADEN) | (1<<ADSC) | (1<<ADATE) | (1<<ADPS2) | (1<<ADPS1));

	SPI_ENABLE;
	SDInit();					// we know the card is present, init it

	pwdswcount = DEBOUNCE_COUNT;
	wrtswcount = DEBOUNCE_COUNT;

	ShowStatus();

	while (1)
	{
		_delay_ms(DEBOUNCE_DELAY_MS);
		ReadSwitches();
	}
	return  0;						// should never happen
}


static void  ShowStatus(void)
{
	int8_t				pwd;
	volatile int8_t				wrt;

	pwd = CheckPWDState();
	wrt = CheckWRTState();
	SPI_DISABLE;
	_delay_ms(10);
	if (wrt)  WRTPROT_LED_ON;
	else	  WRTPROT_LED_OFF;
	if (pwd)  PWDPROT_LED_ON;
	else	  PWDPROT_LED_OFF;
}


static void  ReadSwitches(void)
{
	uint8_t				r;

	r = (uint8_t)ADCH;			// use only top 8 bits (bits 10:2)
	if (r > 0x60)				// both switches open
	{
#ifdef BUILD_PWD_LOCKER
		if (pwdswcount == 0)
		{
			DESELECT();				// make sure *SS is high before enabling SPI
			SPI_ENABLE;
			if (CheckPWDState())  ModifyPWD(MASK_CLR_PWD);
			else				  ModifyPWD(MASK_SET_PWD);
			ModifyPWD(MASK_LOCK_UNLOCK);
			ShowStatus();
			pwdswcount = DEBOUNCE_COUNT;
		}
#endif

#ifdef BUILD_WRT_LOCKER
		if (wrtswcount == 0)
		{
			DESELECT();				// make sure *SS is high before enabling SPI
			SPI_ENABLE;
			ReadCSD();
			csd[14] = csd[14] ^ 0x10;	// toggle bit 12 of CSD (temp lock)
			WriteCSD();
			SDInit();					// forced reinit of the card
			ShowStatus();
			wrtswcount = DEBOUNCE_COUNT;
		}
#endif
	}
	else if (r > 0x50)				// only PWDLOCK switch closed
	{
		if (pwdswcount)  pwdswcount--;
	}
	else if (r > 0x28)				// only WRTLOCK switch closed
	{
		if (wrtswcount)  wrtswcount--;
	}
}



static int8_t  CheckPWDState(void)
{
	cardstatus[0] = sd_send_command(SD_SEND_STATUS, 0);	// normal response byte (Byte 1)
	cardstatus[1] = xchg(0xff);					// card status bits (Byte 2)
	xchg(0xff);									// burn the CRC
	DESELECT();									// always deselect when done!
	return (cardstatus[1] & 0x01);
}


static int8_t  CheckWRTState(void)
{
	ReadCSD();
	return  (csd[14] & 0x10);		// return current state of write-lock (maybe)
}


/*
 *  xchg      exchange a byte of data with the SD card via host's SPI bus
 */
static  uint8_t  xchg(uint8_t  c)
{
	SPDR = c;
	while ((SPSR & (1<<SPIF)) == 0)  ;
	return  SPDR;
}



static void  SDInit(void)
{
	int					i;
	int8_t				response;

/*
 *  Begin initialization by sending CMD0 and waiting until SD card
 *  responds with In Idle Mode (0x01).  If the response is not 0x01
 *  within a reasonable amount of time, there is no SD card on the bus.
 */
	DESELECT();							// start with card CS high (deslected)
	for (i=0; i<10; i++)				// send several clocks while card power stabilizes
		xchg(0xff);

	for (i=0; i<0x10; i++)
	{
		response = sd_send_command(SD_GO_IDLE, 0);	// send CMD0 - go to idle state
		if (response == 1)  break;
	}
	if (response != 1)
	{
		return;
	}

	sd_send_command(SD_SET_BLK_LEN, 512);		// always set block length (CMD6) to 512 bytes

	response = sd_send_command(SD_SEND_IF_COND, 0x1aa);	// probe to see if card is SDv2 (SDHC)
	if (response == 0x01)						// if card is SDHC...
	{
		for (i=0; i<4; i++)						// burn the 4-byte response (OCR)
		{
			xchg(0xff);
		}
		for (i=20000; i>0; i--)
		{
			response = sd_send_command(SD_ADV_INIT, 1UL<<30);
			if (response == 0)  break;
		}
	}
	else
	{
		response = sd_send_command(SD_READ_OCR, 0);
		if (response == 0x01)
		{
			for (i=0; i<4; i++)					// OCR is 4 bytes
			{
				xchg(0xff);					// burn the 4-byte response (OCR)
			}

			for (i=20000; i>0; i--)
			{
				response = sd_send_command(SD_INIT, 0);
				if (response == 0)  break;
			}
			sd_send_command(SD_SET_BLK_LEN, 512);
		}
	}

	xchg(0xff);								// send 8 final clocks

/*
 *  At this point, the SD card has completed initialization.  The calling routine
 *  can now increase the SPI clock rate for the SD card to the maximum allowed by
 *  the SD card (typically, 20 MHz).
 */
}



#if 0
static int8_t  SDInit(void)
{
	int					i;
	int8_t				response;

	sdtype = SDTYPE_UNKNOWN;			// assume this fails
/*
 *  Begin initialization by sending CMD0 and waiting until SD card
 *  responds with In Idle Mode (0x01).  If the response is not 0x01
 *  within a reasonable amount of time, there is no SD card on the bus.
 */
	DESELECT();							// start with card CS high (deslected)
	for (i=0; i<10; i++)				// send several clocks while card power stabilizes
		xchg(0xff);

	for (i=0; i<0x10; i++)
	{
		response = sd_send_command(SD_GO_IDLE, 0);	// send CMD0 - go to idle state
		if (response == 1)  break;
	}
	if (response != 1)
	{
		return  SDCARD_NO_DETECT;
	}

	sd_send_command(SD_SET_BLK_LEN, 512);		// always set block length (CMD6) to 512 bytes

	response = sd_send_command(SD_SEND_IF_COND, 0x1aa);	// probe to see if card is SDv2 (SDHC)
	if (response == 0x01)						// if card is SDHC...
	{
		for (i=0; i<4; i++)						// burn the 4-byte response (OCR)
		{
			xchg(0xff);
		}
		for (i=20000; i>0; i--)
		{
			response = sd_send_command(SD_ADV_INIT, 1UL<<30);
			if (response == 0)  break;
		}
		sdtype = SDTYPE_SDHC;
	}
	else
	{
		response = sd_send_command(SD_READ_OCR, 0);
		if (response == 0x01)
		{
			for (i=0; i<4; i++)					// OCR is 4 bytes
			{
				xchg(0xff);					// burn the 4-byte response (OCR)
			}

			for (i=20000; i>0; i--)
			{
				response = sd_send_command(SD_INIT, 0);
				if (response == 0)  break;
			}
			sd_send_command(SD_SET_BLK_LEN, 512);
			sdtype = SDTYPE_SD;
		}
	}

	xchg(0xff);								// send 8 final clocks

/*
 *  At this point, the SD card has completed initialization.  The calling routine
 *  can now increase the SPI clock rate for the SD card to the maximum allowed by
 *  the SD card (typically, 20 MHz).
 */
	return  SDCARD_OK;					// if no power routine or turning off the card, call it good
}
#endif





static  void		ReadCSD(void)
{
	uint8_t			i;

	for (i=0; i<16; i++)  csd[i] = 0;

	sd_send_command(SD_SEND_CSD, 0);
	for (i=0; i<100; i++)				// wait for card to send 0xfe (start of data)
	{
		if (xchg(0xff) == 0xfe)  break;
		_delay_ms(1);
	}

/*
 *  Not checking for 0xfe response, just assume this works.  (doh!)
 */
	for (i=0; i<16; i++)
	{
		csd[i] = xchg(0xff);
	}
	xchg(0xff);							// burn the CRC
	DESELECT();
}


static void		WriteCSD(void)
{
	int8_t				r;
	uint16_t			i;

	sd_send_command(SD_PROGRAM_CSD, 0);
	xchg(0xfe);							// send data token marking start of data block

	for (r=0; r<15; r++)				// for all 15 data bytes in CSD...
	{
   		xchg(csd[r]);					// send each byte via SPI
	}
	xchg(0x01);							// send a dummy CRC

	xchg(0xff);							// ignore dummy checksum
	xchg(0xff);							// ignore dummy checksum

	i = 0xffff;							// max timeout
	while (!xchg(0xff) && (--i))  ;		// wait until we are not busy
	DESELECT();
}



static void  ModifyPWD(uint8_t  mask)
{
	int8_t						r;
	uint16_t					i;

	mask = mask & 0x07;					// top five bits MUST be 0, do not allow forced-erase!
	r = sd_send_command(SD_LOCK_UNLOCK, 0);
	xchg(0xfe);							// send data token marking start of data block

	xchg(mask);							// always start with required command
	xchg(GLOBAL_PWD_LEN);				// then send the password length

	for (i=0; i<512; i++)				// need to send 512-byte block for CMD42
	{
		if (i < GLOBAL_PWD_LEN)			// if haven't sent entire pwd yet...
		{
			r = pgm_read_byte(&(GlobalPWDStr[i]));
		}
		xchg(r);
	}

	xchg(0xff);							// ignore dummy checksum (required!)
	xchg(0xff);							// ignore dummy checksum (required!)

	i = 0xffff;							// max timeout
	while (!xchg(0xFF) && (--i))  ;		// wait until we are not busy

	DESELECT();							// always deselect when done!
}


/*
 *  ==========================================================================
 *
 *  sd_send_command      send raw command to SD card, return response
 *
 *  This routine accepts a single SD command and a 4-byte argument.  It sends
 *  the command plus argument, adding the appropriate CRC.  It then returns
 *  the one-byte response from the SD card.
 *
 *  For advanced commands (those with a command byte having bit 7 set), this
 *  routine automatically sends the required preface command (CMD55) before
 *  sending the requested command.
 *
 *  Upon exit, this routine returns the response byte from the SD card.
 *  Possible responses are:
 *    0xff	No response from card; card might actually be missing
 *    0x01  SD card returned 0x01, which is OK for most commands
 *    0x?? 	other responses are command-specific
 */
static  int8_t  sd_send_command(uint8_t  command, uint32_t  arg)
{
	uint8_t				response;
	uint8_t				i;
	uint8_t				crc;

	if (command & 0x80)					// special case, ACMD(n) is sent as CMD55 and CMDn
	{
		command = command & 0x7f;		// strip high bit for later
		response = sd_send_command(CMD55, 0);	// send first part (recursion)
		if (response > 1)  return response;
	}

	DESELECT();
	xchg(0xff);
	SELECT();							// enable CS
	xchg(0xff);

    xchg(command | 0x40);				// command always has bit 6 set!
	xchg((unsigned char)(arg>>24));		// send data, starting with top byte
	xchg((unsigned char)(arg>>16));
	xchg((unsigned char)(arg>>8));
	xchg((unsigned char)(arg&0xff));
	crc = 0x01;							// good for most cases
	if (command == SD_GO_IDLE)  crc = 0x95;			// this will be good enough for most commands
	if (command == SD_SEND_IF_COND)  crc = 0x87;	// special case, have to use different CRC
    xchg(crc);         					// send final byte                          

	for (i=0; i<10; i++)				// loop until timeout or response
	{
		response = xchg(0xff);
		if ((response & 0x80) == 0)  break;	// high bit cleared means we got a response
	}

/*
 *  We have issued the command but the SD card is still selected.  We
 *  only deselect the card if the command we just sent is NOT a command
 *  that requires additional data exchange, such as reading or writing
 *  a block.
 */
	if ((command != SD_READ_OCR) &&
		(command != SD_SEND_CSD) &&
		(command != SD_SEND_STATUS) &&
		(command != SD_SEND_IF_COND) &&
		(command != SD_LOCK_UNLOCK) &&
		(command != SD_PROGRAM_CSD))
	{
		DESELECT();							// all done
		xchg(0xff);							// close with eight more clocks
	}

	return  response;					// let the caller sort it out
}

