/*
 * linux/driver/char/gs4500.c
 *
 *      Scanner driver for GS4500: version 2.0
 *      26.2.97 Pedro Maia Alves (l41096@alfa.ist.utl.pt)
 *              Jan Willamowius (jan@janhh.shnet.org)
 *
 *      Changes:
 *      - Minor changes to the driver to work with a GS4500A: 
 *        + In GS4500A the IRQ and DMA are selected by software so if the 
 *          IRQ, DMA or PORT are specified we don't do auto-detection.
 *      - Add a new driver mode to use the ability of the GS4500A scanner
 *        to read data without moving the scanner nor to move it, which I 
 *        called SINGLE-SHOT-MODE
 *
 *      Scanner driver for GS4500: version 1.6
 *      16.4.96 Jan Willamowius (jan@janhh.shnet.org)
 *
 *      Changes:
 *      - added interrupt handling (thanks to Rick Lyons for
 *        the patch)
 *      - updated for kernel 1.3.88
 *
 *      Scanner driver for GS4500: version 1.5
 *      3.1.96 Jan Willamowius (jan@janhh.shnet.org)
 *
 *      Changes:
 *      - moved to device major 16
 *      - updated for kernel 1.3.45 (and up) and ELF
 *      - made gcc 2.7.2 happy
 *
 *      Scanner driver for GS4500: version 1.4
 *      19.5.95 Jan Willamowius (jan@janhh.shnet.org)
 *
 *      Changes:
 *      - modified for kernels > 1.1.80 and modutils-1.1.87
 *      - safe allocation of DMA buffer (finally)
 *      - added request_region()
 *
 *      Scanner driver for GS4500: version 1.3
 *      14.12.94 Jan Willamowius
 *
 *      Changes:
 *      - Check if system has more than 8 MB ram on module load
 *        and give a warning.
 *      - updated request_dma for newer kernels
 *      - fixed cli()/sti()
 *
 *      Scanner driver for GS4500: version 1.2
 *      5.10.94 Jan Willamowius
 *
 *      Changes:
 *      - changed buffer allocation to make
 *        loadable module work
 *      - changed ioctls to be compatible with M105, AC4096 and
 *        Logitech driver
 *
 *
 *      Scanner driver for GS4500: version 1.1
 *      1.10.94 Jan Willamowius
 *
 *      Changes:
 *      - added automatic detection what io adresses and
 *        dma channel is set on the interface board
 *      - fixed a bug with dma channel selection
 *      - scanner dosn't flash when driver loads
 *      - changed device major to 26 like the other handheld
 *        scanner drivers use (avoids conflict with ibcs and
 *        sound driver)
 *      - added fsync to file_operations table (= NULL)
 *      - adapted scandemo from logitech driver for gs4500
 *      - added the option to create a loadable kernel module
 *      - small changes here and there...
 *
 *      Supported Scanner:
 *      - Genius GeniScan GS4500
 *      - Genius GeniScan GS4500A
 *      ? Genius GeniScan GS4000 (DOS drivers for the GS4000 work
 *        with my GS4500 so I'd assume this driver works with a
 *        GS4000)
 *
 *      
 *      Scanner driver for GS4500: version 1.0
 *
 *      (C) Copyright 1992,1993 Richard Lyons
 *            pclink@qus102.qld.tne.oz.au
 *            rick@razorback.brisnet.org.au
 *
 */


#ifdef MODULE
#include <linux/module.h>
#else /* provide dummies */
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif

#include "scanner.h"
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/ioport.h>
#include <linux/wait.h>
#include <linux/timer.h>
#include <linux/fcntl.h>

#include <asm/dma.h>
#include <asm/segment.h>

#ifdef GS4500A
#define SCANNER_NAME    "Genius GS4500A"
#else
#define SCANNER_NAME    "Genius GS4500"
#endif

#ifndef SCANNER_MAJOR
#define	SCANNER_MAJOR	16	/* major device */
#endif

#define BUFFER_SIZE	2048	/* size of buffer for scanned data */

struct ioports {
  int data;
  int status;
  int control;
  int dmahold;
};

static struct ioports gs_port[] =
{
  {0x272, 0x273, 0x27a, 0x27b},
  {0x2e1, 0x2e2, 0x2e3, 0x2e4},
  {0x371, 0x372, 0x373, 0x374},
  {0x3e1, 0x3e2, 0x3e3, 0x3e4}
};

static int gs_dma = -1, gs_irq = -1, gs_data = -1;
static int gs_status = 0, gs_control = 0, gs_dmahold = 0;

static int isopen = 0;
static int resolution = -1;
static int dpi[] = {100, 200, 300, 400};
static int bytesperline[] = {52, 104, 157, 206};
static int gs_mode[] = {0xF0, 0xA0, 0x80, 0xE0};
static unchar ctrlbyte;

#ifdef GS4500A
static int bytesperlinesingle[] = {121, 121, 181, 242};
#endif


static struct modeinfo_struct scan_info;

static struct scanner_capabilities scc_gs4500 =
{

  SCC_HARDSWITCH,		/* O_monochrome mode */
  SCC_HARDSWITCH,		/* O_greyscale  mode */
  0,				/* O_true-color mode */

  SCC_HARDSWITCH | SCC_SOFTDETECT,	/* Options for x resolution */
  SCC_HARDSWITCH | SCC_SOFTDETECT,	/* Options for y resolution */

  0,				/* Options for brightness */
  {0, 0},
  0,				/* Options for contrast */
  {0, 0},
  0,				/* Options for hue */
  {0, 0},
  0,				/* Options for saturation */
  {0, 0},

  105,				/* Width in mm */
  0,				/* Length in mm (unlimited) */
  0				/* No options */
};

static struct scanner_type sctxt = {"Genius", "gs4500"};

static char *startbuf, *endbuf, *head, *tail;
static int buflen, linecnt;
static int scanning = 0, looping = 0;

static int istoosoon;
static int lightoff;

static struct wait_queue *gs4500_wait = NULL;

static void gs4500_timer (unsigned long ignored);

#define PAUSE HZ / 10
#define POLLPAUSE 1

static struct timer_list gs4500_timer_list =
{NULL, NULL, 0, 0, gs4500_timer};


/* startscan
 * This function programs the dma and tells the scanner to start 
 * scanning.
 */
static void 
startscan (int size)
{
  unsigned long flags;

  if (scanning)			/* Don't do nothing if we are already scanning */
    return;

  if (gs_dma) {
    /* Program the dma */
    save_flags (flags);		/* Save the current state here... */
    cli ();
    disable_dma (gs_dma);
    clear_dma_ff (gs_dma);
    set_dma_mode (gs_dma, DMA_MODE_READ);
    set_dma_addr (gs_dma, (unsigned int) head);
    set_dma_count (gs_dma, size);
    enable_dma (gs_dma);
    restore_flags (flags);	/* Use this instead of sti() */
  }
  else
    linecnt = 0;

  outb (0, gs_dmahold);		/* Enable DMA */

  scanning = 1;
}



/* linecomplete
 * This function should be called after a line is read. If we aren't in
 * single shot mode, start to scan the next line.
 */
static void 
linecomplete (void)
{
  int count;

  scanning = 0;

#ifdef GS4500A
  if (scan_info.sc_mode & SCM_SINGLESHOT) {
    head += scan_info.bpl * 6;
    tail += scan_info.bpl * 5;
  }
  else {
#endif
    if ((count = head - tail) < 0)
      count += buflen;

    if (buflen - count >= scan_info.bpl) {
      if ((head += scan_info.bpl) >= endbuf)
	head -= buflen;
    }
    else
      printk ("GS4500: scan overrun\n");

    startscan (scan_info.bpl);

#if GS4500A
  }
#endif
}



/* gs4500_interrupt 
 * When a line is totaly transfered from the scanner a interrupt
 * occurs. This routine wakes the program that is waiting for the
 * scanline to be completly transfered.
 */
static void
gs4500_interrupt (int irq, void *dev_id, struct pt_regs *regs)
{
  if (scanning) {
    if (gs_dma) {
      if ((inb (DMA1_STAT_REG) & (1 << gs_dma))) {
	linecomplete ();
	if (looping)
	  looping = 0;
	wake_up_interruptible (&gs4500_wait);
      }
    }
    else {
      /*
       *   This is not right.  There should be a bit somewhere that indicates
       *   That there is data still to be read from gs_data, but I can't
       *   figure out what it is.  I've tried a number of solutions,
       *   but bytes are still falling down the cracks, so I leave it
       *   to somebody with more time/better documentation to fix.
       *   This only applies to an IRQ only setup.
       */

      head[linecnt++] = inb (gs_data);
      if (linecnt == scan_info.bpl) {
	linecomplete ();
	wake_up_interruptible (&gs4500_wait);
      }
    }
  }
}




/* gs4500_timer 
 * When we are in single shot mode it must be a pause before the light 
 * is turned on again.
 */
static void
gs4500_timer (unsigned long ignored)
{
  del_timer (&gs4500_timer_list);

  if (looping) 
    gs4500_interrupt (0, NULL, NULL);

  if (looping) {
    gs4500_timer_list.expires = jiffies + POLLPAUSE;
    add_timer (&gs4500_timer_list);
  }
  if (istoosoon) {
    istoosoon = 0;
    wake_up_interruptible (&gs4500_wait);
  }
}



/* waitforscan
 * This function is called to wait for a scan line. In single shot mode
 * we must keep checking the DMA for the end of the transfer. In normal 
 * mode we just sleep until the interrupt routine wakes us up.
 */
static void
waitforscan (void)
{

  if ((scan_info.sc_mode & SCM_SINGLESHOT) || !gs_irq) {
    if (gs_dma) {
      while (!(inb (DMA1_STAT_REG) & (1 << gs_dma))) {
	if (current->signal & ~current->blocked)
	  return;
	else
	  schedule ();
      }
      linecomplete ();
    }
  }
  else {
    interruptible_sleep_on (&gs4500_wait);
  }
}





/* stopscan
 * This function is called if we want to stop scanning 
 */
static void
stopscan (void)
{
  unsigned long flags;

  if (scanning) {
    scanning = 0;

    if (gs_dma) {

      save_flags (flags);
      cli ();
      disable_dma (gs_dma);
      clear_dma_ff (gs_dma);
      restore_flags (flags);

    }

    looping = 0;

  }
}



/* gs4500_detect
 * check for interface card and checks DMA and IO settings 
 */
static int
gs4500_detect (void)
{
  int set = -1;
  int val = 0;

  if (gs_data >= 0) {

    for (set = 3; set >= 0 && gs_port[set].data != gs_data; set--);

    if (set >= 0) {

      gs_status = gs_port[set].status;
      gs_control = gs_port[set].control;
      gs_dmahold = gs_port[set].dmahold;

      check_region (gs_data, (gs_dmahold - gs_data) + 1);
      val = inb (gs_status);
      if (val == 0xff) {
	printk ("GS4500: Invalid port.\n");
	return -1;
      }
    }

  }
  else {

    for (set = 3; set >= 0; set--) {
      check_region (gs_port[set].data,
		    (gs_port[set].dmahold - gs_port[set].data) + 1);
      val = inb (gs_port[set].status);
      if (val != 0xff)
	break;
    }

    if (set >= 0) {
      gs_data = gs_port[set].data;
      gs_status = gs_port[set].status;
      gs_control = gs_port[set].control;
      gs_dmahold = gs_port[set].dmahold;
    }

  }

  if (set < 0) {
    printk ("GS4500: You must select a valid port for the scanner.\n");
    return -1;
  }

  if (val & 0x80)
    printk ("GS4500: Plug the scanner in!\n");

  if (gs_dma == -1) {

    switch (val & 0x0A) {
    case 0x00:
      printk ("GS4500: Both DMA1 and DMA3 jumpered!\n");
      return -1;
    case 0x0A:
      gs_dma = 0;
      break;
    case 0x08:
      gs_dma = 1;
      break;
    case 0x02:
      gs_dma = 3;
      break;

    }
  }
  else {
#ifdef GS4500A
    if (gs_dma != 1 && gs_dma != 3) {
      printk ("GS4500A: You must select DMA channel 1 or 3.\n");
      return -1;
    }
#endif
  }

  if (gs_irq == -1) {
    switch (val & 0x50) {
    case 0x00:
      printk ("GS4500: Both IRQ3 and IRQ5 jumpered!\n");
      return -1;
    case 0x50:
      gs_irq = 0;
      break;
    case 0x40:
      gs_irq = 3;
      break;
    case 0x10:
      gs_irq = 5;
      break;
    }
  }
  else {
#ifdef GS4500A
    if (gs_irq != 5 && gs_irq != 9) {
      printk ("GS4500A: You must select IRQ 5, 9.\n");
      return -1;
    }
#endif
  }

  if (gs_irq == 0 && gs_dma == 0) {
    printk ("GS4500: no way to get data from scanner.\n");
    return -1;
  }

  ctrlbyte = (gs_dma == 3 ? 2 : 0) | (gs_irq == 9 ? 4 : 0);

  return 0;			/* -1 means no device found */
}


/* gs4500_resolution
 * Checks which resolution is selected on the scanner
 */
static int
gs4500_resolution (void)
{
  int loops, tries;
  int cc = 0;

  outb (ctrlbyte | 1, gs_control);	/* Turn the scanner on */
  outb (1, gs_dmahold);		/* Disables DMA */
  for (tries = 4; --tries;) {
    /* wait until AR is 1 fail after MANY reads */
    for (loops = 100000; --loops && !(inb (gs_status) & 0x80););
    /* wait until AR is 0 fail after MANY reads */
    if (loops) {
      for (loops = 100000; --loops && ((cc = inb (gs_status)) & 0x80););
      if (cc == inb (gs_status))
	break;
    }
  }
  outb (ctrlbyte, gs_control);	/* Turn the scanner off */
  return (tries && loops) ? cc : -EIO;
}




/* gs4500_open
 * Opening the scanner file 
 */
static int
gs4500_open (struct inode *inode, struct file *file)
{
  int cc;

  if ((file->f_flags & O_ACCMODE) != O_RDONLY)
    return -EINVAL;

  if (isopen)
    return -EBUSY;

  isopen = 1;
  if ((cc = gs4500_resolution ()) < 0) {
    isopen = 0;
    return cc;
  }

  switch (cc & 0xA4) {		/* determine dpi */
  case 0x24:
    resolution = 0;
    break;
  case 0x20:
    resolution = 1;
    break;
  case 0x04:
    resolution = 2;
    break;
  case 0x00:
    resolution = 3;
    break;
  default:
    isopen = 0;
    return -EIO;
  }

  scan_info.xdpi = scan_info.ydpi = dpi[resolution];
  scan_info.depth = 1;
  scan_info.left = scan_info.right = 0;
  scan_info.top = scan_info.bottom = 0;
  scan_info.bright = scan_info.contrast = 0;
  scan_info.hue = scan_info.sat = 0;
  scan_info.options = 0;
  scan_info.sc_mode = SCM_MONO;

  scan_info.bpl = bytesperline[resolution];
  buflen = ((BUFFER_SIZE / scan_info.bpl) * scan_info.bpl);

  head = tail = startbuf;
  endbuf = startbuf + buflen;
  MOD_INC_USE_COUNT;

  istoosoon = 1;
  gs4500_timer_list.expires = jiffies + PAUSE;
  add_timer (&gs4500_timer_list);

  lightoff = 1;

  return 0;
}


/* gs4500_release
 * Closing the scanner file 
 */
static void
gs4500_release (struct inode *inode, struct file *file)
{
  if (isopen) {
    outb (ctrlbyte, gs_control);

    stopscan ();
    isopen = 0;
    MOD_DEC_USE_COUNT;
  }

  del_timer (&gs4500_timer_list);

}


/*
 *    gs4500_read() and gs4500_ioctl() assume that they are only
 *      called after a successful gs4500_open().
 */


/* gs4500_read_normal 
 * Reading in normal mode. The image is transfered as the scanner moves. If
 * it's the first read, turn the light on. 
 */
static int
gs4500_read_normal (struct inode *inode, struct file *file, char *buf, int size)
{
  int scanned = 0;
  int chars, scancount;

  if (istoosoon) {
    del_timer (&gs4500_timer_list);
    istoosoon = 0;
  }

  while (size) {

    if ((scancount = head - tail) < 0)
      scancount += buflen;

    if (file->f_flags & O_NONBLOCK) {
      if (!scancount)
	return -EAGAIN;
      if (size > scancount)
	size = scancount;
    }

    while (!scancount) {

      if (lightoff) {
	outb (ctrlbyte | gs_mode[resolution] | 1, gs_control);
	outb (1, gs_dmahold);	/* disable DMA */
	lightoff = 0;
      }

      startscan (scan_info.bpl);
      waitforscan ();

      if (current->signal & ~current->blocked)
	return -EAGAIN;

      if ((scancount = head - tail) < 0)
	scancount += buflen;
    }

    if (tail + scancount > endbuf)
      scancount = endbuf - tail;

    chars = scancount < size ? scancount : size;
    memcpy_tofs (buf, tail, chars);
    if ((tail += chars) >= endbuf)
      tail -= buflen;
    buf += chars;
    scanned += chars;
    size -= chars;
  }
  return scanned;
}


#ifdef GS4500A

/* gs4500_read_singleshot (NEW)
 * If there are no data in the buffer start scanning a new line.
 * The first five lines of scanning don't have useful information. So
 * we read six lines and ignore the first five. The buffer is not used
 * in a circular way like in the normal read because the scanner is off
 * when processing data. That's the reason we always start scanning to
 * the begining of the buffer.
 * If there is data return it.
 *
 */
static int
  gs4500_read_singleshot
  (struct inode *inode, struct file *file, char *buf, int size) {
  int scancount;

  scancount = head - tail;

  if (!scancount) {

    if (istoosoon) {
      if (file->f_flags & O_NONBLOCK)
	return -EAGAIN;
      interruptible_sleep_on (&gs4500_wait);
    }

    if (!scanning) {

      outb (ctrlbyte | 9, gs_control);	/* turn scanner on */
      outb (1, gs_dmahold);	/* disable DMA */

      head = tail = startbuf;
      startscan (scan_info.bpl * 6);

      if (file->f_flags & O_NONBLOCK)
	return -EAGAIN;

    }

    waitforscan ();

    if (current->signal & ~current->blocked)
      return -EAGAIN;

    outb (ctrlbyte, gs_control);

    istoosoon = 1;
    gs4500_timer_list.expires = jiffies + PAUSE;
    add_timer (&gs4500_timer_list);

    scancount = scan_info.bpl;

  }

  if (size > scancount)
    size = scancount;

  memcpy_tofs (buf, tail, size);
  tail += size;

  return size;
}


/* gs4500_read
 * This function is just a front end that selects from single shot mode
 * and normal mode.
 */
static int
gs4500_read (struct inode *inode, struct file *file, char *buf, int size)
{
  if (scan_info.sc_mode & SCM_SINGLESHOT)
    return gs4500_read_singleshot (inode, file, buf, size);

  return gs4500_read_normal (inode, file, buf, size);
}

#endif


/* gs4500_select
 * Make a "select" on scanner.
 * In single shot mode just sees if the is data in the buffer.
 * In normal mode if there is no data start a new scan and wait 
 * that it's ready.
 *
 */
static int
gs4500_select (struct inode *inode, struct file *file, int type, select_table * table)
{

  if (type == SEL_IN) {
    if (head != tail)
      return 1;

#ifdef GS4500A
    if (!(scan_info.sc_mode & SCM_SINGLESHOT)) {
#endif
      if (istoosoon) {
	del_timer (&gs4500_timer_list);
	istoosoon = 0;
      }

      if (lightoff) {
	outb (ctrlbyte | gs_mode[resolution] | 1, gs_control);
	outb (1, gs_dmahold);	/* disable DMA */
	lightoff = 0;
      }

      startscan (scan_info.bpl);

      if (!gs_irq && !looping) {
	looping = 1;
	gs4500_timer_list.expires = jiffies + POLLPAUSE;
	add_timer (&gs4500_timer_list);
      }

      if (table)
	select_wait (&gs4500_wait, table);

      return 0;
#ifdef GS4500A
    }
#endif
  }
  return 0;
}



static int
select_scanmode (long mode)
{
#ifdef GS4500A

  if (!(mode & SCM_SINGLESHOT)) {
    if (scan_info.sc_mode & SCM_SINGLESHOT) {
      del_timer (&gs4500_timer_list);
      lightoff = 1;
      if (scanning) {
	stopscan ();
	outb (ctrlbyte, gs_control);
      }

      scan_info.bpl = bytesperline[resolution];
      buflen = ((BUFFER_SIZE / scan_info.bpl) * scan_info.bpl);
      head = tail = startbuf;
      endbuf = startbuf + buflen;
      scan_info.sc_mode &= ~SCM_SINGLESHOT;
    }
  }
  else {
    if (!(scan_info.sc_mode & SCM_SINGLESHOT)) {
      lightoff = 1;
      if (scanning) {
	stopscan ();
	del_timer (&gs4500_timer_list);
	outb (ctrlbyte, gs_control);

	istoosoon = 1;
	gs4500_timer_list.expires = jiffies + PAUSE;
	add_timer (&gs4500_timer_list);
      }

      scan_info.bpl = bytesperlinesingle[resolution];
      head = tail = startbuf;
      scan_info.sc_mode |= SCM_SINGLESHOT;
    }
  }

  if (mode & ~SCM_SINGLESHOT)
    return -EINVAL;

  return 0;
#else
  return -EINVAL;
#endif
}


/* gs4500_ioctl
 * ioctl function for the scanner
 *
 */
static int
gs4500_ioctl (struct inode *inode, struct file *file, unsigned int cmd,
	      unsigned long arg)
{

  switch (cmd) {
  case HSIOCGOPT:
    break;			/* no options supported */
  case HSIOCGSCC:
    if (arg)
      memcpy_tofs ((char *) arg, (char *) &scc_gs4500,
		   sizeof (struct scanner_capabilities));
    break;
  case HSIOCGSCT:
    if (arg)
      memcpy_tofs ((char *) arg, (char *) &sctxt,
		   sizeof (struct scanner_type));
    break;
  case HSIOCGMOD:
    if (arg)
      memcpy_tofs ((char *) arg, (char *) &scan_info,
		   sizeof (struct modeinfo_struct));
    break;
  case HSIOCGBUF:
    break;			/* change of buffersize not supported */
  case HSIOCSBUF:
    if (arg)
      put_fs_long ((unsigned long) 1, arg);
    break;
  case HSIOCGSIB:
    if (arg)
      put_fs_long ((unsigned long) 0, arg);
    break;
  case HSIOCMODE:
    return select_scanmode (arg);
  default:
    return -EINVAL;
    break;
  }

  return 0;
}

static struct file_operations gs4500_fops =
{
  NULL,				/* lseek */
#ifdef GS4500A
  gs4500_read,			/* read */
#else
  gs4500_read_normal,
#endif
  NULL,				/* write */
  NULL,				/* readdir */
  gs4500_select,		/* select */
  gs4500_ioctl,			/* ioctl */
  NULL,				/* mmap */
  gs4500_open,			/* open */
  gs4500_release,		/* release */
  NULL				/* fsync */
};


/* gs4500_init
 *  Requests the port region, the dma and the irq based on the data
 *  returned by gs4500_detect
 */
static struct file_operations *
gs4500_init (void)
{

  if (gs4500_detect () == 0) {

    request_region (gs_data, (gs_dmahold - gs_data) + 1, "gs4500");

    if (gs_irq) {
      if (request_irq (gs_irq, gs4500_interrupt, SA_INTERRUPT,
		       "gs4500", NULL) != 0) {
	printk ("GS4500: unable to get IRQ %d\n", gs_irq);
	return NULL;
      }
    }

    if (gs_dma) {
      if (request_dma (gs_dma, "gs4500") != 0) {
	printk ("GS4500: unable to get DMA %d\n", gs_dma);
	return NULL;
      }
    }

    printk ("GS4500: ");
    if (gs_dma)
      printk ("DMA %d ", gs_dma);
    if (gs_irq)
      printk ("IRQ %d ", gs_irq);
    printk ("at at 0x%04x - 0x%04x\n", gs_data, gs_dmahold);

    /* The scanner must be turned on on initialization to select
     * the DMA and the IRQ. */

    outb (ctrlbyte | 1, gs_control);
    outb (1, gs_dmahold);	/* disable DMA */
    outb (ctrlbyte, gs_control);

    return &gs4500_fops;
  }
  else {
    printk ("GS4500: no usable scanner found.\n");
    return NULL;
  }
}

#ifdef MODULE

/* init_module
 * Module initialization
 */

int
init_module (void)
{
  struct file_operations *f;

  printk ("Genius GS4500 scanner driver, version 2.0\n");
#ifdef GS4500A
  printk ("GS4500: using GS4500A extentions.\n");
#endif

  if ((f = gs4500_init ()) == NULL) {
    printk ("GS4500: Aborting module initialization\n");
    return -EIO;
  }

  if (register_chrdev (SCANNER_MAJOR, SCANNER_NAME, f)) {
    printk ("GS4500: cannot register major number %d\n", SCANNER_MAJOR);
    return -EIO;
  }

  startbuf = (char *) kmalloc (BUFFER_SIZE, GFP_KERNEL | GFP_DMA);
  if (startbuf == NULL) {
    printk ("GS4500: cannot allocate buffer");
    return -EIO;
  }

  return 0;
}


/* cleanup_module
 * Cleaning up the module
 */
void
cleanup_module (void)
{
  if (MOD_IN_USE)
    printk ("GS4500: busy, remove delayed\n");
  else {
    printk ("Removing Genius GS4500 driver from memory\n");

    unregister_chrdev (SCANNER_MAJOR, SCANNER_NAME);
    release_region (gs_data, (gs_dmahold - gs_data) + 1);
    kfree_s (startbuf, BUFFER_SIZE);
    if (gs_dma)
      free_dma (gs_dma);
    if (gs_irq)
      free_irq (gs_irq, NULL);
  }
}

#else /* part of the kernel */

unsigned long
scanner_init (unsigned long kmem_start)
{
  struct file_operations *f;

  if ((f = gs4500_init ()) != NULL) {
    if (register_chrdev (SCANNER_MAJOR, SCANNER_NAME, f)) {
      printk ("GS4500: cannot register major number %d\n", SCANNER_MAJOR);
      return kmem_start += BUFFER_SIZE;
    }

    return kmem_start += BUFFER_SIZE;
  }

}
#endif
