/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*    Kurt Maerker       Kurt_Maerker@vnet.ibm.com                          */
/*                                                              03/03/97    */
/*  Command_S.cmd  -  TCP/IP Socket Server with sessions                    */
/*                    A basic command processor                             */
/*                    implementing DIR, TYPE, and QUIT commands             */
/*                    * unknown commands are handled                        */
/*                    with additional HELLO command                         */
/*                    * the list of valid commands is maintained automatic. */
/*                                                                          */
/*--------------------------------------------------------------------------*/

aCmdServer = .CommandServer~new(1922)  /* create command server at port 1922*/
aCmdServer~startAccepting              /* method inherited from "tcpServer" */    
aCmdServer~shutdown                    /* shutdown the server               */                    

/****************************************************************************/

::REQUIRES "servers.frm"               /* Load the servers framework        */   

/*--------------------------------------------------------------------------*/
/* CommandServer Class definition                                           */
/*--------------------------------------------------------------------------*/
::CLASS CommandServer SUBCLASS tcpServer PUBLIC

/*--------------------------------------------------------------------------*/
::METHOD NewClient UNGUARDED           /* Over-write superclass method      */    
  expose sSocket port 
  use arg cSocket                      /* Input parameter: a client socket  */

                                       /* Create a session at server        */ 
  session = .Session~new(cSocket)
  session~start(logon)                 /* and start it                      */

                                       /* NewClient must return NOTHING!    */                                   
  return                               /* otherwise it blocks accepting     */
                                       /* new clients!                      */
                                          
/*--------------------------------------------------------------------------*/
/* Session Class definition                                                 */
/*--------------------------------------------------------------------------*/
::CLASS Session

/*--------------------------------------------------------------------------*/
::METHOD init
  expose cSocket Platform NL
  use arg cSocket                      /* Parameter: client socket          */

  platform = cSocket~ReceiveData       /* Receive platform id from client   */
  if platform = 'LINUX' then
    NL = x2c('0a')                     /* 'new line' character for UNIX     */   
  else
    NL = x2c('0d0a')                   /* 'new line' characters for nonUNIX */   
                                       
/*--------------------------------------------------------------------------*/
::METHOD logon 
  expose cSocket Platform NL

                                       /* create a command processor        */
  processor = .CommandProcessor~new(cSocket, Platform, NL)

  do forever                           /* Session handling loop             */ 
    CommandLine = cSocket~ReceiveData  /* Receive command line from client  */
    if \cSocket~stillOpen then leave   /* if problems, end client session   */ 
                                       /* parse into command and options    */
    Parse Var CommandLine Command Option
    Command = Command~translate        /* translate command to upper and    */
    Option = option~strip              /* strip off leading/trailing blanks */

    say "Processing" Command "with option:" option "(client:" cSocket~HostName ||")"
    cSocket~SendData(">>>Begin_of_transmission<<<" || NL)
    cSocket~SendData("Processing" Command "with option:" option || NL ) 

                                       /* Create message for the processor  */
    Message = .message~new(Processor, Command, 'I', Option)
    Message~send                       /* now send it to him                */
                                       /* Send end of answer back to client */
    if Message~result = ">>>End_of_session<<<" then do
      cSocket~SendData(Message~result) 
      leave
    end                                /* Initiate server shutdown          */ 
    cSocket~SendData(">>>End_of_transmission<<<")
  end

  cSocket~Close                        /* Close client's socket             */  

/*--------------------------------------------------------------------------*/
/* CommandProcessor Class definition                                        */
/*--------------------------------------------------------------------------*/
::CLASS CommandProcessor

/*--------------------------------------------------------------------------*/
::METHOD init                          /* Command processor initialisation  */
  expose cSocket Platform NL
  use arg cSocket, Platform, NL        /* set client socket and 'new line'  */

/*--------------------------------------------------------------------------*/
::METHOD Dir                           /* Method handling the DIR command   */ 
  expose cSocket NL Platform
  use arg option

  if Platform = "LINUX" then           /* determine platform specific list  */
    "ls" option "| RXQUEUE"
  else
    "DIR" option "| RXQUEUE"           /* route the result to RXqueue       */

  do while QUEUED()>0                  /* read from RXqueue until it's empty*/
    parse pull line
    cSocket~SendData(line || NL)       /* send each line to the client      */
  end 

  return ">>>Continue<<<"              /* the session can continue          */

/*--------------------------------------------------------------------------*/
::METHOD Type                          /* Method handling the type command  */  
  expose cSocket NL                    
  use arg file

  if file = "" then                    /* send message to client            */
    cSocket~SendData("You have to specify a filename with TYPE" NL) 

  else do
    stream = .stream~new(file)         /* create a stream object for file   */

    if stream~query("EXISTS") = "" then/* doeas file exist?                 */ 
      cSocket~SendData("The file '"|| file || "' does not exist" NL) 
    else do
      line = stream~linein             /* now read stream line by line      */  
      do while line \= ""
        cSocket~SendData(line || NL)   /* and send it to the client         */
        line = stream~linein
      end
    end 
  end 

  return ">>>Continue<<<"              /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD Hello                         /* Method handling hello command     */ 
  expose cSocket NL                     
                                       /* Do whatever is necessary for this */
  thisHost = .tcpHost~new
  if thisHost~name = cSocket~Hostname then 
    cSocket~SendData("It's me," thisHost~name || NL)
  else 
    cSocket~SendData("Hello" cSocket~HostName || "! My name is" thisHost~name || NL)
  return ">>>Continue<<<"              /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD Quit                          /* Method handling the quit command  */
  expose cSocket NL                     
  return ">>>End_of_session<<<"

/*--------------------------------------------------------------------------*/
::METHOD Unknown                       /* Method handling all unknown cases */
  expose cSocket NL
  use arg Command
                                       /* Were there socket problems?       */ 
  if Command < 0 & command~length > 0 then do  
    Say "Lost client on socket" cSocket "; session closed"
    return ">>>End_of_session<<<"      /* tell the session to end           */ 
  end
  else do                              /* Signal command is unsupported     */
    if arg() = 1 then                  /* Immediate Unknown Command         */
      Command = "UNKNOWN"
    else                               /* Unsup'd command routed to unknown */
      Say "--> Actually processing UNKNOWN command"

    cSocket~SendData("Invalid command:" Command || NL) 
    cSocket~SendData("Valid commands are:" cmdlist() || NL ) 
    return ">>>Continue<<<"            /* the session can continue          */ 
  end

/*--------------------------------------------------------------------------*/
::ROUTINE cmdlist                      /* Generic fct providing cmdList     */
                                       /* define set of hidden commands     */
  hiddenCmds = .set~of("UNKNOWN", "INIT") 
                                       /* get instance methods of class     */
  commandSupplier = .CommandProcessor~methods(.nil)
  cmds = commandSupplier~index         /* get first command from supplier   */ 
  commandSupplier~next                 /* move to the next supplier entry   */ 


  do while commandSupplier~available   /* process all supplier entries      */
    cmd = commandSupplier~index        /* get command from the supplier     */
    if \hiddenCmds~hasindex(cmd) then  /* if a hidden command, skip         */
      cmds = cmds || "," cmd           /* include this command in the list  */
    commandSupplier~next               /* move to the next supplier entry   */ 
  end
  return cmds                          /* return the list of commands       */
