/*===================================================================*/
/*  REXX RPC Server-Stub                  (AIX, Linux, OS/2, WIN32)  */
/*                                                                   */
/*  establishService:      procedure expose srvSock sockList host    */
/*                                          port                     */
/*  runService:            procedure expose srvSock callTxtBuf.      */
/*                                          sockList sockTranslate.  */
/*  acceptNewClient:       procedure expose srvSock sockList         */
/*                                          sockTranslate.           */
/*  handleNastyRequest:    procedure expose sockList sockTranslate.  */
/*                                          callTxtBuf.              */
/*  shutDownService:       procedure expose sockList                 */
/*  createRPCSocket:       procedure                                 */
/*  addSocketToList:       procedure expose sockList sockTranslate.  */
/*  removeSocketFromList:  procedure expose sockList                 */
/*  netError:              procedure                                 */
/*                                                                   */
/*===================================================================*/

SIGNAL ON HALT

main:
   if RxFuncQuery('SockDropFuncs') \= 0 then do
      if RxFuncAdd('SockLoadFuncs', 'rxsock', 'SockLoadFuncs') = 0 then
         call SockLoadFuncs
      else do
         say "ERROR - couldn't register function package 'rxsock'!"
         exit 1
      end
   end


   parse arg host port              /* fetch params.                 */

    if host = '' | symbol('host') \= 'VAR' then /* host name or dotted IP addr. */
        host = 127.0.0.1;
    if port = '' | symbol('port') \= 'VAR' then /* port number of RPC service.  */
        port = 5000;

    srvSock         = ''   /* The number of the server socket, that  */
                           /* is listening for incoming connection   */
                           /* requests.                              */

    sockList        = ''   /* String with numbers of all used sockets*/

    sockTranslate.  = 0    /* Array of booleans,indexed by socket    */
                           /* number, indicating if ASCII-EBCDIC     */
                           /* translation is to be performed on that */
                           /* socket.                                */

    callTxtBuf.     = ''   /* Buffer that holds incoming messages for*/
                           /* each socket.                           */

    call initVars                            /* init global variables*/

    if establishService() < 0 then do        /* init RPC environment */
        say "sorry...couldn't establish service!"
        say "  establish-service-RC = "rc
        say "  socket interface RC  = "SockSock_Errno()
        signal bye
    end

    call runService                          /* run the service.     */

halt:
    call shutDownService            /* de-initialize RPC enironment. */
bye:
    say 'good bye...'
    exit 0

/*===================================================================*/
/*                              BELOW IS YOUR CODE                   */
/*===================================================================*/
initVars:
    return

reverseString: procedure
    parse arg text;
    return reverse(text);


multiplyString_1: procedure
    parse arg nr, text;

    resText = '';
    do nr
        resText = resText||text
    end
    return resText;


multiplyString_2: procedure
    parse arg nr text;

    resText = '';
    do nr
        resText = resText||text
    end
    return resText;

/*REXX STRSRV----------------------------------------------------------------*/

/*===================================================================*/
/*                              ABOVE IS YOUR CODE                   */
/*===================================================================*/

/*___________________________________________________establishService__
 *  return values:
 *        0 : successful established service
 *       -1 : error occured
 */
establishService:

    srvSock = createRPCSocket()        /* create server socket.      */
    call addSocketToList srvSock, 0

    address.family  = "AF_INET"        /* build server address.      */
    address.port    = port
    address.addr    = host

    rc = SockBind(srvSock, address.)   /* bind the server socket to  */
    if rc <> 0 then                    /* an IP address and port.    */
        return -1

    /*  complete binding necessary for a socket to accept connections*/
    /*  and create a connection request queue for incoming requests. */

    rc = SockListen(srvSock, 128)
    if rc <> 0 then
        return -1

    say "Initialization complete!"
    say "Service is offered on  host =" host
    say "                   and port =" port
    say "Listening for clients on socket '"srvSock"'"

    return 0

/*_________________________________________________________runService__
 */
runService:

    do forever
        /* check, if there are any connection request pending or if  */
        /* any of the already accepted clients has sent some data.   */

        rList.0 = words(sockList)
        do i = 1 to rList.0
            rList.i = word(sockList, i)
        end
        nrOfReadySockets = SockSelect('rList.','','', 1)

        if nrOfReadySockets > 0 then do      /* process each socket  */
            do i = 1 to rList.0              /* that has been found  */
                rSock = rList.i              /* to be ready for      */
                if rSock = srvSock then      /* reading.             */
                    call acceptNewClient
                else
                    call handleNastyRequest rSock
            end
        end
    end
    return

/*____________________________________________________acceptNewClient__
 */
acceptNewClient:

    say '-> now trying to accept a new client...'
    clSock = SockAccept(srvSock)
    if clSock < 0 then do
        say 'WARNING - a requested client connection has been REFUSED.'
        say '   socketRC =' SockSock_ErrNo()
    end
    else do
        say '...ok - got him! New client is connected on socket 'clSock
        call addSocketToList clSock
    end

    return clSock

/*_________________________________________________handleNastyRequest__
 */
handleNastyRequest:
    parse arg sock

    say "-> now looking what client connected at socket '"sock"'",
                                                      "has sent..."

    if symbol('callTxtBuf.'sock) = 'VAR' then   /* is there already  */
        lastTxt = callTxtBuf.sock               /* something in the  */
    else                                        /* buffer?           */
        lastTxt = ''

    inTxt = ''
    rc = SockRecv(sock, 'inTxt', 1024)          /* receive next KB   */

    if rc <= 0 then do                    /* client has disconnetcted*/
        say "....hmm, suppose he's gone...shutting down the socket"
        call SockShutdown sock, 0
        call removeSocketFromList sock
        return
    end

    callTxT = lastTxt || inTxT                  /* concatenate with  */
                                                /* previous rest     */

    do while pos('~~', callTxt) > 0             /* handle calls in   */
                                                /* the message       */
        parse var callTxt cmd '(' opt ')~~' callTxt

        say "processing: call" cmd opt
        INTERPRET   "call" cmd opt";",
                    "retVal = result;"

        retVal = retVal || '~~'

        rc = SockSend(sock, retVal)
        if rc < 0 then
            call netError
    end
    callTxtBuf.sock = callTxt

    return

/*____________________________________________________shutDownService__
 */
shutDownService:

    say '--> shutting down all sockets in socketList:'
    do i = 1 to words(sockList)
        sock = word(sockList, i)
        call removeSocketFromList sock
        if SockSoClose(sock) = 0 then
            ext = 'is down.'
        else
            ext = 'had problems -> RC = 'rc
        say right(sock, 5)': 'ext
    end

    return

/*____________________________________________________createRPCSocket__
 */
createRPCSocket: procedure

   /* create a new socket for the TCP/IP protocol.
    */
    socket = SockSocket('AF_INET', 'SOCK_STREAM', 'IPPROTO_TCP')
    if rc < 0 then
        return -1

   /* 'keep-alive' ensures that the connection won't get closed
    *          even if there has been no traffic for a longer time.
    */
    rc = SockSetSockOpt(socket, 'SOL_SOCKET', 'SO_KEEPALIVE', 1)
    if rc <> 0 then
        return -1

   /* 'linger' mode makes sure that data that is still to be sent
    *          won't get lost when a closeSocket call occurs.
    *          -> it blocks the application at socket closing for
    *             the specified amount of seconds.
    */
    rc = SockSetSockOpt(socket, 'SOL_SOCKET', 'SO_LINGER', 1 0)
    if rc <> 0 then
        return -1

   /* DISabled(OFF) 'non-blocking' mode makes I/O statements block
    *          the execution of the program until they've been
    *          satisfied, because I/O has been TCP/IP-acknowledged
    *          or an optionally specified timeout has been reached.
    *
    * ENabled(ON) 'non-blocking' mode makes I/O statements return
    *          promptly a RC = -1 instead of blocking execution.
    */
    rc = SockIOCtl(socket, 'FIONBIO', 1)
    if rc <> 0 then
        return -1

    return socket

/*____________________________________________________addSocketToList__
 */
addSocketToList:
    parse arg newSocket

    sockList = sockList newSocket
    return

/*_______________________________________________removeSocketFromList__
 */
removeSocketFromList:
    parse arg remSocket

    sockList = delStr(sockList,,
                      pos(remSocket, sockList),,
                      length(remSocket) + 1)
    return

/*___________________________________________________________netError__
 */
netError:
    say copies('*', 50)
    say "Network Error occured:"
    say "  socket error nr   = "SockSock_ErrNo()
    say copies('*', 50)
    return
