/********************************************************/
/*                                                      */
/* SHOWDATE.CMD - IBM REXX Sample Program               */
/*                                                      */
/* Display the last modification date of a document in  */
/* the WWW.                                             */
/*                                                      */
/* Parameters:                                          */
/*   URL: full document specification to check          */
/*                                                      */
/* ---------------------------------------------------- */
/* (C) Copyright IBM Corp. 1997 - All Rights Reserved.  */
/*                                                      */
/* DISCLAIMER OF WARRANTIES.  The following [enclosed]  */
/* code is sample code created by IBM Corporation. This */
/* sample code is not part of any standard or IBM       */
/* product and is provided to you solely for the        */
/* purpose of assisting you in the development of your  */
/* applications.  The code is provided "AS IS", without */
/* warranty of any kind.  IBM shall not be liable for   */
/* any damages arising out of your use of the sample    */
/* code,  even if they have been advised of the         */
/* possibility of such damages.                         */
/*                                                      */
/********************************************************/
Parse Arg URL

/* Load REXX Socket library if not already loaded       */
If RxFuncQuery("SockLoadFuncs") Then
 Do
   Call RxFuncAdd "SockLoadFuncs","RXSOCK","SockLoadFuncs"
   Call SockLoadFuncs
 End

/* retrieve the header of the document specified by URL */
Header = GetHeader(URL)

If Length(Header) \= 0 Then
  Do
    /* header could be read, find date                  */
    DocDate = GetModificationDate(Header)
    Say "Document date is:" DocDate
  End
Else
  Say "Document information could not be retrieved."

Exit


/********************************************************/
/*                                                      */
/* Function:  Connect                                   */
/* Purpose:   Create a socket and connect it to server. */
/* Arguments: Server - server name, may contain port no.*/
/* Returns:   Socket number if successful, -1 otherwise */
/*                                                      */
/********************************************************/
Connect: Procedure
  Parse Arg Server

  /* if the servername has a port address specified     */
  /* then use this one, otherwise use the default http  */
  /* port 80                                            */
  Parse Var Server Server ":" Port
  If Port = "" Then
    Port = 80

  /* resolve server name alias to dotted IP address     */
  rc = SockGetHostByName(Server, "Host.!")
  If rc = 0 Then
   Do
     Say "Unable to resolve server:" Server
     Return -1
   End

  /* create a TCP socket                                */
  Socket = SockSocket("AF_INET", "SOCK_STREAM", "0")
  If Socket < 0 Then
   Do
     Say "Unable to create socket"
     Return -1
   End

  /* connect the new socket to the specified server     */
  Host.!family = "AF_INET"
  Host.!port = Port
  rc = SockConnect(Socket, "Host.!")
  If rc < 0 Then
   Do
     Say "Unable to connect to server:" Server
     Call Close Socket
     Return -1
   End

  Return Socket


/********************************************************/
/*                                                      */
/* Function:  SendCommand                               */
/* Purpose:   Send a command via the specified socket   */
/*            and return the full response to caller.   */
/* Arguments: Socket - active socket number             */
/*            Command - command string                  */
/* Returns:   Response from server or empty string if   */
/*            failed.                                   */
/*                                                      */
/********************************************************/
SendCommand: Procedure
  Parse Arg Socket, Command

  /* append two pairs of CRLF to end the command string */
  Command = Command || "0D0A0D0A"x
  BytesSent = SockSend(Socket, Command)
  Response = ""
  Do Forever
    BytesRcvd = SockRecv(Socket, "RcvData", 1024)
    If BytesRcvd <= 0 Then
      Leave
    Response = Response || RcvData
  End

  Return Response


/********************************************************/
/*                                                      */
/* Procedure: Close                                     */
/* Purpose:   Close the specified socket.               */
/* Arguments: Socket - active socket number             */
/* Returns:   nothing                                   */
/*                                                      */
/********************************************************/
Close: Procedure
  Parse Arg Socket

  Call SockShutDown Socket, 2
  Call SockClose Socket
  Return


/********************************************************/
/*                                                      */
/* Function:  GetHeader                                 */
/* Purpose:   Request the header for the specified URL  */
/*            from the network.                         */
/* Arguments: URL - fully specified document locator    */
/* Returns:   Full header of specified document or      */
/*            empty string if failed (also if no header */
/*            exists).                                  */
/*                                                      */
/********************************************************/
GetHeader: Procedure
  Parse Arg URL

  /* Isolate server name and document name, document    */
  /* name is always preceeded with a slash              */
  Parse Var URL "http://" Server "/" Document
  Document = "/" || Document

  Socket = Connect(Server)
  If Socket = -1 Then
    Return ""

  Command = "HEAD" Document "HTTP/1.0"
  Header = SendCommand(Socket, Command)
  Call Close Socket
  Return Header


/********************************************************/
/*                                                      */
/* Function:  GetModificationDate                       */
/* Purpose:   Find the last-modified date in the passed */
/*            header and return just the date.          */
/* Arguments: Header - full header of document          */
/* Returns:   Date string when document was last        */
/*            modified or empty string if date was not  */
/*            found.                                    */
/*                                                      */
/********************************************************/
GetModificationDate: Procedure
  Parse Arg Header

  /* isolate date string and strip all unwanted chars   */
  Parse Upper Var Header "LAST-MODIFIED:" ModDate "0A"x
  ModDate = Strip(ModDate)
  ModDate = Strip(ModDate,,"0D"x)

  Return ModDate
