/********************************************************/
/*                                                      */
/* URLCHECK.CMD - IBM REXX Sample Program               */
/*                                                      */
/* Check a list of WWW documents for their expiry and   */
/* create a HTML document listing the WWW documents     */
/* grouped by changed, unchanged and commented out      */
/* documents.                                           */
/*                                                      */
/* Parameters:                                          */
/*   URLList: name of file containing the URL list      */
/*   HTMLFile: name of HTML output file with doc links  */
/*                                                      */
/* ---------------------------------------------------- */
/* (C) Copyright IBM Corp. 1996 - All Rights Reserved.  */
/*                                                      */
/* DISCLAIMER OF WARRANTIES.  The following [enclosed]  */
/* code is sample code created by IBM Corporation. This */
/* sample code is not part of any standard or IBM       */
/* product and is provided to you solely for the        */
/* purpose of assisting you in the development of your  */
/* applications.  The code is provided "AS IS", without */
/* warranty of any kind.  IBM shall not be liable for   */
/* any damages arising out of your use of the sample    */
/* code,  even if they have been advised of the         */
/* possibility of such damages.                         */
/*                                                      */
/********************************************************/
Parse Arg URLList HTMLFile

/* Load REXX Socket library if not already loaded       */
If RxFuncQuery("SockLoadFuncs") Then
 Do
   Call RxFuncAdd "SockLoadFuncs","RXSOCK","SockLoadFuncs"
   Call SockLoadFuncs
 End

/* check all URLs in the specified file for expiration  */
URLS.0 = 0
Changed = ""
Unchanged = ""
Commented = ""
Call CheckURLs URLList
Call WriteHTML HTMLFile, Changed, Unchanged, Commented
Exit


/********************************************************/
/*                                                      */
/* Function:  Connect                                   */
/* Purpose:   Create a socket and connect it to server. */
/* Arguments: Server - server name, may contain port no.*/
/* Returns:   Socket number if successful, -1 otherwise */
/*                                                      */
/********************************************************/
Connect: Procedure
  Parse Arg Server

  /* if the servername has a port address specified     */
  /* then use this one, otherwise use the default http  */
  /* port 80                                            */
  Parse Var Server Server ":" Port
  If Port = "" Then
    Port = 80

  /* resolve server name alias to dotted IP address     */
  rc = SockGetHostByName(Server, "Host.!")
  If rc = 0 Then
   Do
     Say "Unable to resolve server:" Server
     Return -1
   End

  /* create a TCP socket                                */
  Socket = SockSocket("AF_INET", "SOCK_STREAM", "0")
  If Socket < 0 Then
   Do
     Say "Unable to create socket"
     Return -1
   End

  /* connect the new socket to the specified server     */
  Host.!family = "AF_INET"
  Host.!port = Port
  rc = SockConnect(Socket, "Host.!")
  If rc < 0 Then
   Do
     Say "Unable to connect to server:" Server
     Call Close Socket
     Return -1
   End

  Return Socket


/********************************************************/
/*                                                      */
/* Function:  SendCommand                               */
/* Purpose:   Send a command via the specified socket   */
/*            and return the full response to caller.   */
/* Arguments: Socket - active socket number             */
/*            Command - command string                  */
/* Returns:   Response from server or empty string if   */
/*            failed.                                   */
/*                                                      */
/********************************************************/
SendCommand: Procedure
  Parse Arg Socket, Command

  /* append two pairs of CRLF to end the command string */
  Command = Command || "0D0A0D0A"x
  BytesSent = SockSend(Socket, Command)
  Response = ""
  Do Forever
    BytesRcvd = SockRecv(Socket, "RcvData", 1024)
    If BytesRcvd <= 0 Then
      Leave
    Response = Response || RcvData
  End

  Return Response


/********************************************************/
/*                                                      */
/* Procedure: Close                                     */
/* Purpose:   Close the specified socket.               */
/* Arguments: Socket - active socket number             */
/* Returns:   nothing                                   */
/*                                                      */
/********************************************************/
Close: Procedure
  Parse Arg Socket

  Call SockShutDown Socket, 2
  Call SockClose Socket
  Return


/********************************************************/
/*                                                      */
/* Function:  GetHeader                                 */
/* Purpose:   Request the header for the specified URL  */
/*            from the network.                         */
/* Arguments: URL - fully specified document locator    */
/* Returns:   Full header of specified document or      */
/*            empty string if failed (also if no header */
/*            exists).                                  */
/*                                                      */
/********************************************************/
GetHeader: Procedure
  Parse Arg URL

  /* Isolate server name and document name, document    */
  /* name is always preceeded with a slash              */
  Parse Var URL "http://" Server "/" Document
  Document = "/" || Document

  Socket = Connect(Server)
  If Socket = -1 Then
    Return ""

  Command = "HEAD" Document "HTTP/1.0"
  Header = SendCommand(Socket, Command)
  Call Close Socket
  Return Header


/********************************************************/
/*                                                      */
/* Function:  GetModificationDate                       */
/* Purpose:   Find the last-modified date in the passed */
/*            header and return just the date.          */
/* Arguments: Header - full header of document          */
/* Returns:   Date string when document was last        */
/*            modified or empty string if date was not  */
/*            found.                                    */
/*                                                      */
/********************************************************/
GetModificationDate: Procedure
  Parse Arg Header

  /* isolate date string and strip all unwanted chars   */
  Parse Upper Var Header "LAST-MODIFIED:" ModDate "0A"x
  ModDate = Strip(ModDate)
  ModDate = Strip(ModDate,,"0D"x)

  Return ModDate


/********************************************************/
/*                                                      */
/* Procedure: CheckURLs                                 */
/* Purpose:   Check the modification dates of all URLs  */
/*            listed in the specified file. If the date */
/*            has changed, update the list file with    */
/*            the new date.                             */
/* Arguments: URLFile - file containing URL list        */
/* Returns:   nothing                                   */
/*                                                      */
/********************************************************/
CheckURLs: Procedure Expose URLS. Changed Unchanged,
                            Commented
  Parse Arg URLFile

  Index = 0
  Do While Lines(URLFile)
    /* read line with URL and last modification date    */
    URLLine = LineIn(URLFile)

    /* remember line for later update of file           */
    Index = Index + 1
    URLS.0 = Index
    URLS.Index = URLLine

    /* if first character is not a "#" then process URL */
    If SubStr(URLLine, 1, 1) \= "#" Then
     Do
       /* retrieve header for specified URL             */
       Parse Var URLLine URL ModDate
       Header = GetHeader(URL)

       If Length(Header) \= 0 Then
        Do
          /* header could be read, find date            */
          DocDate = GetModificationDate(Header)

          If Length(ModDate) = 0 | ModDate \= DocDate Then
           Do
             /* this URL has been changed, add to list  */
             /* of changed URLs and update the date     */
             Changed = Changed Index
             URLS.Index = URL DocDate
           End
          Else
            /* add index to list of unchanged URLs     */
            Unchanged = Unchanged Index
        End
       Else
         /* add index to list of unchanged URLs        */
         Unchanged = Unchanged Index
     End
    Else
      /* add index to list of all commented out URLs   */
      Commented = Commented Index
  End

  /* close input stream, erase it and then rewrite it   */
  Call Stream URLFile, "C", "CLOSE"
  "@DEL" URLFile

  Do Index = 1 To URLS.0
    Call LineOut URLFile, URLS.Index
  End
  Call Stream URLFile, "C", "CLOSE"
  Return


/********************************************************/
/*                                                      */
/* Procedure: WriteHTML                                 */
/* Purpose:   Create a new HTML document with links to  */
/*            the input URLs grouped by modification.   */
/* Arguments: HTML - output filename                    */
/*            Changed - list of changed URL indices     */
/*            Unchanged - list of unchanged URL indices */
/*            Commented - list of commented URL indices */
/* Returns:   nothing                                   */
/*                                                      */
/********************************************************/
WriteHTML: Procedure Expose URLS.
  Parse Arg HTML, Changed, Unchanged, Commented

  /* write new HTML document with links to URLs         */
  "@DEL" HTML "1>NUL 2>NUL"

  Call LineOut HTML, "<html><head>"
  Call LineOut HTML, "<title>My link list</title>"
  Call LineOut HTML, "</head><body>"

  Call LineOut HTML, "<h1>Changed documents</h1>"
  Call FormatURLList HTML, Changed
  Call LineOut HTML, "<h1>Unchanged documents</h1>"
  Call FormatURLList HTML, Unchanged
  Call LineOut HTML, "<h1>Commented documents</h1>"
  Call FormatURLList HTML, Commented

  Call LineOut HTML, "<p><i>Documents checked at",
               Date() "on" Time() "</i>"
  Call LineOut HTML, "</body></html>"
  Return


/********************************************************/
/*                                                      */
/* Procedure: FormatURLList                             */
/* Purpose:   Format a list of URL indices into a HTML  */
/*            formatted list with links to the URLs.    */
/* Arguments: HTML - output filename                    */
/*            List - list of indices                    */
/* Returns:   nothing                                   */
/*                                                      */
/********************************************************/
FormatURLList: Procedure Expose URLS.
  Parse Arg HTML, List

  /* are there any indices in the list?                 */
  If Words(List) > 0 Then
   Do
    Do Index = 1 To Words(List)
      Idx = Word(List, Index)
      Parse Var URLS.Idx URL ModDate
      URL = Strip(URL, "L", "#")

      Call LineOut HTML, "<br><a href=""" || URL || """>"
      Call LineOut HTML, URL || "</a>"
      If Length(ModDate) > 0 Then
        Call LineOut HTML, ", last modified at" ModDate
    End
   End
  Else
    Call LineOut HTML, "<p><i>no documents in list</i><p>"
  Return
