/*==========================================================================*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*  (c) Copyright IBM Corporation 1999                                      */
/*                                                                          */
/*  Ftp.frm  -  Framework providing basic FTP services                      */
/*                                                                          */
/*  Methods:                                                                */
/*    connect       - connect to remote host                                */
/*    logoff        - logoff connection                                     */
/*    binary        - set binary transfer mode                              */
/*    ascii         - set ascii transfer mode                               */
/*    ping          - ping host                                             */
/*    ls            - list (short form)                                     */
/*    dir           - list (long form)                                      */
/*    get           - get a file from remote host                           */
/*    put           - put a file to remote host                             */
/*    putunique     - put a file uniquely to remote host                    */
/*    append        - append to file previously put to remote host          */
/*    delete        - delete a remote file                                  */
/*    rename        - rename a remote file                                  */
/*    cd            - change remote directory                               */
/*    mkdir         - make a remote directory                               */
/*    rmdir         - remove a remote directory                             */
/*    proxy         - move file from one to another remote host             */
/*    pwd           - display current remote directory                      */
/*    sys           - display remote system                                 */
/*    quote         - send quote information to remote host                 */
/*    site          - send site information to remote host                  */
/*    setactivemode - set active mode on or off                             */
/*    version       - display RxFtp version                                 */
/*    errno         - provide FtpErrNo value                                */
/*    lSLASH        - local slash character                                 */
/*    rSLASH        - remote slash character                                */
/*    lSysType      - local system type                                     */
/*    rSysType      - remote system type                                    */
/*    debug         - debug mode                                            */
/*                                                                          */
/*  Public Routines:                                                        */
/*    message       - generate message                                      */
/*    ensure        - ensure a path representation                          */
/*                                                                          */
/*  Requirements:                                                           */
/*    RxFtp library (Rexx interface to TCP Ftp library)                     */
/*                                                                          */
/*==========================================================================*/
                                       /* Load RxFtp interface              */
if RxFuncQuery("FtpLoadFuncs") then do
  rc = RxFuncAdd("FtpLoadFuncs", "rxftp", "FTPLOADFUNCS")
  rc = "FtpLoadFuncs"()
end
                                       /* FtpMsg pool definition            */
ftpMsg = .directory~new
                                       -- Ftp messages
ftpMsg[FTPSERVICE] = "unknown service"
ftpMsg[FTPHOST] = "unknown host"
ftpMsg[FTPSOCKET] = "unable to obtain socket"
ftpMsg[FTPCONNECT] = "unable to connect to server"
ftpMsg[FTPLOGIN] = "login failed"
ftpMsg[FTPABORT] = "transfer stopped"
ftpMsg[FTPLOCALFILE] = "problem opening local file"
ftpMsg[FTPDATACONN] = "problem initializing connection"
ftpMsg[FTPCOMMAND] = "command failed"
ftpMsg[FTPPROXYTHIRD] = "proxy server does not support third party transfers"
ftpMsg[FTPNOPRIMARY] = "no primary connection for proxy transfer"
                                       -- Ping messages
ftpMsg[PINGREPLY] = "host does not reply"
ftpMsg[PINGSOCKET] = "unable to obtain socket"
ftpMsg[PINGPROTO] = "unknown protocol ICMP"
ftpMsg[PINGSEND] = "send failed"
ftpMsg[PINGRECV] = "receive failed"
ftpMsg[PINGHOST] = "unknown host"

.local~ftpMsg = ftpMsg                 -- put it in local directory

/*--------------------------------------------------------------------------*/
/* tcpFtp Class definition                                                  */
/*--------------------------------------------------------------------------*/
::CLASS tcpFtp public

/*--------------------------------------------------------------------------*/
::METHOD init                          /* tcpFtp initializer                */
  expose descriptor notClosed

  notClosed = 0
  descriptor = '<userid>@<host>'
  parse source system .                -- determine local characteristics
  if system = 'AIX' | system = 'LINUX' then do
    self~lSlash = '/'
    self~lSysType = 'UNIX'
  end
  else do
    self~lSlash = '\'
    self~lSysType = system
  end
  self~debug = .false                  -- debug mode is off initially

/*--------------------------------------------------------------------------*/
::METHOD connect                       /* Setup the connection              */
  expose FtpErrNo descriptor notClosed
  use arg host, userid, password, account

  if arg() < 3 then do
    say "Parameters missing; host, userid, and password have to be specified!"
    return 1                           -- parameter error
  end

  if arg(4, 'o') then                  -- define connection to user at host
    rc = ftpSetUser(host, userid, password)
  else
    rc = ftpSetUser(host, userid, password, account)
  if rc then do                        -- is connection specification valid?
    descriptor = userid'@'host

    rc = ftpSys(sysType)               -- get the remote system type
    if rc \= -1 then do
      sysType = sysType~word(1)
      self~rSysType = sysType
      if sysType = 'UNIX' then         -- determine remote slash char
        self~rSlash = '/'
      else
        self~rSlash = '\'
      rc = FtpSetBinary('Binary')      -- initialize default to binary
      notClosed = 1
    end
    else do                            -- connection problem
      if self~debug then
        say Message(FtpErrNo, 'connect' host userid)
      rc = 0
    end
  end
  return 1 - rc                        --  return 0 - ok,  1 - error

/*--------------------------------------------------------------------------*/
::METHOD string                        /* The default string                */
  expose descriptor

  return "Ftp:"descriptor

/*--------------------------------------------------------------------------*/
::METHOD lSlash ATTRIBUTE              /* The local slash character         */

/*--------------------------------------------------------------------------*/
::METHOD rSlash ATTRIBUTE              /* The remote slash character        */

/*--------------------------------------------------------------------------*/
::METHOD lSysType ATTRIBUTE            /* The local system type             */

/*--------------------------------------------------------------------------*/
::METHOD rSysType ATTRIBUTE            /* The remote system type            */

/*--------------------------------------------------------------------------*/
::METHOD debug ATTRIBUTE               /* debug mode (.true or .false)      */

/*--------------------------------------------------------------------------*/
::METHOD Binary                        /* Set binary transmission mode      */

  return 1 - ftpSetBinary('Binary')

/*--------------------------------------------------------------------------*/
::METHOD Ascii                         /* Set Ascii transmission mode       */

  return 1 - ftpSetBinary('Ascii')

/*--------------------------------------------------------------------------*/
::METHOD ping                          /* Ping a remote host                */
  expose FtpErrNo descriptor
  use arg host, length

  if arg(1, 'o') then
    host =  descriptor~substr(1 + descriptor~lastpos('@')) -- default host
  if arg(2, 'o') then
    length = 5

  value = ftpPing(host, length)
  if \value~datatype('N') then do
    say Message(value, 'ping' host)
    return -1                          -- indicate error
  end
  return value                         -- return the mean response time

/*--------------------------------------------------------------------------*/
::METHOD ls                            /* List files in current directory   */
  expose FtpErrNO
  use arg pattern                      -- search pattern

  if arg() = 0 then
    pattern = '*'
                                       -- get directory info in short format
  if ftpLs(pattern, rlist.) \= 0 then do
    if FtpErrno \= 'FTPCOMMAND' then do-- nothing found is no error!
      if self~debug then
        say Message(FtpErrno, 'ls' pattern)
      return -1
    end
  end
                                       -- put stem content into array
  return StemToArray(rlist.)           -- return the array of filenames

/*--------------------------------------------------------------------------*/
::METHOD dir                           /* List files in current directory   */
  expose FtpErrNO
  use arg pattern                      -- search pattern

  if arg() = 0 then
    pattern = '*'
                                       -- get directory info in long format
  if ftpDir(pattern, rlist.) \= 0 then do
    if self~debug then
      say Message(FtpErrno, 'dir' pattern)
    return -1
  end
                                       -- put stem content into array
  return StemToArray(rlist.)           -- return the array

/*--------------------------------------------------------------------------*/
::METHOD get                           /* Get a remote file                 */
  expose FtpErrNO
  use arg localFile, remoteFile

  if arg(2, 'o') then
    remoteFile = localFile
  if arg(3, 'o') then
    rc = ftpGet(localFile, remoteFile)
  else
    rc = ftpGet(localFile, remoteFile, arg(3))
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'get' localFile remotefile)
  return rc

/*--------------------------------------------------------------------------*/
::METHOD put                           /* Put a file to remote directory    */
  expose FtpErrNO
  use arg localFile, remoteFile

  if arg(2, 'o') then
    remoteFile = localFile
  if arg(3, 'o') then
    rc = ftpPut(localFile, remoteFile)
  else
    rc = ftpPut(localFile, remoteFile, arg(3))
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'put' localFile remoteFile)
  return rc

/*--------------------------------------------------------------------------*/
::METHOD putUnique                     /* Putunique file to remote directory*/
  expose FtpErrNO
  use arg localFile, remoteFile

  if arg(2, 'o') then
    remoteFile = localFile
  if arg(3, 'o') then
    rc = ftpPutUnique(localFile, remoteFile)
  else
    rc = ftpPutUnique(localFile, remoteFile, arg(3))
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'putunique' localFile remoteFile)
  return rc

/*--------------------------------------------------------------------------*/
::METHOD append                        /* Append a file to remote file      */
  expose FtpErrNO
  use arg localFile, remoteFile

  if arg(2, 'o') then
    remoteFile = localFile
  if arg(3, 'o') then
    rc = ftpAppend(localFile, remoteFile)
  else
    rc = ftpAppend(localFile, remoteFile, arg(3))
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'append' localFile RemoteFile)
  return rc

/*--------------------------------------------------------------------------*/
::METHOD delete                        /* Delete file from remote directory */
  expose FtpErrNO
  use arg remoteFile

  rc = ftpDelete(remoteFile)
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'delete' remotefile)
  return rc

/*--------------------------------------------------------------------------*/
::METHOD rename                        /* Rename file in remote directory   */
  expose FtpErrNO
  use arg oldName, newName

  rc = ftpRename(oldName, newName)
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'rename' oldName newName)
  return rc

/*--------------------------------------------------------------------------*/
::METHOD cd                            /* Change remote directory           */
  expose FtpErrNO
  use arg newDir

  rc = ftpChDir(newDir)
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'cd' newDir)
  return rc

/*--------------------------------------------------------------------------*/
::METHOD mkdir                         /* Make a remote directory           */
  expose FtpErrNO
  use arg newDir

  rc = ftpMkDir(newDir)
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'mkdir' newDir)
  return rc

/*--------------------------------------------------------------------------*/
::METHOD rmdir                         /* remove a remote directory         */
  expose FtpErrNO
  use arg remDir

  rc = ftpRmDir(remDir)
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'rmdir' remDir)
  return rc

/*--------------------------------------------------------------------------*/
::METHOD proxy                         /* copy file from host S to host     */
  expose FtpErrNO
  use arg host, userid, password, account, shost, suserid, spassword, saccount, file, sfile

  if arg() < 9 then do
    say Message("Parameter(s) missing; 10 are required!", 'proxy')
    return -1                          -- parameter error
  end

  if arg(10, 'o') then
    sfile = file                       -- use same name as target

  if arg(11, 'o') then                 -- Create a specific connection
    rc = ftpProxy(host, userid, password, account,,     -- target system
                  shost, suserid, spassword, saccount,, -- source system
                  file, sfile)                          -- target, source file
  else
    rc = ftpProxy(host, userid, password, account,,     -- target system
                  shost, suserid, spassword, saccount,, -- source system
                  file, sfile,,                         -- target, source file
                  arg(11))                              -- 'Binary' or 'Ascii'
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'proxy' host userid 'file' file 'from' shost suserid 'file' sfile)
  return rc

/*--------------------------------------------------------------------------*/
::METHOD pwd                           /* get current remote directory      */
  expose FtpErrNO

  if ftpPwd(curDir) \= 0 then do
    if self~debug then
      say Message(FtpErrno, 'pwd')
    return -1
  end
  return changestr('"', curDir, "")~word(1)

/*--------------------------------------------------------------------------*/
::METHOD sys                           /* get remote system type            */
  expose FtpErrNO

  rc = ftpSys(sysType)
  if rc = -1 & self~debug then
    say Message(FtpErrno, 'sys')
  return rc

/*--------------------------------------------------------------------------*/
::METHOD quote                         /* send quote string to remote host  */
  expose FtpErrNO
  use arg quote

  rc = ftpQuote(quote)
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'quote')
  return rc

/*--------------------------------------------------------------------------*/
::METHOD site                         /* send site string to remote host    */
  expose FtpErrNO
  use arg site

  rc = ftpSite(site)
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'site')
  return rc

/*--------------------------------------------------------------------------*/
::METHOD setactivemode                /* set active transfer mode           */
  expose FtpErrNO
  use arg mode

  rc = ftpSetActiveMode(mode)
  if rc \= 0 & self~debug then
    say Message(FtpErrno, 'setactivemode')
  return rc

/*--------------------------------------------------------------------------*/
::METHOD version                       /* get RxFtp version                 */
  expose FtpErrNO

  rc = ftpVersion(version)
  if rc = 0 then
    rc = version
  return rc

/*--------------------------------------------------------------------------*/
::METHOD logoff                        /* Logoff all FTP connections        */
  expose FtpErrNo notClosed

  if notClosed then do
    if ftpLogoff() \= 0 then do
      if self~debug then
        say Message(FtpErrno, 'logoff')
      return -1
    end
    notClosed = 0
  end
  return 0

/*--------------------------------------------------------------------------*/
::METHOD uninit                        /* Uninitialize the FTP connections  */
  expose FtpErrNO notClosed

  if notClosed then self~Logoff

/*--------------------------------------------------------------------------*/
::METHOD errno                         /* Provide the FtpErrNo value        */
  expose FtpErrNo
  return FtpErrNo

/*--------------------------------------------------------------------------*/
::ROUTINE StemToArray                  /* convert stem into an array        */
  use arg stem.

  array = .array~new(stem.0)           -- create a suitable array
  do i = 1 to stem.0                   -- and fill it from the stem
    array[i] = stem.i
  end
  return array                         -- return the array

/*--------------------------------------------------------------------------*/
::ROUTINE Message PUBLIC               /* generate message                  */
  use arg Errno, method

  msg = .ftpMsg[Errno]                 -- lookup the message code
  if .nil == msg then do               -- if not found among the messages
    if Errno~datatype('N') then        -- is it a number?
      msg = 'Error code:' Errno        --   display it as error code
    else
      msg = Errno                      --   display it as text
  end
  if arg(2, 'e') then
    msg = method':' msg                -- prefix the message text
  return msg

/*--------------------------------------------------------------------------*/
::ROUTINE ensure PUBLIC                /* Ensure path representation        */
  use arg path, slash                  /* any path spec, target slash       */

  if path~pos('/') = 0 then do
    if path~pos('\') \= 0 then do
      if slash \= '\' then             -- modify path
        return path~changestr('\', slash)
    end
  end
  else do
    if slash \= '/' then               -- modify path
      return path~changestr('/', slash)
  end
  return path

/*--------------------------------------------------------------------------*/
/* End of tcpFtp Class definition                                           */
/*--------------------------------------------------------------------------*/