/*==========================================================================*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*  (c) Copyright IBM Corporation 1999                                      */
/*                                                                          */
/*  xFtp.cls - An Extended FTP Support Class                                */
/*                                                                          */
/*  Remote methods:                                                         */
/*    mget             - get multiple files matching pattern                */
/*    tget             - get the whole directory tree                       */
/*    mput             - put multiple file matching pattern                 */
/*    tput             - put the whole directory tree                       */
/*    mdel             - delete multiple files matching pattern             */
/*    tdel             - delete the whole directory tree                    */
/*    getRDirList      - get remote directory list                          */
/*    genRDirTree      - generate remote directory tree                     */
/*    xcd              - create and change directory                        */
/*                                                                          */
/*  Local methods:                                                          */
/*    lcd              - local change directory and verify                  */
/*    xLcd             - local create and change directory                  */
/*    lmdel            - delete files from local directory matching pattern */
/*    ltdel            - delete the whole local directory tree              */
/*    getLDirList      - get local directory list                           */
/*    genLDirTree      - generate local directory tree                      */
/*    showLDir         - show local directory contents                      */
/*                                                                          */
/*  Private methods:                                                        */
/*    mgetd            - get multiple files returning subdirectories        */
/*    mputd            - put multiple files returning subdirectories        */
/*    mdeld            - delete multiple files returning subdirectories     */
/*    lmdeld           - delete local multiple files returning subdirs      */
/*                                                                          */
/*  Public Routines: none                                                   */
/*                                                                          */
/*  Requirements:                                                           */
/*    ftp.frm  - FTP Support Framework, a Base FTP Support Class            */
/*                                                                          */
/*==========================================================================*/

/*--------------------------------------------------------------------------*/
/*                                                                          */
/* Valid method invocations:                                                */
/*                                                                          */
/*   getRDirList([[pattern][, [mode]]])                                     */
/*   getRDirList([[rdir][, [mode]]])                                        */
/*   getLDirList([[pattern][, [mode]]])                                     */
/*   getLDirList([[rdir][, [mode]]])                                        */
/*   xcd(name)                                                              */
/*                                                                          */
/*   mget([[pattern], [rdir]])                                              */
/*   mgetd()                                                                */
/*   tget([[rdir], [ldir]])                                                 */
/*                                                                          */
/*   mput([[pattern], [rdir]])                                              */
/*   mputd()                                                                */
/*   tput([[ldir], [rdir]])                                                 */
/*                                                                          */
/*   mdel([[pattern], [rdir]])                                              */
/*   mdeld()                                                                */
/*   tdel([rdir])                                                           */
/*                                                                          */
/*   genLDirTree(pathArray)                                                 */
/*   genRDirTree(pathArray)                                                 */
/*                                                                          */
/*   lcd(path)                                                              */
/*   xLcd(path)                                                             */
/*   lmdel([pattern])                                                       */
/*   lmdeld()                                                               */
/*   ltdel([ldir])                                                          */
/*                                                                          */
/*   showLDir([[pattern], [mode]]])                                         */
/*   showLDir([[ldir], [mode]]])                                            */
/*                                                                          */
/* with:                                                                    */
/*   pattern   - search pattern using '*' as wildcard character             */
/*   rdir      - remote directory name                                      */
/*   ldir      - local directory name                                       */
/*   path      - directory path                                             */
/*   pathArray - an array od directory paths                                */
/*   mode      - 'F' - files, 'D' - directories                             */
/*   name      - simple name                                                */
/*                                                                          */
/*--------------------------------------------------------------------------*/
 
::REQUIRES "ftp.frm"

/*--------------------------------------------------------------------------*/
/* xFtp Class definition                                                    */
/*--------------------------------------------------------------------------*/
::CLASS xFtp SUBCLASS tcpFtp PUBLIC

::METHOD init                          /* Initialize xFtp base object       */
  self~init:super

/*--------------------------------------------------------------------------*/
::METHOD mget                          /* Get multiple remote files         */ 
  use arg pattern, rdir                /* optional parm 1: search pattern   */
                                       /* optional parm 2: remote directory */
  if arg(1, 'o') then                  -- default pattern
    pattern = '*'                      -- get all files

  rchange = .false                     -- no remote directory change so far

  if arg(2, 'e') then do
    curDir = self~pwd                  -- remember the current directory
    rc = self~cd(rdir)                 -- change remote directory to rdir
    if rc = -1 then signal error       -- signal problem
    rchange = .true                    -- remote directory change
  end   
    
  list = self~getRDirList(pattern, 'F')-- retrieve list of files/directories
  if list \= -1 & list \= 0 then       -- a list is received 
  do file over list                    -- go over the list  
    rc = self~get(file)                -- get this file from remote host
    if rc = -1 then leave              -- if error, abend transfer  
  end
  else rc = list  

  if rchange then do
    rx = self~cd(curDir)               -- change remote directory back                      
    if rc \= rx then rc = -1           -- propagate the error
  end

error:
  if rc = -1 then say message(self~errno, 'mget')
  return rc                            -- return success "0" or error "-1"

/*--------------------------------------------------------------------------*/
::METHOD mgetd PRIVATE                 /* Multiple get returning subdirs    */
                                
  dirs = .set~new                      -- create empty directory set

  list = self~getRDirList()            -- retrieve list of files/directories
  if list \= -1 & list \= 0 then       -- a list is received 
  do file over list                    -- go over the list                 
    if file~word(1) then do            -- check for directory   
      dir = file~word(2)               -- ... get its name,
      dirs~put(dir)                    -- ... put it in dirs
      rc = directory(dir)              -- try to change into dir
      if rc \= '' then                 -- if it existed already...  
        rc = directory("..")           -- fine, change back
      else do                          -- no, create it
        rc = sysMkDir(dir)            
        if rc \= 0 then do 
          say "Error creating local directory" dir"; code:" rc  
          return -1                    -- error 
        end
      end
      rc = 0                           -- ok, directory exists at this point              
    end  
    else do                            -- no, handle file entry 
      rc = self~get(file~word(2))      -- get this file from remote host
      if rc = -1 then leave            -- if problem, abend transfer 
    end                                
  end
  else rc = list                       -- communicate problem

  if rc \= 0 then do                   -- error case
    say message(self~ErrNo, 'mgetd')     
    return -1                          -- return error "-1"
  end                       
  if dirs~items > 0 then return dirs   -- return directory list, if any
  return 0                             -- return success "0"

/*--------------------------------------------------------------------------*/
::METHOD tget                          /* Get whole directory tree          */ 
  use arg rdir, ldir

  rchange = .false                     -- no remote directory change so far
  lchange = .false                     -- no local directory change so far

  if arg(1, 'e') then do               -- argument 1 specifies remote directory
    rc = self~cd(rdir)                 -- change remote directory to this 
    if rc = -1 then signal error       -- signal any problems
    rchange = .true                    -- remote directory change 
  end  

  if arg(2, 'e') then do               -- argument 2 specifies local directory
    if self~xlcd(ldir) then do         -- change to local directory ok?
      say message('Error creating local directory' ldir, "tget") 
      rc = -1                          -- signal problem  
      signal error                     -- abend transfer
    end 
    lchange = .true                    -- local directory change
  end
  else do                              -- no local dir specified 
    if arg(1, 'e') then do             -- if remote dir specified ...
      ldir = rdir                      -- use remote dir as default
      if self~xlcd(ldir) then do       -- change to local directory ok?
        say message('Error creating local directory' ldir, "tget") 
        rc = -1                        -- signal problem  
        signal error                   -- abend transfer  
      end 
      lchange = .true                  -- local directory change
    end
  end
   
  dirs = self~mgetd()                  -- get all files and directory list
  if dirs \= 0 & dirs \= -1 then       -- if there is a subdir list
  do dir over dirs                     -- handle all members of every subdir
    rc = self~tget(dir)                -- get all files and subdirectories
    if rc = -1 then leave              -- if error, abend transfer
  end
  else rc = dirs                       -- mget error or empty list

  if lchange then rx = directory("..") -- return to previous local directory

error:
  if rchange then self~cd("..")        -- return to previous remote directory
  if rc = -1 then say message(self~ErrNo, "tget") 
  return rc

/*--------------------------------------------------------------------------*/
::METHOD mput                          /* Put multiple files to remote dir. */ 
  use arg pattern, rdir                /* optional parm 1: search pattern   */
                                       /* optional parm 2: remote directory */
  if arg(1, 'o') then                  -- default pattern
    pattern = '*'                      -- put all files

  rchange = .false                     -- no remote directory change so far

  if arg(2, 'e') then do
    curDir = self~pwd                  -- remember the current directory
    rc = self~cd(rdir)                 -- change remote directory to rdir
    if rc = -1 then signal error       -- signal problem
    rchange = .true                    -- remote directory change
  end   
    
  rc = sysfiletree(pattern, 'flist', 'FO')  
  if rc = 0 then do                    
    if flist.0 = 0 then                -- is list empty?
      say message('Nothing to transfer', 'mput')  
    else do i = 1 to flist.0           -- handle all files in list
      fname = flist.i~substr(1 + flist.i~lastpos(self~lSLASH))        
      rc = self~put(fname)             -- put each file fname
      if rc = -1 then leave            -- check, if error abend transfer        
    end  
  end
  else rc = -1                         -- sysfiletree error

  if rchange then do
    rx = self~cd(curDir)               -- change remote directory back                      
    if rc \= rx then rc = -1           -- propagate the error
  end

error:
  if rc = -1 then say message(self~errno, 'mput')
  return rc                            -- return success "0" or error "-1"

/*--------------------------------------------------------------------------*/
::METHOD mputd PRIVATE                 /* Multiple put returning subdirs    */

  dirs = .set~new                      -- create empty directory set

  rc = sysfiletree('*', 'flist', 'FO') -- get list of local files 
  if rc = 0 then do i = 1 to flist.0   -- go over the list of files
    file = flist.i~substr(1 + flist.i~lastpos(self~lSLASH))
    rc = self~put(file)                -- put each file 
    if rc = -1 then signal error       -- if error, abend transfer
  end

  rc = sysfiletree('*', 'dlist', 'DO') -- get list of local directories 
  if rc = 0 then
  do i = 1 to dlist.0                  -- walk over the list of directories
    dir = dlist.i~substr(1 + dlist.i~lastpos(self~lSLASH))
    dirs~put(dir)                      -- put directory name into dirs
    rc = self~cd(dir)                  -- try changing to it
    if rc = 0 then do                  -- ok, it exists already
      rc = self~cd("..")               -- go back 
      if rc = -1 then leave 
    end 
    else do                            -- no, directory doesn't exist
      rc = self~mkdir(dir)             -- ... so create it
      if rc = -1 then do
        say "Error creating remote directory" dir
        leave
      end  
    end 
  end

error:   
  if rc \= 0 then do                   -- error case
    say message(self~ErrNO, 'mputd')     
    return -1                          -- return error "-1"
  end 
  if dirs~items > 0 then return dirs   -- return directory set, if any
  return 0                             -- return success "0"

/*--------------------------------------------------------------------------*/
::METHOD tput                          /* Put whole directory tree          */
  use arg ldir, rdir                   /* optional parm 1: local directory  */
                                       /* optional parm 2: remote directory */
  rchange = .false                     -- no remote directory change so far
  lchange = .false                     -- no local directory change so far

  if arg(1, 'e') then do               -- argument 1 specifies local directory
    rc = self~lcd(ldir)                -- change local directory to this 
    if rc = -1 then signal error       -- signal any problems
    lchange = .true                    -- local directory change 
  end  

  if arg(2, 'e') then do               -- argument 2 specifies remote directory
    if self~xcd(rdir) then do          -- change to remote directory ok?
      say message('Error creating local directory' rdir, "tput") 
      rc = -1                          -- signal problem  
      signal error                     -- abend transfer
    end 
    rchange = .true                    -- remote directory change
  end
  else do                              -- no remote dir specified 
    if arg(1, 'e') then do             -- if local dir specified ...
      rdir = ldir                      -- use local dir as default
      if self~xcd(rdir) then do        -- change to remote directory ok?
        say message('Error creating remote directory' rdir, "tput") 
        rc = -1                        -- signal problem  
        signal error                   -- abend transfer  
      end 
      rchange = .true                  -- remote directory change
    end
  end
   
  dirs = self~mputd()                  -- put all files and directory list
  if dirs \= 0 & dirs \= -1 then       -- if there is a subdir list
  do dir over dirs                     -- handle all members of every subdir
    rc = self~tput(dir)                -- put all files and subdirectories
    if rc = -1 then leave              -- if error, abend transfer
  end
  else rc = dirs                       -- mputd error or empty list

  if rchange then self~cd("..")        -- return to previous remote directory

error:
  if lchange then rx = directory("..") -- return to previous local directory
  if rc = -1 then say message(self~ErrNo, "tput")
  return rc

/*--------------------------------------------------------------------------*/
::METHOD mdel                          /* Delete multiple remote files      */ 
  use arg pattern, rdir                /* optional parm 1: search pattern   */
                                       /* optional parm 2: remote directory */
  rchange = .false                     -- no remote directory change so far

  if arg(1, 'o') then                  -- default pattern
    pattern = '*'                      -- delete all files
  if arg(2, 'e') then do               -- argument 2 specifies remote directory
    curDir = self~pwd                  -- remember the current directory
    rc = self~cd(rdir)                 -- change remote directory to this 
    if rc = -1 then signal error       -- signal any problems
    rchange = .true                    -- remote directory change 
  end    
    
  list = self~getRDirList(pattern, 'F')-- retrieve list of files
  if list \= -1 & list \= 0 then       -- a list is received 
  do file over list                    -- go over the list  
    rc = self~delete(file)             -- delete this file from remote host
    if rc = -1 then leave              -- check, if error abend operationd
  end
  else rc = list                       -- getRDirList error or empty list

  if rchange then self~cd(curDir)      -- return to previous remote directory

error:
  if rc = -1 then say message(self~ErrNo, "mdel")                        
  return rc                            -- return success "0" or error "-1"

/*--------------------------------------------------------------------------*/
::METHOD mdeld PRIVATE                 /* Multiple Delete returning subdirs */

  dirs = .set~new                      -- create empty directory set
  rc = 0
  list = self~getRDirList()            -- retrieve list of files/directories
  if list \= -1 & list \= 0 then       -- a list is received 
  do file over list                    -- go over the list                 
    if file~word(1) then               -- check for directory   
      dirs~put(file~word(2))           -- yes, put its name into dirs        
    else do                            -- no, handle file entry
      rc = self~delete(file~word(2))   -- no, delete this file from remote host
      if rc = -1 then leave            -- any problems? 
    end                                
  end
  else rc = list                       -- communicate problem

  if rc \= 0 then do                   -- error case
    say message(self~ErrNo, 'mdeld')     
    return -1                          -- return error 
  end                    
  if dirs~items > 0 then return dirs   -- return directory list, if any
  return 0                             -- return success

/*--------------------------------------------------------------------------*/
::METHOD tdel                          /* Delete whole directory tree       */ 
  use arg rdir                         /* optional parm: remote directory   */

  if arg(1, 'e') then do               -- argument 1 specifies remote directory
    rc = self~cd(rdir)                 -- change remote directory to this 
    if rc = -1 then signal error       -- signal any problems
  end  

  dirs = self~mdeld()                  -- delete all files and return subdirs
  if dirs \= 0 & dirs \= -1 then       -- if there is a subdir list
  do dir over dirs                     -- handle all members of every subdir
    rc = self~tdel(dir)                -- delete all files and return subdirs
    if rc = -1 then leave              -- if error, abend operation 
  end
  else rc = dirs                       -- mdeld error

  if rc \= -1 & arg(1, 'e') then do
    rc = self~cd("..")                 -- return to previous remote directory
    if rc = 0 then                     -- if ok, resume operation
      rc = self~rmdir(rdir)            -- remove this directory 
  end 

error:
  if rc = -1 then say message(self~ErrNo, "tdel")
  return rc

/*--------------------------------------------------------------------------*/
::METHOD getRDirList                   /* get remote unified file list      */
  use arg pattern, mode                /* search pattern and mode           */
                                       /* returns array of entries: d name  */ 
  if arg(1, 'o') then                 
    if self~rSysType = 'OS/2' then     -- seems a bug in OS/2 
      pattern = '*'                    --    requiring an '*'
    else 
      pattern = ''
  else if self~rSysType = 'OS/2' then 
    if pattern = '' then 
       pattern = '*'                   --    required by OS/2

  if arg(2, 'o') then                  -- <d name> with d: 1 dir or 0 file
    mode = '*'
  else if arg(2, 'e') then do
    mode = translate(arg(2))              
    if mode~verify('FD*') > 0 | mode~length \= 1 then do
      say 'Invalid mode parameter; allowed are *, F, and D'
      return -1
    end
  end 
  list = self~dir(pattern)             -- retrieve the directory list
  if list \= -1 & list \= 0 then do    -- valid non-empty list received
    rList = .set~new()                 -- provide a result list
    if self~rSysType = 'OS/2' then     -- for OS/2 systems
    do entry over list
      parse var entry . attr . . name
      select                           -- check for directory
        when mode = '*' then do        -- include files and directories...    
          if attr = 'DIR' then 
            dirind = .true
          else dirind = .false
          rList~put(dirind name)       -- put new entry into list 
        end         
        when mode = 'F' then           -- include just files ...
          if attr \= 'DIR' then 
            rList~put(name)            -- put new file name into list 
        when mode = 'D' then           -- include just directories...
          if attr = 'DIR' then 
            rList~put(name)            -- put new directory name into list  
      end   
    end   
    else do entry over list            -- for Unix and Windows type systems
      parse var entry attr . . . . . . . name
      if name = '' | name = '.' | name = '..' then iterate
      select                           -- check for directory
        when mode = '*' then do        -- include files and directories...
          if attr~substr(1, 1) = 'd' then 
            dirind = .true
          else dirind = .false
          rList~put(dirind name)       -- put new entry into list  
        end 
        when mode = 'F' then           -- include just files ...
          if attr~substr(1, 1) \= 'd' then
            rList~put(name)            -- put new file name into list  
        when mode = 'D' then           -- include just directories...   
          if attr~substr(1, 1) = 'd' then
            rList~put(name)            -- put new directory name into list 
      end   
    end

    if rList~items > 0 then             
      return rList                     -- return the unified result list
    else rc = 0                        
  end
  else rc = list
  return rc                            -- return '-1' error, '0' empty list

/*--------------------------------------------------------------------------*/
::METHOD genRDirTree                   /* Generate a remote directory tree  */

  use arg paths                        /* Array of paths (w trailing '/'! ) */
                                       /* - any representation, with \ or / */ 
  rc = self~pwd()                      -- that's the root of the tree   
  if rc = -1 then signal error
  rc = ensure(rc, self~rSLASH)                         
  rootDir = rc                         -- remember rootdir
  if rc~substr(rc~length) = self~rSLASH then
    root = rc~substr(1,rc~length-1)    -- remove leading slash and  
  else
    root = rc                          -- remember relative root
  rootdir = root
  do path over paths                   -- walk over all path specs
    cDir = root                        -- start at root of tree  
    path = path~changestr('\', '/')    -- unify path representation -> '/'  
    pos = path~pos('/')                -- locate end of dir name    

    do while pos \= 0                  -- verify the whole path       
      dir = path~substr(1, pos-1)      -- get next directory        
      ndir = cDir'/'dir                -- and append it to current 
      rc = self~cd(ndir)               -- try to make it current    
      if rc = -1 then do               -- does it exist?            
        rc = self~mkdir(dir)           -- No, create this directory 
        if rc \= 0 then leave
        rc = self~cd(dir)              -- try to make it current 
        if rc \= 0 then leave            
      end                
      cDir = ndir                      -- ndir is now current
      path = path~substr(pos+1)        -- drop directory from path  
      pos = path~pos('/')              -- get next directory         
    end  

    rc  = self~cd(rootDir)             -- go back to root dir       
    if rc \= 0 then leave
  end

error:
  if rc = -1 then say message(self~ErrNo, "genRDirTree")   
  return abs(rc)                       -- 0 success, 1 error 

/*--------------------------------------------------------------------------*/
::METHOD getLDirList                   /* get local unified file list       */
  use arg pattern, mode                /* search pattern and mode           */
                                       /* returns array of entries:[d ]name */ 
  if arg(1, 'o') then                  
    pattern = '*'                      -- default pattern

  lchange = .false                     -- no local directory changed so far
  if '*'~verify(pattern) then do 
    rootDir = directory()              -- remember where we are
    rc = directory(pattern)
    if rc = '' then return -1          -- error changing directory
    pattern = '*'                      -- take all in this directory 
    lchange = .true                    -- local directory changed
  end 

  rList = .set~new()                   -- provide a result list

  if arg(2, 'e') then do
    mode = translate(arg(2))              
    if mode~verify('FD*') > 0 | mode~length \= 1 then do
      say 'Invalid mode parameter; allowed are *, F, and D'
      return -1
    end
    if mode = '*' then signal all
    mode = mode'O'

    rc = sysFileTree(pattern,list.,mode)-- retrieve the directory list
    if rc = 0 then do 
      do i = 1 to list.0  
        entry = list.i
        name = entry~substr(1 + entry~lastpos(self~lSLASH))
        rList~put(name)                -- put new entry into list 
      end  
    end
  end
  else do

all:
    rc = sysFileTree(pattern, list.)   -- retrieve the directory list   
    if rc = 0 then do                  -- for OS/2 and Windows 
      if self~lSysType = 'OS/2' | self~lSysType~abbrev("Windows") then
      do i = 1 to list.0  
        entry = list.i
        parse var entry . . . attr name 
        name = name~substr(1 + name~lastpos(self~lSLASH))
        if attr~substr(2, 1) = 'D' then 
          dirind = .true
        else dirind = .false
        rList~put(dirind name)         -- put new entry into list 
      end
      else do i = 1 to list.0          -- for AIX and Linux  
        entry = list.i
        parse var entry . . . attr name  
        name = name~substr(1 + name~lastpos(self~lSLASH))
        if attr~substr(1, 1) = 'd' then 
          dirind = .true
        else dirind = .false
        rList~put(dirind name)         -- put new entry into list 
      end         
    end   
  end 

  if lchange then rx = directory(rootDir) -- change dir back to where we were 
 
  if rc = 0 then do
    if rList~items > 0 then             
      return rList                     -- return the unified result list
    return 0 
  end 
  return -1                            -- error '-1', empty list '0'

/*--------------------------------------------------------------------------*/
::METHOD genLDirTree                   /* Generate a local directory tree   */
  use arg paths                        /* Array of paths (w trailing '/'! ) */

  rootDir = directory()                -- remember the root of tree   
  rc = 0
  do path over paths                   -- walk over all paths 
    cDir = rootDir                     -- start at root of tree 
    path = path~changestr('\', '/')    -- unify path representation -> '/'
    pos = path~pos('/')                -- locate end of dir name    
    do while pos \= 0                  -- verify the whole path
      dir = path~substr(1, pos - 1)    -- get next directory        
      ndir = cDir||self~lSLASH||dir    -- and append it to current 
      cDir = directory(ndir)           -- try to make it current    
      if cDir \= '' then do            -- does it exist? 
        rc = SysMkdir(dir)             -- No, create this directory 
        if rc \= 0 then leave
        cDir = directory(dir)          -- now change to it          
      end 
      path = path~substr(pos + 1)      -- drop directory from path  
      pos = path~pos('/')              -- get next directory 
    end  
    rootDir = directory(rootDir)       -- go back to root directory       
  end

  return rc                            -- 0 success, >0  error 

/*--------------------------------------------------------------------------*/
::METHOD lmdel                         /* Delete files from local directory */
  use arg pattern                      /* optional parm: search pattern     */

  if arg(1, 'o') then                  -- default pattern
    pattern = '*'                      -- delete all files

  rc = sysfiletree(pattern, 'list')    -- get list of files/dirs 
  if rc = 0 then do
    if self~lSysType = 'Linux' | self~lSysType = 'AIX' then
                                       /* for Linux and AIX                 */ 
    do i = 1 to list.0                 -- go over the list of files/dirs
      entry = list.i
      parse var entry . . . attr path  -- parse the entry
      name = path~substr(1 + path~lastpos(self~lSLASH))
      if attr~substr(1, 1) \= 'd' then do -- is it a file?     <drwxrwxrwx>
        rc = sysfiledelete(name)       -- delete this file
        if rc \= 0 then leave          -- if error, abort the operation
      end
    end
                                       /* for OS/2 and Windows95/NT         */ 
    else do i = 1 to list.0            -- go over the list of files/dirs
      entry = list.i
      parse var entry . . . attr path  -- parse the entry
      name = path~substr(1 + path~lastpos(self~lSLASH))
      if attr~substr(2, 1) \= 'D' then do  -- is it a file?    <-D--->
        rc = sysfiledelete(name)       -- delete this file
        if rc \= 0 then leave          -- if error, abort the operation
      end 
    end
  end

  if rc \= 0 then do                   -- error case
    say 'Error deleting files in local directory, "lmdel"'     
    return rc                          -- return error 
  end                        
  return 0                             -- return success

/*--------------------------------------------------------------------------*/
::METHOD lmdeld PRIVATE                /* Delete files from local directory */

  dirs = .set~new                      -- create a directory set

  rc = sysfiletree('*', 'list')        -- get list of files/dirs
  if rc = 0 then do
    if self~lSysType = 'Linux' | self~lSysType = 'AIX' then
                                       /* for Linux and AIX                 */
    do i = 1 to list.0                 -- go over the list of files/dirs
      entry = list.i
      parse var entry . . . attr path  -- parse the entry
      name = path~substr(1 + path~lastpos(self~lSLASH))
      if attr~substr(1, 1) = 'd' then  -- is it a directory?   <drwxrwxrwx>
        dirs~put(name)                 -- put directory name into dirs
      else do                          -- it's a file
        rc = sysfiledelete(name)       -- delete this file
        if rc \= 0 then leave          -- if error, abort the operation
      end
    end
                                       /* for OS/2 and Windows95/NT         */       
    else do i = 1 to list.0            -- go over the list of files/dirs
      entry = list.i
      parse var entry . . . attr path  -- parse the entry
      name = path~substr(1 + path~lastpos(self~lSLASH))
      if attr~substr(2, 1) = 'D' then  -- is it a directory?   <-D--->
        dirs~put(name)                 -- put directory name into dirs
      else do                          -- it's a file
        rc = sysfiledelete(name)       -- delete this file
        if rc \= 0 then leave          -- if error, abort the operation
      end
    end
  end

  if rc \= 0 then do                   -- error case
    say 'Error deleting local directory'     
    return rc                          -- return error 
  end                        
  if dirs~items > 0 then return dirs   -- return directory list, if any
  return 0                             -- return success

/*--------------------------------------------------------------------------*/
::METHOD ltdel                         /* Delete whole local directory tree */
  use arg ldir                         /* optional parm: local directory    */
  if arg(1, 'e') then do               -- argument 1 specifies remote directory
    rc = self~lcd(ldir)                -- change local directory to this
    if rc = 1 then signal error        -- signal any problems
  end  

  dirs = self~lmdeld()                 -- delete all files and return subdirs
  if dirs \= 0 & dirs \= -1 then       -- if there is a subdir list
  do dir over dirs                     -- handle all members of every subdir
    rc = self~ltdel(dir)               -- delete all files and return subdirs
    if rc = -1 then leave              -- if error, abend operation 
  end
  else rc = dirs                       -- mdeld error

  if rc \= -1 & arg(1, 'e') then do
    rc = directory("..")               -- return to previous local directory
    if rc \= '' then                   -- if ok, resume operation
      rc = sysrmdir(ldir)              -- remove this directory 
    else rc = 1
  end 

error:
  if rc \= 0 then say message("Error deleting local directory tree" ldir, "ltdel")   
  return rc

/*--------------------------------------------------------------------------*/
::METHOD delLDirTree                   /* Delete a local directory tree     */
  use arg ldir                         /* local directory name              */
 
  if arg(1, 'e') then do               -- argument 1 specifies local directory
    rc = self~lcd(ldir)                -- change local directory to this 
    if rc = -1 then signal error       -- signal any problems
  end  

  dirs = self~delLDir()                -- delete all files and return subdirs
  if dirs \= 0 & dirs \= -1 then       -- if there is a subdir list
  do dir over dirs                     -- handle all members of every subdir
    rc = self~delLDirTree(dir)         -- delete all files and return subdirs
    if rc = -1 then leave              -- if error, abend operation 
  end
  else rc = dirs                       -- delLDir error or empty dirs

  if rc \= -1 & arg(1, 'e') then do 
    ndir = directory("..")             -- return to previous local directory 
    if ndir \= '' then                 -- if ok, resume operation
      rc = sysrmdir(ldir)              -- remove this directory 
  end   

error:
  return rc                            -- 0 success, >0  error 

/*--------------------------------------------------------------------------*/
::METHOD lcd                           /* Local change directory            */
  use arg dir                          /* directory spec                    */

  slash =  self~lSLASH                 -- determine local slash
  dir = ensure(dir, slash)             -- ensure local path representation
  ndir = directory(dir)                -- change to this local directory
  if ndir = '' then
    return 1                           -- return error 
verify:
  pos = dir~lastpos(slash)
  if pos = 0 then do                   -- is directory spec simple?
    if dir = ndir~substr(1 + ndir~lastpos(slash)) then
      return 0        
  end
  else do                              -- verify complex spec
    if dir~length = pos then do
      dir = dir~substr(1, pos - 1)
      signal verify 
    end
    spos = ndir~length - dir~length
    if spos = 0 then 
      return 0                         -- ok
    if dir = ndir~substr(spos + 1) & ndir~substr(spos, 1) = slash then
      return 0                         -- ok                   
  end
  return 1                             -- return error

/*--------------------------------------------------------------------------*/
::METHOD xcd                           /* Extended change directory         */
  use arg rdir                         /* directory name                    */

  rc = self~cd(rdir)                   -- change to this remote directory 
  if rc \= 0 then do                   -- change ok?
    rc = self~mkdir(rdir)              -- no, try to create it
    if rc \= 0 then 
      return rc                        -- abend transfer 
    rc = self~cd(rdir)                 -- change to this remote directory ok?
  end 
  return rc                            -- success 0, error -1 

/*--------------------------------------------------------------------------*/
::METHOD xlcd                          /* Extended local change directory   */
  use arg ldir                         /* directory spec                    */

  rc = self~lcd(ldir)                  -- change to this local directory 
  if rc then do                        -- change ok?
    rc = sysMkDir(ldir)                -- no, try to create it
    if rc \= 0 then 
      return 1                         -- abend transfer 
    rc = self~lcd(ldir)                -- change to this local directory ok?
  end 
  return rc

/*--------------------------------------------------------------------------*/
::METHOD showldir                      /* Show local directory              */
  use arg pattern                      /* search pattern [and mode]         */

  if arg(1, 'o') then                  -- default pattern
    pattern = '*'                      -- show all files and/or dirs

  if arg(2, 'e') then do               -- mode: 'D' - dirs or 'F' - files
    mode = translate(arg(2))     
    if mode~verify('FD') > 0 | mode~length \= 1 then do
      say 'Invalid mode parameter; allowed are F and D'
      return -1                        -- parameter error
    end
    mode = mode'O'
  end 
  if arg(2, 'o') then do               -- get list of files and dirs
    say '-----dir("'pattern'")-------'
    rc = sysfiletree(pattern, 'list') 
  end  
  else do                              -- get list of files or dirs
    say '-----dir("'pattern'", "'arg(2)'")-------'
    rc = sysfiletree(pattern, 'list', mode)
  end   
  if rc = 0 then do i = 1 to list.0    -- go over the list returned
    say list.i~substr(1 + list.i~lastpos(self~lSLASH))
  end
  return rc                            -- 0 - success, >0 - error

/*--------------------------------------------------------------------------*/
/* End of xFtp Class definition                                             */
/*--------------------------------------------------------------------------*/
