/**MOD+***********************************************************************/
/* Module:    csample2.c                                                     */
/*                                                                           */
/* Purpose:   CPI-C sample application                                       */
/*                                                                           */
/* (C) COPYRIGHT DATA CONNECTION LIMITED 1991, 1992, 1995                    */
/*                                                                           */
/**MOD-***********************************************************************/

/*****************************************************************************/
/* The transaction programs CSAMPLE1 and CSAMPLE2 enable a user to browse    */
/* thru a file on another system.  The user of CSAMPLE1 is presented with a  */
/* single data block at a time, in hex and character format.  After each     */
/* block a user can request the next block, request the previous block, or   */
/* quit.  CSAMPLE1 (the invoking TP) sends a filename to CSAMPLE2 (the       */
/* invoked TP).  If CSAMPLE2 locates the file, it returns the first block to */
/* CSAMPLE1, otherwise it issues deallocate, and terminates.  If CSAMPLE1    */
/* receives a block it displays it on the screen and waits for a user        */
/* prompt; if it receives an indication that CSAMPLE2 has terminated, it too */
/* terminates.  If the user asks for the next block when CSAMPLE2 has sent   */
/* the last one, CSAMPLE2 wraps to the beginning of file.  Similarly,        */
/* CSAMPLE2 wraps to send the last block if the previous one is requested    */
/* when the first block is displayed.  Neither program attempts to recover   */
/* from errors.  A 'bad' return code from CPIC causes the program to         */
/* terminate with an explanatory message.                                    */
/*                                                                           */
/* Note that the numeric values in the data transferred between the two      */
/* programs are stored as arrays of 4 characters, rather than as integers.   */
/* CSAMPLE2 uses putlong() to put an integer into the data block in this     */
/* format.  This avoids problems if the two programs are running on          */
/* different machines with different byte ordering.                          */
/*****************************************************************************/

#define BLOCKSIZ 256                /* size of data blocks sent to CSAMPLE1 */
#define ISIZE    4                  /* size of an integer - 4 bytes         */
#define TRUE     1
#define FALSE    0

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <unistd.h>

#include <cmc.h>

/*****************************************************************************/
/* remove comments from following line to implement debugging to screen      */
/*****************************************************************************/
/* #define DEBUGGING */

#define boolean int


/**STRUCT+********************************************************************/
/* Structure:   BLOCK_TO_SEND                                                */
/*                                                                           */
/* Description: data to send to partner application.                         */
/*                                                                           */
/* Note:        all numeric values are stored as arrays of 4 characters.     */
/*****************************************************************************/
typedef struct
{
   char      block_number[ISIZE];      /* this block number (1 - num_blocks) */
   char      num_blocks[ISIZE];        /* number of blocks in file           */
   char      block_size[ISIZE];        /* bytes in this block                */
   char      data_block[BLOCKSIZ];     /* data in block                      */
} BLOCK_TO_SEND;
/**STRUCT-********************************************************************/

FILE *  fhandle;                       /* handle of file to be read and sent */

int          block_wanted;                  /* range 0 to (num_blocks - 1)   */
                                            /* updated when prompt received. */
                                            /* used by get_block().          */
unsigned int num_blocks;              /* set by get_fname() when file opened */

unsigned char               conv_id[8];
CM_REQUEST_TO_SEND_RECEIVED rts_rcv;
unsigned char               buffer[sizeof(BLOCK_TO_SEND)];
CM_INT32                    req_len = 40;
CM_DATA_RECEIVED_TYPE       data_rcv;
CM_INT32                    rcv_len;
CM_STATUS_RECEIVED          stat_rcv;
CM_DEALLOCATE_TYPE          deal_type = CM_DEALLOCATE_ABEND;
CM_RETURN_CODE              retcode;

void      putlong ();
int       main ();
void      initialize ();
void      send ();
boolean   receive ();
boolean   get_fname ();
char      get_prompt ();
int       get_block ();
void      deallocate ();
void      exit_error ();


/**PROC+**********************************************************************/
/* Name:      putlong                                                        */
/*                                                                           */
/* Purpose:   Put an integer into the data block as 4 characters             */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    IN      source - long integer value                            */
/*            OUT     dest   - character string to go into data block        */
/*                             representing the integer value                */
/*                                                                           */
/**PROC-**********************************************************************/
void putlong (source, dest)
long   source;
char * dest;
{
  int       ii;
  for ( ii = 0 ; ii < ISIZE ; ii++ )
  {
    *dest++ = source;
    source >>= 8;
  }
}


/**PROC+**********************************************************************/
/* Name:      main                                                           */
/*                                                                           */
/* Purpose:   Entry point to application                                     */
/*                                                                           */
/* Returns:   0                                                              */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
int main()
{
  char      prompt;
  char      fname[65];
  boolean   conversing = TRUE;
  boolean   first_time = TRUE;
  int       bytes_read;

#ifdef DEBUGGING
  printf ("CSAMPLE2: main: top\n");
#endif

  initialize();

  /***************************************************************************/
  /* Repeat this loop until a bad prompt is received                         */
  /***************************************************************************/
  do
  {
    if (receive())
    {

#ifdef DEBUGGING
      printf ("CSAMPLE2: main: receive was OK\n");
#endif

      if (data_rcv == CM_COMPLETE_DATA_RECEIVED)
      {
        if (first_time)
        {
          /*******************************************************************/
          /* Get the filename of file to scan through                        */
          /*******************************************************************/

#ifdef DEBUGGING
          printf ("CSAMPLE2: main: first time through, get filename\n");
#endif

          if (get_fname(fname, sizeof(fname)-1))
          {
            /*****************************************************************/
            /* First block => do an 'F'                                      */
            /*****************************************************************/
            prompt = 'F';
            first_time = FALSE;
          }
          else
          {
            /*****************************************************************/
            /* Get filename failed - quit now                                */
            /*****************************************************************/

#ifdef DEBUGGING
            printf ("CSAMPLE2: get_fname failed - deallocating now\n");
#endif

            deallocate();
            conversing = FALSE;
          }
        }
        else
        {
          prompt = get_prompt();
        }
      }
      if (stat_rcv == CM_SEND_RECEIVED)
      {
        if (prompt == 'F')
        {
          /***************************************************************/
          /* First time this sets block_wanted from -1 to 0              */
          /***************************************************************/
          block_wanted++;
        }
        else if (prompt == 'B')
        {
          block_wanted--;
        }
        else
        {
          printf( "CSAMPLE2: Unknown Prompt hex %02x received\n", prompt );
          conversing = FALSE;
        }
        if (conversing)
        {
          bytes_read = get_block ((BLOCK_TO_SEND *) buffer);

          /***********************************************************/
          /* Send the data, allowing for num_blocks, block number    */
          /* and block size in the length to be sent                 */
          /***********************************************************/
           send( buffer, bytes_read +
                 (int)(((BLOCK_TO_SEND*)buffer)->data_block)-(int)buffer);

        }
      }
    }
    else
    {
      conversing = FALSE;
    }
  }
  while (conversing);

#ifdef DEBUGGING
  printf ("CSAMPLE2: main: conversation deallocated - ending now\n");
#endif

  fclose (fhandle);
  return (0);
}


/**PROC+**********************************************************************/
/* Name:      initialize                                                     */
/*                                                                           */
/* Purpose:   Initialize CPI-C conversation                                  */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void initialize()
{
#ifdef DEBUGGING
   printf ("CSAMPLE2: initialize: top\n");
#endif

  /***************************************************************************/
  /* Issue CMACCP call                                                       */
  /***************************************************************************/
  cmaccp (conv_id, &retcode);

  if (retcode != CM_OK )
  {
    exit_error(retcode);
  }

  block_wanted = -1;

#ifdef DEBUGGING
  printf ("CSAMPLE2: initialize: end\n");
#endif

}


/**PROC+**********************************************************************/
/* Name:      send                                                           */
/*                                                                           */
/* Purpose:   Send data to partner application.                              */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    IN   datastring - data to send                                 */
/*            IN   datalen    - length to send                               */
/*                                                                           */
/* Operation: Issue CPIC call cmsend.  Exit with error message if not CM_OK. */
/*                                                                           */
/**PROC-**********************************************************************/
void send (datastring, datalen)
unsigned char CM_PTR datastring;
CM_INT32     datalen;
{

#ifdef DEBUGGING
  printf ("CSAMPLE2: send: top with datalen %d\n", datalen);
#endif

  cmsend (conv_id, datastring, &datalen, &rts_rcv, &retcode);

  if (retcode != CM_OK)
  {
    exit_error (retcode);
  }

#ifdef DEBUGGING
  printf ("CSAMPLE2: send: end\n");
#endif

}


/**PROC+**********************************************************************/
/* Name:      receive                                                        */
/*                                                                           */
/* Purpose:   Receive data from partner application.                         */
/*                                                                           */
/* Returns:   TRUE/FALSE                                                     */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/* Operation: Issue CPIC call cmrcv. Return FALSE if error or if deallocate  */
/*            received.                                                      */
/*                                                                           */
/**PROC-**********************************************************************/
boolean receive()
{

#ifdef DEBUGGING
  printf ("CSAMPLE2: receive: top\n");
#endif

  cmrcv (conv_id, buffer, &req_len, &data_rcv, &rcv_len,
         &stat_rcv, &rts_rcv, &retcode);

  if (retcode == CM_DEALLOCATED_NORMAL)
  {
    return (FALSE);
  }
  else
  {
    if (retcode != CM_OK)
    {
      exit_error (retcode);
    }
  }

#ifdef DEBUGGING
  printf ("CSAMPLE2: receive: end\n");
#endif

  return (TRUE);
}



/**PROC+**********************************************************************/
/* Name:      prompt                                                         */
/*                                                                           */
/* Purpose:   Extract and check prompt from partner application.             */
/*                                                                           */
/* Returns:   Prompt character, or ' '.                                      */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/* Operation: prompt should have been read in to segment addressed by        */
/*            buffer.  Extract and store it in prompt.  Test to see that     */
/*            length is 1.  If so return with char, else return with ' ';    */
/*                                                                           */
/**PROC-**********************************************************************/
char get_prompt()
{
  int ch;

#ifdef DEBUGGING
  printf ("CSAMPLE2: get_prompt: top\n");
#endif

  ch = *buffer;
  if (rcv_len == 1 && ( ch == 'F' || ch == 'B'))
  {
    return (ch);
  }
  else
  {
    return (' ');
  }

#ifdef DEBUGGING
  printf ("CSAMPLE2: get_prompt: end\n");
#endif

}



/**PROC+**********************************************************************/
/* Name:      get_fname                                                      */
/*                                                                           */
/* Purpose:   Extract filename from buffer, open file and read its size.     */
/*                                                                           */
/* Returns:   TRUE / FALSE                                                   */
/*                                                                           */
/* Params:    OUT    fname  - filename                                       */
/*            IN     maxlen - max filename length                            */
/*                                                                           */
/* Operation: filename should have been read in to segment addressed by      */
/*            buffer.  Extract and store it in fname.  Try to open file.  If */
/*            successful, get file size and return TRUE.  If not successful, */
/*            return FALSE.                                                  */
/**PROC-**********************************************************************/
boolean get_fname (fname, maxlen)
char *fname;
int   maxlen;
{
   long filesize;
   char * type = "r";
   unsigned short rc = 0;

#ifdef DEBUGGING
  printf("CSAMPLE2: get_fname: top, %d length recd, data %13.0Fs.\n",
         (int) rcv_len, (char *) buffer);
#endif

  if (rcv_len > maxlen)
  {

#ifdef DEBUGGING
    printf ("CSAMPLE2: get_fname: rcv_len %d > maxlen %d\n", rcv_len, maxlen);
#endif

    return (FALSE);
  }

  memcpy(fname, buffer, (int)rcv_len);
  fname [(int)rcv_len] = '\0';

  fhandle = (FILE *)fopen( fname, type );
  if ( fhandle != NULL)
  {
    rc = TRUE;

#ifdef DEBUGGING
    printf ("CSAMPLE2: get_fname: Opened file %s OK\n", fname);
#endif

    /*************************************************************************/
    /* go to end of file and find offset from beginning - this gives size    */
    /* of file.  Then calulate no.  of blocks.                               */
    /*************************************************************************/
    fseek ( fhandle, 0L, 2 );
    filesize = ftell ( fhandle ) ;
    num_blocks = ( filesize + BLOCKSIZ - 1L ) / BLOCKSIZ;
  }
  else
  {

#ifdef DEBUGGING
    printf ("CSAMPLE2: get_fname: Failed to open file %s\n", fname);
#endif

    rc = FALSE;
  }


#ifdef DEBUGGING
   printf ("CSAMPLE2: get_fname: size of file %s was %d, no. of blocks %d\n",
       fname, filesize, num_blocks );
#endif

   return( rc );
}


/**PROC+**********************************************************************/
/* Name:      get_block                                                      */
/*                                                                           */
/* Purpose:   Read a block of data from file                                 */
/*                                                                           */
/* Returns:   Number of bytes read                                           */
/*                                                                           */
/* Params:    OUT    psblock - block to be sent                              */
/*                                                                           */
/* Operation: External variable block_wanted holds the block number to read. */
/*            Check it for range.  Read the block, store the block_wanted    */
/*            number and the block ready for sending.                        */
/*                                                                           */
/**PROC-**********************************************************************/
int get_block(psblock)
BLOCK_TO_SEND *psblock;
{
   char  buf[ BLOCKSIZ ];
   int   bytes_read = 0;

#ifdef DEBUGGING
   printf ("CSAMPLE2: get_block: top, read block number %d\n", block_wanted );
#endif

   /**************************************************************************/
   /* make sure block wanted is within range 0 - num_blocks, then seek it    */
   /* and read the information.                                              */
   /**************************************************************************/

   block_wanted = block_wanted < 0 ? num_blocks - 1 :
                                     block_wanted % num_blocks;

   fseek( fhandle, block_wanted * BLOCKSIZ, 0 );

   if ( bytes_read = fread( buf, 1, BLOCKSIZ, fhandle )  )
   {

#ifdef DEBUGGING
     printf ("CSAMPLE2: get_block: read %d bytes for block %d\n",
                                       bytes_read, block_wanted );
#endif

      memcpy( psblock->data_block, buf, bytes_read );
      putlong (block_wanted + 1, psblock->block_number);
      putlong (num_blocks, psblock->num_blocks);
      putlong (bytes_read, psblock->block_size);

   }
   else
   {
      printf( "CSAMPLE2: get_block: file read failed\n" );
   }

   return( bytes_read );
}


/**PROC+**********************************************************************/
/* Name:      deallocate                                                     */
/*                                                                           */
/* Purpose:   Deallocate converstion                                         */
/*                                                                           */
/* Returns:   None                                                           */
/*                                                                           */
/* Params:    None                                                           */
/*                                                                           */
/* Operation: Issue cmsdt, cmdeal.                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void deallocate()
{

#ifdef DEBUGGING
  printf ("CSAMPLE2: deallocate: top\n");
#endif

  cmsdt(conv_id, &deal_type, &retcode);
  if (retcode != CM_OK)
  {
    exit_error(retcode);
  }
  else
  {
    cmdeal(conv_id, &retcode);
    if (retcode != CM_OK)
    {
      exit_error(retcode);
    }
  }

#ifdef DEBUGGING
  printf ("CSAMPLE2: deallocate: end\n");
#endif
}


/**PROC+**********************************************************************/
/* Name:      exit_error                                                     */
/*                                                                           */
/* Purpose:   Display error message and exit application.                    */
/*                                                                           */
/* Returns:   exits with -1                                                  */
/*                                                                           */
/* Params:    IN   rc  - CPI-C return code.                                  */
/*                                                                           */
/* Operation:                                                                */
/*                                                                           */
/**PROC-**********************************************************************/
void exit_error(rc)
CM_INT32 rc;
{

  switch ((int) (rc))
  {
    case CM_OK:
      printf("CSAMPLE2: CPI-C returned: CM_OK\n");
      break;
    case CM_ALLOCATE_FAILURE_NO_RETRY:
      printf("CSAMPLE2: CPI-C returned: CM_ALLOCATE_FAILURE_NO_RETRY\n");
      break;
    case CM_ALLOCATE_FAILURE_RETRY:
      printf("CSAMPLE2: CPI-C returned: CM_ALLOCATE_FAILURE_RETRY\n");
      break;
    case CM_CONVERSATION_TYPE_MISMATCH:
      printf("CSAMPLE2: CPI-C returned: CM_CONVERSATION_TYPE_MISMATCH\n");
      break;
    case CM_PIP_NOT_SPECIFIED_CORRECTLY:
      printf("CSAMPLE2: CPI-C returned: CM_PIP_NOT_SPECIFIED_CORRECTLY\n");
      break;
    case CM_SECURITY_NOT_VALID:
      printf("CSAMPLE2: CPI-C returned: CM_SECURITY_NOT_VALID\n");
      break;
    case CM_SYNC_LEVEL_NOT_SUPPORTED_LU:
      printf("CSAMPLE2: CPI-C returned: CM_SYNC_LEVEL_NOT_SUPPORTED_LU\n");
      break;
    case CM_SYNC_LEVEL_NOT_SUPPORTED_PGM:
      printf("CSAMPLE2: CPI-C returned: CM_SYNC_LEVEL_NOT_SUPPORTED_PGM\n");
      break;
    case CM_TPN_NOT_RECOGNISED:
      printf("CSAMPLE2: CPI-C returned: CM_TPN_NOT_RECOGNISED\n");
      break;
    case CM_TP_NOT_AVAILABLE_NO_RETRY:
      printf("CSAMPLE2: CPI-C returned: CM_TP_NOT_AVAILABLE_NO_RETRY\n");
      break;
    case CM_TP_NOT_AVAILABLE_RETRY:
      printf("CSAMPLE2: CPI-C returned: CM_TP_NOT_AVAILABLE_RETRY\n");
      break;
    case CM_DEALLOCATED_ABEND:
      printf("CSAMPLE2: CPI-C returned: CM_DEALLOCATED_ABEND\n");
      break;
    case CM_DEALLOCATED_NORMAL:
      printf("CSAMPLE2: CPI-C returned: CM_DEALLOCATED_NORMAL\n");
      break;
    case CM_PARAMETER_ERROR:
      printf("CSAMPLE2: CPI-C returned: CM_PARAMETER_ERROR\n");
      break;
    case CM_PRODUCT_SPECIFIC_ERROR:
      printf("CSAMPLE2: CPI-C returned: CM_PRODUCT_SPECIFIC_ERROR\n");
      break;
    case CM_PROGRAM_ERROR_NO_TRUNC:
      printf("CSAMPLE2: CPI-C returned: CM_PROGRAM_ERROR_NO_TRUNC\n");
      break;
    case CM_PROGRAM_ERROR_PURGING:
      printf("CSAMPLE2: CPI-C returned: CM_PROGRAM_ERROR_PURGING\n");
      break;
    case CM_PROGRAM_ERROR_TRUNC:
      printf("CSAMPLE2: CPI-C returned: CM_PROGRAM_ERROR_TRUNC\n");
      break;
    case CM_PROGRAM_PARAMETER_CHECK:
      printf("CSAMPLE2: CPI-C returned: CM_PROGRAM_PARAMETER_CHECK\n");
      break;
    case CM_PROGRAM_STATE_CHECK:
      printf("CSAMPLE2: CPI-C returned: CM_PROGRAM_STATE_CHECK\n");
      break;
    case CM_RESOURCE_FAILURE_NO_RETRY:
      printf("CSAMPLE2: CPI-C returned: CM_RESOURCE_FAILURE_NO_RETRY\n");
      break;
    case CM_RESOURCE_FAILURE_RETRY:
      printf("CSAMPLE2: CPI-C returned: CM_RESOURCE_FAILURE_RETRY\n");
      break;
    case CM_UNSUCCESSFUL:
      printf("CSAMPLE2: CPI-C returned: CM_UNSUCCESSFUL\n");
      break;
    case CM_DEALLOCATED_ABEND_SVC:
      printf("CSAMPLE2: CPI-C returned: CM_DEALLOCATED_ABEND_SVC\n");
      break;
    case CM_DEALLOCATED_ABEND_TIMER:
      printf("CSAMPLE2: CPI-C returned: CM_DEALLOCATED_ABEND_TIMER\n");
      break;
    case CM_SVC_ERROR_NO_TRUNC:
      printf("CSAMPLE2: CPI-C returned: CM_SVC_ERROR_NO_TRUNC\n");
      break;
    case CM_SVC_ERROR_PURGING:
      printf("CSAMPLE2: CPI-C returned: CM_SVC_ERROR_PURGING\n");
      break;
    case CM_SVC_ERROR_TRUNC:
      printf("CSAMPLE2: CPI-C returned: CM_SVC_ERROR_TRUNC\n");
      break;
    default:
      printf("CSAMPLE2: CPI-C returned unrecognised return_code %d", (int) rc);
      break;
  }

  exit (-1);
}
