/****************************************************************************/
/* Sample OS/2 Semaphore Functions                                          */
/*                                                                          */
/* (C) Copyright 1990, 1992 Data Connection Ltd.                            */
/*                                                                          */
/* These are provided to assist porting OS/2 LUA applications to UNIX.      */
/* They are described in the LUA Programmer's Guide.                        */
/*                                                                          */
/* This code is provided as a sample only.                                  */
/*                                                                          */
/* This file contains a simple implementation of the basic OS/2 semaphore   */
/* functions.  They are suitable for use in a single process for            */
/* synchronising the caller of an LUA function with an LUA callback         */
/* routine.                                                                 */
/*                                                                          */
/* This file also contains sample code for the callback procedure that is   */
/* called by the SNAP-IX LUA when a verb completes. SNAP-IX provides an     */
/* alternative entry point to the LUA interface, called RUI_SEM, for use    */
/* when porting an OS/2 LUA application which uses semaphores. This entry   */
/* point is typically invoked by including the following statements in the  */
/* application source :                                                     */
/*                                                                          */
/* #include <lua_c.h>                                                       */
/* extern void (*callback)();                                               */
/* #define RUI(X) { DosSemSet (X->common.lua_post_handle); \                */
/*                  RUI_SEM (X, callback) }                                 */
/*                                                                          */
/* See the LUA Programmer's Guide for further information.                  */
/*                                                                          */
/* This code relies on the fact that a semaphore used only by LUA can be    */
/* changed only in the context of the main program, or of a SIGPOLL signal  */
/* catcher.                                                                 */
/*                                                                          */
/* The following entry points are provided:                                 */
/****************************************************************************/
int DosMuxSemWait();
int DosSemClear();
int DosSemRequest();
int DosSemSet();
int DosSemWait();

/****************************************************************************/
/* These functions use UNIX signals to implement OS/2 semaphores within a   */
/* single process.  More sophisticated versions are required for use        */
/* between processes.                                                       */
/*                                                                          */
/* Define the various types required including the DosMuxSemWait struct.    */
/* Also declare static variables used for implementing timeouts.            */
/****************************************************************************/
#include <svigsig.h>
#include <memory.h>
#include <lua_c.h>

struct           sem0
{
  int pad;
  unsigned long* sem;
};

struct           sem1
{
  int count;                         /* Number of semaphores to wait on     */
  struct sem0 list[16];              /* List of semaphore handles           */
};

/****************************************************************************/
/* The following variables and functions implement a timeout function.      */
/* AlarmSig is set up as a signal catcher for SIGALRM signals.              */
/* When a timeout is triggered any previous signal catcher is called and a  */
/* SIGPOLL signal generated.  This SIGPOLL signal is used to indicate that  */
/* a semaphore value may have changed.                                      */
/*                                                                          */
/* SetTimeout is used to set up a timeout value.  The single parameter is:  */
/*    0  do not wait                                                        */
/*   -1  wait indefinitely                                                  */
/*    n  wait n milliseconds.                                               */
/* Note that the minimum timeout period is 1 second.                        */
/****************************************************************************/
static void (*old_alarm)();          /* Previous SIGALRM catcher            */
static int alarm_set = 0;            /* Flag for SIGALRM catcher set up     */
static int wait;                     /* Timeout pending flag                */



static int AlarmSig(sig)
int sig;
{
  if ((old_alarm != SIG_DFL)   &&
      (old_alarm != SIG_IGN)   &&
      (old_alarm != SIG_HOLD))
  {
    (*old_alarm)(sig);
  }

  wait = 0;
  kill(getpid(), SIGPOLL);
}


static void SetTimeout(time)
long time;
{
  if (time == 0)
  {
    wait = 0;
  }

  else
  {
    if (time != -1)
    {
      /**********************************************************************/
      /* Set up ALARM signal catcher once only                              */
      /**********************************************************************/
      if (alarm_set == 0)
      {
        old_alarm = sigset(SIGALRM, AlarmSig);
        alarm_set = 1;
      }

      time /= 1000;

      if (time == 0)
      {
        time = 1;
      }

      /**********************************************************************/
      /* Generate a SIGALRM at end of timeout period                        */
      /**********************************************************************/
      alarm((unsigned int)(time));
    }

    wait = 1;
  }
}


/****************************************************************************/
/* Simple unconditional semaphore operations.                               */
/****************************************************************************/
int DosSemSet(sem)
long *sem;
{
  *sem = 1;
  return(0);
}


int DosSemClear(sem)
long *sem;
{
  *sem = 0;
  return(0);
}


/****************************************************************************/
/* DosSemWait waits for a semaphore to clear, or timeout.                   */
/****************************************************************************/
int DosSemWait(sem, time)
long *sem;
long time;
{
  SetTimeout(time);

  sighold (SIGPOLL);
  while (*sem && wait)
  {
    sigpause(SIGPOLL);
    sighold (SIGPOLL);
  }
  sigrelse (SIGPOLL);

  if (wait)
  {
    /************************************************************************/
    /* Semaphore clear - cancel SIGALRM.                                    */
    /************************************************************************/
    alarm(0);
    return(0);
  }

  else
  {
    /************************************************************************/
    /* Either a timeout has occurred or the 'time' parameter was zero.      */
    /************************************************************************/
    if (*sem)
    {
      return(-1);
    }

    else
    {
      return(0);
    }
  }
}


/****************************************************************************/
/* DosSemRequest waits for a semaphore to clear and then sets it.           */
/****************************************************************************/
int DosSemRequest(sem, time)
long *sem;
long time;
{
  int rc;

  rc = DosSemWait(sem, time);
  if (rc)
  {
    return(rc);
  }

  else
  {
    *sem = 1;
    return(0);
  }
}


/****************************************************************************/
/* DosMuxSemWait waits for any one of a list of semaphores to clear.        */
/****************************************************************************/
int DosMuxSemWait(which, semlst, time)
int *which;
struct sem1 *semlst;
long time;
{
  int  j;
  int  semclear = 0;

  SetTimeout(time);
  *which = 0;

  sighold (SIGPOLL);
  while ((!semclear) && wait)
  {
    /************************************************************************/
    /* Check whether any of the specified semaphores is clear.              */
    /************************************************************************/
    for (j=0; j<semlst->count; j++)
    {
      if (*semlst->list[j].sem == 0)
      {
        *which   = j;
        semclear = 1;
        break;
      }
    }

    /************************************************************************/
    /* If no semaphore is clear then wait for a signal (possibly a timeout).*/
    /************************************************************************/
    if (!semclear)
    {
      sigpause(SIGPOLL);
      sighold (SIGPOLL);
    }
  }
  sigrelse (SIGPOLL);

  if (wait)
  {
    /************************************************************************/
    /* Semaphore clear - cancel SIGALRM.                                    */
    /************************************************************************/
    alarm(0);
    return(0);
  }

  else
  {
    /************************************************************************/
    /* Either a timeout has occurred or the 'time' parameter was zero.      */
    /************************************************************************/
    for (j=0; j<semlst->count; j++)
    {
      if (*semlst->list[j].sem == 0)
      {
        *which   = j;
        semclear = 1;
        break;
      }
    }

    if (semclear)
    {
      return(0);
    }

    else
    {
      return(-1);
    }
  }
}


/****************************************************************************/
/* Sample callback procedure.                                               */
/*                                                                          */
/* The SNAP-IX LUA informs an application that a verb has completed by      */
/* invoking a supplied callback procedure. An OS/2 LUA application which    */
/* has been written to have verb completion notified by the clearing of a   */
/* semaphore could be ported to UNIX using the RUI_SEM alternative entry    */
/* point to the SNAP-IX LUA interface, together with this callback          */
/* procedure.                                                               */
/****************************************************************************/
void callback(vcbptr)
LUA_VERB_RECORD *vcbptr;
{
  if (vcbptr->common.lua_post_handle != 0L)
  {
    DosSemClear(vcbptr->common.lua_post_handle);
  }
}

/* luasems.c */
