/**MOD+***********************************************************************/
/* Module:    asample2.c                                                     */
/*                                                                           */
/* Purpose:   TP2, APPC program browser                                      */
/*                                                                           */
/* The transaction programs TP1 and TP2 enable a user to browse through a    */
/* file on another system.  The user of TP1 is presented with a single data  */
/* block at a time, in hex and character format.  After each block a user    */
/* can request the next block, request the previous block, or quit.          */
/* TP1 (the invoking TP) sends a filename to TP2 (the invoked TP).  If TP2   */
/* locates the file, it returns the first block to TP1, otherwise it issues  */
/* deallocate, and terminates.  If TP1 receives a block it displays it on    */
/* the screen and waits for a user prompt; if it receives an indication      */
/* that TP2 has terminated, it too terminates.                               */
/* If the user asks for the next block when TP2 has sent the last one, TP2   */
/* wraps to the beginning of file.  Similarly, TP2 wraps to send the last    */
/* block if the previous one is requested when the first block is displayed. */
/* Neither program attempts to recover from errors.  A 'bad' return code     */
/* from APPC causes the program to terminate with an explanatory message.    */
/*                                                                           */
/*  COPYRIGHT DATA CONNECTION LIMITED 1989-2006                             */
/*                                                                           */
/**MOD-***********************************************************************/

#include <appc_c.h>
#include <acssvcc.h>
#define INCL_SUB
#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>
#include <unistd.h>

/*****************************************************************************/
/* remove comments from following line to implement debugging to screen      */
/*****************************************************************************/
/* #define DEBUGGING */

typedef int  boolean;
#define TRUE  1
#define FALSE 0

/*****************************************************************************/
/* Flipping macros.                                                          */
/* These are required to pass control information between TP1 and TP2 even   */
/* if the TPs are running on systems with different byte ordering.           */
/* Include the file svconfig.h to get the correct value for PFLIPRQD.        */
/* If this file is not available, set PFLIPRQD to 1 for OS's with Intel      */
/* byte ordering - otherwise, set to 0.                                      */
/*****************************************************************************/
#include <svconfig.h>
#if PFLIPRQD==1
#ifdef _LP64
#define PFLIPI(X) (unsigned short)(((X >> 8) & 0x00ff) | ((X << 8) & 0xff00))

#define PLFXIPI(X) (PFLIPI(((unsigned long)X & 0x0000ffff)) << 16 | \
                    PFLIPI( (unsigned long)X >> 16))

#define PLFLIPI(X) ((((unsigned long)PLFXIPI(X)) << 32)) | PLFXIPI((X >> 32))
#else
#define PFLIPI(X) (unsigned short)(((X >> 8) & 0x00ff) | ((X << 8) & 0xff00))

#define PLFLIPI(X) (PFLIPI(((unsigned long)X & 0x0000ffff)) << 16 | \
                    PFLIPI( (unsigned long)X >> 16))
#endif
#else

#define PFLIPI(X) (X)
#define PLFLIPI(X) (X) 

#endif

/*****************************************************************************/
/* structures required by APPC                                               */
/*****************************************************************************/
struct receive_allocate         rec_alloc;
struct mc_send_data             mcsend;
struct mc_receive_and_wait      mcrec;
struct mc_deallocate            mcdealloc;
struct tp_ended                 tpended;
struct convert                  conv_block;              /* for convert verb */


/*****************************************************************************/
/* Data area                                                                 */
/*****************************************************************************/
unsigned char    sharebufptr[512];
FILE * filestream;                    /* pointer to file to be read and sent */

unsigned char   tp_id[ 8 ];   /* TP identifier returned by receive_allocate, */
                               /* passed on all subsequent APPC calls        */
AP_UINT32       conv_id;      /* conversation ID passed by receive_allocate  */

long int    block_wanted;                   /* range 0 to (num_blocks - 1)   */
                                            /* updated when prompt received. */
                                            /* used by get_block().          */
long int num_blocks;                  /* set by get_fname() when file opened */

typedef struct                         /* data to be sent to partner TP      */
{
   size_t block_number;                /* this block number (1 - num_blocks) */
   size_t num_blocks;                  /* number of blocks in file           */
   size_t block_size;                  /* bytes in this block                */
   size_t data_block[ 256 ];
} BLOCK_TO_SEND;

void  initialize();
void  send();
boolean receive_wait();
void  deallocate();
void  exit_error();
void  terminate();
boolean get_fname();
char  get_prompt();
long int   get_block();

/**PROC+**********************************************************************/
/* Name:      main                                                           */
/*                                                                           */
/* Purpose:   Handles protocols for file access end of file browser          */
/*                                                                           */
/* Returns:   0  if tp stops conversing for any reason                       */
/*                                                                           */
/* Params:    NONE                                                           */
/*                                                                           */
/* Operation: This tp is auto started by the other tp. It receives the       */
/*            allocate, and then receive_and_waits for the filename. If this */
/*            is invalid, it deallocate_abends, or otherwise sends the first */
/*            block, and send permission back, so that it may receive        */
/*            instructions as to which block the user wants next. This is    */
/*            sent and the next instruction awaited.                         */
/*                                                                           */
/**PROC-**********************************************************************/
int main()
{
   char      prompt;
   char      fname[ 65 ];
   boolean   conversing = TRUE;
   boolean   first_time = TRUE;
   long int       bytes_read;

   initialize();
   do
   {
      if( receive_wait() )
      {

         switch( mcrec.what_rcvd )
         {

            case AP_DATA_COMPLETE_SEND :
               if ( first_time )
               {
                  if (get_fname(fname, sizeof( fname ) - 1 ) )
                  {
                     /********************************************************/
                     /* If first time, simulate a forward 1 block command    */
                     /********************************************************/
                     prompt = 'F';
                     first_time = FALSE;
                  }
                  else
                  {
                     deallocate();
                     conversing = FALSE;
                  }
               }
               else
               {
                  /***********************************************************/
                  /* If not first time, read command from receive buffer     */
                  /***********************************************************/
                  prompt = get_prompt();
               }
               if( prompt == 'F' )
               {
                  block_wanted++;
               }
               else if ( prompt == 'B' )
               {
                  block_wanted--;
               }
               else
               {
                  printf( "TP2: Unknown Prompt hex %02x received\n", prompt );
                  conversing = FALSE;
               }
               if ( conversing )
               {
                  bytes_read = get_block( (BLOCK_TO_SEND *) sharebufptr );

                  /***********************************************************/
                  /* Convert all integers to line format before sending      */
                  /***********************************************************/
                  ((BLOCK_TO_SEND*)sharebufptr)->block_number =
                    PLFLIPI(((BLOCK_TO_SEND*)sharebufptr)->block_number);
                  ((BLOCK_TO_SEND*)sharebufptr)->num_blocks =
                    PLFLIPI(((BLOCK_TO_SEND*)sharebufptr)->num_blocks);
                  ((BLOCK_TO_SEND*)sharebufptr)->block_size =
                    PLFLIPI(((BLOCK_TO_SEND*)sharebufptr)->block_size);

                  /***********************************************************/
                  /* Send the data, allowing for num_blocks, block number    */
                  /* and block size in the length to be sent                 */
                  /***********************************************************/
                  send( sharebufptr, bytes_read + (unsigned char *)
                      (((BLOCK_TO_SEND*)sharebufptr)->data_block)-sharebufptr);

                  /***********************************************************/
                  /* Convert all integers back from line format now          */
                  /***********************************************************/
                  /* Repair all ints from line format */
                  ((BLOCK_TO_SEND*)sharebufptr)->block_number =
                    PLFLIPI(((BLOCK_TO_SEND*)sharebufptr)->block_number);
                  ((BLOCK_TO_SEND*)sharebufptr)->num_blocks =
                    PLFLIPI(((BLOCK_TO_SEND*)sharebufptr)->num_blocks);
                  ((BLOCK_TO_SEND*)sharebufptr)->block_size =
                    PLFLIPI(((BLOCK_TO_SEND*)sharebufptr)->block_size);
               }
               break;

            default :
               break;
         }
      }
      else
      {
         conversing = FALSE;
      }

   } while( conversing );
   terminate();
   return( 0 );
}


/**PROC+**********************************************************************/
/* Name:      initialize                                                     */
/*                                                                           */
/* Purpose:   Initializes the converstion                                    */
/*                                                                           */
/* Returns:   NONE                                                           */
/*                                                                           */
/* Params:    NONE                                                           */
/*                                                                           */
/* Operation: Sets up lu_alias and tp_name and issues a receive_allocate verb*/
/*                                                                           */
/**PROC-**********************************************************************/
void initialize()
{

   unsigned char  tp_name[ 64 ];

#ifdef DEBUGGING
   printf( "TP2: initialize: top\n" );
#endif


   memset( &rec_alloc, (int ) '\0', sizeof( rec_alloc ) );

   rec_alloc.opcode = AP_RECEIVE_ALLOCATE;

   memset( tp_name, ( int ) ' ', sizeof( tp_name ) );           /* 64 blanks */
   memcpy( tp_name, "TPNAME2", 7 );

   /**************************************************************************/
   /* convert ASCII tp_name to EBCDIC rec_alloc.tp_name                      */
   /**************************************************************************/
   memset(&conv_block, 0, sizeof(conv_block));
   conv_block.opcode    = SV_CONVERT;
   conv_block.opext     = 0;
   conv_block.direction = SV_ASCII_TO_EBCDIC;
   conv_block.char_set  = SV_AE;
   conv_block.len       = sizeof( tp_name );
   conv_block.source    = tp_name;
   conv_block.target    = rec_alloc.tp_name;
#ifdef DEBUGGING
   printf( "TP2: initialize: calling ACSSVC_C\n" );
#endif
   ACSSVC_C( &conv_block );

#ifdef DEBUGGING
   printf( "TP2: initialize: calling APPC\n" );
#endif

   APPC( &rec_alloc );

#ifdef DEBUGGING
   printf( "TP2: initialize: returned from APPC\n" );
#endif

   if ( rec_alloc.primary_rc  == AP_OK )
   {
      /***********************************************************************/
      /* save tp id                                                          */
      /***********************************************************************/
      memcpy( tp_id, rec_alloc.tp_id, sizeof( tp_id ) );
      conv_id = rec_alloc.conv_id;
   }
   else
   {
      exit_error( rec_alloc.primary_rc );
   }

   block_wanted = -1;

#ifdef DEBUGGING
   printf( "TP2: initialize: end\n" );
#endif
}


/**PROC+**********************************************************************/
/* Name:      send                                                           */
/*                                                                           */
/* Purpose:   Issue an mc_send call                                          */
/*                                                                           */
/* Returns:   NONE                                                           */
/*                                                                           */
/* Params:    IN   datalen      length of data to be sent                    */
/*            IN   *datastring  data to be sent                              */
/*                                                                           */
/* Operation: NOTE: This procedure uses the standard function of the send    */
/*                  verb. Compare with send in asample1 where multiple verbs */
/*                  may be issued with a single call.                        */
/*                                                                           */
/**PROC-**********************************************************************/
void send( datastring, datalen )

   unsigned char * datastring;
   unsigned    datalen;
{

#ifdef DEBUGGING
   printf( "TP2: send: top with datalen %d\n", datalen );
#endif

   memset(&mcsend, 0, sizeof(mcsend));
   mcsend.opcode  = AP_M_SEND_DATA;
   mcsend.opext   = AP_MAPPED_CONVERSATION;
   memcpy( mcsend.tp_id, tp_id, sizeof( tp_id ) );
   mcsend.conv_id = conv_id;
   mcsend.dlen    = datalen;
   mcsend.dptr    = datastring;

   APPC( &mcsend );

   if ( mcsend.primary_rc != AP_OK )
   {
      exit_error( mcsend.primary_rc );
   }

#ifdef DEBUGGING
   printf( "TP2: send: end\n" );
#endif

}


/**PROC+**********************************************************************/
/* Name:      receive_wait                                                   */
/*                                                                           */
/* Purpose:   Issue an mc_receive_and_wait for filename or command           */
/*                                                                           */
/* Returns:   TRUE   - data received                                         */
/*            FALSE  - user terminated asample1                              */
/*                                                                           */
/* Params:    NONE                                                           */
/*                                                                           */
/* Operation: NOTE: This is the extended usage of receive verbs where status */
/*                  e.g. AP_SEND and data can be received together on 1 call */
/*                  of the verb. See asample1 for the standard usage.        */
/*                                                                           */
/**PROC-**********************************************************************/
boolean receive_wait()
{

#ifdef DEBUGGING
   printf( "TP2: receive_wait: top\n" );
#endif
   memset(&mcrec, 0, sizeof(mcrec));
   mcrec.opcode     = AP_M_RECEIVE_AND_WAIT;
   mcrec.opext      = AP_MAPPED_CONVERSATION;
   memcpy (mcrec.tp_id, tp_id, sizeof( tp_id ) );
   mcrec.conv_id    = conv_id;
   mcrec.rtn_status = AP_YES;
   mcrec.max_len    = 64;
   mcrec.dptr       = sharebufptr;
   APPC( &mcrec );

   if ( mcrec.primary_rc == AP_DEALLOC_NORMAL )
   {
      return( FALSE );
   }
   else if ( mcrec.primary_rc != AP_OK )
   {
      exit_error( mcrec.primary_rc );
   }

#ifdef DEBUGGING
   printf( "TP2: receive_wait: end\n" );
#endif
   return( TRUE );
}


/**PROC+**********************************************************************/
/* Name:      get_prompt                                                     */
/*                                                                           */
/* Purpose:   Reads the data buffer to get the next instruction              */
/*                                                                           */
/* Returns:   F    - send the next block of file                             */
/*            B    - send the previous block of file                         */
/*           ' '   - no command found                                        */
/*                                                                           */
/* Params:    NONE                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
char get_prompt()
{
   int    ch;

#ifdef DEBUGGING
   printf( "TP2: get_prompt: top\n" );
#endif
   ch = *sharebufptr;
   if ( mcrec.dlen == 1 && ( ch == 'F' || ch == 'B' ) )
   {
      return( ch );
   }
   else
   {
      return( ' ' );
   }

}


/**PROC+**********************************************************************/
/* Name:      get_fname                                                      */
/*                                                                           */
/* Purpose:   Reads name of and opens file to be browsed                     */
/*                                                                           */
/* Returns:   FALSE  failed to open the file                                 */
/*            TRUE   successfully opened the file                            */
/*                                                                           */
/* Params:    OUT   *fname  name of file to browse                           */
/*            IN    maxlen  maximum permitted length of filename             */
/*                                                                           */
/**PROC-**********************************************************************/
boolean get_fname( fname, maxlen )
   char * fname;
   int    maxlen;
{
   boolean rc;
   long int filesize;
   char * type = "r";

#ifdef DEBUGGING
   printf(
   "TP2: get_fname: top, %d length recd, data %13.0Fs.\n",
             mcrec.dlen, ( char * ) sharebufptr );
#endif
   if ( mcrec.dlen > (unsigned short) maxlen )
   {
      return( FALSE );
   }

   memcpy( fname, sharebufptr, mcrec.dlen );
   fname[ mcrec.dlen ] = '\0';
   filestream = fopen ( fname, type );
   if ( filestream != NULL)
   {
      rc = TRUE;
      /***********************************************************************/
      /* go to end of file and find offset from beginning - this gives size  */
      /* of file. Then calulate no. of 256 byte blocks.                      */
      /***********************************************************************/
      fseek ( filestream, 0L, 2 );
      filesize = ftell ( filestream ) ;
      num_blocks = ( filesize + 255L ) / 256;
   }
   else
   {
      rc = FALSE;
   }


#ifdef DEBUGGING
   printf( "TP2: get_fname: size of file %s was %d, no. of blocks %d\n",
       fname, filesize, num_blocks );
#endif

   return( rc );
}


/**PROC+**********************************************************************/
/* Name:      get_block                                                      */
/*                                                                           */
/* Purpose:   Reads block of data from file                                  */
/*                                                                           */
/* Returns:   bytes_read   the number of bytes read from the file            */
/*                                                                           */
/* Params:    IN/OUT *psblock  structure of data to be sent and length info  */
/*                                                                           */
/* Operation: External block_wanted holds the number of the required block.  */
/*            This is checked for validity, then the block is read, and the  */
/*            structure updated.                                             */
/*                                                                           */
/**PROC-**********************************************************************/
long int  get_block( psblock )
   BLOCK_TO_SEND * psblock;
{
   char  buf[ 256 ];
   size_t bytes_read = 0;

#ifdef DEBUGGING
   printf( "TP2: get_block: top to read block number %d\n", block_wanted );
#endif

   /**************************************************************************/
   /* make sure block wanted is within range 0 - num_blocks, then seek it    */
   /* and read the information.                                              */
   /**************************************************************************/

   block_wanted = block_wanted < 0 ? num_blocks - 1 :
                                     block_wanted % num_blocks;

   fseek( filestream, block_wanted * 256L, 0 );

   if ( bytes_read = fread( buf, 1, 256, filestream )  )
   {

      memcpy( psblock->data_block, buf, bytes_read );
      psblock->block_number = block_wanted + 1;
      psblock->num_blocks   = num_blocks;
      psblock->block_size   = bytes_read;


#ifdef DEBUGGING
      printf(
  "TP2: fread has read %d bytes for block %d\n", bytes_read, block_wanted );
#endif
   }
   else
   {
      printf( "TP2: file read failed\n" );
   }

   return( bytes_read );
}


/**PROC+**********************************************************************/
/* Name:      deallocate                                                     */
/*                                                                           */
/* Purpose:   Issues a deallocate abend                                      */
/*                                                                           */
/* Returns:   NONE                                                           */
/*                                                                           */
/* Params:    NONE                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void deallocate()
{

#ifdef DEBUGGING
   printf( "TP2: deallocate: end\n" );
#endif

   memset(&mcdealloc, 0, sizeof(mcdealloc));
   mcdealloc.opcode       = AP_M_DEALLOCATE;
   mcdealloc.opext        = AP_MAPPED_CONVERSATION;
   memcpy( mcdealloc.tp_id, tp_id, sizeof( tp_id ) );
   mcdealloc.conv_id      = conv_id;
   mcdealloc.dealloc_type = AP_ABEND;
   APPC( &mcdealloc );

   if ( mcdealloc.primary_rc != AP_OK )
   {
      exit_error( mcdealloc.primary_rc );
   }

#ifdef DEBUGGING
   printf( "TP2: deallocate: end\n" );
#endif
}


/**PROC+**********************************************************************/
/* Name:      exit_error                                                     */
/*                                                                           */
/* Purpose:   Produces error message for APPC error conditions               */
/*                                                                           */
/* Returns:   -1   Error has occurred                                        */
/*                                                                           */
/* Params:    rc   the APPC return code                                      */
/*                                                                           */
/**PROC-**********************************************************************/
void exit_error( rc )
   int    rc;
{

   switch( rc )
   {
      case AP_ALLOCATION_ERROR :
         printf( "TP2: APPC returned allocation error\n" );
         break;

      case AP_CANCELLED :
         printf( "TP2: APPC returned cancelled\n" );
         break;

      case AP_COMM_SUBSYSTEM_NOT_LOADED :
         printf( "TP2: APPC returned Comm Subsystem Not Loaded\n" );
         break;

      case AP_DEALLOC_ABEND :
         printf( "TP2: APPC returned Dealloc_abend\n" );
         break;

      case AP_DEALLOC_NORMAL :
         printf( "TP2: APPC returned Dealloc_normal\n" );
         break;
      case AP_PARAMETER_CHECK :
         printf( "TP2: APPC returned Parameter Check\n" );
         break;

      case AP_STATE_CHECK :
         printf( "TP2: APPC returned State Check\n" );
         break;

      case AP_TP_BUSY :
         printf( "TP2: APPC returned TP_Busy\n" );
         break;

      case AP_UNSUCCESSFUL :
         printf( "TP2: APPC returned Unsuccessful\n" );
         break;

      default :
         printf( "TP2: APPC returned primary_rc %04X\n", rc );
         break;
   }
   exit( -1 );
}


/**PROC+**********************************************************************/
/* Name:      terminate                                                      */
/*                                                                           */
/* Purpose:   Issues a tp_ended verb and closes the file                     */
/*                                                                           */
/* Returns:   NONE                                                           */
/*                                                                           */
/* Params:    NONE                                                           */
/*                                                                           */
/**PROC-**********************************************************************/
void terminate()
{

#ifdef DEBUGGING
   printf( "TP2: terminate: top\n" );
#endif

   fclose ( filestream );

   memset(&tpended, 0, sizeof(tpended));
   tpended.opcode = AP_TP_ENDED;
   memcpy( tpended.tp_id, tp_id, sizeof( tp_id ) );
   APPC( &tpended );

   if ( tpended.primary_rc != AP_OK )
   {
      exit_error( tpended.primary_rc );
   }

#ifdef DEBUGGING
   printf( "TP2: terminate: end\n" );
#endif
}
