-----------------------------------------------------------------------
--(c) Copyright IBM Corporation 2006  All rights reserved.           --
--                                                                   --
--This sample program is owned by International Business Machines    --
--Corporation or one of its subsidiaries ("IBM") and is copyrighted  --
--and licensed, not sold.                                            --
--BY ACCESSING, COPYING, OR USING THIS SAMPLE PROGRAM, YOU AGREE TO  --
--THE TERMS OF THE AGREEMENT TITLED "International License Agreement --
--for Non-Warranted db2perf Programs" LOCATED IN THE FILE NAMED      --
--"license.txt".                                                     --
--                                                                   --
-- db2perf_plans.db2                                                 --
-- Steve Rees - srees@ca.ibm.com                                     --
--                                                                   --
-- Analyzes explain tables to identify biggest hitters in terms of   --
-- total cost and IO cost.   Also identifies indexes that aren't     --
-- used in the current explain plans.                                --
--                                                                   --
-----------------------------------------------------------------------

CALL db2perf_quiet_drop('PROCEDURE db2perf_plans')@

CALL db2perf_quiet_drop('TABLE db2perf_plans_report')@
CREATE TABLE db2perf_plans_report( line INTEGER, message VARCHAR(128) )@


-----------------------------------------------------------------------
-- Create a scratch table db2perf_plans_ranking, where we store information  
-- about the statements we're analyzing.

CALL db2perf_quiet_drop('TABLE db2perf_plans_ranking')@
CREATE TABLE db2perf_plans_ranking( 
      EXPLAIN_REQUESTER VARCHAR(128),
      EXPLAIN_TIME      TIMESTAMP,
      SOURCE_NAME       VARCHAR(128),
      SOURCE_SCHEMA     VARCHAR(128),
      SOURCE_VERSION    VARCHAR(64),
      STMTNO            INTEGER,
      SECTNO    	INTEGER,
      short_text        VARCHAR(100),
      total_ranking     INTEGER,
      io_ranking        INTEGER )@


-----------------------------------------------------------------------
-- db2perf_plans
-----------------------------------------------------------------------

CREATE PROCEDURE db2perf_plans()
DYNAMIC RESULT SETS 1
BEGIN
  DECLARE line INTEGER DEFAULT 0;
  
  DECLARE start 	      INTEGER;
  DECLARE done  	      INTEGER;
  DECLARE stmt_buffer 	      VARCHAR(100);

  DECLARE row_number 	      SMALLINT;
  DECLARE v_explain_requester VARCHAR(128);
  DECLARE v_explain_time      TIMESTAMP;
  DECLARE v_source_name       VARCHAR(128);
  DECLARE v_source_schema     VARCHAR(128);
  DECLARE v_source_version    VARCHAR(64);
  DECLARE v_stmtno            INTEGER;
  DECLARE v_sectno	      INTEGER;
  DECLARE v_total_cost        DOUBLE;
  DECLARE v_io_cost           DOUBLE;

  DECLARE total_ranking SMALLINT;
  DECLARE io_ranking    SMALLINT;
  DECLARE short_text    VARCHAR(100);

  DECLARE short_schema_source CHAR(32);
  DECLARE statement_text      CLOB(2M);

  DECLARE table_name   VARCHAR(128);
  DECLARE schema       VARCHAR(128);
  DECLARE index_name   VARCHAR(128);
  DECLARE count        INTEGER;
  DECLARE header_done  INTEGER;

  DECLARE at_end INT DEFAULT 0;

  ----------------------------------------------------------------------
  -- Declare a cursor to fetch the top 10 most expensive statements
  -- from explain_statement.  We need a self-join because the cost
  -- is stored in the optimized ('P') record, and the text is stored in
  -- the original ('O') record.

  DECLARE top10_total_cost CURSOR FOR
    SELECT 
        CAST( row_number() OVER (ORDER BY p.total_cost DESC) AS SMALLINT) as row_num,
	p.total_cost,
	o.explain_requester,
	o.explain_time,
	o.source_name,
	o.source_schema,
	o.source_version,
	o.stmtno,
	o.sectno,
        o.statement_text
    FROM explain_statement o, explain_statement p
    WHERE o.explain_level     = 'O'
      AND p.explain_level     = 'P'
      AND o.EXPLAIN_REQUESTER = p.EXPLAIN_REQUESTER
      AND o.EXPLAIN_TIME      = p.EXPLAIN_TIME
      AND o.SOURCE_NAME       = p.SOURCE_NAME
      AND o.SOURCE_SCHEMA     = p.SOURCE_SCHEMA
      AND o.SOURCE_VERSION    = p.SOURCE_VERSION
      AND o.STMTNO            = p.STMTNO
    ORDER by p.total_cost desc
    FETCH first 10 rows only;

  ----------------------------------------------------------------------
  -- Declare a cursor to fetch the top 10 most expensive statements
  -- by IO.   Here we join explain_statement (for the statement text)
  -- and explain_operator (for the IO cost of the topmost operator,
  -- the RETURN.

  DECLARE top10_io_cost CURSOR FOR
    SELECT 
        CAST( row_number() OVER (ORDER BY o.io_cost DESC) AS SMALLINT) as row_num,
	o.io_cost,
	s.explain_requester,
	s.explain_time,
	s.source_name,
	s.source_schema,
	s.source_version,
	s.stmtno,
	s.sectno,
        s.statement_text
    FROM explain_statement s, explain_operator o
    WHERE s.explain_level     = 'O'
      AND o.operator_type     = 'RETURN'
      AND s.EXPLAIN_REQUESTER = o.EXPLAIN_REQUESTER
      AND s.EXPLAIN_TIME      = o.EXPLAIN_TIME
      AND s.SOURCE_NAME       = o.SOURCE_NAME
      AND s.SOURCE_SCHEMA     = o.SOURCE_SCHEMA
      AND s.SOURCE_VERSION    = o.SOURCE_VERSION
      AND s.STMTNO            = o.STMTNO
    ORDER by o.io_cost desc
    FETCH first 10 rows only;

  DECLARE ranking_cursor CURSOR FOR
    SELECT
      total_ranking,
      io_ranking,
      short_text
    FROM db2perf_plans_ranking
    ORDER BY total_ranking, io_ranking;

  
  ----------------------------------------------------------------------
  -- The following 3 cursors are used to find the tables referred to
  -- in the plans, the indexes defined on them, and then the indexes
  -- that aren't actually used in the plans here.

  DECLARE table_cursor CURSOR FOR
    SELECT distinct object_schema, object_name
    FROM   explain_object
    WHERE  object_type = 'TA';

  DECLARE defined_idx_cursor CURSOR FOR
    SELECT indname
    FROM   syscat.indexes
    WHERE  tabname   = table_name
      AND  tabschema = schema;

  DECLARE unused_idx_cursor CURSOR FOR
    SELECT distinct object_name
    FROM   explain_object
    WHERE  object_type = 'IX'
      AND  object_name = index_name
      AND  object_schema = schema;

  DECLARE report_cursor CURSOR WITH RETURN TO CALLER FOR
    SELECT message
    FROM db2perf_plans_report
    ORDER BY line;
 
  DECLARE CONTINUE HANDLER FOR NOT FOUND
    SET at_end = 1;

  DELETE FROM db2perf_plans_report;
  DELETE FROM db2perf_plans_ranking;

  OPEN top10_total_cost;

  INSERT INTO db2perf_plans_report 
  	VALUES ( line, ' ' ); 
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 'Top 10 most expensive statements - total cost' );  
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, ' ' ); 
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 'Rank  Cost       Source                          Section' ); 
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 
	'---------------------------------------------------------------------------------------------------------------------' ); 
  SET line=line+1;

  SET at_end = 0;
  FETCH top10_total_cost 
    INTO row_number,
         v_total_cost,
	 v_explain_requester,
	 v_explain_time,
	 v_source_name,
	 v_source_schema,
	 v_source_version,
	 v_stmtno,
	 v_sectno,
	 statement_text;

  WHILE at_end = 0 DO
    ----------------------------------------------------------------------
    -- For each of the statements in the top 10, write its information to the ranking table.

    INSERT INTO db2perf_plans_ranking
	VALUES ( v_explain_requester, 
		 v_explain_time, 
		 v_source_name, 
		 v_source_schema, 
		 v_source_version, 
		 v_stmtno, 
		 v_sectno, 
		 substr(statement_text,1,100), 
		 row_number,
		 NULL );

    ----------------------------------------------------------------------
    -- One by one, write the details about the top 10 total cost statements
    -- to the output table.

    SET short_schema_source = rtrim( substr(v_source_schema,1,16)) || '.' || rtrim( substr(v_source_name,1,16));
    INSERT INTO db2perf_plans_report VALUES ( line,
	CHAR( row_number )
    ||  CHAR( CAST( v_total_cost AS INTEGER ) )
    ||  short_schema_source
    ||  CHAR( v_sectno ) );    SET line=line+1;

    SET start = 1;
    SET done  = 0;
    SET stmt_buffer = substr(statement_text,start,100);


    INSERT INTO db2perf_plans_report VALUES ( line,' ' ); SET line=line+1;


    ----------------------------------------------------------------------
    -- Often times the most expensive statements are very long too, so we
    -- write out the statement text in pieces, 100 bytes at a time.

    WHILE done = 0 DO
      INSERT INTO db2perf_plans_report VALUES
	    ( line, '                 ' || rtrim(ltrim(stmt_buffer)) );
      SET line = line+1;

      IF start+100 >= length(statement_text) THEN
	SET done = 1;
      ELSE
        ----------------------------------------------------------------------
	-- Not done yet - get the next piece.
	SET start = start + 100;
	SET stmt_buffer = substr(statement_text,start,100);
      END IF;

    END WHILE;

    INSERT INTO db2perf_plans_report VALUES ( line,' ' ); SET line=line+1;

    ----------------------------------------------------------------------
    -- The previous statement is done, get the next one.
    FETCH top10_total_cost 
      INTO row_number,
	   v_total_cost,
	   v_explain_requester,
	   v_explain_time,
	   v_source_name,
	   v_source_schema,
	   v_source_version,
	   v_stmtno,
	   v_sectno,
	   statement_text;

  END WHILE;

  CLOSE top10_total_cost;



  ----------------------------------------------------------------------
  -- Repeat the above process iwth the top 10 statements in IO cost.

  OPEN top10_io_cost;

  INSERT INTO db2perf_plans_report 
  	VALUES ( line, ' ' ); 
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 'Top 10 most expensive statements - I/O cost' );  
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, ' ' ); 
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 'Rank  Cost       Source                          Section' ); 
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 
	'---------------------------------------------------------------------------------------------------------------------' ); 
  SET line=line+1;

  SET at_end = 0;
  FETCH top10_io_cost 
    INTO row_number,
         v_io_cost,
	 v_explain_requester,
	 v_explain_time,
	 v_source_name,
	 v_source_schema,
	 v_source_version,
	 v_stmtno,
	 v_sectno,
	 statement_text;

  WHILE at_end = 0 DO

    ----------------------------------------------------------------------
    -- Above, for total cost, we inserted rows into the ranking table for the top 10.
    -- Here, we start by assuming the "top N" row in IO is alrady in the ranking
    -- table, and we can update it (to record the ranking in terms of IO.)  
    -- If the update fails, we insert.

    UPDATE db2perf_plans_ranking r
	SET   r.io_ranking        = row_number
	WHERE r.explain_requester = v_explain_requester
	  AND r.explain_time      = v_explain_time
	  AND r.source_name       = v_source_name
	  AND r.source_schema     = v_source_schema
	  AND r.source_version    = v_source_version
	  AND r.stmtno            = v_stmtno
	  AND r.sectno            = v_sectno;

    IF at_end = 1 THEN
      INSERT INTO db2perf_plans_ranking
	  VALUES ( v_explain_requester, 
		   v_explain_time, 
		   v_source_name, 
		   v_source_schema, 
		   v_source_version, 
		   v_stmtno, 
		   v_sectno, 
		   substr(statement_text,1,100), 
		   NULL,
		   row_number );
    END IF;

    SET short_schema_source = rtrim( substr(v_source_schema,1,16)) || '.' || rtrim( substr(v_source_name,1,16));
    INSERT INTO db2perf_plans_report VALUES ( line,
	CHAR( row_number )
    ||  CHAR( CAST( v_io_cost AS INTEGER ) )
    ||  short_schema_source
    ||  CHAR( v_sectno ) );    SET line=line+1;

    SET start = 1;
    SET done  = 0;
    SET stmt_buffer = substr(statement_text,start,100);

    INSERT INTO db2perf_plans_report VALUES ( line,' ' ); SET line=line+1;

    ----------------------------------------------------------------------
    -- Insert the wrapped statement text as well

    WHILE done = 0 DO
      INSERT INTO db2perf_plans_report VALUES
	    ( line, '                 ' || rtrim(ltrim(stmt_buffer)) );
      SET line = line+1;

      IF start+100 >= length(statement_text) THEN
	SET done = 1;
      ELSE
	SET start = start + 100;
	SET stmt_buffer = substr(statement_text,start,100);
      END IF;

    END WHILE;

    INSERT INTO db2perf_plans_report VALUES ( line,' ' ); SET line=line+1;

    FETCH top10_io_cost 
      INTO row_number,
         v_io_cost,
	 v_explain_requester,
	 v_explain_time,
	 v_source_name,
	 v_source_schema,
	 v_source_version,
	 v_stmtno,
	 v_sectno,
	 statement_text;

  END WHILE;
  CLOSE top10_io_cost;



  ----------------------------------------------------------------------
  -- Now that we have Top 10 by total cost and IO cost determined
  -- and recorded, we can go back and pull out the Top 10 overall.
  -- That is, statements that are in the union of these sets.

  INSERT INTO db2perf_plans_report 
  	VALUES ( line, ' ' ); 
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 'Comparative ranking by total cost & I/O cost' );  
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, ' ' ); 				 
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 'Total I/O' );             
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 'Cost  Cost' );            
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 'Rank  Rank  Statement' ); 
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 
	'---------------------------------------------------------------------------------------------------------------------' ); 
  SET line=line+1;


  OPEN ranking_cursor;

  SET at_end = 0;
  FETCH ranking_cursor 
    INTO total_ranking,io_ranking,short_text;

  WHILE at_end = 0 DO
    
    INSERT INTO db2perf_plans_report VALUES ( line,
	COALESCE( CHAR( total_ranking ), '      ' )
    ||  COALESCE( CHAR( io_ranking ), '      ' )
    ||  short_text );
    SET line=line+1;

    FETCH ranking_cursor 
      INTO total_ranking,io_ranking,short_text;
  END WHILE;

  CLOSE ranking_cursor;


  ----------------------------------------------------------------------
  -- Now we report all unused indexes

  INSERT INTO db2perf_plans_report 
  	VALUES ( line, ' ' ); 
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, ' ' ); 
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 'Tables with unreferenced indexes in these plans' );  
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, ' ' ); 
  SET line=line+1;
  INSERT INTO db2perf_plans_report 
  	VALUES ( line, 
	'---------------------------------------------------------------------------------------------------------------------' ); 
  SET line=line+1;


  OPEN table_cursor;

  SET at_end = 0;
  FETCH table_cursor 
    INTO schema,table_name;

  WHILE at_end = 0 DO

    ----------------------------------------------------------------------
    -- For each table listed in the explain tables, get all defined indexes

    SET header_done=0;

    OPEN defined_idx_cursor;

    FETCH defined_idx_cursor
      INTO index_name;

    WHILE at_end = 0 DO

      ----------------------------------------------------------------------
      -- For each table and index, we see how many times that index was referred to.

      SELECT count(*)
      into   count
      FROM   explain_object
      WHERE  object_type   = 'IX'
	AND  object_name   = index_name
	AND  object_schema = schema;
      
      IF count = 0 THEN
        
        ----------------------------------------------------------------------
	-- We found an index that isn't used, even though its base table is referred to.
	-- Report the unused index name.

	IF header_done = 0 THEN
	  INSERT INTO db2perf_plans_report 
	  	VALUES ( line,'Table: '||substr(rtrim(schema)||'.'||rtrim(table_name),1,80) );
	  SET line=line+1;
	  SET header_done=1;
	END IF;

        INSERT INTO db2perf_plans_report VALUES 
		( line,'       '||substr(rtrim(index_name),1,80) );
	SET line=line+1;

      END IF;
 
      SET at_end = 0;
      FETCH defined_idx_cursor
	INTO index_name;
    END WHILE;

    CLOSE defined_idx_cursor;

    SET at_end = 0;
    FETCH table_cursor 
      INTO schema,table_name;

  END WHILE;

  CLOSE table_cursor;


  OPEN report_cursor;

END@
