/*---------------------------------------------------------------------
--(c) Copyright IBM Corporation 2006  All rights reserved.           --
--                                                                   --
--This sample program is owned by International Business Machines    --
--Corporation or one of its subsidiaries ("IBM") and is copyrighted  --
--and licensed, not sold.                                            --
--BY ACCESSING, COPYING, OR USING THIS SAMPLE PROGRAM, YOU AGREE TO  --
--THE TERMS OF THE AGREEMENT TITLED "International License Agreement --
--for Non-Warranted db2perf Programs" LOCATED IN THE FILE NAMED      --
--"license.txt".                                                     --
--                                                                   --
-- db2perf_sanity.c                                                  --
-- Steve Rees - srees@ca.ibm.com                                     --
--                                                                   --
-- Performs basic sanity checks on database configuration            --
--                                                                   --
---------------------------------------------------------------------*/

#include <stdio.h>
#include <sqlenv.h>
#include <sqlcli.h>
#include <sqlca.h>
#include <db2ApiDf.h>
#include "utilcli.h"

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#ifdef WIN32 
#define DEFAULT_BP_SIZE 250
#else
#define DEFAULT_BP_SIZE 1000
#endif

int main( int argc, char **argv )
{
  int i,j;
  struct sqlca sqlca;
  SQLHENV  henv;
  SQLHDBC  hdbc;
  SQLHSTMT hstmt;

  char dbname[SQL_ALIAS_SZ+1];
  char dbpath[SQL_DRIVE_SZ + SQL_INAME_SZ+1];
  db2CfgParam cfgParameters[7];
  db2Cfg cfgStruct;

  db2DbDirOpenScanStruct 		dbDirOpenParmStruct;
  db2DbDirCloseScanStruct 		dbDirCloseParmStruct;
  struct db2DbDirNextEntryStruct 	dbDirNextEntryParmStruct;
  struct db2DbDirInfo 			*dbEntry = NULL;
  char 					*dbDirPath = NULL;
  db2Uint16 				dbDirHandle = 0;
  db2Uint16 				dbEntryNb = 0;
  db2Uint32 				versionNumber = db2Version820;
  int 					logpath_matched = FALSE;



  short 	logbufsz;
  char     	logpath[SQL_PATH_SZ];
  short 	num_iocleaners;
  short 	num_ioservers;
  sqlint32 	buffpage;
  short 	mincommit;
  sqlint32 	num_poolagents;
  sqlint32 	maxagents;

  SQLRETURN rc;



  if( argc != 2 )
  {
    printf( "Usage: %s <dbname>\n",argv[0] );
    return 1;
  }


  /*
  ** Get the database name from the command line arguments
  */
  for( i=0; i<SQL_ALIAS_SZ && argv[1][i]; ++i )
    dbname[i] = toupper(argv[1][i]);
  for( ; i<SQL_ALIAS_SZ; ++i )
    dbname[i] = ' ';
  dbname[SQL_ALIAS_SZ] = '\0';

  printf("\ndb2perf_sanity\n");
  printf("==============\n\n");
  printf("Running sanity tests on configuration for database %s\n\n\n",dbname);

  rc = SQLAllocHandle (SQL_HANDLE_ENV, SQL_NULL_HANDLE, &henv);
  ENV_HANDLE_CHECK(henv, rc);

  rc = SQLAllocHandle (SQL_HANDLE_DBC, henv, &hdbc);
  DBC_HANDLE_CHECK(hdbc, rc);

  rc = SQLSetConnectAttr (hdbc, SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_OFF, 0);
  DBC_HANDLE_CHECK(hdbc, rc);


  rc = SQLConnect (hdbc, (SQLCHAR *)dbname, SQL_NTS, (SQLCHAR *)"",
                     SQL_NTS, (SQLCHAR *)"", SQL_NTS);
  DBC_HANDLE_CHECK(hdbc, rc);

  rc = SQLAllocHandle (SQL_HANDLE_STMT, hdbc, &hstmt);
  DBC_HANDLE_CHECK(hdbc, rc);


  /*
  ** Get basic configuration parameters for the current database:
  ** log buffer size, log path, number of page cleaners, number of prefetchers,
  ** size of the bufferpool, and setting of mincommit.
  */
  cfgParameters[0].flags = 0;
  cfgParameters[0].token = SQLF_DBTN_LOGBUFSZ;
  cfgParameters[0].ptrvalue = (void*)&logbufsz;

  cfgParameters[1].flags = 0;
  cfgParameters[1].token = SQLF_DBTN_LOGPATH;
  cfgParameters[1].ptrvalue = (void*)&logpath;

  cfgParameters[2].flags = 0;
  cfgParameters[2].token = SQLF_DBTN_NUM_IOCLEANERS;
  cfgParameters[2].ptrvalue = (void*)&num_iocleaners;

  cfgParameters[3].flags = 0;
  cfgParameters[3].token = SQLF_DBTN_NUM_IOSERVERS;
  cfgParameters[3].ptrvalue = (void*)&num_ioservers;

  cfgParameters[4].flags = 0;
  cfgParameters[4].token = SQLF_DBTN_BUFF_PAGE;
  cfgParameters[4].ptrvalue = (void*)&buffpage;

  cfgParameters[5].flags = 0;
  cfgParameters[5].token = SQLF_DBTN_MINCOMMIT;
  cfgParameters[5].ptrvalue = (void*)&mincommit;

  cfgStruct.numItems = 6;
  cfgStruct.flags = db2CfgDatabase;
  cfgStruct.paramArray = cfgParameters;
  cfgStruct.dbname = dbname;

  db2CfgGet(versionNumber, (void *)&cfgStruct, &sqlca);
  DB2_API_CHECK("db2CfgGet");

  /*
  ** We also get a couple of database manager configuration parameters: number of
  ** pool agents and max agents.
  */
  cfgParameters[0].flags = 0 ;
  cfgParameters[0].token = SQLF_KTN_NUM_POOLAGENTS;
  cfgParameters[0].ptrvalue = (void*)&num_poolagents;

  cfgParameters[1].flags = 0 ;
  cfgParameters[1].token = SQLF_KTN_MAXAGENTS;
  cfgParameters[1].ptrvalue = (void*)&maxagents;

  cfgStruct.numItems = 2;
  cfgStruct.paramArray = cfgParameters;
  cfgStruct.flags = db2CfgDatabaseManager;
  cfgStruct.dbname = dbname;

  db2CfgGet(versionNumber, (void *)&cfgStruct, &sqlca);
  DB2_API_CHECK("db2CfgGet");


  /*
  ** First, we want to compare the log path against the default
  ** database path.  If the log is still in the default database
  ** path, we'll suggest to move it.
  */

  dbDirOpenParmStruct.piPath = dbDirPath;
  dbDirOpenParmStruct.oHandle = dbDirHandle;
  db2DbDirOpenScan(versionNumber,
                   &dbDirOpenParmStruct,
                   &sqlca);

  DB2_API_CHECK("Database Directory -- Open");

  dbDirNextEntryParmStruct.iHandle = dbDirHandle;
  dbDirNextEntryParmStruct.poDbDirEntry = dbEntry;
  for (dbEntryNb = 1; dbEntryNb <= dbDirOpenParmStruct.oNumEntries; dbEntryNb++)
  {
    /* 
    ** Get next database directory entry 
    */
    db2DbDirGetNextEntry(versionNumber,
                         &dbDirNextEntryParmStruct,
                         &sqlca);

    DB2_API_CHECK("Database Directory -- Read");


    dbEntry = dbDirNextEntryParmStruct.poDbDirEntry;

    if( !memcmp( dbEntry->alias, dbname, SQL_ALIAS_SZ ) ||
        !memcmp( dbEntry->dbname,dbname, SQL_DBNAME_SZ ) )
    {
      /*
      ** We've found the database we're sanity checking in the directory list.  Concatenate the drive
      ** and DB path.
      */

      for( i=0,j=0; i<SQL_DRIVE_SZ && j<sizeof(dbpath)-1 && dbEntry->drive[i] != ' '; ++i,++j )
        dbpath[j] = dbEntry->drive[i];
      for( i=0; i<SQL_INAME_SZ && j<sizeof(dbpath)-1 && dbEntry->intname[i] != ' '; ++i,++j )
        dbpath[j] = dbEntry->intname[i];
      dbpath[j] = '\0';

      for( logpath_matched=FALSE, i=0; i<sizeof(dbpath) && i<sizeof(logpath) ; ++i )
      {
        if( !logpath[i] || !dbpath[i] || logpath[i] != dbpath[i] )
 	{
	  /*
	  ** We've hit the end of the parts of the log & directory paths that match.
	  ** If it looks like one path is a sub-path of the other, then report
	  ** a warning.
	  */
	  if( ((dbpath [i] == '/' || dbpath[i]  == '\\') && logpath[i] == '\0')
	   || ((logpath[i] == '/' || logpath[i] == '\\') && dbpath[i]  == '\0')
	   || (logpath[i] == '\0' && dbpath[i]  == '\0') )
	    logpath_matched = TRUE;
          break;
	}
      }

      break;
    }
  }



  /*******************
  ** LOGBUFSZ
  *******************/

  printf("LOGBUFSZ:\n");
  if( logbufsz < 128 )
  {
    printf("  Warning:\n");
    printf("    The log buffer size (logbufsz) is currently %d.\n",logbufsz);
    printf("  Recommendation:\n");
    printf("    The generally recommended size is 128 or greater.\n\n\n");
  }
  else
    printf("  Passed\n\n\n");



  /*******************
  ** LOGPATH
  *******************/

  printf("Log path:\n");
  if( logpath_matched )
  {
    printf("  Warning:\n");
    printf("    The transaction log is currently located in '%s',\n" 
	   "    which seems to be under the database path '%s'.\n",
	   logpath,dbpath);
    printf("  Recommendation:\n");
    printf("    In general, the transaction log should be located on its own device(s) if possible.\n\n\n");
  }
  else
    printf("  Passed\n\n\n");




  /*******************
  ** NUM_IOCLEANERS
  *******************/

  printf("NUM_IOCLEANERS:\n");
  {
     sqlint32   dirty_steals_per_10k;
     sqlint64	transactions;


     /*
     ** Before we complain about the number of cleaners, we look to see if cleaning
     ** is working well or not.  One good indicator is whether we're seeing dirty
     ** page steal triggers.  Get those from teh database snapshot table function.
     */
     SQLCHAR stmt[] = "select "
		      "COMMIT_SQL_STMTS, "
		      "cast(10000 * cast(POOL_DRTY_PG_STEAL_CLNS as DOUBLE) / (COMMIT_SQL_STMTS+1) as INTEGER) "
		      "from table( snapshot_database( cast(null as varchar(256)),-1 )) as t";
     
     rc = SQLPrepare(hstmt, stmt, SQL_NTS);
     STMT_HANDLE_CHECK(hstmt, hdbc, rc);

     rc = SQLExecute(hstmt);
     STMT_HANDLE_CHECK(hstmt, hdbc, rc);

     rc = SQLBindCol(hstmt, 1, SQL_C_LONG, &dirty_steals_per_10k, 0, NULL);
     STMT_HANDLE_CHECK(hstmt, hdbc, rc);
     rc = SQLBindCol(hstmt, 2, SQL_C_UBIGINT, &transactions, 0, NULL);
     STMT_HANDLE_CHECK(hstmt, hdbc, rc);

     rc = SQLFetch(hstmt);
     STMT_HANDLE_CHECK(hstmt, hdbc, rc);

     rc = SQLFreeStmt(hstmt, SQL_CLOSE);
     STMT_HANDLE_CHECK(hstmt, hdbc, rc);

     if( (transactions > 1000) && (dirty_steals_per_10k > 1) && (num_iocleaners < 3) )
     {
        /*
	** We're seeing activity in the database, plus we're also seeing > 1 dirty steal per 10k transactions,
	** and NUM_IOCLEANERS is lower than 3.   Complain.
	*/
	printf("  Warning:\n");
	printf("    The database is experiencing dirty page steal triggers (%d per 10,000 Tx).\n",dirty_steals_per_10k);
	printf("    The system has %d page cleaners configured (NUM_IOCLEANERS) which may be too low if dirty steals are occurring.\n",num_iocleaners);
	printf("  Recommendation:\n");
	printf("    Increase NUM_IOCLEANERS to at least the default value of 3, or run db2 autoconfigure, \n");
	printf("    and check for dirty steal triggers again in the Database Snapshot.\n\n\n");
     }
     else if( num_iocleaners < 1 )
     {
        /*
	** We also complain if someone sets the number of cleaners to zero.  This is possible but not advised.
	*/
	printf("  Warning:\n");
	printf("    The database is has no page cleaners defined (NUM_IOCLEANERS).  \n");
	printf("  Recommendation:\n");
	printf("    In general at least 1 should be defined.  DB2 autoconfigure will provide\n");
	printf("    a reasonable value.  \n\n");
     }
     else
       printf("  Passed\n\n\n");
  }



  /*******************
  ** NUM_IOSERVERS
  *******************/

  printf("NUM_IOSERVERS:\n");
  if( num_ioservers < 3 )
  {
    /*
    ** We want to make sure there are at least the default 3 prefetchers, since they're used in
    ** other things beside prefetching (eg BACKUP & RESTORE)
    */
    printf("  Warning:\n");
    printf("    The database has only %d prefetchers (NUM_IOSERVERS) defined.\n",num_ioservers);
    printf("  Recommendation:\n");
    printf("    These processes assist with backup / restore operations, so even if\n");
    printf("    no tablescans are intended to be run on the database, increasing NUM_IOSERVERS\n");
    printf("    may be helpful.   DB2 autoconfigure will provide a reasonable value.\n\n\n");
  }
  else
    printf("  Passed\n\n\n");



  /*******************
  ** BUFFPAGE & NPAGES
  *******************/

  printf("BUFFPAGE & NPAGES:\n");
  if( buffpage == DEFAULT_BP_SIZE )
  {
    char       bpname[129] = "";
    SQLINTEGER sqllen=0;

    SQLCHAR stmt[128];
    sprintf( stmt,"select BPNAME from syscat.bufferpools where npages = -1 or npages = %d or npages = -2",DEFAULT_BP_SIZE);
     
    rc = SQLPrepare(hstmt, stmt, SQL_NTS);
    STMT_HANDLE_CHECK(hstmt, hdbc, rc);

    rc = SQLExecute(hstmt);
    STMT_HANDLE_CHECK(hstmt, hdbc, rc);

    rc = SQLBindCol(hstmt, 1, SQL_C_CHAR, bpname, 128, &sqllen);
    STMT_HANDLE_CHECK(hstmt, hdbc, rc);

    rc = SQLFetch(hstmt);

    if( rc == SQL_SUCCESS )
    {
      /*
      ** We get into this path if BUFFPAGE is left at the default, and if NPAGES is either set to -1 (meaning, 
      ** use BUFFPAGE) or if it's set to the defalt (very small size).   Complain.
      */
      printf("  Warning:\n");
      printf("    BUFFPAGE seems to be left at the default value of %d, but the following\n",DEFAULT_BP_SIZE);
      printf("    bufferpools still have NPAGES set to either -1 or to %d, so they still have the default size\n",		DEFAULT_BP_SIZE);

      while( rc != SQL_NO_DATA_FOUND )
      {
        char *p;
        for( p=bpname; (*p != '\0') && (*p != ' '); ++p )
	  ;
        *p = '\0';
        printf("      %s\n",bpname);

        rc = SQLFetch(hstmt);
      }

      printf("  Recommendation:\n");
      printf("    Use ALTER BUFFERPOOL to set NPAGES to the desired value for all bufferpools.\n\n\n");
    }
    else
      printf("  Passed\n\n\n");
	
    rc = SQLFreeStmt(hstmt, SQL_CLOSE);
    STMT_HANDLE_CHECK(hstmt, hdbc, rc);

  }
  else
  {
    /*
    ** In this path, BUFFPAGE is no longer set to the default, so we want to make sure NPAGES is set to
    ** -1 in order to pick up BUFFPAGE.
    */

    char       bpname[129] = "";
    SQLINTEGER sqllen=0;
    sqlint32   npages;

    SQLCHAR stmt[] = "select BPNAME,NPAGES from syscat.bufferpools where npages > -1";
     
    rc = SQLPrepare(hstmt, stmt, SQL_NTS);
    STMT_HANDLE_CHECK(hstmt, hdbc, rc);

    rc = SQLExecute(hstmt);
    STMT_HANDLE_CHECK(hstmt, hdbc, rc);

    rc = SQLBindCol(hstmt, 1, SQL_C_CHAR, bpname, 128, &sqllen);
    STMT_HANDLE_CHECK(hstmt, hdbc, rc);
    rc = SQLBindCol(hstmt, 2, SQL_C_LONG, &npages, 0, 0);
    STMT_HANDLE_CHECK(hstmt, hdbc, rc);

    rc = SQLFetch(hstmt);

    if( rc == SQL_SUCCESS )
    {
      printf("  Warning:\n");
      printf("    BUFFPAGE is set to %d, but the following bufferpools have NPAGES set to\n",buffpage);
      printf("    a specific value, so they will not pick up BUFFPAGE\n");
      printf("      NPAGES\t\tBPNAME\n");
      printf("      ------\t\t-----------------------------------------------\n");

      while( rc != SQL_NO_DATA_FOUND )
      {
        char *p;
        for( p=bpname; (*p != '\0') && (*p != ' '); ++p )
	  ;
        *p = '\0';
        printf("      %d\t\t%s\n",npages,bpname);

        rc = SQLFetch(hstmt);
      }
      printf("  Recommendation:\n");
      printf("    Use ALTER BUFFERPOOL to set NPAGES to the desired value for all bufferpools.\n");
      printf("    Don't mix NPAGES and BUFFPAGE.\n\n\n");
    }
    else
      printf("  Passed\n\n\n");
	
    rc = SQLFreeStmt(hstmt, SQL_CLOSE);
    STMT_HANDLE_CHECK(hstmt, hdbc, rc);

  }



  /*******************
  ** MINCOMMIT
  *******************/

  printf("MINCOMMIT: \n");
  if( mincommit > 1 )
  {
    /*
    ** MINCOMMIT <> 1 is rarely a really good idea, so from a sanity check perspective,
    ** we complain if we see it set otherwise.
    */
    printf("  Warning:\n");
    printf("    Mincommit is currently set to %d instead of the default of 1.\n",mincommit);
    printf("    A mincommit value greater than 1 is usually only appropriate if there are hundreds of connections\n");
    printf("    executing transactions at a very high rate.\n");
    printf("  Recommendation: \n");
    printf("    Set mincommit to 1 unless thorough testing has been done to arrive at the current setting\n\n");
  }
  else
    printf("  Passed\n\n\n");



  /*******************
  ** NUM_POOLAGENTS
  *******************/

  printf("NUM_POOLAGENTS:\n");
  {
     sqlint64	agents_top = 0;

     SQLCHAR stmt[] = "select "
		      "agents_top "
		      "from table( snapshot_database( cast(null as varchar(256)),-1 )) as t";
     
     rc = SQLPrepare(hstmt, stmt, SQL_NTS);
     STMT_HANDLE_CHECK(hstmt, hdbc, rc);

     rc = SQLExecute(hstmt);
     STMT_HANDLE_CHECK(hstmt, hdbc, rc);

     rc = SQLBindCol(hstmt, 1, SQL_C_LONG, &agents_top, 0, NULL);
     STMT_HANDLE_CHECK(hstmt, hdbc, rc);

     rc = SQLFetch(hstmt);
     STMT_HANDLE_CHECK(hstmt, hdbc, rc);

     rc = SQLFreeStmt(hstmt, SQL_CLOSE);
     STMT_HANDLE_CHECK(hstmt, hdbc, rc);

     if( num_poolagents != -1)
     {
       if( agents_top > num_poolagents )
       {
       	  /*
	  ** If there have been a greater number of agents created than NUM_POOLAGENTS,
	  ** then we're probably seeing agent creation & destruction, which is very
	  ** expensive.  Draw attention to that - if this happens a lot, we may want to 
	  ** tune this up a bit.
	  */
	  printf("  Warning:\n");
	  printf("    At peak, this DB2 instance has had %d agents created, \n",agents_top);
	  printf("    which is greater than the current NUM_POOLAGENTS value of %d\n",num_poolagents);
	  printf("  Recommendation:\n");
	  printf("    Agents created beyond NUM_POOLAGENTS will be destroyed & recreated on connect boundaries.\n");
	  printf("    If the number of agents in the system is frequently above NUM_POOLAGENTS, consider increasing\n");
	  printf("    NUM_POOLAGENTS.\n\n\n");
       }
       else if( num_poolagents < 0.2 * maxagents )
       {
          /*
	  ** Even if we don't have hard evidence of going over NUM_POOLAGENTS, if is is
	  ** much smaller than MAXAGENTS, that's an indication that this situation may
	  ** occur.    Advise the user to be on the lookout...
	  */
	  printf("  Warning:\n");
	  printf("    This DB2 instance has NUM_POOLAGENTS set to %d, which is much lower than MAXAGENTS (%d).\n",
		  num_poolagents, maxagents);
	  printf("    This may cause excessive agent creation / destruction when system load increases.\n");
	  printf("  Recommendation:\n");
	  printf("    Monitor the number of agents in the system during normal operation.   If it is regularly\n");
	  printf("    above NUM_POOLAGENTS, consider increaing NUM_POOLAGENTS, or running DB2 autoconfigure.\n\n\n");
       }
     }
     else
       printf("  Passed\n\n\n");
  }

}
