/**
 * ADV_sample:  The Network Dispatcher HTTP advisor
 * 
 * 
 * This class defines a sample custom advisor for Network Dispatcher.  Like all
 * advisors, this custom advisor extends the function of the advisor base, called ADV_Base.
 * It is the advisor base that actually performs most of the advisor's functions, 
 * such as reporting  loads back to the Network Dispatcher for use in the 
 * Network Dispatcher's weight algorithm.  The advisor base also performs socket connect 
 * and close operations and provides send and receive 
 * methods for use by the advisor.  The advisor itself is used only for
 * sending and receiving data to and from the port on the server being advised.               
 * The TCP methods within the advisor base
 * are timed to calculate the load.  A flag within the constructor in the ADV_base		     
 * overwrites the existing load with the new load returned from the advisor if desired.
 *
 * Note: Based on a value set in the constructor,  the advisor base supplies
 * the load to the weight algorithm at specified intervals.  If the actual
 * advisor has not completed so that it can return a valid load, the advisor base uses
 * the previous load.
 *  
 * NAMING 
 * 
 * The naming convention is as follows:
 *
 *  - The file must be located in the following Network Dispatcher directories:
 *   
 *    - For dispatcher - nd/dispatcher/lib/CustomAdvisors/ (nd\dispatcher\lib\CustomAdvisors on NT)
 *    - For CBR        - nd/cbr/lib/CustomAdvisors/ (nd\cbr\lib\CustomAdvisors on NT)
 *
 * - The Advisor name must be preceded with "ADV_".  The advisor can 
 *    be started with only the name, however; for instance, the "ADV_sample"
 *    advisor can be started with "sample".
 *
 * - The advisor name must be in lowercase.
 *
 *  With these rules in mind, therefore, this sample is referred to as:
 *  
 *              <base directory>/lib/CustomAdvisors/ADV_sample.class.
 *
 *
 * Advisors, as with the rest of Network Dispatcher, must be compiled with the prereq version of Java.
 * To ensure access to Network Dispatcher classes, make sure that the ibmnd.jar 
 * file (located in the lib subdirectory of the base directory) is included in the system's CLASSPATH.
 *
 *
 * Methods provided by ADV_Base:
 * 
 * - ADV_Base (Constructor):
 *
 *   - Parms
 *     - String sName = Name of the advisor
 *     - String sVersion = Version of the advisor
 *     - int iDefaultPort = Default port number to advise on
 *     - int iInterval = Interval on which to advise on the servers
 *     - String sDefaultLogFileName = Unused.  Must be passed in as "".
 *     - boolean replace = True - replace the load value being calculated by the advisor base
 *	                          False - add to the load value being calculated by the advisor base
 *   - Return
 *     - Constructors do not have return values.
 *
 * Because the advisor base is thread based, it has several other methods available
 * for use by an advisor.  These methods can be referenced using the CALLER parameter passed in getLoad().
 *
 * These methods are as follows:
 * 
 * - send - Send a packet of information on the established socket connection to the server on
 *          the specified port.
 *   - Parms
 *     - String sDataString - The data to be sent is sent in the form of a string
 *   - Return
 *     - int RC - Whether the data was sucessfully sent or not: zero indicates data was 
 *                sent; a negative integer indicates an error.
 * 
 * - receive - Receive information from the socket connection.
 *   - Parms
 *     - StringBuffer sbDataBuffer - The data received during the receive call
 *   - Return
 *     - int RC - Whether the data was successfully received or not; zero indicates data was 
 *                sent; a negative integer indicates an error.
 *
 * If the function provided by the advisor base is
 * not sufficient, you can create the appropriate function within the advisor and
 * the methods provided by the advisor base will then be ignored.  
 *
 * An important question regarding
 * the load returned is whether to apply it to the load being generated
 * within the advisor base, or to replace it; there are valid instances of both situations.
 * 
 * This sample is essentially the Network Dispatcher HTTP advisor.  It functions very simply:
 * a send request--an http head request--is issued.  Once a response is received, the 
 * getLoad method terminates, flagging the advisor base to stop timing the request.  The method 
 * is then complete.  The information returned is not parsed; the load is based on the time required 
 * to perform the send and receive operations.
 */

package CustomAdvisors;
import com.ibm.internet.nd.advisors.*;

public class ADV_sample extends ADV_Base implements ADV_MethodInterface
{
  String COPYRIGHT = "(C) Copyright IBM Corporation 1997, All Rights Reserved.\n";

  static final String  ADV_NAME              = "Sample";
  static final int     ADV_DEF_ADV_ON_PORT   = 80;
  static final int     ADV_DEF_INTERVAL      = 7;

  // Note: Most server protocols require a carriage return ("\r") and line feed ("\n")
  //       at the end of messages.  If so, include them in your string here.
  static final String  ADV_SEND_REQUEST      = 
    "HEAD / HTTP/1.0\r\nAccept: */*\r\nUser-Agent: " +
    "IBM_Network_Dispatcher_HTTP_Advisor\r\n\r\n";

  /**
   * Constructor.
   *
   * Parms:  None; but the constructor for ADV_Base has several parameters that must be passed to it.
   *
   */
  public ADV_sample()
  {
    super( ADV_NAME,
		   "2.0.0.0-03.27.98",
           ADV_DEF_ADV_ON_PORT,
           ADV_DEF_INTERVAL,
           "",     // not used
           false);
    super.setAdvisor( this );
  }


  /**
   * ADV_AdvisorInitialize
   *
   * Any Advisor-specific initialization that must take place after the advisor base is started.
   * This method is called only once and is typically not used.
   */
  public void ADV_AdvisorInitialize()
  {
    return;
  }


  /**
   * getLoad()
   *
   * This method is called by the advisor base to complete the advisor's operation, based
   * on details specific to the protocol.  In this sample advisor, only a single send and receive are
   * necessary; if more complex logic is necessary, multiple sends and receives 
   * can be issued.  For example, a response might be received and parsed.  Based on the information 
   * learned thereby, another send and receive could be issued.
   *
   * Parameters:
   * 
   * - iConnectTime - The current load as it refers to the length of time it took to
   *                  complete the connection to the server through the specified port.
   *
   * - caller - A reference to the advisor base class where the Network Dispatcher-supplied 
   *            methods are to perform simple TCP requests, mainly send and receive.
   *
   * Results:
   *
   * - The load - A value, expressed in milliseconds, that can either be added to the existing load,
   *   or that can replace the existing load, as determined by the constructor's "replace" flag.
   *
   *   The larger the load, the longer it took the server to respond; therefore, the higher 
   *   the weight will be within Network Dispatcher regarding load balancing.
   *
   *   If the value is negative, an error is assumed.  An error from an advisor indicates that
   *   the server the advisor is trying to reach is not accessible and has been identified as being down.
   *   Network Dispatcher will not attempt to load balance to a server that is down.  
   *   Network Dispatcher will resume load balancing to the server when a positive value is received.
   *
   *   A value of zero is typically not returned; Network Dispatcher handles a load of zero in a special way.
   *   Zero is assumed to indicate an unknown status, and Network Dispatcher gives the server a high weight in
   *   response.
   */
  public int getLoad(int iConnectTime, ADV_Thread caller)
  {
	int iRc;
	int iLoad = ADV_HOST_INACCESSIBLE;  // -1

	// Send tcp request
	iRc = caller.send(ADV_SEND_REQUEST);
	if (iRc >= 0)
	{
		// Perform a receive
		StringBuffer sbReceiveData = new StringBuffer("");
		iRc = caller.receive(sbReceiveData);

		// If the receive is successful, a load of zero is returned.  This is because the "replace" flag
		// is set to false, indicating that the load built within the base advisor is to be used.
		// Since nothing was done with the returned data, additional load is not necessary.

		// Note: it is known that the advisor base load will not be zero, therefore a zero load will
		// not be returned for use in calculating the weight.
		if (iRc >= 0)
		{
			iLoad = 0;
		}
	}
	return iLoad;
  }

} // End - ADV_sample
