#!/usr/bin/ksh
#
#*********************************************************************
#
#SAMPLE NAME:  migVerify.ksh
#
#USAGE:        migVerify.ksh -C|-T MIGRATION_SCRIPT
#
#ENVIRONMENT VARIABLES:
#                       CMVC:
#                         CMVC_FAMILY
#                         TC_FAMILY
#                       TEAMC:
#                         TC_FAMILY_CLIENT
#                         TC_FAMILY
#                         MIGEXPORT_TOP
#
#DESCRIPTION:  This script interrogates the migration list to create
#              the list of releases to be verified.   Input is either
#              CMVC (-C) or TeamConnection (-T), and the name of the
#              migration list file.
#              The migration list file must contain all "migrate FileView"
#              entries for the migration.
#
#*********************************************************************
#             IBM VisualAge TeamConnection Enterprise Server
#                     Version 3 Release 0
#
#                       5622-717
#    (C) Copyright, IBM Corp., 1998. All Rights Reserved.
#             Licensed Materials - Property of IBM
#
#         US Government Users Restricted Rights
#     - Use, duplication or disclosure restricted by
#         GSA ADP Schedule Contract with IBM Corp.
#
#              IBM is a registered trademark of
#        International Business Machines Corporation
#*********************************************************************
#
#           NOTICE TO USERS OF THE SOURCE CODE EXAMPLES
#
# INTERNATIONAL BUSINESS MACHINES CORPORATION PROVIDES THE SOURCE CODE
# EXAMPLES, BOTH INDIVIDUALLY AND AS ONE OR MORE GROUPS, "AS IS" WITHOUT
# WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT
# LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
# PARTICULAR PURPOSE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE
# OF THE SOURCE CODE EXAMPLES, BOTH INDIVIDUALLY AND AS ONE OR MORE GROUPS,
# IS WITH YOU.  SHOULD ANY PART OF THE SOURCE CODE EXAMPLES PROVE
# DEFECTIVE, YOU (AND NOT IBM OR AN AUTHORIZED DEALER) ASSUME THE ENTIRE
# COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
#
#*********************************************************************

function migVerify_usage
{
   print -u2 "\
usage:   migVerify -C|-T MIGRATION_SCRIPT\n\
purpose: Verify the migration of a CMVC or TeamConnection V2 family to V3\n\
details: -C: Migrate from CMVC\n\
         -T: Migrate from TeamConnection V2\n\
         MIGRATION_SCRIPT: File containing the \"migrate FileView\" (CMVC)\n\
         or \"migrate PartView\" (Teamc) commands\n\

variables:    TC_FAMILY=TeamConnection familyName
              CMVC_FAMILY=CMVC familyName@host@port
              MIGEXPORT_TOP=/my/exported/directory/forCMVCextracts
                 This is the top portion of the directory that CMVC and TC will 
                 extract the release files during verification.
                 The MIGEXPORT_TOP must be an exported directory when migrating
                 from CMVC. Follow these instructions to export the directory.                       

notes:        Use smitty on AIX as root to export the directory for                 
              MIGEXPORT_TOP.

              smitty
                Communications Applications and Services
                    NFS
                        NetWork File Systems 
                              Add Directory to Exports list
              Specify the following:
 
              Pathname of directory to export       /some/existing/directory
              Mode                                  read-write
              Export directory now                  both        
"

   # Delete temporary list file
   if [[ -f tmp.$$ ]]
   then
      rm -f tmp.$$
   fi

   return
}

#######################################################################
# partCount REL
# check number of parts migrated CMVC/TeamConnection by specified release
function partCount
{
   print "counting file records, release: $rel"
   let count=0
   let count2=0
   count=$(teamc report -view partfullview -wh "releasename = '${1}'" -raw | awk -F\| '{ print $8 }' | sort -u | wc -l)
   # Process from CMVC or TeamConnection
   if [[ $VIEW = "File" ]]
   then
      x=1
      count2=$(Report -view fileview -wh "releasename = '${1}'" -raw | awk -F\| '{ print $8 }' | wc -l)
   else
      x=1
      count2=$(teamc report -view partfullview -wh "releasename = '${1}' $TC_FAMILY_CLIENT " -raw | awk -F\| '{ print $8 }' | sort -u | wc -l)
   fi

   if (( $count == $count2 ))
   then
      print "SUCCESS: Release $1, Committed version, part count identical: $count" | tee -a summary.file
      return 0
   else
      print "ERROR: Release $1, Committed version, part count differs" | tee -a summary.file
      print "         Source part count: ${count2}" | tee -a summary.file
      print "         Target part count: ${count}" | tee -a summary.file
      return 1
   fi
}

#######################################################################
# changeCount REL
# check number of changeView records migrated CMVC/TeamConnection by specified release
function changeCount
{
   print "counting change records, release: $rel"
   let count=0
   let count2=0
   count=$(teamc report -view changeView -where "releasename = '${1}'" -raw | awk -F\| '{ print $8 }' | wc -l)
   # Process from CMVC or TeamConnection
   if [[ $VIEW = "File" ]]
   then
      x=1
      count2=$(Report -view changeView -wh "releasename = '${1}'" -raw | awk -F\| '{ print $8 }' | wc -l)
   else
      x=1
      count2=$(teamc report -view changeView -wh "releasename = '${1}' -family $TC_FAMILY_CLIENT " -raw | awk -F\| '{ print $8 }' | wc -l)
   fi

   if (( $count == $count2 ))
   then
      print "SUCCESS: Release $1, change count identical:                  $count" | tee -a summary.file
      return 0
   else
      print "ERROR: Release $1, change count differs"  | tee -a summary.file
      print "         Source change count: ${count2}" | tee -a summary.file
      print "         Target change count: ${count}" | tee -a summary.file
      return 1
   fi
}

#######################################################################
function rel_extract
{
   print "Extracting release: $rel" | tee -a summary.file
   let rc=0
   let RelRC=0
   gid=$(id -g)
   uid=$(id -u)
   if [[ $VIEW = "File" ]]
   then
      SOURCE_DIR=${MIGEXPORT_TOP}/cmvc/extract/$rel
      if [[ -d $SOURCE_DIR ]]
      then
          rm -rf $SOURCE_DIR
      fi
      mkdir  -p $SOURCE_DIR
      chmod -R 777 $SOURCE_DIR
      set -x
      Release -extract $rel -nokeys -root $SOURCE_DIR -node $(hostname) -committed -dmask 777 -uid $uid -gid $gid -verbose
#      Release -extract $rel -nokeys -root $HOME/cmvc/extract/$rel -node nomount -committed -verbose
      set +x
      rc=$?
      RelRC=$rc
   else
      SOURCE_DIR=${MIGEXPORT_TOP}/TCv2/extract/$rel
      if [[ -d $SOURCE_DIR ]]
      then
          rm -rf $SOURCE_DIR
      fi
#      mkdir -p $SOURCE_DIR
#      chmod -R 777 $SOURCE_DIR
      set -x
      teamc release -extract $rel -nokeys -root $SOURCE_DIR -crlf -verbose
      set +x
      rc=$?
      RelRC=$rc
   fi

   TARGET_DIR=${MIGEXPORT_TOP}/TCv300/extract/$rel
   if [[ -d $TARGET_DIR ]]
   then
      rm -rf $TARGET_DIR
   fi

#   mkdir $TARGET_DIR
#   chmod -R 777 $TARGET_DIR
   set -x
   teamc release -extract $rel -nokeys -root $TARGET_DIR -crlf -verbose
   set +x
   rc=$?
   let RelRC=RelRC+rc

   return $RelRC
}

#######################################################################
# Part content verification
function rel_diff
{
   TMPFILE=/tmp/TMPFILE.$$

   let DiffRC=0

   # Delete temporary list file
   if [[ -f /tmp/diff_err.$$ ]]
   then
      rm -f /tmp/diff_err.$$
   fi
   diff_err=/tmp/diff_err.$$
   exec 3> /tmp/diff_err.$$

   print "Comparing release contents: $rel"  | tee -a summary.file

#   if [[ $VIEW = "File" ]]
#   then
#      SOURCE_DIR=$HOME/cmvc/extract/$rel
#   else
#      SOURCE_DIR=$HOME/TCv2/extract/$rel
#   fi
#   TARGET_DIR=$HOME/TCv300/extract/$rel

   currentDir=$PWD
   Tdiff=$currentDir/Tdiff.$$
   Fdiff=$currentDir/Fdiff.$$
   cd $SOURCE_DIR
   find .  -type f -print |
      while read record
      do
         cmvcpart=${record#*/}
         if [[ ! -f ${SOURCE_DIR}/${cmvcpart} ]]
         then
            print "ERROR: ${SOURCE_DIR}/${cmvcpart} does not exists"
         else
            bdiff ${cmvcpart} ${TARGET_DIR}/${cmvcpart} >${TMPFILE} 2>&1
         fi

#        Checking to see if there is any output from the bdiff to TMPFILE
         if [[ -s ${TMPFILE} ]]
         then
            print "ERROR: part $cmvcpart is different"
            print -u3 $cmvcpart
            let DiffRC=DiffRC+1
         fi
      done

   # If there were any file differences recorded in diff_err file
   if [[ -s ${diff_err} ]]
   then
      print  "ERROR: $(cat /tmp/diff_err.$$ | wc -l) file differences identified in release: $rel" | tee -a ${currentDir}/summary.file
      let num=0
      print "======================================================" >> ${Tdiff}
      print "PROCESSING RELEASE: $rel" >> ${Tdiff}
      print "======================================================" >> ${Tdiff}
      print "=======================================================" >> ${Fdiff}
      print " Release $rel,  Files that  Do Not compare  " >> ${Fdiff}
      print "=======================================================" >> ${Fdiff}
      while read file
      do
         let num=num+1
         print " " >> ${Tdiff}
         print  "==================================================" >> ${Tdiff}
         print  "File comparison # $num: $file" >> ${Tdiff}
         print  "==================================================" >> ${Tdiff}
         print " " >> ${Tdiff}
         print "${file}" >> ${Fdiff}
         diff ${SOURCE_DIR}/${file} ${TARGET_DIR}/${file} >> ${Tdiff} 2>&1
      done < ${diff_err}
   fi
   cd $currentDir
   return $DiffRC
}


#######################################################################
# Main
# - Test input
#######################################################################
if (( $# < 1 ))
then
   migVerify_usage
   return 0
fi

if (( $# < 2 ))
then
   print " "
   print -u2 "ERROR: Migration parameters missing"
   print  " "
   migVerify_usage
   return 1
fi

if [[ "$1" = "-C" ]]
then
   VIEW=File
else
   if [[ "$1" = "-T" ]]
   then
      VIEW=Part
   else
      print " "
      print -u2 "ERROR: Migration parameter 1 must be \"C\" or \"T\""
      print " "
      migVerify_usage
      return 1
   fi
fi

if [[ ! -f $2 ]]
then
   print " "
   print -u2 "ERROR: Migration List file $2 is missing"
   print " "
   migVerify_usage
   return 1
fi

if [[ $VIEW = "File" ]]
then
   if [[ -z "${CMVC_FAMILY}" ]]
   then
      print " "
      print -u2 "ERROR: CMVC_FAMILY must be set"
      print " "
      migVerify_usage
      return 1
   fi
else
   if [ -z "${TC_FAMILY_CLIENT}" ]
   then
      print " "
      print -u2 "ERROR: TC_FAMILY_CLIENT must be set"
      print " "
      migVerify_usage
      return 1
   fi
fi

# Required for all verifications
if [[ -z "${TC_FAMILY}" ]]
then
   print " "
   print -u2 "ERROR: TC_FAMILY must be set"
   print " "
   migVerify_usage
   return 1
fi

# Required for all verifications
if [[ -z "${MIGEXPORT_TOP}" ]]
then
   print " "
   print -u2 "ERROR: MIGEXPORT_TOP must be set"
   print " "
   migVerify_usage
   return 1
fi

# Summary file
if [[ -f summary.file ]]
then
   rm -f summary.file
fi
   currDir=$PWD
   # Delete differences file
   if [[ -f $currDir/Tdiff.$$ ]]
   then
      rm -f $currDir/Tdiff.$$
   fi

# Get List of Releases
if [[ $VIEW = "File" ]]
then
  grep -i ${VIEW}View $2 | egrep -v ^# | awk '{print $6 }' | sed "s/\'//g" > tmp.$$
else
  grep -i ${VIEW}View $2 | egrep -v ^# | awk '{print $4 }' > tmp.$$
fi

if (( $(cat tmp.$$ | wc -l) == 0 ))
then
   print -u2 "ERROR: No \"migrate ${VIEW}View where releaseName = 'RELEASE' \"\n\
       entries in $2\n\
"
   migVerify_usage
   return 1
fi

let TotalRC=0
while read rel
do
   print "==============================================="  | tee -a summary.file
   print "Verifying File Migration Count, Release: $rel"  | tee -a summary.file
   print "==============================================="  | tee -a summary.file
   #
   partCount $rel
   let TotalRC=TotalRC+$?
   #
   changeCount $rel
   let TotalRC=TotalRC+$?
done < tmp.$$

# Total differences file
if [[ -f Tdiff.$$ ]]
then
   rm -f Tdiff.$$
fi
exec 4> Tdiff.$$

#
#let TotalRC=0
while read rel
do
   print "==============================================="  | tee -a summary.file
   print "Verifying File Contents, Release: $rel"  | tee -a summary.file
   print "==============================================="  | tee -a summary.file

   rel_extract $rel
   rc=$?
   # If extracts are successful, compare contents of each part
   if (( $rc == 0 ))
   then
      rel_diff $rel
      rc=$?
   else
      print "ERROR: Release extract failed: $rel" | tee -a summary.file
      print "ERROR: Verification of File content skipped: $rel" | tee -a summary.file
      rc=1
      let TotalRC=TotalRC+1
   fi

   if (( $rc == 0 ))
   then
      print "SUCCESS: Release $rel, part contents are identical" | tee -a summary.file
   fi
   for i in $SOURCE_DIR $TARGET_DIR
   do
      if [[ -d $i ]]
      then
         print "Cleaning up $i"
         rm -rf $i
      fi
   done
   let TotalRC=TotalRC+rc

done< tmp.$$

print "==============================================="  | tee -a summary.file
print "Migration Verification Complete"  | tee -a summary.file
print "==============================================="  | tee -a summary.file
print "Verification summary is in : summary.file"
# Check results of comparison
if (( $TotalRC > 0 ))
then
   print -u2 "ERRORS Detected, Total Errors: $TotalRC" | tee -a summary.file
   print  "ERRORS Detected, Total Errors: $TotalRC" >> summary.file
   if [[ -s Tdiff.$$ ]]
   then
      print -u2 "File comparison data is in: Tdiff.$$"
      print -u2 "File difference report is in: Fdiff.$$"
      print  "File comparison data is in: Tdiff.$$" >> summary.file
      print  "File difference report is in: Fdiff.$$" >> summary.file
   fi
else
   print  "Successful Verfication" | tee -a summary.file
fi

# Delete temporary list file
if [[ -f tmp.$$ ]]
then
   rm -f tmp.$$
fi

# Delete temporary list file
if [[ -f /tmp/diff_err.$$ ]]
then
   rm -f /tmp/diff_err.$$
fi

# Delete temporary list file
if [[ -f /tmp/TMPFILE.$$ ]]
then
   rm -f /tmp/TMPFILE.$$
fi

for i in ${MIGEXPORT_TOP}/cmvc/extract ${MIGEXPORT_TOP}/TCv2/extract ${MIGEXPORT_TOP}/TCv300/extract
do
   if [[ -d $i ]]
   then
      print "Cleaning up $i"
      rm -rf $i
   fi
done

exit
#end of file

