#!/usr/bin/ksh

# This program takes no parameters. It is used to generate shell script files on Unix to
# Delete levelMembers from Levels in the integrate state prior to migrating to TC.
# It also generates a shell script to Add teh levelMembers back into TC once the migration
#of Levels are complete.

function usage
{
   print -u2 'usage:  processDefects DATABASE  '
   print  "           details: DATABASE = DB2|ORACLE|ORACLE7|SYBASE|INFORMIX"
   exit 1
}

#main
#set -x
curDir=$PWD
TMPFILE=/tmp/tmpfile.$$
RenDef=$curDir/renameDefectsCMVC.ksh
OUTFILE=$curDir/renameDefectsCMVC.log

if [[ -f ${OUTFILE} ]]
  then
   rm -f ${OUTFILE}
fi
if [[ -f ${TMPFILE} ]]
  then
   rm -f ${TMPFILE}
fi
if [[ -f ${RenDef} ]]
  then
   rm -f ${RenDef}
fi

if [ "$#" -ne 1 ]
  then usage
fi

if [[ -z "${CMVC_FAMILY}" ]]
  then
     print " "
     print -u2 "ERROR: CMVC_FAMILY must be set"
     print " "
     return 1
   fi

DATABASE=${1}
case "${DATABASE}" in

  ORACLE|ORACLE7|oracle|oracle7 )
    SQL="${ORACLE_HOME}/bin/sqlplus"
    U_STRING=""
    COMMIT_STRING=""
    PASSWORD_STRING="/${ORACLE_PASS}"
    SEMI_STRING=";"
    INIT_CMD="${SQL} ${U_STRING} ${FAMILY}${PASSWORD_STRING}"
    ;;

  DB2|db2 )
    SQL="${DB2_HOME}/sqllib/bin/db2"
    U_STRING=" -o -t +v"
    COMMIT_STRING=""
    PASSWORD_STRING=""
    SEMI_STRING=";"
    INIT_CMD="${SQL} ${U_STRING}"
    ;;

  INFORMIX|informix )
    SQL="${INFORMIXDIR}/bin/isql"
    U_STRING=""
    COMMIT_STRING="commit work;"
    PASSWORD_STRING=""
    SEMI_STRING=";"
    INIT_CMD="${SQL} ${U_STRING} ${FAMILY} ${PASSWORD_STRING}"
    ;;

  SYBASE|sybase )
    SQL="${SYBASE}/bin/isql"
    U_STRING=""
    COMMIT_STRING="go"
    PASSWORD_STRING=" -P ${SYBASE_PASS}"
    SEMI_STRING=""
    INIT_CMD="${SQL} ${U_STRING} ${PASSWORD_STRING}"
    ;;

  * )
    print "Unrecognized DATABASE $DATABASE" | tee -a $OUTFILE
    usage
    ;;

esac

Report -raw -view TrackView -where "defectName in (select name from releaseView) OR defectName in (select name from levelView)"  | awk -F\| '{ print $2 }' > ${TMPFILE}

rc=$?
if (( $rc > 0 ))
  then
  print "ERROR: Could not generate the list of Defects to be renamed."
  print "Verify connectivity to your CMVC family and that you are a SuperUser"
  print " "
  return 1
fi


if [[ -s ${TMPFILE} ]]
  then
   print "processing Defects"
   print  " "
   echo "########################################################################"  > ${RenDef}
   echo "# Run this script on your CMVC source family to rename defects that have"  >> ${RenDef}
   echo "# the same name as an existing release or existing level."                 >> ${RenDef}
   echo "# This must be executed prior  to migrating to TeamConnection Version 3"  >> ${RenDef}
   echo "########################################################################" >> ${RenDef}

   echo "${INIT_CMD} | tee  ${OUTFILE} "  >> ${RenDef}
   if [ ${DATABASE} = "DB2" -o ${DATABASE} = "db2" ]
   then
      echo  "connect to ${CMVC_FAMILY} 2>&1 | tee -a ${OUTFILE} "          >> ${RenDef}
   fi

   while read defect
   do
      echo "db2 \"update defects set name='${defect}D' where name='${defect}'\" 2>&1 | tee -a ${OUTFILE} "  >> ${RenDef}
      echo "${SEMI_STRING}"  >> ${RenDef}
      echo "${COMMIT_STRING}"  >> ${RenDef}
   done < ${TMPFILE}

   if [ ${DATABASE} = "DB2" -o ${DATABASE} = "db2" ]
   then
     echo  "commit 2>&1 | tee -a ${OUTFILE}"                  >> ${RenDef}
     echo  "connect reset 2>&1 | tee -a ${OUTFILE}"           >> ${RenDef}
   fi

   echo "print \"Results are logged in file: ${OUTFILE}\" "     >> ${RenDef}

else
  print " There Are No Defects  required to be renamed prior to Migration"
  return 0
fi

print "Successful Completion !!!"
print "-----------------------------------------"
if [[ -s ${RenDef} ]]
  then
  print " "
  print " This program did not modify your CMVC or TeamConnection families. It only created the "
  print " the following shell script to be executed by your Family Administrators."
  print " "
  print " =========  review the comments in the generated script file prior to executing. ========"
  print " "
  print "generated script to RENAME Defects from CMVC: ${RenDef}"
fi
exit 0

