#!/usr/bin/ksh
# NAME:    reassign_work.ksh
# PURPOSE:
#   This utility will reassign incomplete work from one active CMVC login
#   to another active CMVC login. Incomplete work can be defined as:
# defects owned/originated
# features owned/originated
# verification
# components
# releases
# levels
# environments
# test records
# size records
# approval
# approver
# tracks
# fix records
# notifications deleted
# access deleted
# login deleted
# NOTE:
#   This script must be run by a CMVC superuser id.


################################################################################
function usage
{
        print -u2 "USAGE: ${PROG} -f <FAMILY> -s <FROMUSER> -t <TOUSER>"
        print -u2 "WHERE: FAMILY      is the CMVC family to process"
        print -u2 "       FROMUSER    is the CMVC login to be deleted"
        print -u2 "       TOUSER      is the CMVC login to assign work to"
        print -u2 "NOTE : script must be run by a SUPERUSER"
        exit 1
}
################################################################################
function initialize
{
	RC=0
        if [ -z "${FAMILY}" ]
          then  print -u2 "<<ERROR>> FAMILY name required"
                RC=1
          else  CMVC_FAMILY=${FAMILY}
        fi
        WHOAMI=`id -u -n`
        SUPERUSER=`Report -view users -wh "login = '${WHOAMI}' AND superuser = 'yes'" -raw`
        if [ -z "${SUPERUSER}" ]
          then  print -u2 "<<ERROR>> LOGIN invalid or not SUPERUSER"
                RC=1
        fi
        if [ -z "${FROMUSER}" ]
          then  print -u2 "<<ERROR>> FROMUSER name required"
                RC=1
	  else	JUNK=`Report -view users -where "login = '${FROMUSER}' AND dropdate is NULL" -raw`
		if [ -z "${JUNK}" ]
		  then	print -u2 "<<ERROR>> FROMUSER login invalid"
			RC=1
		fi
	fi
        if [ -z "${TOUSER}" ]
          then  print -u2 "<<ERROR>> TOUSER name required"
                RC=1
	  else	JUNK=`Report -view users -where "login = '${TOUSER}' AND dropdate is NULL" -raw`
		if [ -z "${JUNK}" ]
		  then	print -u2 "<<ERROR>> TOUSER login invalid"
			RC=1
		fi
	fi
        OUTFILE=/tmp/outfile.$$
        BADFILE=/tmp/badfile.$$
}
################################################################################
function crank
{
	print "${FROMUSER}|${TOUSER}|${cmd}" >>${OUTFILE}
	print "           ${cmd}"
	eval ${cmd} 2>/dev/null
	if [ $? -ne 0 ]
	  then	print "${cmd}" >>${BADFILE}
	fi
}
################################################################################
function process
{
    print "Deleting login: ${FROMUSER}"
    print "OUTPUT FILE: ${OUTFILE}"
    print "FAILED CMDS: ${BADFILE}"
	print "<<STATUS>> Defect records owned"
	Report -view defectview -wh "ownerlogin='${FROMUSER}' and state != 'closed'" -raw |
	while read record
	  do 
   		DEFECTNAME=`print ${record} | cut -d\| -f2`
   		cmd="Defect -assign ${DEFECTNAME} -owner '${TOUSER}'"
		crank
	  done
	print "<<STATUS>> Defect records originated"
	Report -view defectview -where "originlogin = '${FROMUSER}' and state != 'closed'" -raw |
	while read record
	  do 
   		DEFECTNAME=`print ${record} | cut -d\| -f2`
   		cmd="Defect -modify ${DEFECTNAME} -originator '${TOUSER}'"
		crank
	  done
	print "<<STATUS>> Feature records owned"
	Report -view featureview -where "ownerlogin = '${FROMUSER}' and state != 'closed'" -raw |
	while read record
	  do 
   		FEATURENAME=`print ${record} | cut -d\| -f2`
   		cmd="Feature -assign ${FEATURENAME} -owner '${TOUSER}'"
		crank
	  done
	print "<<STATUS>> Feature records originated"
	Report -view featureview -where "originlogin = '${FROMUSER}' and state != 'closed'" -raw |
	while read record
	  do 
   		FEATURENAME=`print ${record} | cut -d\| -f2`
   		cmd="Feature -modify ${FEATURENAME} -originator '${TOUSER}'"
		crank
	  done
	print "<<STATUS>> Verification records"
	Report -view verifyview -where "userlogin = '${FROMUSER}' and state in ('notReady','ready')" -raw |
	while read record
	  do
   		NAME=`print ${record} | cut -d\| -f1 `
     		cmd="Verify -assign -tester ${FROMUSER} -defect ${NAME}"
		cmd="${cmd} -to '${TOUSER}'"
		crank
	  done
	print "<<STATUS>> Component assignment"
	Report -view compview -where "userlogin = '${FROMUSER}'" -raw |
	while read record
	  do
   		COMPNAME=`print ${record} | cut -d\| -f1`
   		cmd="Component -modify ${COMPNAME} -owner '${TOUSER}'"
		crank
	  done
	print "<<STATUS>> Release assignment"
	Report -view releaseview -where "userlogin ='${FROMUSER}'" -raw |
	while read record
	  do
   		RELEASE=`print ${record} | cut -d\| -f1`
   		cmd="Release -modify ${RELEASE} -owner '${TOUSER}'"
		crank
	  done
	print "<<STATUS>> Level assignment"
	Report -view levelview -where "userlogin ='${FROMUSER}' AND state != 'complete'" -raw |
	while read record
	  do
   		LEVELNAME=`print ${record} | cut -d\| -f1`
   		RELEASE=`print ${record} | cut -d\| -f2`
   		cmd="Level -assign ${LEVELNAME} -to '${TOUSER}' -release"
		cmd="${cmd} ${RELEASE}"
		crank
	  done
	print "<<STATUS>> Environment assignments"
	Report -view envview -wh "userlogin ='${FROMUSER}'" -raw |
	while read record
	  do
   		ENVNAME=`print ${record} | cut -d\| -f1`
   		RELEASE=`print ${record} | cut -d\| -f2`
   		cmd="Environment -delete ${ENVNAME} -release ${RELEASE}"
		crank
   		cmd="Environment -create ${ENVNAME} -release ${RELEASE}"
		cmd="${cmd} -tester ${TOUSER}"
		crank
	  done
	print "<<STATUS>> Notification deletions"
	Report -view notifyview -w "userlogin ='${FROMUSER}'" -raw |
	while read record
	  do
   		INTEREST=`print ${record} | cut -d\| -f6`
   		COMPONENT=`print ${record} | cut -d\| -f1`
   		cmd="Notify -delete -login ${FROMUSER} -component ${COMPONENT}"
		cmd="${cmd} -interest ${INTEREST}"
		crank
	  done
	print "<<STATUS>> Test assignments"
	Report -view testview -where "userlogin ='${FROMUSER}' and state in ('ready','notReady')" -raw |
	while read record
	  do
   		NAME=`print ${record} | cut -d\| -f3`
   		RELEASE=`print ${record} | cut -d\| -f1`
   		ENVIRONMENT=`print ${record} | cut -d\| -f4`
     		cmd="Test -assign -tester ${FROMUSER} -defect ${NAME}"
		cmd="${cmd} -release ${RELEASE} -environment ${ENVIRONMENT}"
		cmd="${cmd} -to '${TOUSER}'"
		crank
	  done
	print "<<STATUS>> Size assignments"
	Report -view sizeview -where "userlogin ='${FROMUSER}' and state = 'ready'" -raw |
	while read record
	  do
   		NAME=`print ${record} | cut -d\| -f1`
   		RELEASE=`print ${record} | cut -d\| -f4`
   		COMPONENT=`print ${record} | cut -d\| -f3`
   		cmd="Size -assign -component ${COMPONENT} -release ${RELEASE}"
		cmd="${cmd} -defect ${NAME} -to '${TOUSER}'"
		crank
	  done
	print "<<STATUS>> Approval assignments"
	Report -view approvalview -where "userlogin ='${FROMUSER}' and state = 'ready'" -raw |
	while read record
	  do
   		NAME=`print ${record} | cut -d\| -f2`
   		RELEASE=`print ${record} | cut -d\| -f3`
     		cmd="Approval -assign -release ${RELEASE} -defect ${NAME}"
		cmd="${cmd} -to '${TOUSER}' -approver ${FROMUSER}"
		crank
	  done
	print "<<STATUS>> Approver assignment"
	Report -view approverview -where "userlogin ='${FROMUSER}'" -raw |
	while read record
	  do
   		RELEASE=`print ${record} | cut -d\| -f1`
   		cmd="Approver -delete -login ${FROMUSER} -release ${RELEASE}"
		crank
   		cmd="Approver -create -login ${TOUSER} -release ${RELEASE}"
		crank
	  done
	print "<<STATUS>> Track assignment"
	Report -view trackview -where "userlogin ='${FROMUSER}' and state != 'complete'" -raw |
	while read record
	  do
   		NAME=`print ${record} | cut -d\| -f2`
   		RELEASE=`print ${record} | cut -d\| -f1`
     		cmd="Track -assign -defect ${NAME} -release ${RELEASE}"
		cmd="${cmd} -to '${TOUSER}'"
		crank
	  done
	print "<<STATUS>> Fix assignment"
	Report -view fixview -where "userlogin ='${FROMUSER}' and state != 'complete'" -raw |
	while read record
	  do
   		NAME=`print ${record} | cut -d\| -f1`
   		RELEASE=`print ${record} | cut -d\| -f2`
   		COMPONENT=`print ${record} | cut -d\| -f3`
     		cmd="Fix -assign -defect ${NAME} -release ${RELEASE}"
		cmd="${cmd} -component ${COMPONENT} -to '${TOUSER}'"
		crank
	  done
	print "<<STATUS>> Access deletion"
	Report -vi accessview -where "userlogin ='${FROMUSER}'" -raw |
	while read record
	  do
   		COMPONENT=`print ${record} | cut -d\| -f1`
   		AUTHORITY=`print ${record} | cut -d\| -f5`
   		cmd="Access -delete -login ${FROMUSER} -component ${COMPONENT}"
		cmd="${cmd} -authority ${AUTHORITY}"
		crank
	  done
	print "<<STATUS>> User deletion"
	cmd="User -delete '${FROMUSER}'"
	crank
}
################################################################################
# main processing
PROG=${0##*/}
while getopts :f:s:t: argument
  do
        case "${argument}" in
          f)    FAMILY=${OPTARG}
                ;;
          s)    FROMUSER=${OPTARG}
                ;;
          t)    TOUSER=${OPTARG}
                ;;
          \?)   usage
                ;;
        esac
  done
initialize
if [ "${RC}" -eq "0" ]
  then  process
  else  usage
fi

# end of file

