#!/usr/bin/ksh
# NAME:    release_complete.ksh
# PURPOSE: 
#   This utility will complete work in a given release so that
#   all file changes are committed.
# NOTE:
#   This script must be run by a CMVC superuser id.
 
################################################################################
function usage
{
	print -u2 "USAGE: ${PROG} -f <FAMILY> -r <RELEASENAME>"
	print -u2 "WHERE: FAMILY      is the CMVC family to process"
	print -u2 "       RELEASENAME is the active CMVC release"
	print -u2 "NOTE : script must be run by a SUPERUSER"
	exit 1
}
################################################################################
function initialize
{
	RC=0
	if [ -z "${FAMILY}" ]
	  then	print -u2 "<<ERROR>> FAMILY name required"
		RC=1
	  else	CMVC_FAMILY=${FAMILY}
	fi
	WHOAMI=`id -u -n`
	SUPERUSER=`Report -view users -wh "login = '${WHOAMI}' AND superuser = 'yes'" -raw`
	if [ -z "${SUPERUSER}" ]
	  then	print -u2 "<<ERROR>> LOGIN invalid or not SUPERUSER"
		RC=1
	fi
	if [ -z "${RELEASE}" ]
	  then	print -u2 "<<ERROR>> RELEASE name required"
		RC=1
	  else	PROCESS=`Report -view releaseview -wh "dropdate is null AND name = '${RELEASE}'" -raw | cut -d\| -f3`
		if [ -z "${PROCESS}" ]
		  then	print -u2 "<<ERROR>>  RELEASE name INVALID or deleted"
			RC=1
		  else	SUBPROCESSES="none track approval fix level test"
		fi
	fi
	OUTFILE=/tmp/outfile.$$
	print "Completing release: ${RELEASE}" >${OUTFILE}
	BADFILE=/tmp/badfile.$$
}
################################################################################
function crank
{
	print "${cmd}" >>${OUTFILE}
	print "           ${cmd}"
	eval ${cmd} 2>/dev/null
	if [ $? -ne 0 ]  && [ "${PROCESS}" != "track" ]
	  then	print "${cmd}" >>${BADFILE}
	fi
}
################################################################################
function process_tracks
{
	# integrate tracks if possible
	print "<<STATUS>> processing tracks"
	Report -view trackview -where "releasename = '${RELEASE}' AND state = 'fix'" -raw |
	while read record
	  do
		defect=`print ${record} | cut -d\| -f2`
		cmd="Track -integrate -defect ${defect} -release ${RELEASE}"
		crank
	  done
}
################################################################################
function process_approvals
{
	# delete any approval records that have not been processed (state ready)
	print "<<STATUS>> processing approvals"
	Report -view approvalview -wh "releasename = '${RELEASE}' AND state = 'ready'" -raw |
	while read record
	  do
		defect=`print ${record} | cut -d\| -f2`
		approver=`print ${record} | cut -d\| -f4`
		cmd="Approval -delete -defect ${defect} -release ${RELEASE}"
		cmd="${cmd} -approver ${approver}"
		crank
	  done
}
################################################################################
function process_fixes
{
	# complete fix records that have associated work (state active/ready)
	print "<<STATUS>> processing fixes"
	Report -view fixview -wh "releasename = '${RELEASE}' AND state in ('active','ready')" -raw |
	while read record
	  do
		defect=`print ${record} | cut -d\| -f1`
		component=`print ${record} | cut -d\| -f3`
		cmd="Fix -complete -defect ${defect} -release ${RELEASE}"
		cmd="${cmd} -component ${component}"
		crank
	  done
}
################################################################################
function process_levels
{
	# remove levelmembers of uncommitted levels
	print "<<STATUS>> processing levelmembers"
	Report -view levelmemberview -wh "releasename = '${RELEASE}' AND levelname in (select name from levelview where commitdate is null)"   \
	 -raw | 
	while read record
	  do
		level=`print ${record} | cut -d\| -f1`
		defect=`print ${record} | cut -d\| -f3`
		cmd="LevelMember -delete -defect ${defect} -release ${RELEASE}"
		cmd="${cmd} -level ${level}"
		crank
	  done
	# remove uncommitted levels
	print "<<STATUS>> processing uneeded levels"
	Report -view levelview -wh "releasename = '${RELEASE}' AND commitdate is NULL" -raw |
	while read record
	  do
		level=`print ${record} | cut -d\| -f1`
		cmd="Level -delete ${level} -release ${RELEASE}"
		crank
	  done
	# one biggie
	print "<<STATUS>> processing cleanup level"
	LEVEL=`date "+%y%m%d"s`
	TYPE=`Report -view config -wh "type = 'leveltype'" -raw | head -1 | cut -d\| -f2`
	cmd="Level -create ${LEVEL} -release ${RELEASE} -type ${TYPE}"
	crank
	print "<<STATUS>> processing cleanup levelmembers"
	Report -view trackview -wh "releasename = '${RELEASE}' AND state = 'integrate'" -raw |
	while read record
	  do
		defect=`print ${record} | cut -d\| -f2`
		cmd="LevelMember -create -level ${LEVEL} -release ${RELEASE}"
		cmd="${cmd} -defect ${defect}"
		crank
	  done
	print "<<STATUS>> processing cleanup level commit"
	cmd="Level -commit ${LEVEL} -release ${RELEASE}"
	crank
	# commit to complete
	print "<<STATUS>> processing level completes"
	Report -view levelview -wh "releasename = '${RELEASE}' AND state = 'commit'" -raw |
	while read record
	  do
		level=`print ${record} | cut -d\| -f1`
		cmd="Level -complete ${level} -release ${RELEASE}"
		crank
	  done
}
################################################################################
function process_test
{
	# abstain from all unprocessed test records (state ready)
	print "<<STATUS>> processing tests"
	Report -view testview -wh "releasename = '${RELEASE}' AND state = 'ready'" -raw |
	while read record
	  do
		defect=`print ${record} | cut -d\| -f3`
		environment=`print ${record} | cut -d\| -f4`
		tester=`print ${record} | cut -d\| -f8`
		cmd="Test -abstain -defect ${defect} -release ${RELEASE}"
		cmd="${cmd} -tester ${tester} -environment ${environment}"
		crank
	  done
}
################################################################################
function process
{
	print "OUTPUT FILE: ${OUTFILE}"
	print "FAILED CMDS: ${BADFILE}"
	for PROCESS in ${SUBPROCESSES}
	  do
		case "${PROCESS}" in
		  none)		;;
		  track)	process_tracks
					;;
		  approval)	process_approvals
				;;
		  fix)		process_fixes
				;;
		  level)	process_levels
				;;
		  test)		process_test
				;;
		esac
	  done
}
################################################################################
# main processing
PROG=${0##*/}
while getopts :f:r: argument
  do
	case "${argument}" in
	  f)	FAMILY=${OPTARG}
	  	;;
	  r)	RELEASE=${OPTARG}
	  	;;
	  \?)	usage
		;;
	esac
  done
initialize
if [ "${RC}" -eq "0" ]
  then	process
  else	usage
fi

# end of file

