/* */
/********************************************************************

NAME: do-build.cmd driverName [ debug | beta ]

SAMPLE INVOCATION:
  rexx do-build.cmd 2001-03-09 debug

PURPOSE:
 This is the top level script to build the application "myapp".

NOTES:
 * A temporary file named "do-ver.out" is used.

CUSTOMIZATION NOTES:
 * See the sections labeled "CUSTOMIZATION REQUIRED".
 * This example has variables that need to be added to the master.list
   file, because they change often and/or are set at build time.
   - They need to be defined in Section "Definition of Dynamic Variables"
   - They need to be added to the master.list in Section "Create  master.list".
   The variables are:
    DRIVER_NAME      -> Identifier which might be related to the build date.
    OPERATING_SYSTEM -> Use "ver" or "uname -a" to find out the OS level.
 * The file "level.list" has the serviceability strings that do not change
   often. The contents of this file are copied into the "master.list".
 * The master.list file is a temporary file that contains all the
   serviceability strings. master.list is generated every single time
   this build script is invoked and it is used inside the main make file
   (myapp-w.mak) which in turn, calls do-what.cmd.

*********************************************************************/

say "do-build.cmd: Starting a build of the myapp application"
say "date: " 
"@date /t"
say ""

/* Verify the input parameters */

parse arg driverName option
usage = "do-build.cmd driverName [ debug | beta ]"


/* Section:  Definition of Dynamic Variables */

/* CUSTOMIZATION REQUIRED: 
   Set up the variables that change frequently:
*/
if (driverName = "") then
do
  say "*** Error: need to specify the driver name."
  say usage
  exit 1
end
else
do
  DRIVER_NAME=driverName
end


/* Issue the "ver" command and store the output in a temporary file named
   "do-ver.out".
   The variable 'OPERATING_SYSTEM' will have the version of the operating system.
*/
verout = "do-ver.out"

"@ver >" verout
 do while lines(verout)
    parse value linein(verout) with OPERATING_SYSTEM
 end
 call lineout verout
 "@del" verout

/* CUSTOMIZATION REQUIRED:
   You need to handle the appropriate processing of input arguments.
   For example, the DEBUG and BETA options may need to be set.
*/

/* Do the processing for the option */

if (option = "") then
do
  FLAGS=""
end
else
do
  if (option = "debug") then
  do
     say "**********ERROR**********"
     say "Set the DEBUG variable in the java source code for myapp "
     say "to 'true' and then call this script without the debug  parameter:"
     say ""
     say "rexx do-build.cmd driverName"
     exit 1
    
  end
  else
  do
     if (option = "beta") then
     do
       say "**********ERROR**********"
       say "Set the BETA variable in the java source code for myapp "
       say "to 'true' and then call this script without the beta parameter."
       say ""
       say "rexx do-build.cmd driverName"
       exit 1
     end
     else
     do
       say "*** Error: wrong option. If specified, it must be: debug or beta"
       say usage
       exit 1
     end
  end
end

/* Section:  Remove files */

/* CUSTOMIZATION REQUIRED:
   You need to remove the old master.list before doing a build.
NOTE:  not really customization.
*/

'@del master.list'            /*generated file with the complete list of strings. */

/* CUSTOMIZATION REQUIRED:
   You need to add variables that do not change often to the file
   "level.list".  This is the input file and is permanent.
   "master.list" is a temporary file that will have all the
   serviceability strings. This is the output file.
*/

/* Open input file */
/* CUSTOMIZATION REQUIRED:
 * If you use a name other than "level.list", then customize the following
   statement.*/
inputFile='level.list'
rcFile = stream( inputFile, C, 'open read' )
if ( rcFile \= "READY:" ) then
do
   say "*** Error while trying to open file: " inputFile
   say "    The return code is: " rcFile
   say "    Exiting now."
   exit 1
end

/* Open output file */
outputFile='master.list'
rcFile = stream( outputFile, C, 'open write' )
if ( rcFile \= "READY:" ) then
do
   say "*** Error while trying to open file: " outputFile
   say "    The return code is: " rcFile
   say "    Exiting now."
   exit 1
end


/* Section:  Create  master.list" */

/* CUSTOMIZATION REQUIRED:
   It is necessary to create the master.list file with the variables
  defined in this script. Follow the format below, which is also used
  in the file level.list which contains serviceability strings that do 
  not change often.  The format is a variableName (such as driverInfo)
  followed by a value which can be several strings.
*/

outLine="   driverInfo Driver: " DRIVER_NAME
temp = LINEOUT(outputFile, outLine);

outLine=" platformInfo This code was generated in " OPERATING_SYSTEM
temp = LINEOUT(outputFile, outLine);

DO UNTIL ( LINES( inputFile ) = 0 )   /* Read until the end of file */
  outLine = LINEIN( inputFile )
  temp = LINEOUT(outputFile,outLine)
END  /* do until */

rc = STREAM(outputFile, C, 'close');
rc = STREAM(inputFile, C, 'close');

/* Section:  Invoke the make file */

say 'nmake -f myapp-w.mak all FLAGS='FLAGS''
'nmake -f myapp-w.mak all FLAGS="'FLAGS'"'
/* end of file */
