#!/usr/bin/ksh
#
# NAME: do-build.ksh driverName [ debug | beta ]
# 
# SAMPLE USAGES:
#   do-build.ksh  driver1
#   do-build.ksh  2001-03-01 debug
#
# PURPOSE:
#  This is the top level script to build the application "myapp".
#
#NOTES:
# * A temporary file named "do-ver.out" is used.
#
#CUSTOMIZATION NOTES:
# * See the sections labeled "CUSTOMIZATION REQUIRED".
# * This example has variables that need to be added to the master.list
#   file, because they change often and/or are set at build time.
#   - They need to be defined in Section "Definition of Dynamic Variables" 
#   - They need to be added to the master.list in Section "Create  master.list".
#   The variables are:
#    DRIVER_NAME      -> Identifier which might be related to the build date.
#    OPERATING_SYSTEM -> Use "ver" or "uname -a" to find out the OS level.
# * The file "level.list" has the serviceability strings that do not change
#   often. The contents of this file are copied into the "master.list".
# * The master.list file is a temporary file that contains all the
#   serviceability strings. master.list is generated every single time
#   this build script is invoked and it is used inside the main make file
#   (myapp-x.mak) which in turn, calls do-what.cmd.
#
#
#******************************************************************

if [ $# -eq 0 ]
then
  print "Error: need to specify driver name."
  print "Usage: do-build.ksh driverName [ debug | beta ]"
  exit 1
fi

print "do-build.ksh: Starting a build of the myapp application"
print "date: `date`"
print ""

# CUSTOMIZATION REQUIRED: Section "Definition"
# * Set up the variables that change frequently:
export DRIVER_NAME="2001-01-16"

# Find out the version and name of the operating system    
                                                           
export OPERATING_SYSTEM=`uname -a`                                
if [ "$OPERATING_SYSTEM" = "" ]                            
then                                                       
   print "*** Error: no output from uname"                 
   exit 1                                                  
fi                                                         



# CUSTOMIZATION REQUIRED:
# * You need to handle the appropriate processing of input arguments.
#   For example, the DEBUG and BETA options are set.

if [ $# -ge 1 ]
then
  typeset -u DRIVER_NAME=$1
  if [ $# -eq 2 ]
  then
    typeset -u OPTION=$2
    case "$OPTION" in
    BETA|beta)
       print "**********ERROR**********"
       print "Set the BETA variable in the java source code for myapp "
       print "to 'true' and then call this script without the beta parameter."
       print ""
       print "./do-build.ksh driverName"
       exit 1

      ;;
    DEBUG|debug)
     print "**********ERROR**********"
     print "Set the DEBUG variable in the java source code for myapp "
     print "to 'true' and then call this script without the debug  parameter:"
     print ""
     print "./do-build.ksh driverName"
     exit 1

      ;;
    *)
      print "Unrecognized option: $OPTION"
      ;;
    esac
  fi
fi

# CUSTOMIZATION REQUIRED:
#You need to remove the old master.list before doing a build.
#NOTE:  not really customization.

rm master.list       # This is a temporary file, with the complete list of strings.

# CUSTOMIZATION REQUIRED:
#   You need to add variables that do not change often to the file
#   "level.list".  This is the input file and is permanent.
#   "master.list" is a temporary file that will have all the
#   serviceability strings. This is the output file.


# CUSTOMIZATION REQUIRED: Section:  "Create master.list"
# * It is necessary to create the master.list file with the variables
#   defined in this script. Follow the format below, which is also used
#   in the file with those serviceability strings that do not change often,
#   such as level.list. The format is a variableName (such as driverInfo) 
#   followed by a value which can be several strings.
print "   driverInfo Driver: $DRIVER_NAME"      >>  master.list
print " platformInfo This code was generated in $OPERATING_SYSTEM" >>  master.list


# CUSTOMIZATION REQUIRED:
# * If you use a name other than "level.list", then customize the following
#   statement.
levelListFile="level.list"
while read line                                                                
do                                                                             
   print $line >> master.list
done < $levelListFile                                                          


# CUSTOMIZATION REQUIRED:
# * Invoke your make file

make -f myapp-x.mak all FLAGS="$FLAGS"

# end of file

