package com.ibm.ulc.application;

/*
 * Copyright (c) 1997 Object Technology International Inc.
 */

import java.util.*;
import com.ibm.ulc.application.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.ORBConnection;
/**
 * Implementation of a default ULCTableModel that can be used for simple applications.
 */
public class GenericTableModel extends ULCTableModel {
	/**
	 * @serial
	 */
	private Vector fData = new Vector();
/**
 * Create a GenericTableModel that uses a Vector for its data collection and a UlcHashtable for each row element.
 */
public GenericTableModel() {
	super(false);
	fData = new Vector();
}
/**
 * Add and return a new instance of a ITableRow object initialized with the specified attributeName and value.
 *
 * @param attributeName	The key of the attribute specified.
 * @param value			The value to be stored at attributeName
 */
public ITableRow addNewRow(String attributeName, Object value) {
	UlcHashtable tableRow = new UlcHashtable();
	tableRow.setValueAt(value, attributeName);
	addRow(tableRow);
	return tableRow;
}
/**
 * Add a ITableRow to myself and notify the UI.
 *
 */
public void addRow(ITableRow tableRow) {
	addRow(tableRow, true);
}
/**
 * Add a ITableRow to myself and notify the UI only if notify is true.
 *
 */
public void addRow(ITableRow tableRow, boolean notify) {
	fData.addElement(tableRow);
	int x = fData.size() - 1;
	if (notify)
		notify(TABLE_MODEL_ROWS_ADDED, null, x, x);
}
/**
 * Add a collection of ITableRows to myself and notify the UI.
 *
 */
public void addRows(Vector vectorOfITableRows) {
	int rowCount = getRowCount();
	for (int i = 0; i < vectorOfITableRows.size(); i++) {
		addRow((ITableRow) vectorOfITableRows.elementAt(i), false);
	}
	notify(TABLE_MODEL_ROWS_ADDED, null, rowCount, getRowCount() - 1);
}
/**
 * Registers the given listener to begin receiving notifications
 * when changes are made to the field.
 * The listener will be notified when rows are edited in the UI and 
 * after the values have been updated locally.
 * If veto is true then the listener is notified of the change 
 * and it is up to the listener to do the actual update to the underlying data.
 *
 * The ULCValueChangedEvent will contain a collection of UlcHashtable entries.
 * <pre>
 * Each entry will contain the following key->values:
 *	"row"		The object/row that has been modified
 *	"rowIndex"	The index of the row.
 *	"columns"	A UlcHashtable containing the attributeName and new value of each column that has been modified or null.
 * </pre>
 *
 * @param listener	The object interested in my changes.
 */
public void addValueChangedListener(IValueChangedListener listener) {
	internalAddListener("setData", listener);
}
/**
 * Change the value at attributeName for the specified rowIndex to the specified value.
 * Notify the UI.
 *
 */
public void changeCellValueAt(String attributeName, int rowIndex, Object value) {
	changeCellValueAt(attributeName, rowIndex, value, true);
}
/**
 * Change the value at attributeName for the specified rowIndex to the specified value.
 * Notify the UI only if notify is true.
 *
 */
public void changeCellValueAt(String attributeName, int rowIndex, Object value, boolean notify) {
	if (attributeName != null) {
		ITableRow tableRow = (ITableRow) fData.elementAt(rowIndex);
		if (tableRow != null) {
			tableRow.setValueAt(value, attributeName);
		}
		if (notify)
			notify(TABLE_MODEL_CELL_CHANGED, attributeName, rowIndex, rowIndex);
	}
}
/**
 * Change the row at rowIndex to the specified ITableRow object.
 * Notify the UI.
 *
 */
public void changeRowAt(int rowIndex, ITableRow tableRow) {
	changeRowAt(rowIndex, tableRow, true);
}
/**
 * Change the row at rowIndex to the specified ITableRow object.
 * Notify the UI only if notify is true.
 *
 */
public void changeRowAt(int rowIndex, ITableRow tableRow, boolean notify) {
	fData.setElementAt(tableRow, rowIndex);
	if (notify)
		notify(TABLE_MODEL_ROWS_CHANGED, null, rowIndex, rowIndex);
}
/**
 * Delete all elements in my collection.  Notify the UI.
 *
 */
public void clear() {
	fData.removeAllElements();
	notify(TABLE_MODEL_CONTENTS_CHANGED, null, -1, -1);
}
/**
 * Return a new instance of a ITableRow object.
 */
public static ITableRow createNewRow() {
	return new UlcHashtable();
}
/**
 * Return a new instance of a ITableRow object initialized with the specified attributeName and value.
 *
 * @param attributeName	The key of the attribute specified.
 * @param value			The value to be stored at attributeName
 */
public static ITableRow createNewRow(String attributeName, Object value) {
	UlcHashtable tableRow = new UlcHashtable();
	tableRow.setValueAt(value, attributeName);
	return tableRow;
}
/**
 * Answer the ITableRow at the specified index in the receiver
 *
 * @param index	The index of the row being accessed.
 */
public Object getRowAt(int index) {
	return fData.elementAt(index);
}
/**
 * Return the number of rows in my collection.
 */
public int getRowCount() {
	if (fData == null)
		return 0;
	return fData.size();
}
/**
 * Return the rows in my collection.
 * WARNING: Callers of this method must never modify this collection
 * directly but must use the provided API. Failing to do so will corrupt this table model
 * and the UI will not remain in sync with the application.
 */
public Vector getRows() {
	return fData;
}
/**
 * Return the value at the specified attributeName and row index.
 */
public Object getValueAt(String attributeName, int rowIndex) {
	ITableRow tableRow = (ITableRow) fData.elementAt(rowIndex);
	if (tableRow != null)
		return tableRow.getValueAt(attributeName);
	else
		return null;
}
/**
 * Insert a ITableRow after the specified ITableRow object and notify the UI.
 */
public void insertRowAfter(ITableRow tableRow) {
	int index = fData.indexOf(tableRow);
	if (index >= 0)
		insertRowAtIndex(index + 1, tableRow, true);
}
/**
 * Insert a ITableRow at the specified row index and notify the UI.
 */
public void insertRowAtIndex(int index, ITableRow tableRow) {
	insertRowAtIndex(index, tableRow, true);
}
/**
 * Insert a ITableRow at the specified row index and notify the UI if notify is true.
 */
public void insertRowAtIndex(int index, ITableRow tableRow, boolean notify) {
	fData.insertElementAt(tableRow, index);
	if (notify)
		notify(TABLE_MODEL_ROWS_ADDED, null, index, index);
}
/**
 * Insert a ITableRow before the specified ITableRow object and notify the UI.
 */
public void insertRowBefore(ITableRow tableRow) {
	int index = fData.indexOf(tableRow);
	if (index >= 0)
		insertRowAtIndex(index, tableRow, true);
}
/**
 * Insert a collection of ITableRow's after the specified ITableRow object and notify the UI.
 */
public void insertRowsAfter(ITableRow tableRow, Vector vectorOfITableRows) {
	if (!vectorOfITableRows.isEmpty()) {
		int index = fData.indexOf(tableRow);
		Enumeration e = vectorOfITableRows.elements();
		int position = index + 1;
		while (e.hasMoreElements()) {
			ITableRow nextRow = (ITableRow) e.nextElement();
			insertRowAtIndex(position, nextRow, false);
			position++;
		}
		notifyRowsAdded(index + 1, index + vectorOfITableRows.size());
	}
}
/**
 * Insert a collection of ITableRow's before the specified ITableRow object and notify the UI.
 */
public void insertRowsBefore(ITableRow tableRow, Vector vectorOfITableRows) {
	if (!vectorOfITableRows.isEmpty()) {
		int index = fData.indexOf(tableRow);
		Enumeration e = vectorOfITableRows.elements();
		while (e.hasMoreElements()) {
			ITableRow nextRow = (ITableRow) e.nextElement();
			insertRowAtIndex(index, nextRow, false);
		}
		notifyRowsAdded(index, index + vectorOfITableRows.size() - 1);
	}
}
/**
 * Notify the UI that the specified attribute at the specified row index has changed.
 */
public void notifyCellChanged(String attributeName, int rowIndex) {
	notify(TABLE_MODEL_CELL_CHANGED, attributeName, rowIndex, rowIndex);
}
/**
 * Notify the UI that my entire contents has changed and all rows need 
 * to be requested once again.
 */
public void notifyContentsChanged() {
	notify(TABLE_MODEL_CONTENTS_CHANGED, null, -1, -1);
}
/**
 * Notify the UI that the specified row has changed and must be refreshed in the UI.
 */
public void notifyRowChanged(int rowIndex) {
	notify(TABLE_MODEL_ROWS_CHANGED, null, rowIndex, rowIndex);
}
/**
 * Notify the UI that rows have been added at the specified indices.
 */
public void notifyRowsAdded(int startRowIndex, int endRowIndex) {
	notify(TABLE_MODEL_ROWS_ADDED, null, startRowIndex, endRowIndex);
}
/**
 * Notify the UI that rows have been removed at the specified indices.
 */
public void notifyRowsRemoved(int startRowIndex, int endRowIndex) {
	notify(TABLE_MODEL_ROWS_REMOVED, null, startRowIndex, endRowIndex);
}
/**
 * The vector ranges will contain a collection of UlcRange objects
 * that represent contigous row indexes that need to be removed.
 *
 */
public void removeRanges(Vector ranges) {
	Enumeration e = ranges.elements();
	int index = 0;
	while (e.hasMoreElements()) {
		UlcRange r = (UlcRange) e.nextElement();
		for (int i = r.fStartIndex; i <= r.fEndIndex; i++) {
			fData.removeElementAt(i - index);
			index = index + 1;
		}
	}
	notify(TABLE_MODEL_ROWS_REMOVED, null, ranges);
}
/**
 * Remove the specified ITableRow and notify the UI.
 */
public void removeRow(ITableRow tableRow) {
	removeRow(tableRow, true);
}
/**
 * Remove the specified ITableRow and notify the UI if notify is true.
 */
public void removeRow(ITableRow tableRow, boolean notify) {
	int i = fData.indexOf(tableRow);
	if (i >= 0) {
		fData.removeElement(tableRow);
		if (notify)
			notify(TABLE_MODEL_ROWS_REMOVED, null, i, i);
	}
}
/**
 * Remove the row at the specified row index and notify the UI.
 */
public void removeRowAt(int rowIndex) {
	removeRowAt(rowIndex, true);
}
/**
 * Remove the row at the specified row index and notify the UI if notify is true.
 */
public void removeRowAt(int rowIndex, boolean notify) {
	fData.removeElementAt(rowIndex);
	if (notify)
		notify(TABLE_MODEL_ROWS_REMOVED, null, rowIndex, rowIndex);
}
/**
 * Remove the specified collection of rows and notify the UI.
 */
public void removeRows(Vector vectorOfITableRows) {
	if (!vectorOfITableRows.isEmpty()) {
		Vector indices = new Vector();
		Enumeration e = vectorOfITableRows.elements();
		while (e.hasMoreElements()) {
			ITableRow nextRow = (ITableRow) e.nextElement();
			int index = fData.indexOf(nextRow);
			if (index >= 0)
				indices.addElement(new Integer(index));
		}
		removeRanges(UlcRange.createFromVector(indices));
	}
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive value changed events.
 * The listener will be notified when rows are edited in the UI and 
 * after the values have been updated locally.
 *
 * The ULCValueChangedEvent will contain a collection of UlcHashtable entries.
 * <pre>
 * Each entry will contain the following key->values:
 *	"row"		The object/row that has been modified
 *	"rowIndex"	The index of the row.
 *	"columns"	A UlcHashtable containing the attributeName and new value of each column that has been modified or null.
 * </pre>
 *
 * @param listener	The object that was interested in my changes.
 */
public void removeValueChangedListener(IValueChangedListener listener) {
	internalRemoveListener("setData", listener);
}
/**
 * The UI updates the application with changes from UI.
 *
 * If the veto flag is not set, use the default implementation of the superclass.
 * @see ULCTableModel#setData
 *
 * If the veto flag is set to true, send all changes to the application. Do not update
 * the UI, as this is the applications responsibility.
 *
 * @param args		Anything		The arguments associated with this request.
 */
protected void setData(Anything args) {
	if (!getVeto()) {
		super.setData(args);
	}
	Vector changeCollection = new Vector();
	Anything rowIds = args.get("rowids");
	if (rowIds != null) {
		for (int i = 0; i < rowIds.size(); i++) {
			Anything changes = args.get("changes");
			Anything aa = changes.get(i);
			int oid = rowIds.get(i).asInt(-1);
			int row = getRowIndexForOid(oid);
			UlcHashtable changeEntry = new UlcHashtable();
			changeEntry.setValueAt(getRowAt(row), "row");
			changeEntry.setValueAt(new Integer(row), "rowIndex");
			if (row >= 0) {
				Anything columns = aa.get("cols");
				UlcHashtable changeCols = new UlcHashtable();
				for (int j = 0; j < columns.size(); j++) {
					Anything column = columns.get(j);
					String colId = column.get("colId", null);
					Anything value = column.get("value");
					if (row >= 0 && colId != null && value != null) {
						changeCols.setValueAt(convert(value), colId);
					}
				}
				changeEntry.setValueAt(changeCols, "columns");
			}
			changeCollection.addElement(changeEntry);
		}
	}
	distributeToListeners("setData", new ULCValueChangedEvent(this, changeCollection));
	return;
}
/**
 * Set the contents of my rows collection to be the vector containing ITableRow objects.
 * Notify the UI that my contents have changed.
 */
public void setRows(Vector vector) {
	fData = vector;
	notifyContentsChanged();
}
/**
 * Set the contents of attributeName at the specified rowIndex to be the specified value.
 * Notify the UI that the cell has changed.
 */
public void setValueAt(Object value, String attributeName, int rowIndex) {
	ITableRow tableRow = (ITableRow) fData.elementAt(rowIndex);
	if (tableRow != null) {
		tableRow.setValueAt(value, attributeName);
		notifyCellChanged(attributeName, rowIndex);
	} else
		return;
}
}
