package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.util.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.ORBConnection;
import java.awt.Color;

/**
 * This is the abstract superclass for all list widgets, hierarchical as well as indexed.
 */
public abstract class ULCAbstractList extends ULCComponent implements IItemListOwner, IEnabler {
	/**
	 * @serial	 
	 */
	protected int fWidth = 0, fHeight = 0, fHeightInRows = 10;
	/**
	 * @serial	 
	 */
	protected ULCAbstractTableModel fModel = null;
	/**
	 * @serial	 
	 */
	protected IItemList fItemList = null;
	/**
	 * @serial	 
	 */
	protected int fSelectionMode; // Single or multiple select	
	/**
	 * @serial	 
	 */
	protected int[] fSelectedOids = new int[0]; // no selection	

	/**
	 * @serial	 
	 */
	protected static final boolean DEBUG = false;
	/**
	 * The <code>int</code>row height in pixels.
	 * @serial	 
	 */
	protected int fRowHeight = -1;
	/**
	 * The selection foreground color for this widget.
	 * @serial	 
	 */
	protected Color fSelectionForeColor;
	/**
	 * The selection background color for this widget.
	 * @serial	 
	 */
	protected Color fSelectionBackColor;
	/**
	 * The set of row models on this widget.
	 * @serial	 
	 */
	protected Vector fRowModels;
	/**
	 * The initially selected items.
	 * @serial	 
	 */
	protected Vector fInitiallySelectedItems;
	/**
	 * The initially selected indices.
	 * @serial	 
	 */
	protected Vector fInitiallySelectedIndices;
/**
 * Constructs a new instance of the receiver.
 *
 */
public ULCAbstractList() {
}
/**
 * Constructs a new instance of the receiver, which displays values from the given tableModel.
 *
 * @param tableModel 		The tableModel that serves as the data source.
 */
public ULCAbstractList(ULCAbstractTableModel tableModel) {
	fModel = tableModel;
	if (fModel != null)
		fModel.addOwner(this);
	setItemList(null);
}
/**
 * Constructs a new instance of the receiver which displays values from the given tableModel.
 * The receiver is built with the width and heightInRows as specified.
 *
 * @param tableModel 	The tableModel that serves as the data source.
 * @param width			The width in pixels for this table.
 * @param heightInRows	The height specified as number of visible rows for this table.
 */
public ULCAbstractList(ULCAbstractTableModel tableModel, int width, int heightInRows) {
	fModel = tableModel;
	if (fModel != null)
		fModel.addOwner(this);
	setItemList(null);
	fWidth = width;
	if (heightInRows > 50) {
		trouble("ULCAbstractList", "too many rows! May be you are specifying pixels instead of rows?");
	} else
		fHeightInRows = heightInRows;
}
/**
 * Constructs a new instance of the receiver which displays values from the given tableModel.
 * The receiver is built with the width, heightInRows and selection mode as specified.
 *
 * @param tableModel	The table model that serves as the data source.
 * @param width			The width in pixels for this table.
 * @param heightInRows	The height specified as number of visible rows for this table.
 * @param selectionMode	The selection mode which can be one of:
 * <pre>
 *  					IDefaults.LIST_SINGLE_SELECTION
 *  					IDefaults.LIST_SINGLE_INTERVAL_SELECTION
 *  					IDefaults.LIST_MULTIPLE_INTERVAL_SELECTION
 * </pre>
 */
public ULCAbstractList(ULCAbstractTableModel tableModel, int width, int heightInRows, int selectionMode) {
	fModel = tableModel;
	if (fModel != null)
		fModel.addOwner(this);
	setItemList(null);
	fWidth = width;
	if (heightInRows > 50) {
		trouble("ULCAbstractTableModel", "too many rows! May be you are specifying pixels instead of rows?");
	} else
		fHeightInRows = heightInRows;
	fSelectionMode = selectionMode;
}
/**
 * Add all form attributes that should be preloaded to the UI to the specified Vector.
 *
 * @param vectorOfPreloadAttributes	Vector	into which the receiver adds the
 *					form attributes it needs to preload.
 *
 */
public void addPreloadFormAttributesInto(Vector vectorOfPreloadAttributes) {
}
/**
 * Add all table attributes that should be preloaded to the UI to the specified Vector.
 * This receiver defines this abstract method for subclasses to implement.
 *
 * @param vectorOfPreloadAttributes	Vector	into which the receiver adds the
 *					table attributes it needs to preload.
 *
 */
abstract public void addPreloadTableAttributesInto(Vector vectorOfPreloadAttributes);
/**
 * Add the specified rowModel to the receiver's collection of rowModels
 */
void addRowModel(ULCAbstractRowModel rowModel) {
	if (fRowModels == null)
		fRowModels = new Vector();
	if (!fRowModels.contains(rowModel))
		fRowModels.addElement(rowModel);
}
/**
 * Compute and answer the columns that should be uploaded completely every time
 * the UI gets the initial rows from the itemlist. If a widget returns a non-empty
 * array, the complete table will be uploaded, complete with the attributes returned
 * by computePreloadedAttributes(). In effect, the array answered by this method modifies
 * the prefetch attribute of itemlists / tablemodels
 *
 */
public String[] computePreloadedColumns() {
	return new String[0];
}
/*
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
abstract public void distributeSelectionChangeEventToListeners(String type);
/**
 * Requests my UI proxy to ensure that the object at <code>object</code> is visible.
 *
 * NOTE: It is recommended that the following methods are
 *       used instead of this one:
 *
 * @see		ULCTableList#ensureIndexIsVisible
 * @see		ULCAbstractTree#ensureNodeIsVisible
 *
 * @param	object	The object of the table to be made visible.
 */
public void ensureItemIsVisible(Object object) {
	int id= internalGetModel().getRowIdFor(object);
	if (id != -1)	// skip if node unavailable
		sendUI("ensureItemIsVisible", new Anything(id));
}
/**
 * Answer the default item list for the receiver, 
 * which is the defaultItemList of the receiver's model.
 *
 */
abstract protected IItemList getDefaultItemList();
/**
 * Gets the height of this widget
 *
 */
public int getHeight() {
	return fHeight;
}
/**
 * Gets the height in rows of this widget
 *
 * @return	The <code>int</code> heightInRows
 */
public int getHeightInRows() {
	return fHeightInRows;
}
/**
 * Answer the receiver's current itemList.
 *
 * The receiver's <code>ULCItemlist</code> is responsible for the order in which
 * the receiver's rows are displayed.
 *
 * @see ULCDefaultItemList
 * @see ULCSortedItemList
 *
 */
public IItemList getItemList() {
	if (fItemList == null)
		return getDefaultItemList();
	else
		return fItemList;
}
/**
 * Returns the row height in pixels.
 * Note: this value is not valid unless explicitly set and is not 
 * updated from the UI.
 * 
 */
public int getRowHeight() {
	return fRowHeight;
}
/**
 * Return the currently selected item or null if nothing is selected
 *
 * @return  The <code>Object</code> currently selected.
 */
public Object getSelectedItem() {
	if (getSelectedOids().length == 0)
		return null;
	else {
		return internalGetModel().getRow(getSelectedOids()[0]);
	}
}
/**
 * Return the currently selected items or an empty Vector if nothing is selected
 *
 * @return  The <code>Vector</code> of objects currently selected.
 */
public Vector getSelectedItems() {
	Vector selectedItems = new Vector();
	for (int i = 0; i < getSelectedOids().length; i++) {
		selectedItems.addElement(internalGetModel().getRow(getSelectedOids()[i]));
	}
	return selectedItems;
}
/**
 * Return the selected rows as the collection of indices
 *
 * @return indices The <code>Vector</code> of indices selected.
 */
protected int[] getSelectedOids() {
	return fSelectedOids;
}
/**
 * Returns the selection background color for this widget.
 *
 * @return The <code>Color</code> that should be used within this widget
 */
public Color getSelectionBackground() {
	return fSelectionBackColor;
}
/**
 * Returns the selection foreground color for this widget.
 *
 * @return The <code>Color</code> that should be used within this widget
 */
public Color getSelectionForeground() {
	return fSelectionForeColor;
}
/**
 * Gets the current selection mode of the <code>ULCTableList</code>
 *
 * @return The selection mode which for table/list can be one of:
 * <pre>
 *  					IDefaults.LIST_SINGLE_SELECTION
 *  					IDefaults.LIST_SINGLE_INTERVAL_SELECTION
 *  					IDefaults.LIST_MULTIPLE_INTERVAL_SELECTION
 * </pre>
 *
 *		or for trees:
 *
  * <pre>
 *  					TREE_SINGLE_SELECTION
 *  					TREE_CONTIGUOUS_SELECTION
 *  					TREE_DISCONTIGUOUS_SELECTION
 * </pre>
 */
public int getSelectionMode() {
	return fSelectionMode;
}
/**
 * Gets the width of this widget
 *
 */
public int getWidth() {
	return fWidth;
}
/*
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		Vector rows = new Vector();
		Anything selection = args.get("rows");
		String type = args.get("type", null);
		if (selection != null) {
			Anything rowIds = selection.get("rowids");
			if (rowIds != null) {
				for (int i = 0; i < rowIds.size(); i++) {
					int oid = rowIds.get(i).asInt(-1);
					if (oid > -1)
						rows.addElement(new Integer(oid));
				}
			}
		}
		setSelectedOids(type, rows);
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Gets the <code>ULCTableModel</code> that will serve as my data source.
 *
 * @return	The <code>ULCTableModel</code>
 */
protected ULCAbstractTableModel internalGetModel() {
	if (fModel == null)
		if (fItemList == null) {
			setDefaultModel();
			return internalGetModel();
		}
		else
			return getItemList().getModel();
	else
		return fModel;
}
/**
 * Set the <code>ULCAbstractTableModel</code> that will serve as my data source.
 * If the model being set is already uploaded to the UI this call can 
 * be used to switch the data source for the widget without requiring additional
 * round trips to the application to retrieve the data.
 *
 * @param tableModel	The <code>ULCAbstractTableModel</code>
 */
protected  void internalSetModel(ULCAbstractTableModel tableModel) {
	ULCAbstractTableModel oldModel = fModel;
	if (tableModel == null)
		setModelItemList(null, null);
	else
		setModelItemList(tableModel, null);
	if (fRowModels != null) {
		if (oldModel != null) {
			for (int i = 0; i < fRowModels.size(); i++) {
				ULCAbstractRowModel rowModel = (ULCAbstractRowModel) fRowModels.elementAt(i);
				rowModel.setWidgetTableModel(null);
			}
		}
		if (fModel != null || fItemList != null) {
			for (int i = 0; i < fRowModels.size(); i++) {
				ULCAbstractRowModel rowModel = (ULCAbstractRowModel) fRowModels.elementAt(i);
				rowModel.setWidgetTableModel(internalGetModel());
			}
		}
	}
}
/**
 * remove the specified rowModel from the receiver's collection of rowModels
 *
 */
void removeRowModel(ULCAbstractRowModel rowModel) {
	if (fRowModels != null && fRowModels.contains(rowModel)) {
		fRowModels.removeElement(rowModel);
		if (fRowModels.size() == 0)
			fRowModels = null;
	}
}
/**
 * Save the reference of the receiver's itemList to the specified
 * Anything. Do nothing if the itemlist is null.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveItemList(Anything a) {
	if (fItemList != null)
		a.put("itemList", fItemList.getRef(fContext));
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	if (fModel != null)
		a.put("model", fModel.getRef(fContext));
	saveItemList(a);
	if (fWidth != 0)
		a.put("w", fWidth);
	if (fHeight != 0)
		a.put("h", fHeight);
	if (fHeightInRows != 10)
		a.put("rows", fHeightInRows);
	if (fSelectionMode != LIST_SINGLE_SELECTION)
		a.put("selectionMode", fSelectionMode);
	if (fRowHeight > 0)
		a.put("rowHeight", fRowHeight);
	if (fSelectionForeColor != null)
		a.put("ofc", internalConvertColor(fSelectionForeColor));
	if (fSelectionBackColor != null)
		a.put("obc", internalConvertColor(fSelectionBackColor));
	if (fInitiallySelectedIndices == null && fInitiallySelectedItems != null && !fInitiallySelectedItems.isEmpty()) {
		setSelectedItems(fInitiallySelectedItems);
		fInitiallySelectedItems= null;
	}
	super.saveState(a);
}
private void sendItemList() {
	if (!isUploaded())
		return;
	Anything args = new Anything();
	args.put("model", fModel.getRef(fContext));
	if (getItemList() != null)
		args.put("itemList", getItemList().getRef(fContext));
	sendUI("setItemList", args);
}
/**
 *  Update the UI with the newly selected items.
 *
 */
abstract protected void sendSelectedItems();
/**
 * Gets the <code>ULCTableModel</code> that will serve as my data source.
 *
 * @return	The <code>ULCTableModel</code>
 */
protected void setDefaultModel() {
	internalSetModel(new ULCEmptyTableModel());
}
/**
 * Sets the height of this widget
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @param height	The <code>int</code> height
 */
public void setHeight(int height) {
	fHeight = height;
}
/**
 * Set the height in rows of this widget
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @param heightInRows	The <code>int</code> heightInRows
 */
public void setHeightInRows(int heightInRows) {
	fHeightInRows = heightInRows;
}
/**
 * Set the <code>IItemList</code> that will serve as intermediary between
 * the receiver and its ULCTableModel.
 *
 * IMPORTANT: If the itemList passed does not know its tableModel, this method will throw
 * an exception. When connecting widgets to an itemList visually the connection 
 * between the itemList and its tableModel must be fired prior to the connection to the
 * widget. The sequence of connection execution can be modified in the builder.
 *
 * if <code>itemList</code> is null, the default itemList of the receiver's <code>ULCTableModel</code>
 * will be used by default.
 *
 * @see #getItemList
 *
 * @param itemList	The <code>IItemList</code> to be used from now on
 */
public void setItemList(IItemList itemList) {
	if (fItemList != itemList) {
		if (itemList == null)
			internalSetModel(fModel);
		else
			setModelItemList(itemList.getModel(), itemList);
	}
}
/**
 * Set the <code>ULCAbstractTableModel</code> subclass that will serve as my data source.
 * If the model being set is already uploaded to the UI this call can 
 * be used to switch the data source for the widget without requiring additional
 * round trips to the application to retrieve the data.
 *
 * @param tableModel	The <code>ULCAbstractTableModel</code>
 */
void setModelItemList(ULCAbstractTableModel tableModel, IItemList itemList) {
	if (fModel == tableModel && fItemList == itemList)
		return; // nothing to do
	if (fItemList != null)
		fItemList.removeOwner(this);
	else {
		if (fModel != null)
			getItemList().removeOwner(this);
	}
	if (fModel != tableModel) {
		if (fModel != null)
			fModel.removeOwner(this);
		fModel = tableModel;
		if (fModel != null)
			fModel.addOwner(this);
	}
	if (itemList == null || (tableModel != null && (itemList == getDefaultItemList())))
		fItemList = null;
	else
		fItemList = itemList;
	if (fModel != null)
		getItemList().addOwner(this);
	setSelectedOids("select", new int[0]);
	sendItemList();
}
/**
 * Set the row height of the tree
 *
 * @param rowHeight The height of a row in the tree.
 */
public void setRowHeight(int rowHeight) {
	if (fRowHeight != rowHeight) {
		fRowHeight = rowHeight;
		sendUI("setRowHeight", new Anything(fRowHeight));
	}
}
/**
 * Set the currently selected item to be the object specified
 *
 * @param  The <code>Object</code>  to be selected.
 */
public void setSelectedItem(Object object) {
	if (object == null)
		setSelectedItems(null);
	else {
		Vector selectedItems= new Vector();
		selectedItems.add(object);
		setSelectedItems(selectedItems);
	}
}
/**
 * Set the currently selected items to be the objects in the specified Vector
 *
 * @param  The <code>Vector</code> of objects to be selected.
 */
public void setSelectedItems(Vector items) {
	if (items == null) {
		fInitiallySelectedItems= null;
		fInitiallySelectedIndices= null;
		setSelectedItems(new Vector());
	}
	else {
		int[] selectedOids= new int[items.size()];
		for (int i= 0; i < items.size(); i++) {
			Object object= items.elementAt(i);
			int oid= internalGetModel().getRowIdFor(object);
			if (oid == -1) {
				// if one oid is not known yet then all must be pending
				fInitiallySelectedItems= (Vector) items.clone();
				fInitiallySelectedIndices= null;
				return;
			}
			selectedOids[i]= oid;			
		}
		setSelectedOids("select", selectedOids);
		if (isUploaded())
			sendSelectedItems();
	}
}
/**
 * Set the currently selected items to be the objects in the specified Vector
 *
 * @param  The <code>Vector</code> of objects to be selected.
 */
public void setSelectedOids(String type, int[] oids) {
	fSelectedOids = oids;
	distributeSelectionChangeEventToListeners(type);
}
/**
 * Set the currently selected items to be the objects in the specified Vector
 *
 * @param  The <code>Vector</code> of objects to be selected.
 */
void setSelectedOids(String type, Vector oids) {
	int[] rowIds = new int[oids.size()];
	for (int i = 0; i < oids.size(); i++)
		rowIds[i] = ((Integer) oids.elementAt(i)).intValue();
	setSelectedOids(type, rowIds);
}
/**
 * Sets the selection background color for this widget.
 *
 * @param color The <code>Color</code> that should be used within this widget
 */
public void setSelectionBackground(Color color) {
	fSelectionBackColor = color;
	if (color == null)
		sendUI("setSelectionBackground", new Anything());
	else
		sendUI("setSelectionBackground", internalConvertColor(fSelectionBackColor));
}
/**
 * Sets the selection foreground color for this widget.
 *
 * @param color The <code>Color</code> that should be used within this widget
 */
public void setSelectionForeground(Color color) {
	fSelectionForeColor = color;
	if (color == null)
		sendUI("setSelectionForeground", new Anything());
	else
		sendUI("setSelectionForeground", internalConvertColor(fSelectionForeColor));
}
/**
 * Set the current selection mode of the <code>ULCTableList</code>
 *
 * @param selectionMode	The selection mode which can be one of:
 * <pre>
 *  					IDefaults.LIST_SINGLE_SELECTION
 *  					IDefaults.LIST_SINGLE_INTERVAL_SELECTION
 *  					IDefaults.LIST_MULTIPLE_INTERVAL_SELECTION
 * </pre>
 * and for trees:
 * <pre>
 *  					TREE_SINGLE_SELECTION
 *  					TREE_CONTIGUOUS_SELECTION
 *  					TREE_DISCONTIGUOUS_SELECTION
 * </pre>

 */
public void setSelectionMode(int selectionMode) {
	if (selectionMode != fSelectionMode) {
		fSelectionMode = selectionMode;
		sendUI("setSelectionMode", new Anything(selectionMode));
	}
}
/**
 * Set the width of this widget
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @param width	The <code>int</code> width
 */
public void setWidth(int width) {
	fWidth = width;
}
}
