package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.ORBConnection;
import java.util.Vector;
/**
 * Implementation of a FormModel which forwards its getValueAt and setValueAt requests
 * to a specific row of a table model.
 * If the tableList property is specified then this row model will automatically track 
 * selection changes in the table, tree or list widget that it is connected to.
 * If the tableList property is not specified then an explicit call to <code>setRow</code> 
 * is needed to specify over which row within the table model that this row model is acting upon.
 */
abstract public class ULCAbstractRowModel extends ULCAbstractModel implements IForm, IEnabler, IModelOwner {
	/**
	 * The current row of the <code>ULCTableModel</code> that I map into.
	 * @serial	 
	 */
	protected int fRowId = -1;
	/**
	 * The <code>ULCTableModel</code> that I will track.
	 * @serial	 
	 */
	protected ULCAbstractTableModel fTableModel = null;
	/**
	 * The <code>ULCAbstractList</code> table/list/tree widget
	 * @serial	 
	 */
	protected ULCAbstractList fWidget = null;
/**
 *  Constructs a new instance of the receiver
 */
public ULCAbstractRowModel() {
}
/**
 * Constructs a new instance of the receiver based on the specified tableModel.
 *
 * @param table The <code>ULCAbstractTableModel</code> whose values I track.
 */
public ULCAbstractRowModel(ULCAbstractTableModel table) {
	fTableModel = table;
	if (fTableModel != null)
		fTableModel.addOwner(this);
}
/**
 * Constructs a new instance of the receiver based on the specified tableModel, for
 * the specified widget.
 *
 * @param table The <code>ULCAbstractTableModel</code> whose values I track.
 * @param tableList 	The <code>ULCAbstractList</code> list widget whose current 
 *						selection the receiver is tracking.
 */
public ULCAbstractRowModel(ULCAbstractTableModel tableModel, ULCAbstractList widget) {
	fTableModel = tableModel;
	fWidget = widget;
	if (fTableModel != null)
		fTableModel.addOwner(this);
}
/**
 * Add the specified attribute to the list of attributes that will be uploaded to the
 * UI even if not requested by the UI when a row is uploaded for the first time. 
 *
 *
 * @param 	attributeName	the <code>String</code> attributeName 
 */
public void addPreloadAttribute(String attributeName) {
	super.addPreloadAttribute(attributeName);
	internalUpdatePreloadAttributes();
}
/**
 * Add all attributes that should be preloaded to the UI to the specified Vector.
 *
 * @param vectorOfPreloadAttributes	The <code>Vector</code> into which the receiver's
 									preloadAttributes should be added.
 */
protected void addPreloadAttributesInto(Vector vectorOfPreloadAttributes) {
	if (!getUploadAllAttributes())
		return;
	addPreloadFormAttributesInto(vectorOfPreloadAttributes);
}
/**
 * Add the table attributes that should be preloaded to the UI to the specified Vector.
 *
 * @param vectorOfPreloadAttributes	Vector	into which the receiver adds the
 *											table attributes it needs to preload.
 */
public void addPreloadTableAttributesInto(Vector vectorOfPreloadAttributes) {
	addPreloadFormAttributesInto(vectorOfPreloadAttributes);
}
/**
 * Send a request to the Table Model to throw away the current edit values of
 * the receiver's current model.
 *
 * Do nothing if oid is -1 or if the tableModel is null
 */
public void cancelInput() {
	if (getTableModel() != null)
		getTableModel().cancelInput(this);
}
/**
 * Set the <code>ULCTableModel</code> that will serve as my data source.
 *
 * @param tableModel	The <code>ULCAbstractTableModel</code>
 */
public void configureTableModel(ULCAbstractTableModel tableModel) {
	if (tableModel != null) {
		tableModel.addOwner(this);
		internalUpdatePreloadAttributes();
	}
}
/**
 * Return the row identifier on which the receiver is functioning as FormModel.
 * Note: The table model must be set before this method can be called.
 */
int getCurrentOid() {
	return fRowId;
}
/**
 * Return the row index on which the receiver is functioning as FormModel.
 * Note: The table model must be set before this method can be called.
 */
public Object getCurrentRow() {
	return getTableModel().getRow(getCurrentOid());
}
/**
 * Return the <code>ULCAbstractList</code> whose current selection the receiver is tracking.
 */
public ULCAbstractList getList() {
	return fWidget;
}
/**
 * Gets the <code>ULCAbstractTableModel</code> that will serve as my data source.
 *
 * @return	The <code>ULCAbstractTableModel</code>
 */
public ULCAbstractTableModel getTableModel() {
	if (getList() != null)
		return getList().internalGetModel();
	else
		return fTableModel;
}
private void internalUpdatePreloadAttributes() {
	if (getTableModel() != null) {
		Vector attrs = getPreloadAttributes();
		for (int i = 0; i < attrs.size(); i++) {
			getTableModel().addPreloadAttribute((String) attrs.elementAt(i));
		}
	}
}
/**
 * Send a request to the Table Model to request the current edit values of
 * the receiver's current model.
 *
 * Do nothing if oid is -1 or if the tableModel is null
 *
 * @return true if the values have been recieved. Returns false if the connection failed
 * 			while waiting for the reply.
 */
public boolean saveInput() {
	if (getTableModel() != null)
		return getTableModel().saveInput(this);
	return true;
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 * Only one of widget and tableModel will be stored in the state because if list is specified, the receiver
 * uses its tableModel. If list is not defined, the tableModel must be set explicitly.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if ((fTableModel != null) && (fWidget == null))
		a.put("tableModel", fTableModel.getRef(fContext));
	if (fWidget != null)
		a.put("w", fWidget.getRef(fContext));
	if (fRowId != -1)
		a.put("rowid", fRowId);
//	internalUpdatePreloadAttributes();
}
/**
 * Update the UI with the new row information.
 *
 * @parm rowid		the oid of the row the receiver is currently using.
 */
abstract protected void sendCurrentOid(int rowId);
/**
 * Specify the row on which the receiver is functioning as FormModel.
 * Note: The table model must be set before this method can be called.
 *
 * @param row	The oid of the row within the <code>ULCTableModel</code> whose values
 *				I now represent.
 */
public void setCurrentOid(int rowId) {
	fRowId = rowId;
	sendCurrentOid(rowId);
}
/**
 * Specify the row on which the receiver is functioning as FormModel.
 * Note: The table model must be set before this method can be called.
 *
 * @param row	The object  whose values the receiver now represents.
 */
public void setCurrentRow(Object row) {
	setCurrentOid(getTableModel().getRowIdFor(row));
}
/**
 * Set the the <code>ULCAbstractList</code> whose current selection the receiver is tracking.
 */
protected void setList(ULCAbstractList list) {
	if (fWidget != null)
		fWidget.removeRowModel(this);
	fWidget = list;
	if (list != null)
		list.addRowModel(this);
	setWidgetTableModel(list.internalGetModel());
}
/**
 * Set the <code>ULCAbstractTableModel</code> that will serve as my data source.
 *
 * @param tableModel	The <code>ULCAbstractTableModel</code>
 */
public void setTableModel(ULCAbstractTableModel tableModel) {
	if (getList() != null) {
		if (tableModel != null && (getTableModel() != tableModel && isUploaded()))
			trouble("setTableModel", "when a list is used, that list's tableModel must be used in row models");
		return;
	}
	if (fTableModel != null)
		fTableModel.removeOwner(this);
	fTableModel = tableModel;
	configureTableModel(tableModel);
}
/**
 * The default behavior causes all this formModel to send all 
 * attributes known to be used to the UI during the saveState.
 * Users can disable this behavior by setting this flag to false.
 *
 * Note: Once this attribute has been set to true we cannot allow
 * it to be set to false since the UI will not request attributes
 * once marked as preload.
 *
 * @param uploadAllAttributes	boolean	Determines whether all attributes
 *										should be uploaded.
 */
public void setUploadAllAttributes(boolean uploadAllAttributes) {
	super.setUploadAllAttributes(uploadAllAttributes);
	internalUpdatePreloadAttributes();
}
void setWidgetTableModel(ULCAbstractTableModel tableModel) {
	configureTableModel(tableModel);
}
}
