package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.Assert;
import com.ibm.ulc.comm.*;

/**
 * The ULC button widget consists of an icon, a label, or both.
 * It fires ULCActionEvents to its listeners.
 */
public class ULCButton extends ULCComponent {
	/**
	 * The ULCIcon if specified to be displayed.
	 * @serial
	 */
	protected ULCIcon fIcon = null;
	/**
	 * The integer value of the Mnemonic character.
	 * @serial	 
	 */
	protected int fMnemonic = -1;
	/**
	 * If the disable on action is set to true then the button is disabled on 
	 * user click. The application is responsible for enabling the button either by explicty
	 * calling setEnable or via an enabler.
	 * @serial	 
	 */
	protected boolean fDisableOnAction = false;
	/**
	 * Any valid ULCTrigger object which will be triggered in the UI when 
	 * this widgets actionPerformed method is called
	 * @serial	 
	 */
	protected ITrigger fTrigger= null;
public ULCButton() {
}
/**
 * Constructs a button with a label.
 */
public ULCButton(String label) {
	super(label);
}
/**
 * Constructs a button with a label and registers the given ActionListener.
 *
 * @param label 	the label string to be displayed.
 * @param listener	the widget that will listen for my action events.
 */
public ULCButton(String label, IActionListener listener) {
	super(label);
	addActionListener(listener);
}
/**
 * Constructs a button with an icon and a label.
 *
 * @param label 	the label string to be displayed.
 * @param icon		the <code>ULCicon</code> that should be displayed along with the label.
 */
public ULCButton(String label, ULCIcon icon) {
	super(label);
	fIcon = icon;
}
/**
 * Constructs a button with an icon and a label and registers the given ActionListener.
 *
 * @param label 	The label string to be displayed.
 * @param icon 		The <code>ULCIcon</code> to be displayed alongside the label.
 * @param listener	The widget interested in my actionEvents.
 */
public ULCButton(String label, ULCIcon icon, IActionListener listener) {
	this(label, listener);
	fIcon = icon;
}
/**
 * Registers the given listener to begin receiving notifications
 * when the Button was pressed.
 *
 * @param listener	The object interested in my actionEvents.
 */
public void addActionListener(IActionListener listener) {
	internalAddListener("action", listener);
}
/**
 * Simulate a button click action. distribute the event to all listeners.
 */
public void click() {
	distributeToListeners("action", new ULCActionEvent(this, getLabel()));
}
/**
 * Gets the button's icon.
 *
 * @return The ULCIcon or null
 */
public ULCIcon getIcon() {
	return fIcon;
}
/**
 * Gets the button's label.
 *
 * @return The label string  
 */
public String getLabel() {
	return fLabel;
}
/**
 * Gets the button's mnemonic.
 *
 * @return The keycode or 0
 */
public int getMnemonic() {
	return fMnemonic;
}
/**
 * Returns the configured ULCTrigger object which will be triggered in the UI when 
 * this widget's actionPerformed method is called.
 *
 * @return ITrigger	 
 */
public ITrigger getTrigger() {
	return fTrigger;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		String type = args.get("type", "???");
		if (type.equals("action")) {
			if (isDisableOnAction())
				fEnabled = false;
			distributeToListeners("action", new ULCActionEvent(this, args.get("cmd", "???")));
		}
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * If the disable on action is set to true then the button is disabled on 
 * user click. The application is responsible for enabling the button either 
 * by explicitly calling setEnable or via an enabler.
 *
 * @return disableOnAction if true button gets disabled each time it is clicked.
 */
public boolean isDisableOnAction() {
	return fDisableOnAction;
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive action events.
 *
 * @param listener	The widget that was registered to receive my actionEvents.
 */
public void removeActionListener(IActionListener listener) {
	internalRemoveListener("action",listener);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fIcon != null)
		a.put("icon", fIcon.getRef(fContext));
	if (fMnemonic != -1)
		a.put("mnemonic", fMnemonic);
	if (fDisableOnAction)
		a.put("disableOnAction", new Anything(fDisableOnAction));
	if (fTrigger != null)
		a.put("trigger", ((ULCProxy) fTrigger).getRef(fContext));
}
/**
 * If the disable on action is set to true then the button is disabled on 
 * user click. The application is responsible for enabling the button either 
 * by explicitly calling setEnable or via an enabler.
 *
 * @param disableOnAction if true button gets disabled each time it is clicked.
 */
public void setDisableOnAction(boolean disableOnAction) {
	if (fDisableOnAction != disableOnAction) {
		fDisableOnAction = disableOnAction;
		sendUI("setDisableOnAction", new Anything(fDisableOnAction));
	}
}
/**
 * Set the icon of the button.
 *
 * @param icon The ULCIcon to be displayed.
 */
public void setIcon(ULCIcon icon) {
	if (icon != fIcon) {
		fIcon = icon;
		sendUI("setIcon", fIcon);
	}
}
/**
 * Sets the widgets's <code>String</code> label.
 * Note: If the label contains the <code>IDefaults.MNEMONIC_INDICATOR</code> then
 * the character after this indicator is treated as the mnemonic character for this widget.
 *
 * @param label the label <code>String</code> of the widget.
 */
public void setLabel(String label) {
	internalSetLabel(label);
}
/**
 * Sets this widgets mnemonic code.
 *
 * @param keyCode The keyCode that represents the character to be used as a mnemonic
 */
public void setMnemonic(int keyCode) {
	if (keyCode != fMnemonic) {
		fMnemonic = keyCode;
		sendUI("setMnemonic", new Anything(fMnemonic));
	}
}
/**
 * Sets the configured ULCTrigger which will be triggered in the UI when 
 * this widgets actionPerformed method is called
 *
 * @param ULCTrigger	 
 */
public void setTrigger(ITrigger trigger) {
	if (trigger != fTrigger) {
		fTrigger = trigger;
		sendUI("setTrigger", (ULCTrigger)fTrigger);
	}
}
}
