package com.ibm.ulc.application;

/*
 * Copyright (c) 1999 Object Technology International Inc.
 */
import com.ibm.ulc.util.UlcObject;
import java.util.Date;
import java.util.Locale;
/**
 * This comparator handles comparison of standard native types like Date, Float, Integer and String.
 * All other object types are compared by using their string representation.
 *
 * Note :  It is assumed that the items provided for comparision to this class are homogeneous. For
 * example, comparing object1 and object2, these two are assumed to be of the same (base) type.
 *
 */
public class ULCDefaultComparator extends UlcObject implements IComparator {
	protected boolean fIgnoreStringCase = false;
	protected Locale fLocale = null;
/**
 * Return a default compartor
 *
 */
public ULCDefaultComparator() {
	this(Locale.getDefault(), false);
}
/**
 * Return a default compartor that ignores the case of strings if ignoreStringCase 
 * is true based on the rules for the locale specified.
 * 
 *
 * @param ignoreStringCase	If true all strings are compared as uppercase.
 * @param locale			The locale to be used for comparing strings
 */
public ULCDefaultComparator(Locale locale, boolean ignoreStringCase) {
	fIgnoreStringCase = ignoreStringCase;
	fLocale = locale;
}
/**
 * Return a default compartor that ignores the case of strings if ignoreStringCase is true
 *
 * @param ignoreStringCase	if true all strings are compared as uppercase.
 */
public ULCDefaultComparator(boolean ignoreStringCase) {
	this(Locale.getDefault(), ignoreStringCase);
}
/**
 * Compares the object at left to the Object right answer 0 if 
 * they are equal -1 if its less and 1 if its greater.
 *
 * Handle common native types Date, Integer, Float, String.
 * All unknown types will be compared using string comparison.
 * Assumes both objects to be of the same base type.
 *
 * @return  the value <code>0</code> if the objects are equivalent
 *			a value less than <code>0</code> if its less; and a
 *          value greater than <code>0</code> if its greater.
 */
public int compare(Object left, Object right) {
	if ((left == null) && (right == null))
		return 0;
	if (left == null)
		return -1;
	if (right == null)
		return 1;
	if (left instanceof Integer) {
		return ((Integer) left).intValue() - ((Integer) right).intValue();
	}
	if (left instanceof Float) {
		return (new Float(((Float) left).floatValue() - ((Float) right).floatValue())).intValue();
	}
	if (left instanceof Date) {
		Date l = (Date) left;
		Date r = (Date) right;
		if (l.equals(r))
			return 0;
		if (l.before(r))
			return -1;
		else
			return 1;
	}
	if (fIgnoreStringCase)
		return left.toString().toUpperCase(fLocale).compareTo(right.toString().toUpperCase(fLocale));
	else
		return (left.toString().compareTo(right.toString()));
}
}
