package com.ibm.ulc.application;

/*
 * Copyright (c) 1999 Object Technology International Inc.
 */

import com.ibm.ulc.application.*;
import com.ibm.ulc.comm.*;
import com.ibm.ulc.util.*;
import java.io.*;
/**
 * Instances of this class represent the name of a file or directory 
 * on the client file system (UIEngine). A file is specified by a pathname, 
 * which can either be an absolute pathname or a pathname relative to the 
 * current working directory. The pathname must follow the naming 
 * conventions of the client platform. 
 */

public class ULCFile extends ULCProxy {
	private Anything fReply = null;

	/**
	 * The system-dependent path separator string on the client system.
	 * 
	 */
	public String fSeparator = null;

	/**
	 * The system-dependent path separator character on the client system.
	 * This character separates the directory and file components in a filename. 
	 *
	 */
	public char fSeparatorChar;

	/**
	 * The system-dependent path separator string on the client system.
	 *
	 */
	public String fPathSeparator = null;

	/**
	 * The system-dependent path separator character on the client system. 
	 * This character is often used to separate filenames in a sequence of 
	 * files given as a "path list".
	 *
	 */
	public char fPathSeparatorChar;

	/**
	 * The system-dependent line separator string on the client system.
	 * This is the value of the line.separator property on the client system.
	 */
	public String fLineSeparator;
public ULCFile() {
}
/**
 * Creates a <code>ULCFile</code> instance that represents the file 
 * on the client system whose pathname is the given path argument. 
 *
 * @param      context the ULCContext upon which the file should 
 *             be created.
 * @param      path   the file pathname.
 * @param      path   the file pathname.
 * @exception  NullPointerException  if the file path is equal to
 *               <code>null</code>.
 * @see        ULCFile#getPath()
 */
public ULCFile(ULCContext context, String path) {
	this(context, null, path);
}
/**
 * Creates a <code>ULCFile</code> instance that represents the file
 * on the client system whose pathname is the pathname of the specified
 * directory, followed by the separator character, followed by the
 * <code>name</code> argument. 
 *
 * @param      context the ULCContext upon which the file stream should 
 *             be created.
 * @param   path   the directory pathname.
 * @param   name   the file pathname.
 * @see     #getPath
 * @see     #fSeparator
 */
public ULCFile(ULCContext context, String path, String name) {
	fContext = context;
	Anything args = new Anything();
	if (path != null)
		args.put("path", path);
	if (name != null)
		args.put("name", name);
	Anything res = syncCall("construct", args);
	fSeparator = res.get("separator", "\\");
	fSeparatorChar = (char) res.get("separatorChar", 0);
	fPathSeparator = res.get("pathSeparator", ";");
	fPathSeparatorChar = (char) res.get("pathSeparatorChar", 0);
	fLineSeparator = res.get("lineSeparator", "\n");
}
/**
 * Tests if the application can read from the specified file. 
 *
 */
public boolean canRead() {
	return syncCall("canRead", null).asBoolean(false);
}
/**
 * Tests if the application can write to this file. 
 *
 */
public boolean canWrite() {
	return syncCall("canWrite", null).asBoolean(false);
}
/**
 * Deletes the file specified by this object.  If the target
 * file to be deleted is a directory, it must be empty for deletion
 * to succeed.
 *
 * @return     <code>true</code> if the file is successfully deleted;
 *             <code>false</code> otherwise.
 *
 */
public boolean delete() {
	return syncCall("delete", null).asBoolean(false);
}
/**
 * Tests if this <code>ULCFile</code> exists. 
 *
 */
public boolean exists() {
	return syncCall("exists", null).asBoolean(false);
}
public void finalize() {
}
/**
 * Returns the absolute pathname of the file represented by this object.
 * If this object represents an absolute pathname, then return the 
 * pathname. Otherwise, return a pathname that is a concatenation of 
 * the current client directory, the separator character, and the 
 * pathname of this file object. 
 *
 * @see     #getPath()
 * @see     #isAbsolute()
 */
public String getAbsolutePath() {
	return syncCall("getAbsolutePath", null).asString("");
}
/**
 * Returns the canonical form of this <code>ULCFile</code> object's pathname.
 * The precise definition of canonical form is system-dependent, but it
 * usually specifies an absolute pathname in which all relative references
 * and references to the current user directory have been completely
 * resolved.  The canonical form of a pathname of a nonexistent file may
 * not be defined.
 *
 * @exception IOException If an I/O error occurs, which is possible because
 * the construction of the canonical path may require filesystem queries.
 *
 */
public String getCanonicalPath() throws IOException {
	Anything res = syncCall("getCanonicalPath", null);
	if (res.isDefined("exception")) {
		throw (IOException) res.get("exception", (Serializable) null);
	}
	return res.asString("");
}
/**
 * Returns the name of the file represented by this object. The name 
 * is everything in the pathame after the last occurrence of the 
 * separator character. 
 *
 * @see     #getPath()
 */
public String getName() {
	return syncCall("getName", null).asString("");
}
/**
 * Returns the parent part of the pathname of this <code>ULCFile</code>
 * object, or <code>null</code> if the name has no parent part. The parent
 * part is generally everything leading up to the last occurrence of the
 * separator character, although the precise definition is system
 * dependent.  On UNIX, for example, the parent part of
 * <code>"/usr/lib"</code> is <code>"/usr"</code>, whose parent part is
 * <code>"/"</code>, which in turn has no parent.  On Windows platforms,
 * the parent part of <code>"c:\java"</code> is <code>"c:\"</code>, which
 * in turn has no parent.
 *
 * @see    ULCFile#getPath()
 * @see    ULCFile#getCanonicalPath()
 * @see    ULCFile#fLineSeparator
 */
public String getParent() {
	return syncCall("getParent", null).asString("");
}
/**
 * Returns the pathname of the file represented by this object.
 *
 */
public String getPath() {
	return syncCall("getPath", null).asString("");
}
/**
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("reply")) {
		fReply = args;
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Tests if the file represented by this <code>ULCFile</code> object is an
 * absolute pathname. The definition of an absolute pathname is system 
 * dependent. For example, on UNIX, a pathname is absolute if its 
 * first character is the separator character. On Windows platforms, 
 * a pathname is absolute if its first character is an ASCII '&#92;' or 
 * '/', or if it begins with a letter followed by a colon. 
 *
 */
public boolean isAbsolute() {
	return syncCall("isAbsolute", null).asBoolean(false);
}
/**
 * Tests if the file represented by this <code>ULCFile</code> 
 * object is a directory. 
 *
 */
public boolean isDirectory() {
	return syncCall("isDirectory", null).asBoolean(false);
}
/**
 * Tests if the file represented by this <code>ULCFile</code> 
 * object is a "normal" file. 
 * <p>
 * A file is "normal" if it is not a directory and, in 
 * addition, satisfies other system-dependent criteria. Any 
 * non-directory file created by a Java application is guaranteed to 
 * be a normal file. 
 *
 */
public boolean isFile() {
	return syncCall("isFile", null).asBoolean(false);
}
/**
 * Returns the time that the file represented by this 
 * <code>ULCFile</code> object was last modified. 
 * <p>
 * The return value is system dependent and should only be used to 
 * compare with other values returned by last modified. It should not 
 * be interpreted as an absolute time. 
 *
 */
public long lastModified() {
	return syncCall("lastModified", null).asLong(-1L);
}
/**
 * Returns the length in bytes of the file represented by this 
 * <code>ULCFile</code> object or <code>0L</code> if the specified file
 * does not exist.
 *
 */
public long length() {
	return syncCall("length", null).asLong(-1L);
}
/**
 * Returns a list of the files in the directory specified by this
 * <code>ULCFile</code> object. This list does not include the current 
 * directory or the parent directory ("<code>.</code>" and "<code>..</code>"
 * on Unix systems). 
 *
 */
public String[] list() {
	return (String[]) syncCall("list", null).asSerializable(null);
}
/**
 * Returns a list of the files in the directory specified by this 
 * <code>ULCFile</code> object, which passes the filter specified by the
 * <code>filter</code> parameter. This list does not include
 * the current directory or the parent directory ("<code>.</code>"
 * and "<code>..</code>" on Unix systems).
 *
 * @param      an array of extensions, if any array element contains
 *             <code>null</code>, directories are also included in the
 *             resulting list.
 *
 * @see     com.ibm.ulc.util.UlcFileFilter
 */
public String[] list(UlcFileFilter filter) {
	Anything args = new Anything();
	args.put("arg1", filter);
	return (String[]) syncCall("list_with_filter", args).asSerializable(null);
}
/**
 * Creates a directory whose pathname is specified by this 
 * <code>ULCFile</code> object. 
 *
 * @return     <code>true</code> if the directory could be created;
 *             <code>false</code> otherwise.
 *
 */
public boolean mkdir() {
	return syncCall("mkdir", null).asBoolean(false);
}
/**
 * Creates a directory whose pathname is specified by this 
 * <code>ULCFile</code> object, including any necessary parent directories.
 *
 * @return     <code>true</code> if the directory (or directories) could be
 *             created; <code>false</code> otherwise.
 *
 */
public boolean mkdirs() {
	return syncCall("mkdirs", null).asBoolean(false);
}
/**
 * Renames the file specified by this <code>ULCFile</code> object to 
 * have the pathname given by the <code>dest</code> argument. 
 *
 * @param      dest  the new filename.
 * @return     <code>true</code> if the renaming succeeds;
 *             <code>false</code> otherwise.
 *
 */
public boolean renameTo(String filename) {
	Anything args = new Anything();
	args.put("arg1", filename);
	return syncCall("renameTo", args).asBoolean(false);
}
synchronized Anything syncCall(String request, Anything args) {
	upload(fContext);
	fReply = null;
	sendUI(request, args);
	while ((fReply == null) && !fContext.processNextRequest(0)) {
		try {
			Thread.sleep(100);
		} catch (Exception e) {
		}
	};
	return fReply;
}
/**
 * returns a logical name for this component.
 * This name is used to locate the other half of this object in the UI Engine.
 * The default implementation extracts the name from the class name by stripping
 * off the package name and an optional prefix "ULC".
 * widgets that are not found in the com.ibm.ulc.ui.swing package should
 * override this method to return the fully qualified class name of the UI class
 * eg: com.ibm.ulc.ui.UIApplication
 *
 * @return The Logical name for this component.
 */
protected String typeString() {
	return "com.ibm.ulc.ui.io.UIFile";
}
}
