package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import com.ibm.ulc.util.Anything;
import java.util.Vector;
/**
 * Common base class for all ULC widgets which bidirectionally tracks an attribute of a FormModel.
 * This class maintains the reference to the FormModel and the name of the tracked attribute.
 * In addition it defines protocol and constants for setting the update policy.
 *
 * @see ULCFormModel
 */
public class ULCFormComponent extends ULCComponent implements IModelOwner {

	/**
	 * The data source that implements the <code>IForm</code> interface.
	 * All requests/changes that this component makes will be via the formModel
	 * @serial
	 */
	protected IForm fFormModel = null;
	/**
	 * The key to be used to retrieve/set changes to the data in the formModel
	 * @serial	 
	 */
	protected String fKey = null;
	/**
	 * The currently configured notification Policy
	 * @serial	 
	 */
	protected int fNotificationPolicy = FORM_NOTIFICATION_ON_FOCUS_CHANGE;
/**
 * Default Constructor for a ULCFormComponent
 */
public ULCFormComponent() {
}
/**
 * Construct a ULCFormComponent that is connected to model.
 *
 * @param formModel 			The <code>IForm</code> data source
 * @param formAttributeName		The attribute key within the formModel
 */
public ULCFormComponent(IForm formModel, String formAttributeName) {
	fKey = formAttributeName;
	fFormModel = formModel;
	if (fFormModel != null)
		fFormModel.addOwner(this);
}
/**
 * Construct a ULCFormComponent that with the specified label.
 *
 * @param label The label <code>String</code> to be displayed.
 */
public ULCFormComponent(String label) {
	super(label);
}
/**
 * Construct a ULCFormComponent that with the specified label connected to model.
 *
 * @param label 			The label <code>String</code> to be displayed.
 * @param formModel 		The <code>IForm</code> data source
 * @param formAttributeName	The attribute key within the formModel
 */
public ULCFormComponent(String label, IForm formModel, String formAttributeName) {
	super(label);
	fKey = formAttributeName;
	fFormModel = formModel;
	if (fFormModel != null)
		fFormModel.addOwner(this);
}
/**
 * Add all attributes that should be preloaded with a form model to the UI to the specified Vector.
 *
 * @return Vector
 */
public void addPreloadFormAttributesInto(Vector vectorOfPreloadAttributes) {
	if (getFormAttributeName() == null)
		return;
	if (!vectorOfPreloadAttributes.contains(getFormAttributeName()))
		vectorOfPreloadAttributes.addElement(getFormAttributeName());
}
/**
 * Add all table attributes that should be preloaded to the UI to the specified Vector.
 * This receiver has no table attributes to preload.
 *
 * @param vectorOfPreloadAttributes	Vector	into which the receiver adds the
 *											table attributes it needs to preload.
 */
public void addPreloadTableAttributesInto(Vector vectorOfPreloadAttributes) {
}
/**
 * Gets the <code>String</code> attribute key within the formModel.
 *
 * @return The <code>String</code> attribute key within my <code>IForm</code>
 */
public String getFormAttributeName() {
	return fKey;
}
/**
 * Gets the <code>ULCFormModel</code> that is currently my data source.
 *
 * @return The <code>IForm</code> data source
 */
public IForm getFormModel() {
	return fFormModel;
}
/**
 * Return the notification policy of this component.
 *
 * @param policy The new notification policy to be set.
 * <pre>
 * The Notification policy can be one of:
 *
 * 	FORM_NOTIFICATION_IMMEDIATE
 *		any changes in the form field are immediately
 *		reported back to the application.
 *
 *	FORM_NOTIFICATION_ON_FOCUS_CHANGE 
 *		changes in the form field are reported back to the
 *		application on focus changes.
 *
 *	FORM_NOTIFICATION_ON_REQUEST	
 *		Changes will be reported back only when the user explicitly calls
 *		<code>saveInput</code> on the FormModel.
 *
 * </pre>
 */
public int getNotificationPolicy() {
	return fNotificationPolicy;
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	if (fKey != null)
		a.put("key", fKey);
	if (fNotificationPolicy != FORM_NOTIFICATION_ON_FOCUS_CHANGE)
		a.put("notificationPolicy", fNotificationPolicy);
	super.saveState(a);
	if (fFormModel != null)
		a.put("formModel", fFormModel.getRef(fContext));
}
/**
 * Sets the <code>String</code> attribute key within the formModel.
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @param The <code>String</code> attribute key within my <code>IForm</code>
 */
public void setFormAttributeName(String formAttributeName) {
	fKey = formAttributeName;
}
/**
 * Set the <code>ULCFormModel</code> that is currently my data source.
 *
 * @param model The <code>IForm</code> data source
 */
public void setFormModel(IForm model) {
	if (fFormModel != null)
		fFormModel.removeOwner(this);
	fFormModel = model;
	if (fFormModel != null)
		fFormModel.addOwner(this);
	sendUI("setFormModel", model);
}
/**
 * Set the notification policy of this component.
 *
 * @param policy The new notification policy to be set.
 * <pre>
 * The Notification policy can be one of:
 *
 * 	FORM_NOTIFICATION_IMMEDIATE
 *		any changes in the form field are immediately
 *		reported back to the application.
 *
 *	FORM_NOTIFICATION_ON_FOCUS_CHANGE 
 *		changes in the form field are reported back to the
 *		application on focus changes.
 *
 *	FORM_NOTIFICATION_ON_REQUEST	
 *		Changes will be reported back only when the user explicitly calls
 *		<code>saveInput</code> on the FormModel.
 *
 * </pre>
 */
public void setNotificationPolicy(int policy) {
	if (fNotificationPolicy != policy) {
		fNotificationPolicy = policy;
		sendUI("setNotificationPolicy", new Anything(fNotificationPolicy));
	}
}
}
