package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import com.ibm.ulc.base.*;
import com.ibm.ulc.comm.*;
import com.ibm.ulc.util.*;
import java.net.*;
/**
 * A ULCHtmlPane implements a Html Display widget.
 * The widget displays the Html page identified by the supplied URL.
 * If <code>veto</code> is enabled then all hyperlinks within the displayed
 * page are sent back to the ULC application for confirmation. The selected 
 * hyperlink will be displayed only if the ULC application then sends the 
 * <code>setPage</code> request.
 */
public class ULCHtmlPane extends ULCComponent {
	/**
	 * @serial	 
	 */
	private boolean fEditable = false;
	/**
	 * @serial	 
	 */
	private boolean fVeto = false;
	/**
	 * @serial	 
	 */
	private URL fUrl = null;
/**
 * Construct a ULCHtmlPane .
 *
 */
public ULCHtmlPane() {
	super("HtmlPane");
}
/**
 * Construct a ULCHtmlPane with the specified label.
 *
 * @param label The <code>String</code> to be used as the label.
 */
public ULCHtmlPane(String label) {
	super(label);
}
/**
 * Construct a ULCHtmlPane with the <code>editable</code> and <code>veto</code> flags specified.
 *
 * @param editable	If true, allows the displayed page to be edited.
 * @param veto 		If true, all hyperlinks selected within the page are
 *					sent to the ULC application for confirmation.
 */
public ULCHtmlPane(boolean editable, boolean veto) {
	fEditable = editable;
	fVeto = veto;
}
/**
 * Registers the given listener to begin receiving notifications
 * when a link has been activated in myself.
 *
 * @param listener	The object interested in my link activation events.
 */
public void addLinkActivatedListener(IHyperLinkListener listener) {
	enableOptionalEvent("la");
	internalAddListener("la", listener);
}
/**
 * Registers the given listener to begin receiving notifications
 * when a link error occurs in myself.
 *
 * @param listener	The object interested in my link error events.
 */
public void addLinkErrorListener(IHyperLinkListener listener) {
	internalAddListener("le", listener);
}
/**
 * Returns the currently displayed url as a string
 *
 */
public String getPage() {
	URL u = getUrl();
	if (u != null)
		return u.toString();
	else
		return null;
}
/**
 * Gets the currently displayed page as a url
 * Note if an exception occurs after the URL has been sent to the UI then the
 * HtmlPane itself will display the error.
 *
 */
public URL getUrl() {
	return fUrl;
}
/**
 * Returns the veto flag which results in every hyperlink
 * selection to be passed to the application for confirmation.
 * The application must call <code>setPage</code> to activate the page
 * if <code>veto</code> is true
 *
 * @return veto	If true, then all hyperlinks selected will be sent to the ULC
 *				application for confirmation.
 */
public boolean getVeto() {
	return fVeto;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		String type = args.get("type", "???");
		if (type.equals("la")) {
			String url = args.get("url", null);
			distributeToListeners("la", new ULCLinkActivatedEvent(this, url));
			return;
		}
		if (type.equals("setPageError")) {
			String url = args.get("url", null);
			String error = args.get("error", null);
			distributeToListeners("le", new ULCLinkErrorEvent(this, url, error));
			return;
		}
	}
	super.handleRequest(conn, request, args);
}
/**
 * Deregisters the given listener from receiving  notifications
 * when a link has been activated in myself.
 *
 * @param listener	The object interested in my link activation events.
 */
public void removeLinkActivatedListener(IHyperLinkListener listener) {
	disableOptionalEvent("la");
	internalRemoveListener("la", listener);
}
/**
 * Deregisters the given listener from receiving notifications
 * when a link error occurs in myself.
 *
 * @param listener	The object interested in my link error events.
 */
public void removeLinkErrorListener(IHyperLinkListener listener) {
	disableOptionalEvent("le");
	internalRemoveListener("le", listener);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fEditable)
		a.put("editable", fEditable);
	if (fVeto)
		a.put("veto", fVeto);
	if (fUrl != null)
		a.put("url", fUrl);
}
/**
 * Sets the currently displayed page to be urlString
 * If an exception occurs when creating a URL from the urlString
 * write an error to the console.
 * Note if an exception occurs after the URL has been sent to the UI then the
 * HtmlPane itself will display the error.
 *
 * @param urlString The URL specified as a <code>String</code>
 */
public void setPage(String urlString) {
	if (urlString != null) {
		try {
			URL u = new URL(urlString);
			setUrl(u);
		}
		catch (Exception e) {
			System.out.println("ULCHtmlPane.setPage: Exception converting to URL " + urlString.toString());
		}
	}
}
/**
 * Sets the currently displayed page to be the contents of htmlText.
 * <code>htmlText</code> must contain valid HTML text.
 *
 * @param htmlText The string containing HTML text.
 */
public void setPageContents(String htmlText) {
	sendUI("setHtmlPageContents", new Anything(htmlText));
}
/**
 * Sets the currently displayed page to be url
 * Note if an exception occurs after the URL has been sent to the UI then the
 * HtmlPane itself will display the error.
 *
 * @param url The URL to be displayed in the HtmlPane.
 */
public void setUrl(URL url) {
	if (fUrl != url) {
		fUrl = url;
		sendUI("setPage", new Anything(url.toString()));
	}
}
/**
 * Sets the veto flag which results in every hyperlink
 * selection to be passed to the application for confirmation.
 * The application must call <code>setPage</code> to activate the page
 * if <code>veto</code> is true
 *
 * @param veto	If true, then all hyperlinks selected will be sent to the ULC
 *				application for confirmation.
 */
public void setVeto(boolean veto) {
	if (fVeto != veto) {
		fVeto = veto;
		sendUI("setVeto", new Anything(fVeto));
	}
}
}
